/***************************************************************************
                          cdownloadmanager.cpp  -  description
                             -------------------
    begin                : Don Mai 16 2002
    copyright            : (C) 2002 by Mathias Kster
    email                : mathen@ketelhot.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>

#include <dclib/dcobject.h>
#include <dclib/cconfig.h>
#include <dclib/cdir.h>
#include <dclib/cservermanager.h>

#ifndef WIN32
#include <unistd.h>
#endif

#include "cdownloadmanager.h"

extern CConfig * pDCLibConfig;
extern CServerManager * pDCLibServerManager;

CDownloadManager * pDownloadManager = 0;

CDownloadManagerInfo::CDownloadManagerInfo()
{
	eType    = DC_MESSAGE_DM_INFO;
	rate_ul  = 0;
	rate_dl  = 0;
	slot_max = 0;
	slot_use = 0;
}

/** */
CDownloadManager::CDownloadManager()
{
	lTransferID    = time(0);
	lUsedSlots     = 0;
	iShutdownState = 0;

	pDownloadManager = this;

	pTransferList      = new CList<CTransfer>;
	pExtraUserSlotList = new CList<CExtraUserSlot>;
	pTransferWaitList  = new CThreadList<DCTransferWait>;
}

/** */
CDownloadManager::~CDownloadManager()
{
	StopListen();

	Stop();

	if (pExtraUserSlotList)
	{
		delete pExtraUserSlotList;
		pExtraUserSlotList = 0;
	}

	if(pTransferList)
	{
		delete pTransferList;
		pTransferList = 0;
	}

	if(pTransferWaitList)
	{
		delete pTransferWaitList;
		pTransferWaitList = 0;
	}
}

/** */
void CDownloadManager::SendFileInfo( DCTransferObject * TransferObject, DCTransferFileObject * TransferFileObject, bool bRemoveFile )
{
	LogThread.Lock();

	CMessageDMFileObject * fo = new CMessageDMFileObject();

	fo->sNick       = TransferObject->sNick;
	fo->sHubName    = TransferObject->sHubName;
	fo->sHubHost    = TransferObject->sHubHost;
	fo->eWaitState  = TransferObject->eState;
	fo->tTimeout    = TransferObject->tTimeout;
	fo->bRemoveFile = bRemoveFile;
	fo->iConnections = TransferObject->iConnections;

	if ( TransferFileObject )
	{
		fo->sRemoteFile      = TransferFileObject->sRemoteFile;
		fo->sRemoteFileName  = TransferFileObject->sRemoteFileName;
		fo->sLocalName       = TransferFileObject->sLocalName;
		fo->sLocalPath       = TransferFileObject->sLocalPath;
		fo->lSize            = TransferFileObject->lSize;
		fo->lStartPosition   = TransferFileObject->lStartPosition;
		fo->lEndPosition     = TransferFileObject->lEndPosition;
		fo->lCurrentPosition = TransferFileObject->lCurrentPosition;
		fo->eFileState       = TransferFileObject->eState;
	}

	fo->eType = DC_MESSAGE_FILE_OBJECT;

	if ( DC_DownloadManagerCallBack(fo) == -1 )
		delete fo;

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendTransferInfo( CTransfer * Transfer, bool remove )
{
	LogThread.Lock();

	CMessageDMTransferObject * to = new CMessageDMTransferObject();

	to->sNick            = Transfer->GetDstNick();
	to->sHost            = Transfer->GetHost();
	to->eState           = Transfer->GetMode();
	to->sRemoteFile      = Transfer->GetDstFilename();
	to->sRemoteFileName  = "";
	to->lSize            = Transfer->GetLength();
	to->lStartPosition   = Transfer->GetStartPosition();
	to->lCurrentPosition = Transfer->GetCurrentPosition();
	to->lEndPosition     = Transfer->GetEndPosition();
	to->lRate            = Transfer->GetTransferrate();
	to->bRemoveTransfer  = remove;

	to->eType = DC_MESSAGE_TRANSFER_OBJECT;

	if ( DC_DownloadManagerCallBack(to) == -1 )
		delete to;

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendSlotInfo( CExtraUserSlot * Object )
{
	LogThread.Lock();

	CMessageDMSlotObject * so = new CMessageDMSlotObject();

	so->sNick      = Object->sNick;
	so->sHubName   = Object->sHubName;
	so->iSlots     = Object->iSlots;
	so->bPermanent = Object->bPermanent;

	so->eType = DC_MESSAGE_SLOT_OBJECT;

	if ( DC_DownloadManagerCallBack(so) == -1 )
		delete so;

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendLogInfo( CString message, CTransfer * Transfer )
{
	LogThread.Lock();

	CString s = "";

	CMessageLog * log = new CMessageLog();

	if ( Transfer != 0 )
	{
		s = "[" + Transfer->GetDstNick() + "] ";
	}
	
	s += message;
	
	log->sMessage = s;
	log->eType    = DC_MESSAGE_LOG;

	if ( DC_DownloadManagerCallBack(log) == -1 )
		delete log;

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendDownloadManagerInfo( CDownloadManagerInfo * dminfo )
{
	LogThread.Lock();

	bool b = FALSE;

	if ( dminfo->rate_dl != DownloadManagerInfo.rate_dl )
		b = TRUE;
	else if ( dminfo->rate_ul != DownloadManagerInfo.rate_ul )
		b = TRUE;
	else if ( dminfo->slot_max != DownloadManagerInfo.slot_max )
		b = TRUE;
	else if ( dminfo->slot_use != DownloadManagerInfo.slot_use )
		b = TRUE;

	if ( b == TRUE )
	{
		CDownloadManagerInfo * dmi = new CDownloadManagerInfo;

		dmi->rate_dl  = DownloadManagerInfo.rate_dl  = dminfo->rate_dl;
		dmi->rate_ul  = DownloadManagerInfo.rate_ul  = dminfo->rate_ul;
		dmi->slot_max = DownloadManagerInfo.slot_max = dminfo->slot_max;
		dmi->slot_use = DownloadManagerInfo.slot_use = dminfo->slot_use;

		if ( DC_DownloadManagerCallBack(dmi) == -1 )
			delete dmi;
	}

	LogThread.UnLock();
}

/** */
void CDownloadManager::FileListDone( CTransfer * Transfer )
{
	DCTransferObject * TransferObject;
	CByteArray ba;

	Lock();

	if ( Transfer->GetBuffer(&ba) != 0 )
	{
		if ( (TransferObject = pDownloadQueue.GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName() )) != 0 )
		{
			TransferObject->sUserFileList.Set((char*)ba.Data(),ba.Size());

			CMessageDMFileListObject * flo = new CMessageDMFileListObject();

			flo->sNick         = TransferObject->sNick;
			flo->sHubName      = TransferObject->sHubName;
			flo->sHubHost      = TransferObject->sHubHost;
			flo->sUserFileList = TransferObject->sUserFileList;

			flo->eType = DC_MESSAGE_FILELIST_OBJECT;

			if ( DC_DownloadManagerCallBack(flo) == -1 )
				delete flo;
		}
	}

	UnLock();
}

/** */
int CDownloadManager::LoadQueue()
{
	int err;
	CStringList * StringList = 0;

	Lock();

	pDownloadQueue.pQueue->Clear();

	err = pDCLibConfig->LoadDCTra(pDownloadQueue.pQueue);

	// send all files over the callback function
	if ( err == 0 )
	{
		while( pDownloadQueue.pQueue->Next( (CObject *&) StringList) )
		{
			DCTransferObject * TransferObject = 0;

			while( StringList->Next( (CObject *&) TransferObject) )
			{
				DCTransferFileObject * TransferFileObject = 0;

				while( TransferObject->pTransferFileList.Next( (CObject *&)TransferFileObject) )
				{
					SendFileInfo( TransferObject, TransferFileObject );
				}
			}
		}
	}

	UnLock();

	Start();

	return err;
}

/** */
int CDownloadManager::SaveQueue()
{
	int err;

	Lock();

	err = pDCLibConfig->SaveDCTra(pDownloadQueue.pQueue);

	UnLock();

	return err;
}

/** */
bool CDownloadManager::InitListen( CString & s )
{
	bool res = FALSE;

	if ( Listen.StartListen( pDCLibConfig->GetTCPListenPort() ) == 0 )
	{
		Listen.SetCallBackFunction(DMListenCallBack);

		res = TRUE;
	}
	else
	{
		s = Listen.GetSocketError();
	}

	return res;
}

/** */
void CDownloadManager::StopListen()
{
	Listen.SetCallBackFunction(0);
	Listen.StopListen();
}

/** */
void CDownloadManager::AddUserSlot( CString nick, CString hubname, int slot, bool permanent )
{
	CExtraUserSlot * ExtraUserSlot;
	Lock();

	ExtraUserSlot = 0;

	while( (ExtraUserSlot=pExtraUserSlotList->Next(ExtraUserSlot)) != 0 )
	{
		if ( (ExtraUserSlot->sNick == nick) && (ExtraUserSlot->sHubName == hubname) )
		{
			if ( (slot == 0) && (!permanent) )
			{
				ExtraUserSlot->iSlots     = slot;
				ExtraUserSlot->bPermanent = FALSE;
			}
			else
			{
				if (ExtraUserSlot->bPermanent)
					ExtraUserSlot->iSlots = 0;
				ExtraUserSlot->iSlots     += slot;
				ExtraUserSlot->bPermanent  = permanent;
			}
			break;
		}
	}

	if ( ExtraUserSlot == 0 )
	{
		ExtraUserSlot = new CExtraUserSlot();
		ExtraUserSlot->sNick      = nick;
		ExtraUserSlot->sHubName   = hubname;
		ExtraUserSlot->iSlots     = slot;
		ExtraUserSlot->bPermanent = permanent;

		pExtraUserSlotList->Add(ExtraUserSlot);
	}

	SendSlotInfo(ExtraUserSlot);

	if ( (ExtraUserSlot->iSlots == 0) && (!ExtraUserSlot->bPermanent) )
	{
		pExtraUserSlotList->Del(ExtraUserSlot);
	}

	UnLock();
}

/** */
bool CDownloadManager::CheckUserSlot( CString nick, CString )
{
	bool res = FALSE;
	CExtraUserSlot * ExtraUserSlot = 0;

	Lock();

	while( (ExtraUserSlot=pExtraUserSlotList->Next(ExtraUserSlot)) != 0 )
	{
		if ( ExtraUserSlot->sNick == nick )
		{
			if ( ExtraUserSlot->iSlots > 0 )
			{
				ExtraUserSlot->iSlots--;

				SendSlotInfo(ExtraUserSlot);

				if ( ExtraUserSlot->iSlots == 0 )
				{
					pExtraUserSlotList->Del(ExtraUserSlot);
				}

				res = TRUE;
			}
			else if ( ExtraUserSlot->bPermanent )
			{
				res = TRUE;
			}

			break;
		}
	}

	UnLock();

	return res;
}

/** disconnect all transfers for a clean shutdown */
void CDownloadManager::Shutdown()
{
	CTransfer * Transfer;

	Listen.StopListen();
	Listen.SetCallBackFunction(0);

	iShutdownState = 1;

	Transfer = 0;

	SendLogInfo("Shutdown download manager ...\n");

	Lock();

	printf("COUNT INCOMING: %ld\n",pTransferList->Count());//fflush(stdout);
	while( (Transfer=pTransferList->Next(Transfer)) != 0 )
	{
		printf("%d\n",Transfer->GetDone());
		Transfer->Disconnect(TRUE);
		
	}

	UnLock();
}

/** */
bool CDownloadManager::TransferClose( CString nick, CString host, CString file )
{
	CTransfer * Transfer;
	bool found=FALSE;

	Lock();

	Transfer = 0;

	while( (Transfer=pTransferList->Next(Transfer)) != 0 )
	{
		if ( (Transfer->GetDstNick() == nick) &&	
		     (Transfer->GetHost() == host) &&
		     (Transfer->GetDstFilename() == file) )
		{
			Transfer->Disconnect(TRUE);
			found = TRUE;
			break;
		}
	}

	UnLock();

	return found;
}

/** */
bool CDownloadManager::TransferCancel( CString nick, CString host, CString file )
{
	CTransfer * Transfer;
	bool found=FALSE;

	Lock();

	Transfer = 0;

	while( (Transfer=pTransferList->Next(Transfer)) != 0 )
	{
		if ( (Transfer->GetDstNick() == nick) &&	
		     (Transfer->GetHost() == host) &&
		     (Transfer->GetDstFilename() == file) )
		{
			Transfer->SendString("$Cancel");
			found = TRUE;
			break;
		}
	}

	UnLock();

	return found;
}

/** */
eDirection CDownloadManager::TransferDirection( CString nick, CString host, CString file )
{
	CTransfer * Transfer;
	eDirection direction=edNONE;

	Lock();

	Transfer = 0;

	while( (Transfer=pTransferList->Next(Transfer)) != 0 )
	{
		if ( (Transfer->GetDstNick() == nick) &&	
		     (Transfer->GetHost() == host) &&
		     (Transfer->GetDstFilename() == file) )
		{
			direction = Transfer->GetSrcDirection();
			break;
		}
	}

	UnLock();

	return direction;
}

/** */
bool CDownloadManager::TransferSetRate( CString nick, CString host, CString file, ulonglong rate )
{
	CTransfer * Transfer;
	bool found=FALSE;

	Lock();

	Transfer = 0;

	while( (Transfer=pTransferList->Next(Transfer)) != 0 )
	{
		if ( (Transfer->GetDstNick() == nick) &&	
		     (Transfer->GetHost() == host) &&
		     (Transfer->GetDstFilename() == file) )
		{
			Transfer->SetRate(rate);
			found=TRUE;
			break;
		}
	}

	UnLock();

	return found;
}

/** */
ulonglong CDownloadManager::TransferGetRate( CString nick, CString host, CString file )
{
	CTransfer * Transfer;
	ulonglong rate = 0;

	Lock();

	Transfer = 0;

	while( (Transfer=pTransferList->Next(Transfer)) != 0 )
	{
		if ( (Transfer->GetDstNick() == nick) &&	
		     (Transfer->GetHost() == host) &&
		     (Transfer->GetDstFilename() == file) )
		{
			rate = Transfer->GetRate();
			break;
		}
	}

	UnLock();

	return rate;
}

/** */
bool CDownloadManager::GetUserFileInfo( CString nick, CString hubname, CString /*hubhost*/, CString remotefile, CUserFileInfo * UserFileInfo )
{
	bool res = FALSE;

	if (!UserFileInfo)
	{
		return res;
	}

	Lock();

	DCTransferObject * TransferObject;
	DCTransferFileObject * TransferFileObject;

	if ( (TransferObject = pDownloadQueue.GetUserTransferObject( nick, hubname )) != 0 )
	{
		UserFileInfo->eWaitState    = TransferObject->eState;
		UserFileInfo->sUserFileList = TransferObject->sUserFileList;

		if ( remotefile != "" )
		{
			if ( TransferObject->pTransferFileList.Get( remotefile, (CObject *&) TransferFileObject ) == 0 )
			{
				UserFileInfo->eFileState = TransferFileObject->eState;
				res = TRUE;
			}
		}
		else
		{
			res = TRUE;
		}
	}

	UnLock();

	return res;
}

/** */
void CDownloadManager::TryConnect( CString nick, CString hubname, CString /*hubhost*/ )
{
	Lock();

	DCTransferObject * TransferObject;

	if ( (TransferObject = pDownloadQueue.GetUserTransferObject( nick, hubname )) != 0 )
	{
		TransferObject->tTimeout = 0;
	}

	UnLock();
}

/** */
bool CDownloadManager::AddTransfer( CString nick, CString userhost, CString hubname, CString hubhost )
{
	bool res = FALSE;
	DCTransferWait * TransferWait = 0;

	printf("Add wait transfer for: '%s' '%s' '%s'\n",nick.Data(),hubname.Data(),hubhost.Data());

	pTransferWaitList->Lock();

	while ( (TransferWait = pTransferWaitList->Next(TransferWait)) != 0 )
	{
		if ( TransferWait->sHubName == hubname )
		{
			if ( ((TransferWait->sNick == nick) && (TransferWait->sNick != "")) ||
			     ((TransferWait->sUserHost == userhost) && (TransferWait->sUserHost != "")))
			{
				// reset timeout
				TransferWait->tTimeout = time(0);

				break;
			}
		}
	}

	if ( TransferWait == 0 )
	{
		TransferWait = new DCTransferWait();
		TransferWait->sNick     = nick;
		TransferWait->sUserHost = userhost;
		TransferWait->sHubName  = hubname;
		TransferWait->sHubHost  = hubhost;
		TransferWait->tTimeout  = time(0);

		pTransferWaitList->Add(TransferWait);
	}

	res = TRUE;

	pTransferWaitList->UnLock();

	return res;
}

/** */
void CDownloadManager::AddTransfer( CString host, int port, CString hubname, CString hubhost )
{
	Lock();

	CTransfer * Transfer = new CTransfer();

	Transfer->SetNick( pDCLibConfig->GetNick() );

	Transfer->SetHubName(hubname);
	Transfer->SetHubHost(hubhost);

	Transfer->SetHost( host, port );
	Transfer->SetRate( pDCLibConfig->GetMaxTransferrate() );

	AddTransfer( "", Transfer->GetHost(), hubname, hubhost );

	pTransferList->Add(Transfer);
	Transfer->SetCallBackFunction(DC_TransferCallBack);

	Transfer->Connect();

	UnLock();
}

/** */
ulonglong CDownloadManager::GetNewTransferID()
{
	lTransferID++;

	if (lTransferID==0)
		lTransferID++;

	return lTransferID;
}

/** */
void CDownloadManager::AddWaitTransfer( CString nick, CString hubname, CString hubhost,
				CString remotename, CString localname,
				CString localpath, CString localrootpath,
				eltMedium medium, ulonglong size,
				ulonglong startposition, ulonglong endposition,
				bool )
{
	DCTransferObject * TransferObject;
	DCTransferFileObject * TransferFileObject = 0;

	printf("add wait transfer: %s %s %s %s %d %s %s %s %Ld %Ld %Ld\n",
		nick.Data(),
		remotename.Data(),
		hubname.Data(),
		hubhost.Data(),
		medium,
		localname.Data(),
		localpath.Data(),
		localrootpath.Data(),
		startposition,
		endposition,
		size );

	Lock();

	CStringList * StringList = pDownloadQueue.GetUserHubList( nick );

	if ( (TransferObject = pDownloadQueue.GetUserTransferObject( nick, hubname )) == 0 )
	{
		TransferObject = new DCTransferObject();

		TransferObject->sNick    = nick;
		TransferObject->sHubHost = hubhost;
		TransferObject->sHubName = hubname;
		TransferObject->eState   = etwsIDLE;
		TransferObject->tTimeout = 0;
		TransferObject->iConnections = 0;

		if ( StringList == 0 )
		{
			StringList = new CStringList();
			pDownloadQueue.pQueue->Add( nick, StringList );
		}

		StringList->Add( hubname, TransferObject );

		printf("transfer add\n");//fflush(stdout);
	}
	else
	{
		TransferFileObject = pDownloadQueue.GetUserFileObject( nick, hubname, remotename );
	}

	// file not found for the nick/hub
	if ( TransferFileObject == 0 )
	{
		TransferFileObject = new DCTransferFileObject();

		TransferFileObject->lSize            = size;
		TransferFileObject->lStartPosition   = startposition;
		TransferFileObject->lCurrentPosition = startposition;
		TransferFileObject->lEndPosition     = endposition;
		TransferFileObject->eMedium          = medium;
		TransferFileObject->sRemoteFile      = remotename;
		TransferFileObject->sRemoteFileName  = CDir(remotename).DirName();
		TransferFileObject->sLocalPath       = localpath;
		TransferFileObject->sLocalName       = localname;
		TransferFileObject->sLocalRootPath   = localrootpath;
		TransferFileObject->eState           = etfsNONE;

		TransferObject->pTransferFileList.Add( remotename, TransferFileObject );

		printf("file add %ld\n",TransferObject->pTransferFileList.Count());//fflush(stdout);
	}
	else
	{
		printf("file found ...\n");fflush(stdout);
	}

	SendFileInfo( TransferObject, TransferFileObject );

	UnLock();
}

/** */
bool CDownloadManager::RemoveQueue( CString nick, CString hubname, CString remotefile )
{
	bool res = FALSE;
	DCTransferFileObject * TransferFileObject = 0;
	DCTransferObject * TransferObject = 0;

	Lock();

	if ( (TransferObject = pDownloadQueue.GetUserTransferObject( nick, hubname )) != 0 )
	{
		if ( remotefile != "" )
		{
			if ( (TransferFileObject = pDownloadQueue.GetUserFileObject( nick, hubname, remotefile )) != 0 )
			{
				if ( TransferFileObject->eState != etfsTRANSFER )
				{
					res = TRUE;
				}
			}
		}
		else
		{
			if ( TransferObject->eState != etwsRUN )
			{
				res = TRUE;
			}
		}

		if ( res == TRUE )
		{
			SendFileInfo( TransferObject, TransferFileObject, TRUE );
		}
	}

	if ( res == TRUE )
	{
		res = pDownloadQueue.DelUserFileObject( nick, hubname, remotefile );
	}

	UnLock();

	return res;
}

/** */
bool CDownloadManager::CheckWaitTransfer( CTransfer * Transfer )
{
	bool res = FALSE;
	DCTransferObject * TransferObject;
	DCTransferWait * TransferWait = 0;

	printf("CheckWaitTransfer I : %s on %s\n",
		Transfer->GetDstNick().Data(),
		Transfer->GetHubName().Data() );

	// search in the wait list ...
	pTransferWaitList->Lock();

	while ( (TransferWait = pTransferWaitList->Next(TransferWait)) != 0 )
	{
		// search for the nick ...
		if ( ((TransferWait->sNick == Transfer->GetDstNick()) && (TransferWait->sNick != "")) ||
		     ((TransferWait->sUserHost == Transfer->GetHost()) && (TransferWait->sUserHost != "")) )
		{
			Transfer->SetHubName(TransferWait->sHubName);
			Transfer->SetHubHost(TransferWait->sHubHost);

			break;
		}
	}

	pTransferWaitList->UnLock();

	printf("CheckWaitTransfer II: %s on %s [%d]\n",
		Transfer->GetDstNick().Data(),
		Transfer->GetHubName().Data(),(TransferWait!=0) );

	Lock();

	if ( (TransferObject = pDownloadQueue.GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName() )) != 0 )
	{
		printf("Waiting: %s on %s %s\n",
			TransferObject->sNick.Data(),
			TransferObject->sHubName.Data(),
			TransferObject->sHubHost.Data() );

		if ( TransferObject->eState == etwsWAIT )
		{
			printf("wait found ...\n");fflush(stdout);

			TransferObject->eState = etwsRUN;
			TransferObject->iConnections++;

			// waiting transfer found ...
			res = TRUE;

			SendFileInfo( TransferObject );
		}
	}

	UnLock();

	if ( (res == FALSE) && (TransferWait == 0) )
	{
		printf("Warning: no wait transfer found for '%s'\n",Transfer->GetDstNick().Data());fflush(stdout);
		Transfer->Disconnect(TRUE);
	}

	if ( TransferWait != 0 )
	{
		pTransferWaitList->Lock();
		pTransferWaitList->Del(TransferWait);
		pTransferWaitList->UnLock();
	}

	return res;
}

/** */
bool CDownloadManager::ChangeDirection( CTransfer * Transfer )
{
	bool res = FALSE;
	DCTransferObject * TransferObject;

	Lock();

	if ( (TransferObject = pDownloadQueue.GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName() )) != 0 )
	{
		printf("Waiting: %s on %s %s\n",
			TransferObject->sNick.Data(),
			TransferObject->sHubName.Data(),
			TransferObject->sHubHost.Data() );

		// change to upload ...
		if ( Transfer->GetSrcDirection() == edDOWNLOAD )
		{
			if ( TransferObject->eState == etwsRUN )
			{
				TransferObject->iConnections--;
				if ( TransferObject->iConnections == 0 )
					TransferObject->eState = etwsIDLE;
				SendFileInfo( TransferObject );

				printf("change transfer -> upload ...\n");fflush(stdout);

				res = TRUE;
			}
			else
			{
				printf("can't change transfer upload ...\n");fflush(stdout);
			}
		}
	}

	UnLock();

	return res;
}

/** */
bool CDownloadManager::SetNextFile( CTransfer * Transfer )
{
	bool res;

	Lock();
	res = SetFile(Transfer);
	UnLock();

	return res;
}

/** */
bool CDownloadManager::SetDirection( CTransfer * Transfer )
{
	int i;
	bool res = FALSE;

	// check incoming transfer limit
	i = pDCLibConfig->GetMaxUpload();

	if ( Transfer->GetSrcDirection() == edUPLOAD )
	{
		if ( i == 0 )
		{
			res = TRUE;
		}
		else if ( i > lUsedSlots )
		{
			res = TRUE;
		}
		else
		{
			res = CheckUserSlot( Transfer->GetDstNick(), Transfer->GetHubName() );
		}

		if (res)
		{
			lUsedSlots++;
		}
	}
	else
	{
		res = TRUE;
	}

	return res;
}

/** */
bool CDownloadManager::SetFile( CTransfer * Transfer )
{
	bool res;
	DCTransferObject * TransferObject;

	res = FALSE;

	if ( (TransferObject = pDownloadQueue.GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName() )) != 0 )
	{
		if ( TransferObject->pTransferFileList.Count() > 0 )
		{
			DCTransferFileObject * TransferFileObject = 0;

			while( TransferObject->pTransferFileList.Next((CObject*&)TransferFileObject) )
			{
				if ( TransferFileObject->eState == etfsNONE )
				{
					printf("set file: '%s'\n",TransferFileObject->sRemoteFile.Data());fflush(stdout);

					CString sPath="",sLocalPath="",sFile="";
					CDir dir;

					if ( TransferFileObject->eMedium == eltFILE )
					{
						// set local download root path
						if ( TransferFileObject->sLocalRootPath != "" )
						{
							sLocalPath = TransferFileObject->sLocalRootPath;
						}
						else
						{
							sLocalPath = pDCLibConfig->GetDownloadFolder();
						}

						printf("DEBUG: root path : '%s'\n", sLocalPath.Data());fflush(stdout);
						printf("DEBUG: local path: '%s'\n", TransferFileObject->sLocalPath.Data());fflush(stdout);

						// add special download path
						if ( TransferFileObject->sLocalPath != "" )
						{
							sPath = dir.SimplePath(TransferFileObject->sLocalPath);
						}

						printf("DEBUG: path: '%s'\n", sPath.Data());fflush(stdout);

						// create the path
						dir.SetPath(sLocalPath);
						if ( dir.CreatePath(sPath) == FALSE )
						{
							TransferFileObject->eState = etfsERROR;
							SendFileInfo( TransferObject, TransferFileObject );
							SendLogInfo( "Create path failed: " + sPath, Transfer );
							printf("DEBUG: create path failed: '%s'\n", (sLocalPath+"/"+sPath).Data());fflush(stdout);
						}
						else
						{
							sFile = sLocalPath + "/" + sPath + "/" + TransferFileObject->sLocalName.Data();
							sFile = dir.SimplePath(sFile);

							printf("DEBUG: file: '%s'\n", sFile.Data());fflush(stdout);

							if ( TransferFileObject->lCurrentPosition > dir.getFileSize(sFile,FALSE) )
							{
								TransferFileObject->eState = etfsERROR;
								SendFileInfo( TransferObject, TransferFileObject );
								SendLogInfo( "Wrong filesize: " + sFile, Transfer );
								printf("DEBUG: wrong file size\n");fflush(stdout);
							}
							else
							{
								printf("DOWNLOAD: '%s' %Ld %Ld %Ld %Ld '%s'\n",
								TransferFileObject->sRemoteFile.Data(),
								TransferFileObject->lCurrentPosition,
								TransferFileObject->lStartPosition,
								TransferFileObject->lEndPosition,
								TransferFileObject->lSize,
								sFile.Data());fflush(stdout);

								res = TRUE;
							}
						}
						fflush(stdout);
					}
					else
					{
						res = TRUE;
					}

					if ( res == TRUE )
					{
						Transfer->SetDone(0);

						// mark file as transfered
						TransferFileObject->eState = etfsTRANSFER;

						Transfer->SetMedium(TransferFileObject->eMedium);

						Transfer->StartDownload(TransferFileObject->sRemoteFile,TransferFileObject->lStartPosition,
									TransferFileObject->lCurrentPosition,TransferFileObject->lEndPosition,TransferFileObject->lSize,sFile);
					}
					else
					{
						continue;
					}

					SendFileInfo( TransferObject, TransferFileObject );

					break;
				}
				else
				{
//					printf("state file: '%s' %d\n",TransferFileItem->sRemoteName.Data(),TransferFileItem->eState);//fflush(stdout);
				}
			}
		}
	}

	if ( res == FALSE )
	{
		// set disconnect timeout ... save remote slot ;-)
		if ( Transfer->GetDone() != 2 )
		{
			Transfer->SetStartTime(time(0));
			Transfer->SetDone(2);
		}
	}

	return res;
}

/** */
void CDownloadManager::UpdateFileState( CTransfer * Transfer, eTransferFileState eState )
{
	DCTransferObject * TransferObject;
	DCTransferFileObject * TransferFileObject;

	Lock();

	if ( (TransferObject = pDownloadQueue.GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName() )) != 0 )
	{
		TransferFileObject = pDownloadQueue.GetUserFileObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetDstFilename() );

		if ( TransferFileObject != 0 )
		{
			if ( TransferFileObject->eState != etfsTRANSFER )
			{
				printf("warning, wrong state in updatefile\n");
			}

			TransferFileObject->eState = eState;
			TransferFileObject->lStartPosition = Transfer->GetStartPosition();
			TransferFileObject->lCurrentPosition = Transfer->GetCurrentPosition();
			TransferFileObject->lEndPosition = Transfer->GetEndPosition();
			TransferFileObject->lSize = Transfer->GetLength();

			printf("%Ld %Ld %Ld %Ld\n",TransferFileObject->lStartPosition,TransferFileObject->lCurrentPosition,TransferFileObject->lEndPosition,TransferFileObject->lSize);////fflush(stdout);

			if ( (TransferFileObject->lCurrentPosition == TransferFileObject->lEndPosition) && (TransferFileObject->lSize != 0) )
			{
				SendLogInfo( "Transfer done '" + TransferFileObject->sRemoteFile + "'", Transfer );
				SendFileInfo( TransferObject, TransferFileObject, TRUE );
				TransferObject->pTransferFileList.Del(TransferFileObject->sRemoteFile);
			}
			else
			{
				SendFileInfo( TransferObject, TransferFileObject );
			}
		}
	}

	UnLock();
}

/** */
void CDownloadManager::Thread( CObject * )
{
	bool bShutdown = FALSE;
	CTransfer * Transfer, * oTransfer;
	int i;
	CDownloadManagerInfo dmi;

	if ( iShutdownState == 2 )
	{
#ifdef WIN32
		Sleep(100);
#else
		usleep(100);
#endif
		return;
	}

	Lock();

	if ( iShutdownState == 1 )
	{
		if ( pTransferList->Count() == 0 )
		{
			bShutdown = TRUE;
		}
	}

	dmi.slot_use = lUsedSlots;
	dmi.slot_max = pDCLibConfig->GetMaxUpload();

	Transfer = oTransfer = 0;

	while( (Transfer=pTransferList->Next(Transfer)) != 0 )
	{
		// done==2 set new file or disconnect
		if ( (Transfer->GetDone() == 2) && (iShutdownState == 0) )
		{
			if ( SetFile(Transfer) == FALSE )
			{
				// no new file ...
				if ( Transfer->GetStartTime() != 0 )
				{
					if ( (time(0)-Transfer->GetStartTime()) >= 60 )
					{
						Transfer->SetStartTime(0);
						Transfer->Disconnect(TRUE);
					}
				}
			}

			oTransfer = Transfer;
		}
		else if ( Transfer->GetDone() == 1 )
		{
			if ( Transfer->GetDstDirection() == edDOWNLOAD )
			{
				lUsedSlots--;
			}

			SendTransferInfo( Transfer, TRUE );

			// update internal transfer state for this user/hubname
			// set to idle (this work only with 1 transfer per user/hubname ...)
			DCTransferObject * TransferObject;

			if ( Transfer->GetSrcDirection() == edDOWNLOAD )
			{
				if ( (TransferObject = pDownloadQueue.GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName() )) != 0 )
				{
					TransferObject->iConnections--;
					if ( TransferObject->iConnections == 0 )
						TransferObject->eState = etwsIDLE;
					TransferObject->tTimeout = time(0);
					SendFileInfo( TransferObject );
				}
			}

			pTransferList->Del(Transfer);

			Transfer = oTransfer;
		}
		else if ( iShutdownState == 0 )
		{
			// update wait queue with current running transfers
			if ( Transfer->GetSrcDirection() == edDOWNLOAD )
			{
				DCTransferFileObject * TransferFileObject;

				if ( (TransferFileObject = pDownloadQueue.GetUserFileObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetDstFilename()) ) != 0 )
				{
					TransferFileObject->lStartPosition = Transfer->GetStartPosition();
					TransferFileObject->lCurrentPosition = Transfer->GetCurrentPosition();
					TransferFileObject->lEndPosition = Transfer->GetEndPosition();
					TransferFileObject->lSize = Transfer->GetLength();
				}
			}

			if ( Transfer->GetSrcDirection() == edUPLOAD )
			{
				dmi.rate_ul += Transfer->GetTransferrate();
			}
			else if ( Transfer->GetSrcDirection() == edDOWNLOAD )
			{
				dmi.rate_dl += Transfer->GetTransferrate();
			}
			oTransfer = Transfer;
		}
		else
		{
			oTransfer = Transfer;
		}
	}

	CStringList * StringList = 0;

	while( pDownloadQueue.pQueue->Next( (CObject *&) StringList) )
	{
		DCTransferObject * TransferObject = 0;

		while( StringList->Next( (CObject *&) TransferObject) )
		{
			if ( (TransferObject->pTransferFileList.Count() == 0) && (TransferObject->eState == etwsIDLE) )
			{
				// empty filelist
				SendFileInfo( TransferObject, 0, TRUE );
				StringList->Del(TransferObject->sHubName);
				break;
			}
			else if ( (TransferObject->eState == etwsWAIT) && (iShutdownState == 0) )
			{
				// resend connection request on wait timeout
				if ( (time(0)-TransferObject->tTimeout) >= pDCLibConfig->GetTransferResendTimeout() )
				{
					// set timeout
					TransferObject->eState = etwsIDLE;
					TransferObject->tTimeout = 0;
					SendFileInfo( TransferObject );
				}
			}
			else if ( (iShutdownState == 0) &&
				( (TransferObject->eState == etwsIDLE) ||
				  (TransferObject->eState == etwsHUBOFFLINE) ||
				  (TransferObject->eState == etwsUSEROFFLINE) ||
				  (TransferObject->eState == etwsUSERBUSY) ||
				  (TransferObject->eState == etwsSENDERROR) ) )
			{
				
				// idle state ... check for timeout to reconnect
				if ( TransferObject->tTimeout == 0 )
				{
					// send connection request to the hub on timeout
					i = pDCLibServerManager->SendConnectionRequest( TransferObject->sNick, TransferObject->sHubName );

					switch (i)
					{
						case 0:
							TransferObject->eState = etwsWAIT;
							break;
						case -1:
							TransferObject->eState = etwsUSEROFFLINE;
							break;
						case -2:
						case -3:
							TransferObject->eState = etwsHUBOFFLINE;
							break;
						case -4:
							TransferObject->eState = etwsSENDERROR;
							break;
						default:
							break;
					}

					SendFileInfo( TransferObject );

					if ( TransferObject->tTimeout != time(0) )
					{
						TransferObject->tTimeout = time(0);
						SendFileInfo( TransferObject );
					}
				}
				else if ( (time(0)-TransferObject->tTimeout) >= pDCLibConfig->GetTransferResponseTimeout() )
				{
					// set timeout
					TransferObject->tTimeout = 0;
					SendFileInfo( TransferObject );
				}
			}
		}
	}

	UnLock();

	if ( iShutdownState == 1 )
	{
		if ( bShutdown == TRUE )
			iShutdownState = 2;
		else
#ifdef WIN32
			Sleep(100);
#else
			usleep(100);
#endif
	}
	else
	{
#ifdef WIN32
		Sleep(100);
#else
		usleep(1000*100);
#endif
	}

	SendDownloadManagerInfo(&dmi);
}

/** callback function */
int CDownloadManager::DMListenCallBack( int handle )
{
	if ( handle == -1 )
	{
		return -1;
	}

	CTransfer * Transfer = new CTransfer();

	Transfer->SetNick( pDCLibConfig->GetNick() );
	Transfer->SetRate( pDCLibConfig->GetMaxTransferrate() );
	Transfer->SetCallBackFunction(DC_TransferCallBack);

	pDownloadManager->Lock();

	if ( Transfer->SetSocket(handle) == 0 )
	{
		pDownloadManager->SendLogInfo("Incoming connection from '"+Transfer->GetHost()+"'\n");
		pDownloadManager->pTransferList->Add(Transfer);
	}
	else
	{
		delete Transfer;
		close(handle);
	}

	pDownloadManager->UnLock();

	return 0;
}

/** */
int CDownloadManager::DC_TransferCallBack( CTransfer * Transfer, CObject * Object )
{
	pDownloadManager->TransferCallBackThread.Lock();

	CDCMessage *DCMsg;

	DCMsg = (CDCMessage*)Object;

	switch ( DCMsg->eType )
	{
		case DC_MESSAGE_MYNICK:
		{
			// check for download or upload
			if ( pDownloadManager->CheckWaitTransfer( Transfer ) == FALSE )
			{
				Transfer->SetSrcDirection(edUPLOAD);
			}
			else
			{
				Transfer->SetSrcDirection(edDOWNLOAD);
			}

			if ( Transfer->GetSrcDirection() != edNONE )
			{
				Transfer->SendDirection( Transfer->GetSrcDirection(), Transfer->GetLevel() );
			}

			break;
		}

		case DC_MESSAGE_GET:
		{
			CMessageGet * msg = (CMessageGet*)Object;

			if ( Transfer->GetSrcDirection() == edUPLOAD )
			{
				pDownloadManager->SendLogInfo( "Upload: " + msg->sFilename, Transfer );

				if ( msg->sFilename == CString(DC_USER_FILELIST) )
				{
					if ( pDCLibConfig->GetShareSize() == 0 )
					{
						Transfer->SendError("File not found.");

						pDownloadManager->SendLogInfo( "Upload (No share): " + msg->sFilename, Transfer );
					}
					else
					{
						CByteArray ba;

						if ( pDCLibConfig->GetShareBuffer( &ba ) == 0 )
						{
							// set send buffer
							Transfer->SetBuffer(&ba);
							// set local transfer medium to buffer
							Transfer->SetMedium(eltBUFFER);
							// start upload
							Transfer->StartUpload( msg->sFilename, pDCLibConfig->GetShareBufferSize(), msg->lPos-1, "" );
						}
					}
				}
				else
				{
					CString s;
					ulonglong  len;
					CDir dir;
					CList<CString> SharedFolders;

					pDCLibConfig->GetSharedFolders(&SharedFolders);

					// search file and upload ...
					if ( (s = dir.GetCorrectFilePath(msg->sFilename.Data(),&SharedFolders)) == "" )
					{
						Transfer->SendError("File not found.");

						pDownloadManager->SendLogInfo( "Upload (File not found): " + msg->sFilename, Transfer );
					}
					else
					{
						len = dir.getFileSize(s,FALSE);

						if ( msg->lPos >= len )
						{
							Transfer->SendError("File allready download.");
						}
						else if ( msg->lPos == 0 )
						{
							Transfer->SendError("Wrong file position.");
						}
						else
						{
							// set local transfer medium to buffer
							Transfer->SetMedium(eltFILE);
							// start upload
							Transfer->StartUpload( msg->sFilename, len, msg->lPos-1, s );
						}
					}
				}
			}
			else
			{
				pDownloadManager->SendLogInfo( "Warning wrong mode", Transfer );
				Transfer->Disconnect(TRUE);
			}

			break;
		}

		case DC_MESSAGE_DIRECTION:
		{
			CMessageDirection * msg = (CMessageDirection*)Object;
			bool bdirec = FALSE;

			printf("DIRECTION: level: LOCAL: %d REMOTE: %d\n",Transfer->GetLevel(), msg->iLevel);
			printf("DIRECTION: direc: LOCAL: %d REMOTE: %d\n",Transfer->GetSrcDirection(), msg->Direction);

			// equal direction ...
			if ( msg->Direction == Transfer->GetSrcDirection() )
			{
				// check the level ...
				if ( msg->iLevel < Transfer->GetLevel() )
				{
					// now we must change the dst direction
					if ( Transfer->GetSrcDirection() == edDOWNLOAD )
					{
						Transfer->SetDstDirection(edUPLOAD);
						bdirec = TRUE;
					}
				}
				else if ( msg->iLevel > Transfer->GetLevel() )
				{
					if ( Transfer->GetSrcDirection() == edDOWNLOAD )
					{
						// we change to upload and re-check for free slots
						if ( pDownloadManager->ChangeDirection(Transfer) == TRUE )
						{
							Transfer->SetSrcDirection(edUPLOAD);
							bdirec = TRUE;
						}
					}
				}
			}
			else
			{
				bdirec = TRUE;
			}

			if ( bdirec == TRUE )
			{
				if ( pDownloadManager->SetDirection(Transfer) == FALSE )
				{
					bdirec = FALSE;
				}
			}

			if ( bdirec == FALSE )
			{
				pDownloadManager->SendLogInfo( "Warning no more free slots", Transfer );
				Transfer->SetDstDirection(edNONE);
				Transfer->SendMaxedOut();
				Transfer->Disconnect(TRUE);
			}
			else
			{
				if ( Transfer->GetSrcDirection() == edDOWNLOAD )
				{
					printf("DIRECTION: download mode ...\n");

					// next file
					if ( pDownloadManager->SetNextFile(Transfer) == FALSE )
					{
						printf("DIRECTION: download mode without destination file -> disconnecting!\n");
						Transfer->Disconnect(TRUE);
					}
				}
				else if ( Transfer->GetSrcDirection() == edUPLOAD )
				{
					printf("DIRECTION: we are in upload mode ...\n");
				}
			}

			break;
		}

		case DC_MESSAGE_CONNECTION_STATE:
		{
			bool remove = TRUE;
			CMessageConnectionState *msg = (CMessageConnectionState*)Object;

			if ( msg->eState == estCONNECTED )
			{
				remove = FALSE;
			}
			else if ( msg->eState == estDISCONNECTED )
			{
				pDownloadManager->SendLogInfo( "Disconnected", Transfer );
			}
			else if ( msg->eState == estSOCKETERROR )
			{
				remove = FALSE;
			}
			else if ( msg->eState == estCONNECTIONTIMEOUT )
			{
				remove = FALSE;
			}

			if ( (remove == TRUE) && (Transfer->GetSrcDirection() == edDOWNLOAD) )
			{
				if ( Transfer->GetMedium() == eltBUFFER )
					Transfer->SetStartPosition(0);
				pDownloadManager->UpdateFileState(Transfer,etfsNONE);
			}

			if ( remove == TRUE )
			{
				Transfer->SetDone(1);
			}

			break;
		}

		case DC_MESSAGE_TRANSFER:
		{
			CMessageTransfer *msg = (CMessageTransfer*)Object;

//			ulonglong rate;
			CString s;
			CString r;

			pDownloadManager->SendTransferInfo(Transfer);

			// filetransfer done
			if ( msg->lTransfered == msg->lLength )
			{
				// download ready
				if ( Transfer->GetSrcDirection() == edDOWNLOAD )
				{
					pDownloadManager->UpdateFileState(Transfer,etfsNONE);

					// show user file list
					if ( Transfer->GetDstFilename() == CString(DC_USER_FILELIST) )
					{
						pDownloadManager->FileListDone(Transfer);
					}

					if ( pDownloadManager->SetNextFile(Transfer) == FALSE )
					{
					}
				}
			}

			break;
		}

		case DC_MESSAGE_FILELENGTH:
		{
//			CMessageFileLength * msg = (CMessageFileLength*)Object;

			pDownloadManager->SendTransferInfo(Transfer);

			break;
		}

		case DC_MESSAGE_MAXEDOUT:
		{
			pDownloadManager->SendLogInfo( "Busy", Transfer );
			Transfer->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_GETLISTLEN:
		{
			Transfer->SendListLen(pDCLibConfig->GetShareBufferSize());
			break;
		}

		case DC_MESSAGE_ERROR:
		{
			CMessageError * msg = (CMessageError*)Object;

			pDownloadManager->SendLogInfo( "Error: " + msg->sError, Transfer );

			if ( Transfer->GetSrcDirection() == edDOWNLOAD )
			{
				pDownloadManager->UpdateFileState(Transfer,etfsERROR);

				// set next file for download
				if ( pDownloadManager->SetNextFile(Transfer) == FALSE )
				{
				}
			}
			break;
		}

		default:
		{
			printf("dctransfer unknown message: %d\n",DCMsg->eType);
			break;
		}
	}

	delete Object;

	pDownloadManager->TransferCallBackThread.UnLock();

	return 0;
}
