/***************************************************************************
                           cdir.cpp  -  description
                             -------------------
    begin                : Tue May 14 2002
    copyright            : (C) 2002 by Mathias Kster
    email                : mathen@ketelhot.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef WIN32
#include <unistd.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#endif

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include "cdir.h"

/** */
CDir::CDir()
{
	sPath  = "";
	sDrive = "";
}

/** */
CDir::CDir( CString path )
{
	SetPath(path);
}

/** */
CDir::~CDir()
{
}

/** */
CString CDir::Path()
{
	return sPath;
}

/** */
void CDir::SetPath( CString path )
{
	CString s;
	long i;

	s = path;

	// store drive letter
	if ( (i = s.Find(':',0)) != -1 )
	{
		sDrive = s.Left(i+1);
		s = s.Mid(i+1,s.Length()-i-1);
	}

	// convert separators to internal
	i = 0;
	while( (i=s.Find('\\',i)) != -1 )
	{
		s.Data()[i] = '/';
		i++;
	}

	// remove last '/'
	if ( ((1+s.FindRev('/')) == s.Length()) && (s.Length() > 1) )
	{
		sPath = s.Left( s.Length()-1 );
	}
	else
	{
		sPath = s;
	}
}

/** */
CString CDir::HomeDirPath()
{
	CString s;

#ifdef WIN32
	s = getenv("USERPROFILE");
#else
	s = getenv("HOME");
#endif
	if ( s == "" )
	{
		s = DIRSEPARATOR;
	}

	return s;
}

/** */
bool CDir::cd( CString path )
{
	sPath = "";

	if ( path == "." )
	{
		return TRUE;
	}

	if ( access( path.Data(), F_OK | R_OK ) == 0 )
	{
		SetPath(path);
		return TRUE;
	}

	return FALSE;
}

/** */
CString CDir::DirName()
{
	int pos = sPath.FindRev('/');

	if(pos==-1)
	{
		return sPath;
	}

	return sPath.Mid( pos + 1, sPath.Length() - 1 - pos );
}

/** */
bool CDir::ReadEntrys( FilterSpec filterspec, CList<CFileInfo> * list )
{
	DIR * dir;
	CFileInfo * FileInfo;
	dirent *file;
	CString s;

	if(!list)
		return FALSE;

	list->Clear();

	dir = opendir( sPath.Data() );

	if ( dir == 0 )
		return FALSE;

	while ( (file = readdir(dir)) != 0 )
	{
		if ( (filterspec == Dirs) && (IsDir(file->d_name)) )
		{
			FileInfo = new CFileInfo();
			FileInfo->name = file->d_name;
			FileInfo->size = 0;
			list->Add(FileInfo);
		}
		else if ( (filterspec == Files) && (IsFile(file->d_name)) )
		{
			FileInfo = new CFileInfo();
			FileInfo->name = file->d_name;
			FileInfo->size = getFileSize(file->d_name);
			list->Add(FileInfo);
		}
	}

	closedir(dir);

	return TRUE;
}

/** */
bool CDir::IsDir( CString s )
{
	struct stat buf;

	if ( stat( (sPath+DIRSEPARATOR+s).Data(), &buf ) != 0 )
		return FALSE;

	if ((buf.st_mode&S_IFMT)&S_IFDIR)
		return TRUE;
	else
		return FALSE;
}

/** */
bool CDir::IsFile( CString s )
{
	struct stat buf;

	if ( stat( (sPath+DIRSEPARATOR+s).Data(), &buf ) != 0 )
	{
		printf("stat error\n");
		return FALSE;
	}

	if ((buf.st_mode&S_IFMT)&S_IFREG)
		return TRUE;
	else
		return FALSE;
}

/** */
ulonglong CDir::getFileSize( CString s, bool rel )
{
	CString p;
	struct stat buf;

	if ( rel )
		p = sPath+DIRSEPARATOR+s;
	else
		p = s;

	if ( stat( p.Data(), &buf ) != 0 )
	{
		perror("CDir::getFileSize: stat: ");
		return FALSE;
	}

	return buf.st_size;
}

/** */
CString CDir::CleanDirPath( CString path )
{
	CString s = path;
	CString search;
	long i=0;

	search  = DIRSEPARATOR;
	search += "..";

	// convert '..' -> '//'
	while( (i=s.Find( search.Data() ,i )) != -1 )
	{
		if ( (s.Data()[i+3] == 0) || (s.Data()[i+4]==DIRSEPARATOR) )
		{
			s.Data()[i+1] = DIRSEPARATOR;
			s.Data()[i+2] = DIRSEPARATOR;
		}
		i++;
	}

	return s;
}

/** */
CString CDir::ConvertSeparators( CString path )
{
	CString s = path;
	long i=0;

	while( (i=path.Find('/',i)) != -1 )
	{
		s.Data()[i] = DIRSEPARATOR;
		i++;
	}

	return s;
}

/** */
CString CDir::SimplePath( CString path )
{
	CString s = path;
	long i;

	i = 0;
	while( (i=s.Find(':',i)) != -1 )
	{
		s.Data()[i] = '/';
		i++;
	}

	i = 0;
	while( (i=s.Find('\\',i)) != -1 )
	{
		s.Data()[i] = '/';
		i++;
	}

	s = ConvertSeparators(s);
	s = CleanDirPath(s);

	return s;
}

/** */
bool CDir::CreatePath( CString path )
{
	int i;
	CString s,s1;

	s  = SimplePath(path);
	s1 = "";

	while ( s != "" )
	{
		if ( (i = s.Find(DIRSEPARATOR)) != -1 )
		{
			if (s1!="")
				s1 = s1 + DIRSEPARATOR + s.Left(i);
			else
				s1 = s.Left(i);
			s  = s.Mid(i+1,s.Length()-1-i);
		}
		else
		{
			if (s1!="")
				s1 = s1 + DIRSEPARATOR + s;
			else
				s1 = s;
			s  = "";
		}

		if ( s1 == "" )
		{
			continue;
		}

		if ( IsDir(s1) == FALSE )
		{
#ifdef WIN32
			if ( mkdir((ConvertSeparators(sDrive+DIRSEPARATOR+sPath)+DIRSEPARATOR+s1).Data()) != 0 )
#else
			if ( mkdir((ConvertSeparators(sPath)+DIRSEPARATOR+s1).Data(),S_IRUSR|S_IWUSR|S_IXUSR) != 0 )
#endif
			{
				printf("mkdir Error: '%s' '%s'\n",strerror(errno),(sPath+DIRSEPARATOR+s1).Data());fflush(stdout);
				return FALSE;
			}
		}
	}

	return TRUE;
}

/** */
CString CDir::GetCorrectFilePath( CString file, CList<CString> * sharedfolders )
{
	int i;
	CString s,s1,sfile,sdir;
	CString * spath;

	if (!sharedfolders)
	{
		return "";
	}

	if ( sharedfolders->Count() <= 0 )
	{
		return "";
	}

	// simple filename
	sfile = SimplePath(file);

	if ( sfile == "" )
	{
		return "";
	}

	// get the first directory
	i = sfile.Find(DIRSEPARATOR);

	if ( i == -1 )
	{
		sdir  = sfile;
		sfile = "";

		// get directory ...
		return "";
	}
	else
	{
		sdir  = sfile.Left(i);
		sfile = sfile.Mid(i+1,sfile.Length()-1-i);
	}

	if ( sfile == "" )
	{
		// get directory ...
		return "";
	}

	spath = 0;
	while( (spath=sharedfolders->Next(spath)) != 0 )
	{
		// directory
		s = (*spath);

		// is valid ?
		if ( cd(s) == TRUE )
		{
			s1 = DirName();

			// wrong path ?
			if ( sdir != s1 )
			{
				continue;
			}

			s1 = DIRSEPARATOR + sfile;

			if ( IsFile(s1) == TRUE )
			{
				// check filesize
				if ( getFileSize(s1) > 0 )
				{
					s += DIRSEPARATOR + sfile;
					return s;
				}
			}
		}
	}

	return "";
}

#ifdef WIN32

#include <malloc.h>
#include <string.h>
#include <errno.h>

/**********************************************************************
* Implement dirent-style opendir/readdir/closedir on Window 95/NT
*
* Functions defined are opendir(), readdir() and closedir() with the
* same prototypes as the normal dirent.h implementation.
*
* Does not implement telldir(), seekdir(), rewinddir() or scandir().
* The dirent struct is compatible with Unix, except that d_ino is
* always 1 and d_off is made up as we go along.
*
* The DIR typedef is not compatible with Unix.
**********************************************************************/

DIR * opendir(const char *dir)
{
	DIR *dp;
	char *filespec;
	long handle;
	int index;

	if(dir==0) return 0;
	filespec = (char*)malloc(strlen(dir) + 2 + 1);
	strcpy(filespec, dir);
	index = strlen(filespec) - 1;
	if (index >= 0 && (filespec[index] == '/' || filespec[index] == '\\'))
	filespec[index] = '\0';
	strcat(filespec, "/*");

	dp = (DIR *)malloc(sizeof(DIR));
	dp->offset = 0;
	dp->finished = 0;
	dp->dir = strdup(dir);

	if ((handle = _findfirst(filespec, &(dp->fileinfo))) < 0) {
		if (errno == ENOENT)
			dp->finished = 1;
		else
			free(filespec);
		return NULL;
	}

	dp->handle = handle;
	free(filespec);

	return dp;
}

struct dirent * readdir(DIR *dp)
{
	if (!dp || dp->finished) return NULL;

	if (dp->offset != 0) {
		if (_findnext(dp->handle, &(dp->fileinfo)) < 0) {
			dp->finished = 1;
			return NULL;
		}
	}
	dp->offset++;

	strncpy(dp->dent.d_name, dp->fileinfo.name, _MAX_FNAME);
	dp->dent.d_ino = 1;
	dp->dent.d_reclen = (unsigned short)strlen(dp->dent.d_name);
	dp->dent.d_off = dp->offset;

	return &(dp->dent);
}

int closedir(DIR *dp)
{
	if (!dp) return 0;
	_findclose(dp->handle);
	if (dp->dir) free(dp->dir);
	if (dp) free(dp);

	return 0;
}
#endif
