/***************************************************************************
                          cconnection.cpp  -  description
                             -------------------
    begin                : Sat Oct 6 2001
    copyright            : (C) 2001 by Mathias Kster
    email                : mathen@ketelhot.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifndef WIN32
#include <stdlib.h>
#include <malloc.h>
#include <unistd.h>
#else
#include <wtypes.h>
#include <winbase.h>
#include <stdlib.h>
#include <malloc.h>
#endif

#include "cconnection.h"

#define BUFFER_SIZE		1024*1024

CConnection::CConnection()
{
	sIP   = "";
	iPort = 411;

	bForceDisconnect = FALSE;

	// set connection timeout
	iConnectTimeout = 60;

	eState = estNONE;

	Start();

	pBuffer = (char*) malloc(BUFFER_SIZE);

	pSendList = new CList<CObject>();
}

CConnection::~CConnection()
{
	Stop();

	CSocket::Disconnect();

	if (pBuffer)
	{
		free(pBuffer);
		pBuffer = 0;
	}

	if (pSendList)
	{
		delete pSendList;
		pSendList = 0;
	}
}

/** */
int CConnection::Connect()
{
	Lock();

	int err = -1;

	eState = estCONNECT;

	bForceDisconnect = FALSE;

	err = 0;

	UnLock();

	return err;
}

/** */
int CConnection::Connect( CString ip, int port, eSocketType sockettype )
{
	Lock();

	sIP   = ip;
	iPort = port;

	SocketType = sockettype;

	UnLock();

	return Connect();
}

/** */
int CConnection::Disconnect( bool force )
{
	int err = -1;

	if ( force )
	{
		err = 0;

		bForceDisconnect = TRUE;
	}
	else
	{
		Lock();

		if ( eState != estNONE )
		{
			eState = estDISCONNECTING;

			err = 0;
		}

		UnLock();
	}

	return err;
}

/** */
int CConnection::SetSocket( int handle, eSocketType sockettype )
{
	CString ip;
	int port;

	if ( handle == -1 )
	{
		return -1;
	}

	if ( sockettype != estTCP )
	{
		return -1;
	}

	if ( eState != estNONE )
	{
		return -1;
	}

	Lock();

	sIP   = "";
	iPort = 0;

	// set socket type
	SocketType = sockettype;

	iHandle = handle;

	// get remote host & port
	if ( GetPeerName( &ip, &port ) == -1 )
	{
		UnLock();
		return -1;
	}

	// set remote host & port
	SetHost( ip, port );

	bForceDisconnect = FALSE;

	eState = estCONNECTED;

	UnLock();

	ConnectionState(estCONNECTED);

	return 0;
}

CString CConnection::GetHost()
{
	CString s;
	char c[20];

#ifdef WIN32
	_snprintf(c,20,"%d",GetPort());
#else
	snprintf(c,20,"%d",GetPort());
#endif

	s = GetIP() + ":" + c;

	return s;
}

void CConnection::Thread( CObject * )
{
	int wait = 10;

	Lock();

	if (bForceDisconnect)
	{
		if ( eState != estNONE )
			eState = estDISCONNECTING;
		bForceDisconnect = FALSE;
	}

	switch(eState)
	{
		case estNONE:
			break;
		case estCONNECT:
			StateConnect();
			if ( eState == estCONNECT )
				wait = 100;
			break;
		case estCONNECTING:
			StateConnecting();
			break;
		case estCONNECTED:
			// read data
			StateRead();
			// send data
			if ( eState == estCONNECTED )
				StateSend();
			// send data from higher level
			if ( eState == estCONNECTED )
				DataSend();
			// check data timeout
			if ( eState == estCONNECTED )
				if ( (time(0)-timeConnection) >= 60 )
					DataTimeout();
			break;
		case estDISCONNECTING:
			StateDisconnect();
			break;
		case estDISCONNECTED:
			break;
		default:
			break;
	}

	UnLock();

#ifdef WIN32
	Sleep(wait);
#else
	usleep(wait);
#endif

}

/** */
void CConnection::StateConnect()
{
	eConnectState ecs;

	timeConnection = time(0);

	ecs = CSocket::Connect( sIP, iPort );

	if ( ecs == ecsERROR )
	{
		ConnectionState(estSOCKETERROR);

		eState = estDISCONNECTING;
	}
	else if ( ecs == ecsSUCCESS )
	{
		eState  = estCONNECTING;
	}
}

/** */
void CConnection::StateConnecting()
{
	int err;

	err = CSocket::IsConnect();

	if ( err < 0 )
	{
		ConnectionState(estSOCKETERROR);

		eState = estDISCONNECTING;
	}
	else if ( err == 1 )
	{
		ConnectionState(estCONNECTED);

		// init data timeout
		timeConnection = time(0);

		eState  = estCONNECTED;
	}
	// check connection timeout
	else if ( (time(0) - timeConnection) >= iConnectTimeout )
	{
		ConnectionState(estCONNECTIONTIMEOUT);

		eState = estDISCONNECTING;
	}
}

/** */
void CConnection::StateRead()
{
	CString s;
	int len;

	if (pBuffer==0)
	{
		return;
	}

	if ( (len = CSocket::Read(pBuffer,BUFFER_SIZE,0,10)) < 0 )
	{
		ConnectionState(estSOCKETERROR);

		eState = estDISCONNECTING;
	}
	else if ( len > 0 )
	{
		// update timeout
		timeConnection = time(0);

		// fix end
		pBuffer[len] = 0;

		// hack
		UnLock();
		DataAvailable(pBuffer,len);
		Lock();
	}
}

/** */
void CConnection::StateSend()
{
	int err = 0;
	CString *s;
	CObject *obj;

	ConnectionThread.Lock();

	if ( pSendList != 0 )
	{
		while( (obj=pSendList->Next(0)) != 0 )
		{
			s = ((CString*)obj);

			if ( s->Length() > 0 )
			{
				err = CSocket::Write(s->Data(),s->Length(),0,10);

				if ( err == 0 )
				{
					break;
				}
				else if ( err == -1 )
				{
					break;
				}
				else if ( err != s->Length() )
				{
					printf("CConnection: warning send %d %ld\n",err,s->Length());fflush(stdout);
					break;
				}
			}

			pSendList->Del(obj);

			// update timeout
			timeConnection = time(0);
		}
	}

	ConnectionThread.UnLock();

	if ( err == -1 )
	{
		ConnectionState(estSOCKETERROR);

		eState = estDISCONNECTING;
	}
}

/** */
void CConnection::StateDisconnect()
{
	CObject *obj;

	CSocket::Disconnect();

	ConnectionThread.Lock();

	if (pSendList!=0)
	{
		while( (obj=pSendList->Next(0)) != 0 )
			pSendList->Del(obj);
	}

	ConnectionThread.UnLock();

	ConnectionState(estDISCONNECTED);

	eState = estNONE;
}

/** */
int CConnection::Write( const char * buffer, int len )
{
	if (pSendList==0)
	{
		return -1;
	}

	if ( (eState != estCONNECTED) &&
	     (eState != estCONNECTING) )
	{
		return -1;
	}

	if ( len <= 0 )
	{
		return -1;
	}

	CString *s = new CString();

	*s = buffer;

	pSendList->Add((CObject*)s);

	return 0;
}

/** */
int CConnection::SendChat( CString sNick, CString s )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "<";
	t += sNick +"> ";
	t += s + "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendVersion( CString sVersion )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$Version ";
	t += sVersion;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendMyInfo( CString sNick, CString sComment, CString sConnectionType, eUserAwayMode eAwayMode, CString sEMail, CString sShare )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$MyINFO $ALL ";
	t += sNick + " " + sComment + "$";
	t += " $"; // ???
	t += sConnectionType;

	switch(eAwayMode)
	{
		case euamAWAY:
			t += "\x2";
			break;
		default:
			t += "\x1";
			break;
	}

	t += "$";
	t += sEMail + "$";
	t += sShare + "$"; // share
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendKey( CString s )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$Key ";
	t += s;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendValidateNick( CString sNick )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$ValidateNick ";
	t += sNick;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendMyNick( CString sNick )
{
	ConnectionThread.Lock();

	int err;
	CString t,s;
	int i,l;

	t  = "$MyNick ";
	t += sNick;
	t += "|";

	t += "$Lock ";

	l = 50+rand()%50;

	for (i=0;i<l;i++)
	{
		t += (char)('%'+rand()%('z'-'%'));
	}

	t += " Pk=";

	s  = "DCGUI-";
	s += VERSION;
	s += "-";
	while(s.Length()<16) s += (char)('%'+rand()%('z'-'%'));

	t += s;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendLock()
{
	ConnectionThread.Lock();

	int err;
	CString t;
	int i,l;

	t = "$Lock ";

	l = 50+rand()%50;

	for (i=0;i<l;i++)
	{
		t += (char)('%'+rand()%('z'-'%'));
	}

	t += " ";
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}


/** */
int CConnection::SendPrivateMessage( CString sNick, CString sTo, CString sMsg )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t = "$To: " + sTo + " From: " + sNick + " $<" + sNick + "> " + sMsg + "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::RequestNickList()
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t = "$GetNickList|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendDirection( eDirection Direction, int level )
{
	ConnectionThread.Lock();

	int err;
	char c[20];
	CString t;

	memset(c,0,20);
#ifdef WIN32
	_snprintf(c,20,"%d",level);
#else
	snprintf(c,20,"%d",level);
#endif

	t  = "$Direction ";

	if (Direction == edUPLOAD)
		t += "Upload";
	else if (Direction == edDOWNLOAD)
		t += "Download";

	t += " ";
	t += c;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendGet( CString file, ulonglong pos )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$Get ";
	t += file;
	t += "$";
	t += CString().setNum(pos);
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendRevConnectToMe( CString sNick, CString sDstNick )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$RevConnectToMe ";
	t += sNick;
	t += " ";
	t += sDstNick;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendSend()
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$Send";
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendError( CString message )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$Error ";
	t += message;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendFileLength( ulonglong len )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$FileLength ";
	t += CString().setNum(len);
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendListLen( ulonglong len )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$ListLen ";
	t += CString().setNum(len);
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendGetListLen()
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$GetListLen";
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendConnectToMe( CString sDstNick, CString host )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$ConnectToMe ";
	t += sDstNick + " ";
	t += host;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendCanceled()
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$Canceled";
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** send string */
int CConnection::SendString( CString message )
{
	ConnectionThread.Lock();

	int err;

	err = Write(message.Data(),message.Length());

	ConnectionThread.UnLock();

	return err;

}

/** send password */
int CConnection::SendPass( CString pass )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$MyPass ";
	t += pass;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** operator kick */
int CConnection::SendKick( CString nick )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$Kick ";
	t += nick;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** operator force move a user */
int CConnection::SendOpForceMove( CString nick, CString host, CString message )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$OpForceMove $Who:";
	t += nick;
	t += "$Where:";
	t += host;
	t += "$Msg:";
	t += message;
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendMaxedOut()
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t  = "$MaxedOut";
	t += "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}

/** */
int CConnection::SendGetInfo( CString sNick, CString sMyNick )
{
	ConnectionThread.Lock();

	int err;
	CString t;

	t = "$GetINFO " + sNick + " " + sMyNick + "|";

	err = Write(t.Data(),t.Length());

	ConnectionThread.UnLock();

	return err;
}
