/*
** iksemel (XML Parser for Jabber)
** Copyright (C) 2000-2001 Gurer Ozen <palpa@jabber.org>
**
** This code is free software; you can redistribute it and/or
** modify it under the terms of GNU Lesser General Public License.
**
** library API
*/

#include "common.h"


/*****  Memory Pools  *****/

ikspool *iks_pool_new(int pagesize)
{
	ikspool *p;

	p = malloc(sizeof(ikspool));
	if(!p) return(NULL);

	p->data = malloc(pagesize);
	if(!p->data)
	{
		free(p);
		return(NULL);
	}

	p->next = NULL;
	p->max = pagesize;
	p->len = 0;

	return p;
}


void *iks_pool_alloc(ikspool *p, int size)
{
	void *mem;
	int i;

	i = size & 8;
	if(i) size += (8 - i);
	i = p->max;

	if(size > i)
	{
		while(p->next) p=p->next;
		p->next = iks_pool_new(size);
		p = p->next;
		if(!p) return(NULL);
		p->len = size;
		return p->data;
	}

	while(p)
	{
		if((p->max - p->len) >= size)
		{
			mem = p->data + p->len;
			p->len += size;
			return mem;
		}

		if(p->next)
			p = p->next;
		else
			p->next = iks_pool_new(i);
	}

	return NULL;
}


char *iks_pool_strdup(ikspool *p, const char *src)
{
	char *dest;
	int len;

	if(!src) return(NULL);

	len = strlen(src);
	if(len == 0) return(NULL);
	dest = iks_pool_alloc(p, len + 1);
	if(!dest) return(NULL);
	memcpy(dest, src, len);
	dest[len] = '\0';

	return dest;
}


void iks_pool_delete(ikspool *p)
{
	ikspool *tmp;

	tmp = p;
	while(tmp)
	{
		tmp = p->next;
		free(p->data);
		free(p);
		p = tmp;
	}
}


/*****  NULL-safe Functions  *****/

char *iks_strdup(const char *src)
{
	if(src) return(strdup(src));
	return NULL;
}


char *iks_strcat(char *dest, const char *src)
{
	if(!src) return(dest);

	while(*src) *dest++ = *src++;
	*dest = '\0';

	return dest;
}


int iks_strcmp(const char *a, const char *b)
{
	if(!a || !b) return(-1);
	return strcmp(a, b);
}


int iks_strcasecmp(const char *a, const char *b)
{
	if(!a || !b) return(-1);
	return strcasecmp(a, b);
}


int iks_strncmp(const char *a, const char *b, int n)
{
	if(!a || !b) return(-1);
	return strncmp(a, b, n);
}


int iks_strncasecmp(const char *a, const char *b, int n)
{
	if(!a || !b) return(-1);
	return strncasecmp(a, b, n);
}


int iks_strlen(const char *src)
{
	if(!src) return(0);
	return (int)strlen(src);
}


/*****  String Stuff  *****/

ikstr *iks_str_new(ikspool *p)
{
	ikstr *s;
	int flag = 0;

	if(!p)
	{
		p = iks_pool_new(512);
		if(!p) return(NULL);
		flag = 1;
	}

	s = iks_pool_alloc(p, sizeof(ikstr));
	if(!s)
	{
		if(flag) iks_pool_delete(p);
		return NULL;
	}

	s->p = p;
	s->flag = flag;
	s->len = 0;
	s->first = NULL;
	s->last = NULL;

	return s;
}


void iks_str_add(ikstr *s, char *src)
{
	struct ikstrnode *sn;
	int len;

	if(!s || !src) return;

	len = strlen(src);
	if(len == 0) return;

	sn = iks_pool_alloc(s->p, sizeof(struct ikstrnode));
	if(!sn) return;
	sn->data = iks_pool_alloc(s->p, len);
	if(!sn->data) return;
	memcpy(sn->data, src, len);
	sn->len = len;

	s->len += len;
	sn->next = NULL;
	if(!s->first) s->first = sn;
	if(s->last) s->last->next = sn;
	s->last = sn;
}


void iks_spool(ikstr *s, ...)
{
	va_list ap;
	char *arg;

	va_start(ap, s);

	while(1)
	{
		arg = va_arg(ap, char *);
		if(((void *)arg) == ((void *)s)) break;
		iks_str_add(s, arg);
	}

	va_end(ap);
}


char *iks_str_print(ikstr *s)
{
	char *ret, *tmp;
	struct ikstrnode *sn;

	if(!s || !s->first) return(NULL);

	ret = iks_pool_alloc(s->p, s->len + 1);

	sn = s->first;
	tmp = ret;
	while(sn)
	{
		memcpy(tmp, sn->data, sn->len);
		tmp += sn->len;
		sn = sn->next;
	}
	*tmp = '\0';

	return ret;
}


void iks_str_delete(ikstr *s)
{
	if(s->flag) iks_pool_delete(s->p);
}


/*****  XML Escaping  *****/

char *iks_escape(ikspool *p, char *src, int len)
{
	char *ret;
	int i, j, nlen;

	if(!src || !p) return(NULL);

	if(len == -1) len = strlen(src);

	nlen = len;
	for(i=0; i<len; i++)
	{
		switch(src[i])
		{
		case '&': nlen += 4; break;
		case '<': nlen += 3; break;
		case '>': nlen += 3; break;
		case '\'': nlen += 5; break;
		case '"': nlen += 5; break;
		}
	}

	if(len == nlen) return(src);

	ret = iks_pool_alloc(p, nlen + 1);
	if(!ret) return(NULL);

	for(i=j=0; i<len; i++)
	{
		switch(src[i])
		{
		case '&': memcpy(&ret[j], "&amp;", 5); j += 5; break;
		case '\'': memcpy(&ret[j], "&apos;", 6); j += 6; break;
		case '"': memcpy(&ret[j], "&quot;", 6); j += 6; break;
		case '<': memcpy(&ret[j], "&lt;", 4); j += 4; break;
		case '>': memcpy(&ret[j], "&gt;", 4); j += 4; break;
		default:
			ret[j++] = src[i];
		}
	}
	ret[j] = '\0';

	return ret;
}


char *iks_unescape(ikspool *p, char *src, int len)
{
	int i,j;
	char *ret;

	if(!p || !src) return(NULL);
	if(!strchr(src, '&')) return(src);
	if(len == -1) len = strlen(src);

	ret = iks_pool_alloc(p, len + 1);
	if(!ret) return(NULL);

	for(i=j=0; i<len; i++)
	{
		if(src[i] == '&')
		{
			i++;
			if(strncmp(&src[i], "amp;", 4) == 0)
			{
				ret[j] = '&';
				i += 3;
			}
			else if(strncmp(&src[i], "quot;", 5) == 0)
			{
				ret[j] = '"';
				i += 4;
			}
			else if(strncmp(&src[i], "apos;", 5) == 0)
			{
				ret[j] = '\'';
				i += 4;
			}
			else if(strncmp(&src[i], "lt;", 3) == 0)
			{
				ret[j] = '<';
				i += 2;
			}
			else if(strncmp(&src[i], "gt;", 3) == 0)
			{
				ret[j] = '>';
				i += 2;
			}
			else
			{
				ret[j] = src[--i];
			}
		}
		else
		{
			ret[j] = src[i];
		}
		j++;
	}
	ret[j] = '\0';

	return ret;
}


/*****  Node Creating & Deleting  *****/

iks *iks_new(const char *name)
{
	ikspool *p;
	iks *x;

	p = iks_pool_new(1024);
	if(!p) return(NULL);

	x = iks_new_pool(p, name);
	if(!x) {
		iks_pool_delete(p);
		return NULL;
	}

	return x;
}


iks *iks_new_pool(ikspool *p, const char *name)
{
	iks *x;

	x = iks_pool_alloc(p, sizeof(iks));
	if(!x) return(NULL);
	memset(x, 0, sizeof(iks));

	x->p = p;
	x->type = IKS_TAG;
	if(name)
	{
		x->name = iks_pool_strdup(p, name);
		if(!x->name) return(NULL);
	}

	return x;
}


iks *iks_insert(iks *x, const char *name)
{
	iks *y;

	if(!x) return(NULL);

	y = iks_new_pool(x->p, name);
	if(!y) return(NULL);

	y->parent = x;
	if(!x->children) x->children=y;
	if(x->lastchild)
	{
		x->lastchild->next = y;
		y->prev = x->lastchild;
	}
	x->lastchild = y;

	return y;
}


iks *iks_insert_cdata(iks *x, const char *data, int len)
{
	iks *y;

	if(!x || !data) return(NULL);

	if(len == -1) len = strlen(data);

	y = x->lastchild;
	if(y && y->type == IKS_CDATA)
	{
		char *tmp;
		tmp = iks_pool_alloc(x->p, y->len + len + 1);
		if(!tmp) return(NULL);
		memcpy(tmp, y->cdata, y->len);
		memcpy(tmp + y->len, data, len);
		tmp[y->len + len] = '\0';
		y->cdata = tmp;
		y->len += len;
	}
	else
	{
		y = iks_insert(x, NULL);
		if(!y) return(NULL);
		y->type = IKS_CDATA;
		y->cdata = iks_pool_alloc(x->p, len + 1);
		if(!y->cdata) return(NULL);
		memcpy(y->cdata, data, len);
		y->cdata[len] = '\0';
		y->len = len;
	}

	return y;
}


iks *iks_insert_attrib(iks *x, const char *name, const char *value)
{
	iks *y;
	int len;

	if(!x || !name || !value) return(NULL);

	y = iks_new_pool(x->p, name);
	if(!y) return(NULL);

	len = strlen(value);
	y->type = IKS_ATTRIBUTE;
	y->cdata = iks_pool_alloc(x->p, len + 1);
	if(!y->cdata) return(NULL);
	memcpy(y->cdata, value, len);
	*(y->cdata + len) = '\0';
	y->len = len;

	y->parent = x;
	if(!x->attribs) x->attribs = y;
	if(x->lastattrib)
	{
		x->lastattrib->next = y;
		y->prev = x->lastattrib;
	}
	x->lastattrib = y;

	return y;
}


iks *iks_insert_node(iks *x, iks *y)
{
	y->parent = x;
	if(!x->children) x->children=y;
	if(x->lastchild)
	{
		x->lastchild->next = y;
		y->prev = x->lastchild;
	}
	x->lastchild = y;

	return y;
}


void iks_hide(iks *x)
{
	iks *y;

	if(!x) return;

	if(x->prev) x->prev->next = x->next;
	if(x->next) x->next->prev = x->prev;

	y = x->parent;
	if(y)
	{
		if(y->children == x) y->children = x->next;
		if(y->lastchild == x) y->lastchild = x->prev;
	}
}


void iks_delete(iks *x)
{
	if(x) iks_pool_delete(x->p);
}


/*****  Node Traversing  *****/

iks *iks_next(iks *x)
{
	if(x) return(x->next);
	return NULL;
}


iks *iks_prev(iks *x)
{
	if(x) return(x->prev);
	return NULL;
}


iks *iks_parent(iks *x)
{
	if(x) return(x->parent);
	return NULL;
}


iks *iks_child(iks *x)
{
	if(x) return(x->children);
	return NULL;
}


iks *iks_attrib(iks *x)
{
	if(x) return(x->attribs);
	return NULL;
}


iks *iks_find(iks *x, const char *name)
{
	iks *y;

	if(!x) return(NULL);

	/* FIX ME: for advanced searchs */

	y = x->children;
	while(y)
	{
		if(y->name && strcmp(y->name, name) == 0) return(y);
		y = y->next;
	}

	return NULL;
}


char *iks_find_cdata(iks *x, const char *name)
{
	iks *y;

	y = iks_find(x, name);
	if(!y) return(NULL);

	y = y->children;
	if(!y) return(NULL);

	return y->cdata;
}


char *iks_find_attrib(iks *x, const char *name)
{
	iks *y;

	if(!x) return(NULL);

	y = x->attribs;
	while(y)
	{
		if(iks_strcmp(y->name, name) == 0) return(y->cdata);
		y = y->next;
	}

	return NULL;
}


/*****  Node Information  *****/

ikspool *iks_pool(iks *x)
{
	if(x) return(x->p);
	return NULL;
}


enum ikstype iks_type(iks *x)
{
	if(x) return(x->type);
	return IKS_NONE;
}


char *iks_name(iks *x)
{
	if(x) return(x->name);
	return NULL;
}


char *iks_cdata(iks *x)
{
	if(x) return(x->cdata);
	return NULL;
}


int iks_cdata_size(iks *x)
{
	if(x) return(x->len);
	return 0;
}


int iks_has_children(iks *x)
{
	if(x && x->children) return(1);
	return 0;
}


int iks_has_attribs(iks *x)
{
	if(x && x->attribs) return(1);
	return 0;
}


/*****  Special  *****/

ikstr *iks_string(ikspool *p, iks *x)
{
	ikstr *s;
	int level=0, dir=0;
	iks *y;

	if(!x || x->type != IKS_TAG) return(NULL);

	s = iks_str_new(p);
	if(!s) return(NULL);
	p = s->p;

	while(1)
	{
		if(dir==0)
		{
			if(x->type == IKS_TAG)
			{
				iks *y;

				iks_str_add(s, "<");
				iks_str_add(s, x->name);
				y = x->attribs;
				while(y)
				{
					iks_spool(s, " ", y->name, "='", iks_escape(p, y->cdata, y->len), "'", s);
					y = y->next;
				}
				if(x->children)
				{
					iks_str_add(s, ">");
					x = x->children;
					level++;
					continue;
				}
				else
					iks_str_add(s, "/>");
			}
			else
				iks_str_add(s, iks_escape(p, x->cdata, x->len));
		}

		y = x->next;
		if(!y)
		{
			x = x->parent;
			level--;
			if(level >= 0)
			iks_spool(s, "</", x->name, ">", s);
			if(level < 1) break;
			dir = 1;
		}
		else
		{
			x = y;
			dir = 0;
		}
	}

	return s;
}


/*****  dom parsers  *****/

static void dom_tstart(iksparser *prs, char *tag, char **atts);
static void dom_tend(iksparser *prs, char *tag);
static void hook_cdata(iksparser *prs, char *data, int len);

iksparser *iks_sax_new(void *udata, iksTagStart *ts, iksTagEnd *te, iksCharData *cd)
{
	iksparser *prs;

	prs = malloc(sizeof(iksparser));
	if(!prs) return(NULL);
	memset(prs, 0, sizeof(iksparser));
#ifdef USE_EXPAT
	prs->expat = XML_ParserCreate(NULL);
	if(!prs->expat)
	{
		free(prs);
		return NULL;
	}
	XML_SetUserData(prs->expat, (void *)prs);
	XML_SetElementHandler(prs->expat, (XML_StartElementHandler)ts, (XML_EndElementHandler)te);
	XML_SetCharacterDataHandler(prs->expat, (XML_CharacterDataHandler)cd);
#endif
	prs->udata = udata;
	prs->udata2 = udata;
	prs->tagStart = ts;
	prs->tagEnd = te;
	prs->charData = cd;

	return prs;
}


iksparser *iks_dom_new(iks **iksptr)
{
	iksparser *prs;

	prs = iks_sax_new(NULL, (iksTagStart *)dom_tstart, (iksTagEnd *)dom_tend, (iksCharData *)hook_cdata);
	if(!prs) return(NULL);
	prs->udata = prs;
	prs->iksptr = iksptr;
	return prs;
}


#ifdef USE_EXPAT
int iks_parse(iksparser *prs, char *data, int len, int finish)
{
	if(XML_Parse(prs->expat, data, len, 0))
		return IKS_RET_OK;

	return IKS_RET_BADXML;
}
#endif


void iks_parser_delete(iksparser *prs)
{
#ifdef USE_EXPAT
	XML_ParserFree(prs->expat);
#endif
	free(prs);
}


/*****  Internal Hooks  *****/

static void insert_attribs(iksparser* p, iks *x, char **atts)
{
	int i=0;

	if(!x || !atts) return;
	while(atts[i])
	{
		iks_insert_attrib(x, atts[i], atts[i+1]);
		i += 2;
	}
}


static void dom_tstart(iksparser *prs, char *tag, char **atts)
{
	iks *x;

	/* FIX ME: handle PIs in parser */
	if(tag[0] == '?') return;

	if(prs->current)
		x = iks_insert(prs->current, tag);
	else
		x = iks_new(tag);
	insert_attribs(prs, x, atts);
	prs->current = x;
}


static void dom_tend(iksparser *prs, char *tag)
{
	iks *x;

	x = iks_parent(prs->current);
	if(x)
		prs->current = x;
	else
		*(prs->iksptr) = prs->current;
}


static void hook_cdata(iksparser *prs, char *data, int len)
{
	if(prs->current)
		iks_insert_cdata(prs->current, data, len);
}


/*****  sax parser  *****/

enum contexts
{
	C_CDATA = 0,
	C_TAG,
	C_ATTRIBUTE,
	C_VALUE,
	C_WHITESPACE,
	C_ENTITY,
	C_COMMENT,
	C_MARKUP,
	C_PI
};

static int parse_xml(iksparser *prs, char *buf, int len);


/* following function sucks, it is lame,
** but as long as it works, i don't care
** -madcat (ode to lazy coder) */

int iks_parse(iksparser *prs, char *data, int len, int finish)
{
	char *buf;
	int diff;

	if(!data || len == 0) return(IKS_RET_OK);

	if(len == -1) len = strlen(data);

	if(prs->buffer)
	{
		buf = malloc(prs->buflen + len);
		if(!buf) return(IKS_RET_NOMEM);
		memcpy(buf, prs->buffer, prs->buflen);
		memcpy(buf + prs->buflen, data, len);

		diff = (int)buf - (int)prs->buffer;
		if(prs->tagname) prs->tagname += diff;
		if(prs->atts)
		{
			int i = 0;
			while(i < (prs->attmax * 2))
			{
				if(prs->atts[i]) prs->atts[i] += diff;
				i++;
			}
		}
		free(prs->buffer);
		prs->buffer = buf;
		buf += prs->buflen;
		prs->buflen += len;
	}
	else
	{
		buf = malloc(len);
		if(!buf) return(IKS_RET_NOMEM);
		memcpy(buf, data, len);
		prs->buffer = buf;
		prs->buflen = len;
	}

	parse_xml(prs, buf, len);

	return IKS_RET_OK;
}


static int parse_xml(iksparser *prs, char *buf, int len)
{
	int pos = 0, old = 0;

	while(pos < len)
	{
		switch(prs->context)
		{
		case C_CDATA:
			if(buf[pos] == '&')
			{
				prs->context = C_ENTITY;
				buf[pos] = '\0';
				if(prs->charData) (prs->charData)(prs->udata, &buf[old], pos - old);
				prs->entpos = 0;
			}
			if(buf[pos] == '<')
			{
				if(old < pos)
				{
					buf[pos] = '\0';
					if(prs->charData) (prs->charData)(prs->udata, &buf[old], pos - old);
					old = pos + 1;
				}
				prs->minpos = pos;
				prs->oldcontext = C_CDATA;
				prs->context = C_TAG;
			}
			break;

		case C_TAG:
			if(!prs->tagname)
			{
				prs->tagname = buf + pos;
				if(buf[pos] == '/')
				{
					prs->tagflag = 2;
					prs->tagname++;
					break;
				}
			}
			if(buf[pos] == ' ' || buf[pos] == '\t' || buf[pos] == '\r' || buf[pos] == '\n')
			{
				buf[pos] = '\0';
				prs->oldcontext = C_ATTRIBUTE;
				prs->context = C_WHITESPACE;
				break;
			}
			if(buf[pos] == '/')
			{
				buf[pos] = '\0';
				prs->tagflag = 1;
			}
			if(buf[pos] == '>')
			{
				buf[pos] = '\0';
				if(prs->tagStart && prs->tagflag != 2) (prs->tagStart)(prs->udata, prs->tagname, NULL);
				if(prs->tagEnd && prs->tagflag) (prs->tagEnd)(prs->udata, prs->tagname);
				prs->tagname = NULL;
				prs->tagflag = 0;
				prs->context = C_CDATA;
				old = pos + 1;
				prs->minpos = -1;
			}
			break;

		case C_ATTRIBUTE:
			if(!prs->atts)
			{
				prs->attmax = 6;
				prs->atts = malloc(sizeof(char *) * 2 * 6);
				if(!prs->atts) return(0);
				memset(prs->atts, 0, sizeof(char *) * 2 * 6);
				prs->attcur = 0;
			}
			else
			{
				if(prs->attcur >= (prs->attmax * 2))
				{
					void *tmp;
					prs->attmax += 4;
					tmp = malloc(sizeof(char *) * 2 * prs->attmax);
					if(!tmp) return(0);
					memset(tmp, 0, sizeof(char *) * 2 * prs->attmax);
					memcpy(tmp, prs->atts, sizeof(char *) * (prs->attcur - 1));
					free(prs->atts);
					prs->atts = tmp;
				}
			}

			if(!prs->atts[prs->attcur])
			{
				prs->atts[prs->attcur] = buf + pos;
			}

			if(buf[pos] == '=')
			{
				buf[pos] = '\0';
				prs->context = C_VALUE;
				break;
			}

			if(buf[pos] == '/') prs->tagflag = 1;
			if(buf[pos] == '>')
			{
				buf[pos] = '\0';
				if(prs->atts) prs->atts[prs->attcur] = NULL;
				if(prs->tagStart) (prs->tagStart)(prs->udata, prs->tagname, prs->atts);
				free(prs->atts);
				prs->atts = NULL;
				prs->attcur = 0;
				if(prs->tagEnd && prs->tagflag) (prs->tagEnd)(prs->udata, prs->tagname);
				prs->tagname = NULL;
				prs->tagflag = 0;
				prs->context = C_CDATA;
				old = pos + 1;
			}
			break;

		case C_VALUE:
			if(buf[pos] == '\'' || buf[pos] == '"')
			{
				buf[pos] = '\0';
				if(prs->valflag)
				{
					prs->valflag = 0;
					prs->oldcontext = C_ATTRIBUTE;
					prs->context = C_WHITESPACE;
					prs->attcur += 2;
					break;
				}
				prs->valflag = 1;
				prs->atts[prs->attcur + 1] = buf + pos + 1;
			}
			break;

		case C_WHITESPACE:
			if(buf[pos] != ' ' && buf[pos] != '\t' && buf[pos] != '\r' && buf[pos] != '\n')
			{
				prs->context = prs->oldcontext;
				pos--;
			}
			break;

		case C_ENTITY:
			if(buf[pos] != ';')
			{
				prs->entity[prs->entpos++] = buf[pos];
			}
			else
			{
				char t = '?';
				prs->entity[prs->entpos] = '\0';
				if(strcmp(prs->entity, "amp") == 0)
					t = '&';
				else if(strcmp(prs->entity, "quot") == 0)
					t = '"';
				else if(strcmp(prs->entity, "apos") == 0)
					t = '\'';
				else if(strcmp(prs->entity, "lt") == 0)
					t = '<';
				else if(strcmp(prs->entity, "gt") == 0)
					t = '>';
				buf[pos] = t;
				old = pos;
				prs->context = C_CDATA;
			}
			break;

		case C_COMMENT:

		case C_MARKUP:

		case C_PI:
			break;

		}
		pos++;
	}

	if(prs->charData && prs->context == C_CDATA && old < pos)
	{
		(prs->charData)(prs->udata, &buf[old], pos - old);
	}

	return 1;
}


/*****  load&save  *****/

iks *iks_load(char *fname)
{
	iksparser *prs;
	iks *x = NULL;
	char *buf;
	FILE *f;
	int len, done = 0;

	if(!fname) return(NULL);

	buf = malloc(2048);
	if(buf)
	{
		prs = iks_dom_new(&x);
		if(prs)
		{
			f = fopen(fname, "r");
			if(f)
			{
				while(!done)
				{
					len = fread(buf, 1, 2048, f);
					if(len < 2048) done = 1;
					iks_parse(prs, buf, len, done);
				}
				fclose(f);
			}
			iks_parser_delete(prs);
		}
		free(buf);
	}

	return x;
}


int iks_save(char *fname, iks *x)
{
	FILE *f;
	ikstr *s;
	char *data;
	int ret;

	s = iks_string(NULL, x);
	data = iks_str_print(s);
	if(!data) return(0);

	f = fopen(fname, "w");
	if(!f)
	{
		iks_str_delete(s);
		return 0;
	}

	ret = fputs(data, f);
	iks_str_delete(s);
	fclose(f);

	if(ret < 0) return(0);
	return 1;
}
