/*
**  Sinek (Media Player)
**  Copyright (c) 2001-2002 Gurer Ozen
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  future library code
*/

#include "common.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/extensions/XShm.h>
#include <X11/Xatom.h>
#include <xine.h>
#include <xine/video_out_x11.h>

#ifndef XShmGetEventBase
extern int XShmGetEventBase(Display *);
#endif

static int MyShmEvent = -1;

static void frame_output_cb(void *this, int vid_w, int vid_h, int *dest_x, int *dest_y, int *dest_w, int *dest_h, int *win_x, int *win_y);
static void dest_size_cb(void *this, int vid_w, int vid_h, int *dest_w, int *dest_h);
extern int translate_point(int x, int y, int *vid_x, int *vid_y);


sinek_type *sinek_new(void)
{
	sinek_type *s;
	const gchar *home;
	gchar *tmp;

	s = malloc(sizeof(sinek_type));
	memset(s, 0, sizeof(sinek_type));

	home = g_get_home_dir();
	tmp = g_strconcat(home, "/.xine/config", NULL);
	s->conf = xine_config_file_init(tmp);
	g_free(tmp);

	return s;
}


void sinek_delete(sinek_type *s)
{
	if(s->xine) xine_exit(s->xine);
	free(s);
}


int sinek_audio_set_output(sinek_type *s)
{
	char *id, **driver_ids;

	driver_ids = xine_list_audio_output_plugins();
	id = s->conf->register_string(s->conf, "audio.driver", "auto", "audio driver to use", NULL, NULL, NULL);
	if(sinek.audio_id) id = sinek.audio_id;

	/* try user preferred driver */
	if(strncmp(id, "auto", 4) != 0 && strncmp(id, "null", 4) != 0)
		s->ad = xine_load_audio_output_plugin(s->conf, id);
	/* if that fails or user choosed auto, try any available driver */
	if(!s->ad && strncmp(id, "null", 4) != 0)
	{
		int i = 0;
		while(driver_ids[i])
		{
			s->ad = xine_load_audio_output_plugin(s->conf, driver_ids[i]);
			if(s->ad) break;
			i++;
		}
	}
	/* if that fails too or user choosed null, disable audio */
	if(!s->ad) id = "null";

	s->conf->update_string(s->conf, "audio.driver", id);

	s->xine = xine_init(s->vd, s->ad, s->conf);
	if(!s->xine)
	{
		printf(_("xine init error\n"));
		return 0;
	}

	return 1;
}


void sinek_video_event(sinek_type *s, void *event)
{
	XEvent *ev = (XEvent *)event;
	xine_input_event_t xinev;
	int x, y;

	switch(ev->type)
	{
		case MotionNotify:
			if(translate_point(((XMotionEvent *)ev)->x, ((XMotionEvent *)ev)->y, &x, &y))
			{
				xinev.event.type = XINE_EVENT_MOUSE_MOVE;
				xinev.button = 0;
				xinev.x = x;
				xinev.y = y;
				xine_send_event(s->xine, (xine_event_t *)&xinev);
			}
			break;

		case ButtonPress:
			if(((XButtonEvent *)ev)->button == 1 && translate_point(((XButtonEvent *)ev)->x, ((XButtonEvent *)ev)->y, &x, &y))
			{
				xinev.event.type = XINE_EVENT_MOUSE_BUTTON;
				xinev.button = 1;
				xinev.x = x;
				xinev.y = y;
				xine_send_event(s->xine, (xine_event_t *)&xinev);
			}
			break;

		case ConfigureNotify:
			s->display_info->win_x = ((XConfigureEvent *)ev)->x;
			s->display_info->win_y = ((XConfigureEvent *)ev)->y;
			s->display_info->win_w = ((XConfigureEvent *)ev)->width;
			s->display_info->win_h = ((XConfigureEvent *)ev)->height;
			break;

		case Expose:
			if(((XExposeEvent *)ev)->count == 0)
			{
				s->vd->gui_data_exchange(s->vd, GUI_DATA_EX_EXPOSE_EVENT, ev);
			}
			break;
	}
	if(ev->type == MyShmEvent)
		s->vd->gui_data_exchange(s->vd, GUI_DATA_EX_COMPLETION_EVENT, ev);
}


int sinek_video_set_output(sinek_type *s, void *display_info)
{
	sinek_x11_type *di = (sinek_x11_type *)display_info;
	char *id, **driver_ids;
	double res_h, res_v;
	x11_visual_t vis;

	s->display_info = di;

	if(XShmQueryExtension(di->display) == True)
		MyShmEvent = XShmGetEventBase(di->display) + ShmCompletion;

	vis.display = di->display;
	vis.screen = di->screen;
	vis.d = di->drawable;
	res_h = (DisplayWidth(vis.display, vis.screen)*1000 / DisplayWidthMM(vis.display, vis.screen));
	res_v = (DisplayHeight(vis.display, vis.screen)*1000 / DisplayHeightMM(vis.display, vis.screen));
	vis.display_ratio = res_h / res_v;
	vis.frame_output_cb = frame_output_cb;
	vis.dest_size_cb = dest_size_cb;
	vis.user_data = (void *)s;
	if(fabs(vis.display_ratio - 1.0) < 0.01) vis.display_ratio = 1.0;

	driver_ids = xine_list_video_output_plugins(VISUAL_TYPE_X11);
	id = s->conf->register_string(s->conf, "video.driver", "auto", "video driver to use", NULL, NULL, NULL);
	if(sinek.video_id) id = sinek.video_id;

	/* try user preferred driver */
	if(strncmp(id, "auto", 4) != 0)
		s->vd = xine_load_video_output_plugin(s->conf, id, VISUAL_TYPE_X11, (void *)&vis);
	/* if that fails or user choosed auto, try any available driver */
	if(!s->vd)
	{
		int i = 0;
		while(driver_ids[i])
		{
			s->vd = xine_load_video_output_plugin(s->conf, driver_ids[i], VISUAL_TYPE_X11, (void *)&vis);
			if(s->vd) break;
			i++;
		}
	}
	/* if that fails too, exit with an error */
	if(!s->vd)
	{
		printf(_("Cannot load video driver.\n"));
		return 0;
	}

	s->conf->update_string(s->conf, "video.driver", id);

	return 1;
}


static void frame_output_cb(void *this, int vid_w, int vid_h, int *dest_x, int *dest_y, int *dest_w, int *dest_h, int *win_x, int *win_y)
{
	sinek_x11_type *di = ((sinek_type *)this)->display_info;

	di->frame_w = vid_w;
	di->frame_h = vid_h;

	if(di->frame_out_cb)
	{
		di->frame_out_cb(di->user_data, di);
	}
	else
	{
		di->out_x = 0;
		di->out_y = 0;
		di->out_w = di->win_w;
		di->out_h = di->win_h;
	}

	*win_x = di->win_x;
	*win_y = di->win_y;
	*dest_x = di->out_x;
	*dest_y = di->out_y;
	*dest_w = di->out_w;
	*dest_h = di->out_h;
}


static void dest_size_cb(void *this, int vid_w, int vid_h, int *dest_w, int *dest_h)
{
	sinek_type *s = (sinek_type *)this;

	*dest_w = s->display_info->out_w;
	*dest_h = s->display_info->out_h;
}


int sinek_open(sinek_type *s, const char *mrl)
{
	if(s->mrl) free(s->mrl);
	s->mrl = strdup(mrl);
	return xine_play(s->xine, s->mrl, 0, 0);
}


void sinek_play(sinek_type *s)
{
	if(!s->mrl) return;

	if(xine_get_status(s->xine) != XINE_PLAY)
		xine_play(s->xine, s->mrl, 0, 0);
	else
		xine_set_speed(s->xine, SPEED_NORMAL);
}


void sinek_seek(sinek_type *s, unsigned long secs)
{
	if(s->mrl && xine_is_stream_seekable(s->xine))
		xine_play(s->xine, s->mrl, 0, secs);
}


void sinek_stop(sinek_type *s)
{
	xine_stop(s->xine);
}
