/*
**  Sinek (Media Player)
**  Copyright (c) 2001-2002 Gurer Ozen
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  preferences
*/

#include "common.h"

#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#endif

enum
{
	COL_NUM,
	COL_TITLE,
	N_COLS
};

static GtkWidget *notebook;
static GtkTooltips *tips;
static GList *page_list = NULL;

struct cfgitem
{
	char *label;
	cfg_entry_t *entry;
	GtkWidget *widget;
	GList *subitems;
};

static void print_usage(void);
static GtkWidget *build_page(GList *items);
static GtkWidget *build_page_list(void);
static void cb_ok(GtkWidget *w, GtkWidget *win);
static void cb_apply(GtkWidget *w, gpointer data);

static char *shortopts = "V:A:u:a:D:G:fbgrshv";
#ifdef HAVE_GETOPT_LONG
static struct option longopts[] =
{
	{ "video-driver", required_argument, 0, 'V' },
	{ "audio-driver", required_argument, 0, 'A' },
	{ "spu-channel", required_argument, 0, 'u' },
	{ "audio-channel", required_argument, 0, 'a' },
	{ "video-display", required_argument, 0, 'D' },
	{ "geometry", required_argument, 0, 'G' },
	{ "fullscreen", 0, 0, 'f' },
	{ "background", 0, 0, 'b' },
	{ "hide-gui", 0, 0, 'g' },
	{ "repeat", 0, 0, 'r' },
	{ "script", required_argument, 0, 's' },
	{ "help", 0, 0, 'h' },
	{ "version", 0, 0, 'v' },
	{ 0, 0, 0, 0 }
};
#endif


void prf_init(int argc, char *argv[])
{
	int c, i;

#ifdef HAVE_GETOPT_LONG
	while((c = getopt_long(argc, argv, shortopts, longopts, &i)) != -1)
#else
	while((c = getopt(argc, argv, shortopts)) != -1)
#endif
	{
		switch(c)
		{
			case 'A':
				sinek.audio_id = g_strdup(optarg);
				break;

			case 'V':
				sinek.video_id = g_strdup(optarg);
				break;

			case 'u':
				sinek.spu_channel = atoi(optarg);
				break;

			case 'a':
				sinek.audio_channel = atoi(optarg);
				break;

			case 'D':
				sinek.video_display_name = strdup(optarg);
				break;

			case 'G':
				if(sscanf(optarg, "%ix%i+%i+%i", &sinek.geo_w, &sinek.geo_h, &sinek.geo_x, &sinek.geo_y) == 4) break;
				sinek.geo_x = 0;
				sinek.geo_y = 0;
				if(sscanf(optarg, "%ix%i", &sinek.geo_w, &sinek.geo_h) == 2) break;
				printf(_("Invalid geometry argument '%s'.\n"), optarg);
				exit(1);

			case 'f':
				sinek.video_mode = VIDEO_FULLSCREEN;
				break;

			case 'b':
				sinek.video_mode = VIDEO_BACKGROUND;
				break;

			case 'g':
				sinek.hide_gui = 1;
				break;

			case 'r':
				sinek.repeat_one = -1;
				break;

			case 's':
				sinek.script = g_strdup(optarg);
				break;

			case 'h':
				print_usage();

			case 'v':
				print_version();
				exit(0);
		}
	}

	sinek.start_mrls = NULL;
	for(; optind < argc; optind++) {
		sinek.start_mrls = g_list_append(sinek.start_mrls,
				g_strdup(argv[optind]));
	}

	print_version();
}


void print_version(void)
{
	printf(_("Sinek v%s (Video Player)\n"), VERSION);
	printf(_("built with xine library v%d.%d.%d, using v%d.%d.%d.\n"),
		XINE_MAJOR_VERSION, XINE_MINOR_VERSION, XINE_SUB_VERSION,
		xine_get_major_version(), xine_get_minor_version(), xine_get_sub_version());
}


static void print_usage(void)
{
	puts(_("Usage: Sinek [OPTIONS] [MRL]"));
	puts(_("  -V, --video-driver <id>        Select video driver."));
	puts(_("  -A, --audio-driver <id>        Select audio driver."));
	puts(_("  -u, --spu-channel <no>         Select SPU subtitle channel."));
	puts(_("  -a, --audio-channel <no>       Select Audio channel."));
	puts(_("  -d, --display <DISPLAY>        X display to use for user interface."));
	puts(_("  -D, --video-display <DISPLAY>  X display to use for video output."));
	puts(_("  -G, --geometry <WxH[+X+Y]>     Set video window coordinates."));
	puts(_("  -f, --fullscreen               Start in fullscreen mode."));
	puts(_("  -b, --background               Play the video in background."));
	puts(_("  -g, --hide-gui                 Hide control panel."));
	puts(_("  -r, --repeat                   Play mrl continously."));
#ifdef USE_GUILE
	puts(_("  -s, --script <file>            Execute scheme script."));
#endif
	puts(_("  -h, --help                     Print this help text and exit."));
	puts(_("  -v, --version                  Print version number and exit."));
#ifndef HAVE_GETOPT_LONG
	puts(_("(long options are not supported on your system)"));
#endif
	puts(_("Report bugs to <sinek-user@lists.sourceforge.net>."));
	exit(0);
}


int prf_build(GtkWidget **win)
{
	GtkWidget *w, *vb, *hb, *bb, *b;
	GList *page;

	w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(w), _("Sinek Settings"));
	gtk_window_set_wmclass(GTK_WINDOW(w), "settings", "sinek");

	tips = gtk_tooltips_new();

	vb = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vb);
	gtk_container_add(GTK_CONTAINER(w), vb);

	hb = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(hb), build_page_list(), FALSE, TRUE, 0);

	notebook = gtk_notebook_new();
	gtk_widget_show(notebook);
	gtk_box_pack_start(GTK_BOX(hb), notebook, TRUE, TRUE, 0);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);
	page = g_list_first(page_list);
	while(page)
	{
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), build_page(page), NULL);
		page = g_list_next(page);
	}

	bb = gtk_hbutton_box_new();
	gtk_widget_show(bb);
	gtk_box_pack_start(GTK_BOX(vb), bb, FALSE, FALSE, 0);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bb), GTK_BUTTONBOX_END);
	gtk_container_set_border_width(GTK_CONTAINER(bb), 5);
	gtk_box_set_spacing(GTK_BOX(bb), 5);

	b = gtk_button_new_from_stock(GTK_STOCK_APPLY);
	gtk_widget_show(b);
	gtk_container_add(GTK_CONTAINER(bb), b);
	g_signal_connect(G_OBJECT(b), "clicked", G_CALLBACK(cb_apply), NULL);

	b = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_widget_show(b);
	gtk_container_add(GTK_CONTAINER(bb), b);
	g_signal_connect_swapped(G_OBJECT(b), "clicked", G_CALLBACK(gtk_widget_hide), GTK_OBJECT(w));

	b = gtk_button_new_from_stock(GTK_STOCK_OK);
	gtk_widget_show(b);
	gtk_container_add(GTK_CONTAINER(bb), b);
	g_signal_connect(G_OBJECT(b), "clicked", G_CALLBACK(cb_ok), (gpointer)w);

	*win = w;
	return WM_NORMAL;
}


static void prf_apply_changes(void)
 {
	GList *page, *item;
	struct cfgitem *citem;
	const gchar *str;
	int val = 0;

	page = g_list_first(page_list);
	while(page)
	{
		item = g_list_first(((struct cfgitem *)page->data)->subitems);
		while(item)
		{
			citem = (struct cfgitem *)item->data;
			switch(citem->entry->type)
			{
				case CONFIG_TYPE_STRING:
					str = gtk_entry_get_text(GTK_ENTRY(citem->widget));
					media->conf->update_string(media->conf, citem->entry->key, (char *)str);
					break;
				case CONFIG_TYPE_BOOL:
					val = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(citem->widget));
					media->conf->update_num(media->conf, citem->entry->key, val);
					break;
				case CONFIG_TYPE_ENUM:
					val = gtk_option_menu_get_history(GTK_OPTION_MENU(citem->widget));
					media->conf->update_num(media->conf, citem->entry->key, val);
					break;
				case CONFIG_TYPE_NUM:
					val = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(citem->widget));
					media->conf->update_num(media->conf, citem->entry->key, val);
					break;
				case CONFIG_TYPE_RANGE:
					val = ((GtkAdjustment *)citem->widget)->value;
					media->conf->update_num(media->conf, citem->entry->key, val);
					break;
			}
			item = g_list_next(item);
		}
		page = g_list_next(page);
	}
}


static void cb_ok(GtkWidget *w, GtkWidget *win)
{
	prf_apply_changes();
	media->conf->save(media->conf);
	gtk_widget_hide(win);
}


static void cb_apply(GtkWidget *w, gpointer data)
{
	prf_apply_changes();
}


static GtkWidget *build_string(struct cfgitem *citem)
{
	GtkWidget *box, *t;

	box = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 2);
	t = gtk_label_new(citem->label);
	gtk_widget_show(t);
	gtk_box_pack_start(GTK_BOX(box), t, FALSE, FALSE, 0);
	t = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(t), citem->entry->str_value);
	gtk_widget_show(t);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), t, citem->entry->description, NULL);
	gtk_box_pack_start(GTK_BOX(box), t, TRUE, TRUE, 0);
	citem->widget = t;

	return box;
}


static GtkWidget *build_checkbox(struct cfgitem *citem)
{
	GtkWidget *t;

	t = gtk_check_button_new_with_label(citem->label);
	gtk_widget_show(t);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(t), citem->entry->num_value);
	citem->widget = t;
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), t, citem->entry->description, NULL);

	return t;
}


static GtkWidget *build_option(struct cfgitem *citem)
{
	GtkWidget *box, *opt, *m, *mi;
	char **enums;
	int i = 0;

	box = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 2);
	m = gtk_label_new(citem->label);
	gtk_widget_show(m);
	gtk_box_pack_start(GTK_BOX(box), m, FALSE, FALSE, 0);

	enums = citem->entry->enum_values;
	opt = gtk_option_menu_new();
	gtk_widget_show(opt);
	m = gtk_menu_new();
	while(enums[i])
	{
		mi = gtk_menu_item_new_with_label(enums[i]);
		gtk_widget_show(mi);
		gtk_menu_shell_append(GTK_MENU_SHELL(m), mi);
		i++;
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(opt), m);
	gtk_option_menu_set_history(GTK_OPTION_MENU(opt), citem->entry->num_value);
	gtk_box_pack_start(GTK_BOX(box), opt, TRUE, TRUE, 0);
	citem->widget = opt;
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), opt, citem->entry->description, NULL);

	return box;
}


static GtkWidget *build_numeric(struct cfgitem *citem)
{
	GtkObject *adj;
	GtkWidget *box, *lab, *spin;

	box = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 2);
	lab = gtk_label_new(citem->label);
	gtk_widget_show(lab);
	gtk_box_pack_start(GTK_BOX(box), lab, FALSE, FALSE, 0);

	adj = gtk_adjustment_new(citem->entry->num_value, INT_MIN, INT_MAX, 1, 0, 0);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
	gtk_widget_show(spin);
	gtk_box_pack_start(GTK_BOX(box), spin, TRUE, TRUE, 0);
	citem->widget = spin;
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), spin, citem->entry->description, NULL);

	return box;
}


static GtkWidget *build_scale(struct cfgitem *citem)
{
	GtkObject *adj;
	GtkWidget *box, *lab, *scale;

	box = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 2);
	lab = gtk_label_new(citem->label);
	gtk_widget_show(lab);
	gtk_box_pack_start(GTK_BOX(box), lab, FALSE, FALSE, 0);

	adj = gtk_adjustment_new(citem->entry->num_value, citem->entry->range_min, citem->entry->range_max, 1, 0, 0);
	scale = gtk_hscale_new(GTK_ADJUSTMENT(adj));
	gtk_widget_show(scale);
	gtk_scale_set_digits(GTK_SCALE(scale), 0);
	gtk_box_pack_start(GTK_BOX(box), scale, TRUE, TRUE, 0);
	citem->widget = (GtkWidget *)adj;
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), scale, citem->entry->description, NULL);

	return box;
}


static GtkWidget *build_page(GList *page)
{
	GtkWidget *sw, *box, *w;
	GList *item;
	struct cfgitem *citem;
	cfg_entry_t *entry;

	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	box = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 10);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), box);

	item = g_list_first(((struct cfgitem *)page->data)->subitems);
	while(item)
	{
		citem = (struct cfgitem *)item->data;
		entry = citem->entry;
		if(entry->description)
		{
			switch(entry->type)
			{
				case CONFIG_TYPE_STRING: w = build_string(citem); break;
				case CONFIG_TYPE_BOOL: w = build_checkbox(citem); break;
				case CONFIG_TYPE_ENUM: w = build_option(citem); break;
				case CONFIG_TYPE_NUM: w = build_numeric(citem); break;
				case CONFIG_TYPE_RANGE: w = build_scale(citem); break;
				default: w = NULL;
			}
			if(w) gtk_box_pack_start(GTK_BOX(box), w, FALSE, FALSE, 0);
		}
		item = g_list_next(item);
	}

	return sw;
}


static void cb_page(GtkTreeSelection *sel, gpointer data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	int row;

	if(gtk_tree_selection_get_selected(sel, &model, &iter))
	{
		gtk_tree_model_get(model, &iter, COL_NUM, &row, -1);
		gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), row);
	}
}


static GtkWidget *build_page_list(void)
{
	GtkWidget *list_view;
	GtkListStore *list_store;
	GtkCellRenderer *cell;
	GtkTreeViewColumn *col;
	GtkTreeIter iter;
	GList *item;
	struct cfgitem *cpage, *citem;
	cfg_entry_t *entry;
	gchar *p;
	int len, num = 0;

	list_store = gtk_list_store_new(N_COLS, G_TYPE_INT, G_TYPE_STRING);
	list_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(list_store));
	cell = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new_with_attributes(_("Category"), cell, "text", COL_TITLE, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), col);
	gtk_widget_show(list_view);

	entry = media->conf->first;
	while(entry)
	{
		p = strchr(entry->key, '.');
		if(p && entry->description)
		{
			len = p - entry->key;
			cpage = NULL;
			item = g_list_first(page_list);
			while(item)
			{
				if(strncmp(((struct cfgitem *)item->data)->label, entry->key, len) == 0)
				{
					cpage = (struct cfgitem *)item->data;
					break;
				}
				item = g_list_next(item);
			}
			if(!cpage)
			{
				cpage = g_malloc0(sizeof(struct cfgitem));
				cpage->label = g_strndup(entry->key, len);
				gtk_list_store_append(list_store, &iter);
				gtk_list_store_set(list_store, &iter, COL_NUM, num, COL_TITLE, (gpointer)cpage->label, -1);
				num++;
				page_list = g_list_append(page_list, (gpointer)cpage);
			}
			citem = g_malloc0(sizeof(struct cfgitem));
			citem->label = g_strdup(entry->key + len + 1);
			citem->entry = entry;
			cpage->subitems = g_list_append(cpage->subitems, (gpointer)citem);
		}
		entry = entry->next;
	}
	g_signal_connect(G_OBJECT(gtk_tree_view_get_selection(GTK_TREE_VIEW(list_view))), "changed", G_CALLBACK(cb_page), NULL);

	return list_view;
}
