/*
**  Sinek (Media Player)
**  Copyright (c) 2001-2002 Gurer Ozen
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  key bindings
*/

#include "common.h"

typedef struct action_struct
{
	char *name;
	char *label;
	cmd_type cmd;
} action;

static action actions[] =
{
	{ "play", N_("Play"), CMD_PLAY },
	{ "pause", N_("Pause"), CMD_PAUSE },
	{ "stop", N_("Stop"), CMD_STOP },
	{ "next", N_("Next"), CMD_NEXT },
	{ "previous", N_("Previous"), CMD_PREVIOUS },
	{ "volume-up", N_("Increase volume"), CMD_VOLUME_UP },
	{ "volume-down", N_("Decrease volume"), CMD_VOLUME_DOWN },
	{ "speed-fast", N_("Increase playback speed"), CMD_SPEED_FAST },
	{ "speed-slow", N_("Decrease playback speed"), CMD_SPEED_SLOW },
	{ "av-offset-down", N_("Speed up video (if it lags behind)"), CMD_AV_OFFSET_DOWN },
	{ "av-offset-up", N_("Slow down video (if audio lags behind)"), CMD_AV_OFFSET_UP },
	{ "toggle-fullscreen", N_("Toggle fullscreen mode"), CMD_TOGGLE_FULLSCREEN },
	{ "zoom-out", N_("Zoom out"), CMD_ZOOM_OUT },
	{ "zoom-in", N_("Zoom in"), CMD_ZOOM_IN },
	{ "zoom-reset", N_("Reset zoom"), CMD_ZOOM_RESET },
	{ "toggle-aspect", N_("Change video aspect"), CMD_TOGGLE_ASPECT },
	{ "toggle-interleave", N_("Toggle interleave"), CMD_TOGGLE_INTERLEAVE },
	{ "toggle-tvmode", N_("Toggle TV mode of DXR3"), CMD_TOGGLE_TVMODE },
	{ "window-1x1", N_("1x1 video size"), CMD_WINDOW_1x1 },
	{ "window-2x2", N_("2x2 video size"), CMD_WINDOW_2x2 },
	{ "window-fullscreen", N_("Fullscreen video"), CMD_WINDOW_FULLSCREEN },
	{ "window-background", N_("Display video in background"), CMD_WINDOW_BACKGROUND },
	{ "window-reduce", N_("Reduce video window"), CMD_WINDOW_REDUCE },
	{ "window-enlarge", N_("Enlarge video window"), CMD_WINDOW_ENLARGE },
	{ "audio-mute", N_("Toggle audio muting"), CMD_TOGGLE_MUTE },
	{ "audio-next", N_("Next audio channel"), CMD_AUDIO_NEXT },
	{ "audio-previous", N_("Previous audio channel"), CMD_AUDIO_PREVIOUS },
	{ "show-playlist", N_("Show playlist window"), CMD_SHOW_PLAYLIST },
	{ "show-tuner", N_("Show tuner window"), CMD_SHOW_TUNER },
	{ "show-about", N_("Show about window"), CMD_SHOW_ABOUT },
	{ "show-controls", N_("Show controls window"), CMD_SHOW_CONTROLS },
	{ "show-video", N_("Show video window"), CMD_SHOW_VIDEO },
	{ "toggle-gui", N_("Toggle all windows"), CMD_TOGGLE_GUI },
	{ "quit", N_("Quit"), CMD_QUIT },
	{ "event-menu1", N_("Menu 1"), CMD_EVENT_MENU1 },
	{ "event-menu2", N_("Menu 2"), CMD_EVENT_MENU2 },
	{ "event-menu3", N_("Menu 3"), CMD_EVENT_MENU3 },
	{ "event-up", N_("Menu up"), CMD_EVENT_UP },
	{ "event-down", N_("Menu down"), CMD_EVENT_DOWN },
	{ "event-left", N_("Menu left"), CMD_EVENT_LEFT },
	{ "event-right", N_("Menu right"), CMD_EVENT_RIGHT },
	{ "event-next", N_("Menu next"), CMD_EVENT_NEXT },
	{ "event-prior", N_("Menu previous"), CMD_EVENT_PRIOR },
	{ "event-angle-next", N_("Menu next angle"), CMD_EVENT_ANGLE_NEXT },
	{ "event-angle-prior", N_("Menu previous angle"), CMD_EVENT_ANGLE_PRIOR },
	{ "event-select", N_("Menu select"), CMD_EVENT_SELECT },
	{ "spu-next", N_("Next subtitle channel"), CMD_SPU_NEXT },
	{ "spu-previous", N_("Previous subtitle channel"), CMD_SPU_PREVIOUS },
	{ "mark-goto-0", N_("Go to mark 0"), CMD_GOTO_MARK_0 },
	{ "mark-goto-1", N_("Go to mark 1"), CMD_GOTO_MARK_1 },
	{ "mark-goto-2", N_("Go to mark 2"), CMD_GOTO_MARK_2 },
	{ "mark-goto-3", N_("Go to mark 3"), CMD_GOTO_MARK_3 },
	{ "mark-goto-4", N_("Go to mark 4"), CMD_GOTO_MARK_4 },
	{ "mark-goto-5", N_("Go to mark 5"), CMD_GOTO_MARK_5 },
	{ "mark-goto-6", N_("Go to mark 6"), CMD_GOTO_MARK_6 },
	{ "mark-goto-7", N_("Go to mark 7"), CMD_GOTO_MARK_7 },
	{ "mark-goto-8", N_("Go to mark 8"), CMD_GOTO_MARK_8 },
	{ "mark-goto-9", N_("Go to mark 9"), CMD_GOTO_MARK_9 },
	{ "mark-set-1", N_("Set mark 1"), CMD_SET_MARK_1 },
	{ "mark-set-2", N_("Set mark 2"), CMD_SET_MARK_2 },
	{ "mark-set-3", N_("Set mark 3"), CMD_SET_MARK_3 },
	{ "mark-set-4", N_("Set mark 4"), CMD_SET_MARK_4 },
	{ "mark-set-5", N_("Set mark 5"), CMD_SET_MARK_5 },
	{ "mark-set-6", N_("Set mark 6"), CMD_SET_MARK_6 },
	{ "mark-set-7", N_("Set mark 7"), CMD_SET_MARK_7 },
	{ "mark-set-8", N_("Set mark 8"), CMD_SET_MARK_8 },
	{ "mark-set-9", N_("Set mark 9"), CMD_SET_MARK_9 },
	{ NULL, NULL, CMD_NONE }
};

#define MOD_SHIFT 1
#define MOD_CONTROL 4
#define MOD_ALT 8

static struct defkeys_struct
{
	cmd_type cmd;
	char *key;
	int modifier;
} defkeys[] =
{
	{ CMD_PLAY, "Return", 0 },
	{ CMD_PAUSE, "space", 0 },
	{ CMD_STOP, "s", 0 },
	{ CMD_NEXT, "Next", 0 },
	{ CMD_PREVIOUS, "Prior", 0 },
	{ CMD_SPEED_FAST, "Up", 0 },
	{ CMD_SPEED_SLOW, "Down", 0 },
	{ CMD_AV_OFFSET_DOWN, "n", 0},
	{ CMD_AV_OFFSET_UP, "m", 0},
	{ CMD_TOGGLE_FULLSCREEN, "f", 0 },
	{ CMD_ZOOM_OUT, "z", 0 },
	{ CMD_ZOOM_IN, "z", MOD_SHIFT },
	{ CMD_ZOOM_RESET, "x", 0 },
	{ CMD_TOGGLE_ASPECT, "a", 0 },
	{ CMD_TOGGLE_INTERLEAVE, "i", 0 },
	{ CMD_TOGGLE_TVMODE, "o", MOD_CONTROL },
	{ CMD_WINDOW_1x1, "1", MOD_CONTROL },
	{ CMD_WINDOW_2x2, "2", MOD_CONTROL },
	{ CMD_WINDOW_FULLSCREEN, "3", MOD_CONTROL },
	{ CMD_WINDOW_BACKGROUND, "4", MOD_CONTROL },
	{ CMD_WINDOW_REDUCE, "less", 0 },
	{ CMD_WINDOW_ENLARGE, "greater", 0 },
	{ CMD_TOGGLE_MUTE, "BackSpace", 0 },
	{ CMD_AUDIO_NEXT, "plus", 0 },
	{ CMD_AUDIO_PREVIOUS, "minus", 0 },
	{ CMD_SHOW_PLAYLIST, "p", 0 },
	{ CMD_SHOW_TUNER, "t", 0 },
	{ CMD_SHOW_ABOUT, "question", 0 },
	{ CMD_SHOW_CONTROLS, "c", 0 },
	{ CMD_SHOW_VIDEO, "v", 0 },
	{ CMD_TOGGLE_GUI, "g", 0 },
	{ CMD_QUIT, "Escape", 0 },
	{ CMD_EVENT_MENU1, "F1", 0 },
	{ CMD_EVENT_MENU2, "F2", 0 },
	{ CMD_EVENT_MENU3, "F3", 0 },
	{ CMD_EVENT_UP, "KP_Up", 0 },
	{ CMD_EVENT_DOWN, "KP_Down", 0 },
	{ CMD_EVENT_LEFT, "KP_Left", 0 },
	{ CMD_EVENT_RIGHT, "KP_Right", 0 },
	{ CMD_EVENT_NEXT, "KP_Next", 0 },
	{ CMD_EVENT_PRIOR, "KP_Prior", 0 },
	{ CMD_EVENT_ANGLE_NEXT, "KP_1", 0 },
	{ CMD_EVENT_ANGLE_PRIOR, "KP_7", 0 },
	{ CMD_EVENT_SELECT, "KP_Enter", 0 },
	{ CMD_SPU_NEXT, "period", 0 },
	{ CMD_SPU_PREVIOUS, "comma", 0 },
	{ CMD_GOTO_MARK_0, "0", 0 },
	{ CMD_GOTO_MARK_1, "1", 0 },
	{ CMD_GOTO_MARK_2, "2", 0 },
	{ CMD_GOTO_MARK_3, "3", 0 },
	{ CMD_GOTO_MARK_4, "4", 0 },
	{ CMD_GOTO_MARK_5, "5", 0 },
	{ CMD_GOTO_MARK_6, "6", 0 },
	{ CMD_GOTO_MARK_7, "7", 0 },
	{ CMD_GOTO_MARK_8, "8", 0 },
	{ CMD_GOTO_MARK_9, "9", 0 },
	{ CMD_SET_MARK_1, "1", MOD_SHIFT },
	{ CMD_SET_MARK_2, "2", MOD_SHIFT },
	{ CMD_SET_MARK_3, "3", MOD_SHIFT },
	{ CMD_SET_MARK_4, "4", MOD_SHIFT },
	{ CMD_SET_MARK_5, "5", MOD_SHIFT },
	{ CMD_SET_MARK_6, "6", MOD_SHIFT },
	{ CMD_SET_MARK_7, "7", MOD_SHIFT },
	{ CMD_SET_MARK_8, "8", MOD_SHIFT },
	{ CMD_SET_MARK_9, "9", MOD_SHIFT },
	{ CMD_NONE, NULL, 0 }
};

typedef struct binding_struct
{
	struct binding_struct *next, *prev;
	char *key;
	int modifier;
	struct action_struct *action;
} binding;

static binding *bindings, *last_binding;

static GtkWidget *key_win;
static GtkWidget *action_wid, *key_wid, *clist_wid;
static int key_row;

static action *find_action_by_event(char *key, int modifier);
static action *find_action_by_name(char *action);
static int load_keys(void);
static void reset_keys(void);
static char *key_name(char *key, int modifier);
void keywin_init(void);


void key_init(void)
{
	int i;

	for(i = 0; actions[i].name; i++)
	{
		actions[i].label = gettext(actions[i].label);
	}

	keywin_init();
	if(!load_keys()) reset_keys();
	gtk_clist_select_row(GTK_CLIST(clist_wid), 0, 0);
}


gboolean key_handle(unsigned long keyval, int modifier)
{
	char *key;
	action *act;

	modifier &= (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK);
	key = gdk_keyval_name(gdk_keyval_to_lower(keyval));
	if(!key) return(FALSE);
	act = find_action_by_event(key, modifier);
	if(!act) return(FALSE);
	execute_cmd(act->cmd);
	return TRUE;
}


cmd_type find_cmd(char *command)
{
	action *act;

	act = find_action_by_name(command);
	if(act) return(act->cmd);
	return CMD_NONE;
}


static action *find_action_by_event(char *key, int modifier)
{
	binding *kb, *tmp = NULL;

	for(kb = bindings; kb; kb = kb->next)
	{
		if(strcasecmp(kb->key, key) == 0)
		{
			if(kb->modifier == modifier) return(kb->action);
			tmp = kb;
		}
	}
	if(tmp && tmp->modifier == 0) return(tmp->action);
	return NULL;
}


static action *find_action_by_name(char *action)
{
	int i;

	for(i = 0; actions[i].name; i++)
	{
		if(strcasecmp(action, actions[i].name) == 0)
			return &actions[i];
	}
	return NULL;
}


static action *find_action_by_cmd(cmd_type cmd)
{
	int i;

	for(i = 0; actions[i].name; i++)
	{
		if(actions[i].cmd == cmd) return(&actions[i]);
	}
	return NULL;
}


static int find_action_no(action *act)
{
	int i;

	for(i = 0; actions[i].name; i++)
	{
		if(act == &actions[i]) return(i);
	}
	return 0;
}


int key_add(action *act, char *key, int modifier)
{
	char *buf[3];
	binding *kb;
	int row;

	kb = malloc(sizeof(binding));
	memset(kb, 0, sizeof(binding));
	kb->action = act;
	kb->key = key;
	kb->modifier = modifier;

	if(!bindings) bindings = kb;
	if(last_binding)
	{
		last_binding->next = kb;
		kb->prev = last_binding;
	}
	last_binding = kb;

	buf[0] = act->label;
	buf[1] = strdup(key_name(key, modifier));
	row = gtk_clist_append(GTK_CLIST(clist_wid), buf);
	gtk_clist_set_row_data(GTK_CLIST(clist_wid), row, (gpointer)kb);
	gtk_clist_columns_autosize(GTK_CLIST(clist_wid));

	return row;
}


void key_remove(binding *kb)
{
	gtk_clist_remove(GTK_CLIST(clist_wid), gtk_clist_find_row_from_data(GTK_CLIST(clist_wid), (gpointer)kb));
	if(bindings == kb) bindings = kb->next;
	if(last_binding == kb) last_binding = kb->prev;
	if(kb->prev) kb->prev->next = kb->next;
	if(kb->next) kb->next->prev = kb->prev;
	free(kb);
}


static void clear_keys(void)
{
	binding *t, *t2;

	for(t = bindings; t; t = t2)
	{
		t2 = t->next;
		key_remove(t);
	}
}


static void reset_keys(void)
{
	int i;

	gtk_clist_freeze(GTK_CLIST(clist_wid));
	clear_keys();
	for(i = 0; defkeys[i].cmd != CMD_NONE; i++)
	{
		key_add(find_action_by_cmd(defkeys[i].cmd), defkeys[i].key, defkeys[i].modifier);
	}
	gtk_clist_thaw(GTK_CLIST(clist_wid));
	gtk_clist_select_row(GTK_CLIST(clist_wid), 0, 0);
}


static int load_keys(void)
{
	action *act;
	char *key;
	int modifier;
	char *fname;
	iks *x, *y;

	fname = g_strconcat(g_get_home_dir(), "/.xine/keys.xml", NULL);
	x = iks_load(fname);
	g_free(fname);
	if(!x || iks_strcmp(iks_name(x), "skb") != 0) return(0);

	for(y = iks_child(x); y; y = iks_next(y))
	{
		if(iks_type(y) == IKS_TAG && iks_strcmp(iks_name(y), "binding") == 0)
		{
			act = find_action_by_name(iks_find_attrib(y, "action"));
			key = iks_strdup(iks_find_cdata(y, "key"));
			modifier = 0;
			if(iks_find(y, "shift")) modifier += MOD_SHIFT;
			if(iks_find(y, "control")) modifier += MOD_CONTROL;
			if(iks_find(y, "alt")) modifier += MOD_ALT;
			key_add(act, key, modifier);
		}
	}
	iks_delete(x);
	return 1;
}


static void save_keys(void)
{
	char *fname;
	iks *x, *y;
	binding *kb;

	fname = g_strconcat(g_get_home_dir(), "/.xine/keys.xml", NULL);
	x = iks_new("skb");
	iks_insert_cdata(x, "\n", 1);
	for(kb = bindings; kb; kb = kb->next)
	{
		y = iks_insert(x, "binding");
		iks_insert_attrib(y, "action", kb->action->name);
		iks_insert_cdata(y, "\n\t", 2);
		iks_insert_cdata(iks_insert(y, "key"), kb->key, -1);
		if(kb->modifier)
		{
			iks_insert_cdata(y, "\n\t", 2);
			if(kb->modifier & MOD_SHIFT) iks_insert(y, "shift");
			if(kb->modifier & MOD_CONTROL) iks_insert(y, "control");
			if(kb->modifier & MOD_ALT) iks_insert(y, "alt");
			iks_insert_cdata(y, "\n", 1);
		}
		else
		{
			iks_insert_cdata(y, "\n", 1);
		}
		iks_insert_cdata(x, "\n", 1);
	}
	iks_save(fname, x);
	iks_delete(x);
	g_free(fname);
	gtk_widget_hide(key_win);
}


static char *key_name(char *key, int modifier)
{
	static char buf[256];
	char *t;

	buf[0] = '\0';
	t = buf;
	if(modifier & MOD_ALT)
	{
		memcpy(t, "Alt ", 4);
		t += 4;
	}
	if(modifier & MOD_CONTROL)
	{
		memcpy(t, "Control ", 8);
		t += 8;
	}
	if(modifier & MOD_SHIFT)
	{
		memcpy(t, "Shift ", 6);
		t += 6;
	}
	strcpy(t, key);

	return buf;
}


static void cb_add_binding(void)
{
	int row;

	row = key_add(&actions[0], "", 0);
	gtk_clist_select_row(GTK_CLIST(clist_wid), row, 0);
	gtk_clist_moveto(GTK_CLIST(clist_wid), row, -1, 1.0, 0);
}


static void cb_remove_binding(void)
{
	binding *kb;

	kb = gtk_clist_get_row_data(GTK_CLIST(clist_wid), key_row);
	key_remove(kb);
}


static void cb_select_binding(GtkWidget *w, gint row, gint col, GdkEventButton *ev, gpointer data)
{
	binding *kb;

	key_row = row;
	kb = gtk_clist_get_row_data(GTK_CLIST(clist_wid), row);
	gtk_list_select_item(GTK_LIST(GTK_COMBO(action_wid)->list), find_action_no(kb->action));
	gtk_entry_set_text(GTK_ENTRY(key_wid), key_name(kb->key, kb->modifier));
}


static gboolean cb_change_key(GtkWidget *w, GdkEventKey *ev, gpointer data)
{
	binding *kb;
	char *keyname;
	int modifier;

	keyname = gdk_keyval_name(gdk_keyval_to_lower(ev->keyval));
	modifier = ev->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK);
	kb = gtk_clist_get_row_data(GTK_CLIST(clist_wid), key_row);
	kb->key = keyname;
	kb->modifier = modifier;
	keyname = key_name(keyname, modifier);
	gtk_entry_set_text(GTK_ENTRY(w), keyname);
	gtk_clist_set_text(GTK_CLIST(clist_wid), key_row, 1, keyname);
	return TRUE;
}


static void cb_change_action(GtkList *list, GtkWidget *child)
{
	binding *kb;
	int i;

	kb = gtk_clist_get_row_data(GTK_CLIST(clist_wid), key_row);
	i = gtk_list_child_position(list, child);
	if(i != -1)
	{
		kb->action = &actions[i];
		gtk_clist_set_text(GTK_CLIST(clist_wid), key_row, 0, kb->action->label);
	}
}


void keywin_init(void)
{
	GtkWidget *vb, *hb, *sw, *t;
	GList *acts = NULL;
	char *titles[] = { _("Action"), _("Key") };
	int i;

	key_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(key_win), _("Sinek Key Bindings"));
	gtk_window_set_wmclass(GTK_WINDOW(key_win), "keybindings", "sinek");

	vb = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vb);
	gtk_container_add(GTK_CONTAINER(key_win), vb);
	gtk_container_set_border_width(GTK_CONTAINER(vb), 4);

	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(vb), sw, TRUE, TRUE, 0);

	clist_wid = gtk_clist_new_with_titles(2, titles);
	gtk_widget_show(clist_wid);
	gtk_container_add(GTK_CONTAINER(sw), clist_wid);
	gtk_container_set_border_width(GTK_CONTAINER(clist_wid), 2);
	g_signal_connect(G_OBJECT(clist_wid), "select_row", G_CALLBACK(cb_select_binding), NULL);

	hb = gtk_hbox_new(TRUE, 3);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, TRUE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(hb), 0);
	t = gtk_button_new_with_label(_("Add Binding"));
	gtk_widget_show(t);
	gtk_box_pack_start(GTK_BOX(hb), t, TRUE, TRUE, 3);
	g_signal_connect(G_OBJECT(t), "clicked", G_CALLBACK(cb_add_binding), NULL);
	t = gtk_button_new_with_label(_("Remove Binding"));
	gtk_widget_show(t);
	gtk_box_pack_start(GTK_BOX(hb), t, TRUE, TRUE, 3);
	g_signal_connect(G_OBJECT(t), "clicked", G_CALLBACK(cb_remove_binding), NULL);

	hb = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hb), 5);

	t = gtk_label_new(_("Action"));
	gtk_widget_show(t);
	gtk_box_pack_start(GTK_BOX(hb), t, FALSE, FALSE, 0);
	action_wid = gtk_combo_new();
	gtk_widget_show(action_wid);
	gtk_box_pack_start(GTK_BOX(hb), action_wid, TRUE, TRUE, 0);
	gtk_editable_set_editable(GTK_EDITABLE(GTK_COMBO(action_wid)->entry), FALSE);
	for(i = 0; actions[i].name; i++)
		acts = g_list_append(acts, (gpointer)actions[i].label);
	gtk_combo_set_popdown_strings(GTK_COMBO(action_wid), acts);
	g_list_free(acts);
	g_signal_connect(G_OBJECT(GTK_COMBO(action_wid)->list), "select_child", G_CALLBACK(cb_change_action), NULL);

	t = gtk_label_new(_("Key"));
	gtk_widget_show(t);
	gtk_box_pack_start(GTK_BOX(hb), t, FALSE, FALSE, 0);
	key_wid = gtk_entry_new();
	gtk_widget_show(key_wid);
	gtk_box_pack_start(GTK_BOX(hb), key_wid, TRUE, TRUE, 0);
	g_signal_connect(G_OBJECT(key_wid), "key_press_event", G_CALLBACK(cb_change_key), NULL);

	hb = gtk_hbutton_box_new();
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, FALSE, 0);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hb), GTK_BUTTONBOX_END);
	gtk_container_set_border_width(GTK_CONTAINER(hb), 5);
	gtk_box_set_spacing(GTK_BOX(hb), 5);

	t = gtk_button_new_with_label(_("Reset to Defaults"));
	gtk_widget_show(t);
	gtk_container_add(GTK_CONTAINER(hb), t);
	g_signal_connect(G_OBJECT(t), "clicked", G_CALLBACK(reset_keys), NULL);

	t = gtk_button_new_from_stock(GTK_STOCK_SAVE);
	gtk_widget_show(t);
	gtk_container_add(GTK_CONTAINER(hb), t);
	g_signal_connect(G_OBJECT(t), "clicked", G_CALLBACK(save_keys), NULL);

	t = gtk_button_new_from_stock(GTK_STOCK_OK);
	gtk_widget_show(t);
	gtk_container_add(GTK_CONTAINER(hb), t);
	g_signal_connect_swapped(G_OBJECT(t), "clicked", G_CALLBACK(gtk_widget_hide), GTK_OBJECT(key_win));

	wm_manage(key_win, WM_NORMAL);
}


void execute_cmd(cmd_type cmd)
{
	xine_event_t xinev;
	int tmp;

	switch(cmd)
	{
		case CMD_NONE:
			break;

		case CMD_QUIT:
			sinek.playing = 0;
			scrsaver_enable();
			sinek_delete(media);
			gtk_main_quit();
			break;

		case CMD_PLAY:
			scrsaver_disable();
			sinek_play(media);
			sinek.pause = 0;
			break;

		case CMD_PAUSE:
			if(sinek.pause)
				xine_set_speed(sinek.xine, SPEED_NORMAL);
			else
				xine_set_speed(sinek.xine, SPEED_PAUSE);
			sinek.pause = ~sinek.pause;
			break;

		case CMD_STOP:
			sinek.playing = 0;
			sinek_stop(media);
			scrsaver_enable();
			break;

		case CMD_NEXT:
			pl_next();
			break;

		case CMD_PREVIOUS:
			pl_prev();
			break;

		case CMD_VOLUME_UP:
			audio_slide_volume(1);
			break;

		case CMD_VOLUME_DOWN:
			audio_slide_volume(-1);
			break;

		case CMD_SPEED_FAST:
			tmp = xine_get_speed(sinek.xine);
			if(tmp != SPEED_PAUSE && tmp < SPEED_FAST_4)
				xine_set_speed(sinek.xine, tmp * 2);
			break;

		case CMD_SPEED_SLOW:
			tmp = xine_get_speed(sinek.xine);
			if(tmp > SPEED_SLOW_4)
				xine_set_speed(sinek.xine, tmp / 2);
			break;

		case CMD_AV_OFFSET_DOWN:
			tmp = xine_get_av_offset(sinek.xine);
			tmp = tmp - 3600;
			xine_set_av_offset(sinek.xine, tmp);
			break;

		case CMD_AV_OFFSET_UP:
			tmp = xine_get_av_offset(sinek.xine);
			tmp = tmp + 3600;
			xine_set_av_offset(sinek.xine, tmp);
			break;

		case CMD_SHOW_ABOUT:
			wm_show(about_build);
			break;

		case CMD_SHOW_TUNER:
			tune_show();
			break;

		case CMD_SHOW_KEYS:
			gtk_widget_show(key_win);
			break;

		case CMD_SHOW_PLAYLIST:
			wm_toggle(pl_build);
			break;

		case CMD_SHOW_VIDEO:
			video_show();
			break;

		case CMD_SHOW_CONTROLS:
			wm_toggle(control_build);
			break;

		case CMD_SHOW_PREFS:
			wm_show(prf_build);
			break;

		case CMD_TOGGLE_GUI:
			if(sinek.hide)
				wm_show_all();
			else
				wm_hide_all();
			sinek.hide = ~sinek.hide;
			break;

		case CMD_TOGGLE_FULLSCREEN:
			if(sinek.video_mode == VIDEO_FULLSCREEN)
				video_set_mode(VIDEO_WINDOW);
			else
				video_set_mode(VIDEO_FULLSCREEN);
			break;

		case CMD_TOGGLE_ASPECT:
			tmp = media->vd->get_property(media->vd, VO_PROP_ASPECT_RATIO);
			media->vd->set_property(media->vd, VO_PROP_ASPECT_RATIO, tmp + 1);
			break;

		case CMD_TOGGLE_INTERLEAVE:
			tmp = media->vd->get_property(media->vd, VO_PROP_INTERLACED);
			media->vd->set_property(media->vd, VO_PROP_INTERLACED, 1 - tmp);
			break;

		case CMD_TOGGLE_TVMODE:
			tmp = media->vd->get_property(media->vd, VO_PROP_ASPECT_RATIO);
			media->vd->set_property(media->vd, VO_PROP_TVMODE, tmp);
			break;

		case CMD_TOGGLE_SUBTITLES:
			if(sinek.osd_subtitles)
				osd_hide();
			else
				osd_show();
			break;

		case CMD_TOGGLE_REPEAT:
			sinek.repeat_one = ~sinek.repeat_one;
			sinek.signal_fake = 1;
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sinek.repeat_tog), sinek.repeat_one);
			sinek.signal_fake = 0;
			update_pix_toggle(sinek.repeat_tog);
			break;

		case CMD_TOGGLE_REPEAT_ALL:
			sinek.repeat_all = ~sinek.repeat_all;
			sinek.signal_fake = 1;
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(sinek.repeat_all_chk), sinek.repeat_all);
			sinek.signal_fake = 0;
			media->conf->update_num(media->conf, "sinek.repeat_all", sinek.repeat_all);
			break;

		case CMD_TOGGLE_MUTE:
			audio_toggle_mute();
			sinek.signal_fake = 1;
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sinek.mute_tog), sinek.mute);
			sinek.signal_fake = 0;
			break;

		case CMD_SPU_NEXT:
			tmp = xine_get_spu_channel(sinek.xine);
			xine_select_spu_channel(sinek.xine, tmp + 1);
			break;

		case CMD_SPU_PREVIOUS:
			tmp = xine_get_spu_channel(sinek.xine);
			xine_select_spu_channel(sinek.xine, tmp - 1);
			break;

		case CMD_AUDIO_NEXT:
			tmp = xine_get_audio_selection(sinek.xine);
			xine_select_audio_channel(sinek.xine, tmp + 1);
			break;

		case CMD_AUDIO_PREVIOUS:
			tmp = xine_get_audio_selection(sinek.xine);
			xine_select_audio_channel(sinek.xine, tmp - 1);
			break;

		case CMD_ZOOM_IN:
			video_zoom(1);
			break;

		case CMD_ZOOM_OUT:
			video_zoom(-1);
			break;

		case CMD_ZOOM_RESET:
			video_zoom(0);
			break;

		case CMD_WINDOW_1x1:
			video_set_mode(VIDEO_WINDOW);
			video_scale_abs(1);
			break;

		case CMD_WINDOW_2x2:
			video_set_mode(VIDEO_WINDOW);
			video_scale_abs(2);
			break;

		case CMD_WINDOW_FULLSCREEN:
			video_set_mode(VIDEO_FULLSCREEN);
			break;

		case CMD_WINDOW_BACKGROUND:
			video_set_mode(VIDEO_BACKGROUND);
			break;

		case CMD_WINDOW_REDUCE:
			video_scale(0.8);
			break;

		case CMD_WINDOW_ENLARGE:
			video_scale(1.2);
			break;

		case CMD_EVENT_MENU1:
			xinev.type = XINE_EVENT_INPUT_MENU1;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_MENU2:
			xinev.type = XINE_EVENT_INPUT_MENU2;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_MENU3:
			xinev.type = XINE_EVENT_INPUT_MENU3;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_UP:
			xinev.type = XINE_EVENT_INPUT_UP;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_DOWN:
			xinev.type = XINE_EVENT_INPUT_DOWN;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_LEFT:
			xinev.type = XINE_EVENT_INPUT_LEFT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_RIGHT:
			xinev.type = XINE_EVENT_INPUT_RIGHT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_NEXT:
			xinev.type = XINE_EVENT_INPUT_NEXT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_PRIOR:
			xinev.type = XINE_EVENT_INPUT_PREVIOUS;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_ANGLE_NEXT:
			xinev.type = XINE_EVENT_INPUT_ANGLE_NEXT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_ANGLE_PRIOR:
			xinev.type = XINE_EVENT_INPUT_ANGLE_PREVIOUS;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_SELECT:
			xinev.type = XINE_EVENT_INPUT_SELECT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_GOTO_MARK_0:
		case CMD_GOTO_MARK_1:
		case CMD_GOTO_MARK_2:
		case CMD_GOTO_MARK_3:
		case CMD_GOTO_MARK_4:
		case CMD_GOTO_MARK_5:
		case CMD_GOTO_MARK_6:
		case CMD_GOTO_MARK_7:
		case CMD_GOTO_MARK_8:
		case CMD_GOTO_MARK_9:
			video_seek(sinek.marks[cmd - CMD_GOTO_MARK_0]);
			sinek.marks[0] = xine_get_current_time(sinek.xine);
			break;

		case CMD_SET_MARK_1:
		case CMD_SET_MARK_2:
		case CMD_SET_MARK_3:
		case CMD_SET_MARK_4:
		case CMD_SET_MARK_5:
		case CMD_SET_MARK_6:
		case CMD_SET_MARK_7:
		case CMD_SET_MARK_8:
		case CMD_SET_MARK_9:
			sinek.marks[cmd - CMD_SET_MARK_1 + 1] = xine_get_current_time(sinek.xine);
			break;
	}
}
