/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.as.test.integration.ws.wsrm;

import java.util.Iterator;
import java.util.Set;
import javax.xml.namespace.QName;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;
import org.jboss.logging.Logger;

/**
 *
 * @author <a href="mailto:rsvoboda@redhat.com">Rostislav Svoboda</a>
 */
public class ReliableCheckHandler implements SOAPHandler<SOAPMessageContext> {

    /*
     * 1 -- Body - CreateSequence
     * 2 -- Body - CreateSequenceResponse
     * 3 -- Header - wsrm:Sequence
     * 4 -- Header - wsrm:SequenceAcknowledgement
     */
    int status = 0;
    private static Logger log = Logger.getLogger(ReliableCheckHandler.class);

    public Set<QName> getHeaders() {
        return null;
    }

    public boolean handleFault(SOAPMessageContext context) {
        handleContext(context);
        return true;
    }

    public boolean handleMessage(SOAPMessageContext context) {
        handleContext(context);
        return true;
    }

    private void handleContext(SOAPMessageContext smc) {
        Boolean outboundProperty = (Boolean) smc.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);
        SOAPMessage message = smc.getMessage();

        if (outboundProperty) {
            log.debug("Outgoing message:");
        } else {
            log.debug("Incoming message:");
        }

        log.debug("-----------");
        try {
            message.writeTo(System.out);
            log.debug("");
        } catch (Exception e) {
            log.debug("Exception in handler: " + e);
        }
        log.debug("-----------");

        SOAPElement firstBodyElement;
        try {
            switch (status % 4) {
                case 0:
                    @SuppressWarnings("unchecked")
                    Iterator<SOAPElement> it = (Iterator<SOAPElement>) message.getSOAPBody().getChildElements();
                    if (it.hasNext()) {
                        firstBodyElement = it.next();
                        final QName createSequenceQName = new QName("http://schemas.xmlsoap.org/ws/2005/02/rm", "CreateSequence");
                        if (!createSequenceQName.equals(firstBodyElement.getElementQName())) {
                            throw new WebServiceException("CreateSequence in soap body was expected, but it contains '"
                                    + firstBodyElement.getElementQName() + "'");
                        }
                        status++;
                    } else {
                        //we could get multiple acknowledments
                        verifySequenceAcknowledgement(message);
                    }
                    break;
                case 1:
                    firstBodyElement = (SOAPElement) message.getSOAPBody().getChildElements().next();
                    final QName createSequenceResponseQName = new QName("http://schemas.xmlsoap.org/ws/2005/02/rm", "CreateSequenceResponse");
                    if (!createSequenceResponseQName.equals(firstBodyElement.getElementQName())) {
                        throw new WebServiceException("CreateSequenceResponse in soap body was expected, but it contains '"
                                + firstBodyElement.getElementQName() + "'");
                    }
                    status++;
                    break;
                case 2:
                    Iterator headerElements = message.getSOAPHeader().getChildElements();
                    boolean found = false;
                    final QName sequenceQName = new QName("http://schemas.xmlsoap.org/ws/2005/02/rm", "Sequence");
                    while (headerElements.hasNext()) {
                        SOAPElement soapElement = (SOAPElement) headerElements.next();
                        if (sequenceQName.equals(soapElement.getElementQName())) {
                            found = true;
                        }
                    }
                    if (!found) {
                        throw new WebServiceException("wsrm:Sequence is not present in soap header");
                    }
                    status++;
                    break;
                case 3:
                    if (verifySequenceAcknowledgement(message)) {
                        status++;
                    }
                    break;
            }
        } catch (SOAPException ex) {
            throw new WebServiceException(ex.getMessage(), ex);
        }
    }

    private boolean verifySequenceAcknowledgement(SOAPMessage message) throws SOAPException {
        Iterator headerElements = message.getSOAPHeader().getChildElements();
        boolean found = false;
        boolean otherRMHeadersFound = false;
        final QName sequenceAckQName = new QName("http://schemas.xmlsoap.org/ws/2005/02/rm", "SequenceAcknowledgement");
        while (headerElements.hasNext()) {
            SOAPElement soapElement = (SOAPElement) headerElements.next();
            if (sequenceAckQName.equals(soapElement.getElementQName())) {
                found = true;
            } else if ("http://schemas.xmlsoap.org/ws/2005/02/rm".equals(soapElement.getNamespaceURI())) {
                otherRMHeadersFound = true;
            }
        }
        //fail if we did not find the sequence ack and the message has other WS-RM headers (hence out of order) or has
        // body contents (hence a non-reliable message is being processed)
        if (!found && (otherRMHeadersFound || message.getSOAPBody().getChildElements().hasNext())) {
            throw new WebServiceException("wsrm:SequenceAcknowledgement is not present in soap header");
        }
        return found;
    }

    public void close(MessageContext context) {
    }
}
