/*
 * JBoss, Home of Professional Open Source.
 * Copyright (c) 2015, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.as.test.integration.web.servlet.nonexistentservletmetric;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.NAME;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.FAILED;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.FAILURE_DESCRIPTION;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.OP;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.OP_ADDR;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.OUTCOME;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.SUCCESS;

import java.net.URL;
import java.util.concurrent.TimeUnit;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.as.arquillian.api.ContainerResource;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.as.test.integration.common.HttpRequest;
import org.jboss.dmr.ModelNode;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;

@RunWith(Arquillian.class)
@RunAsClient
public class NonExistentServletMetricTestCase {

    @ArquillianResource
    private URL url;

    @ContainerResource
    private ManagementClient managementClient;

    @Deployment
    public static WebArchive single() {
        WebArchive war = ShrinkWrap.create(WebArchive.class, "simple.war");
        war.addClasses(HttpRequest.class, SimpleServlet.class);
        return war;
    }

    private static ModelNode getAddress(String servletName) {
        ModelNode address = new ModelNode();
        address.add("deployment", "simple.war");
        address.add("subsystem", "web");
        address.add("servlet", servletName);
        address.protect();
        return address;
    }

    @Test
    public void testExistentServletMetric() throws Exception {
        String s = performCall("simple", "Hello");
        Assert.assertEquals("Simple Servlet called with input=Hello", s);

        ModelNode address = getAddress("SimpleServlet");
        final ModelNode operation = new ModelNode();
        operation.get(OP).set("read-attribute");
        operation.get(OP_ADDR).set(address);
        operation.get(NAME).set("requestCount");

        ModelNode result = managementClient.getControllerClient().execute(operation);
        Assert.assertEquals(SUCCESS, result.get(OUTCOME).asString());
    }

    @Test
    public void testNonExistentServletMetric() throws Exception {
        ModelNode address = getAddress("NonExistentSimpleServlet");

        final ModelNode operation = new ModelNode();
        operation.get(OP).set("read-attribute");
        operation.get(OP_ADDR).set(address);
        operation.get(NAME).set("requestCount");

        ModelNode result = managementClient.getControllerClient().execute(operation);
        Assert.assertEquals(FAILED, result.get(OUTCOME).asString());
        String npeMsg = "JBAS014749: Operation handler failed: java.lang.NullPointerException";
        String msg = "JBAS018002: Failed to get metrics due to non-existent NonExistentSimpleServlet";
        // Should received message to indicate non-existent Servlet instead of Operation failure with NullPointerException
        Assert.assertNotEquals(npeMsg, result.get(FAILURE_DESCRIPTION).asString());
        Assert.assertEquals(msg, result.get(FAILURE_DESCRIPTION).asString());
    }

    private String performCall(String urlPattern, String param) throws Exception {
        URL url = new URL(this.url.toExternalForm() + urlPattern + "?input=" + param);
        return HttpRequest.get(url.toExternalForm(), 10, TimeUnit.SECONDS);
    }
}