"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const findUp = require("find-up");
const webpack = require("webpack");
const constants_1 = require("../constants");
const jsonc_1 = require("../utils/jsonc");
const ExtensionValidator_1 = require("../validation/ExtensionValidator");
const SchemaValidator_1 = require("../validation/SchemaValidator");
exports.loadSchema = (relativePath) => {
    const pkgDir = path.dirname(findUp.sync('package.json', { cwd: __dirname }));
    const schemaPath = [
        path.resolve(pkgDir, 'schema'),
        path.resolve(pkgDir, 'generated/schema'),
    ].find((p) => fs.existsSync(p) && fs.statSync(p).isDirectory());
    return require(path.resolve(schemaPath, relativePath));
};
exports.validateExtensionsFileSchema = (ext, description = constants_1.extensionsFile) => {
    const schema = exports.loadSchema('console-extensions.json');
    return new SchemaValidator_1.SchemaValidator(description).validate(schema, ext);
};
const getPluginManifest = (pkg, ext) => ({
    name: pkg.consolePlugin.name,
    version: pkg.consolePlugin.version,
    displayName: pkg.consolePlugin.displayName,
    description: pkg.consolePlugin.description,
    dependencies: pkg.consolePlugin.dependencies,
    disableStaticPlugins: pkg.consolePlugin.disableStaticPlugins,
    extensions: ext,
});
class ConsoleAssetPlugin {
    constructor(pkg, remoteEntryCallback, skipExtensionValidator = false) {
        this.pkg = pkg;
        this.remoteEntryCallback = remoteEntryCallback;
        this.skipExtensionValidator = skipExtensionValidator;
        this.ext = jsonc_1.parseJSONC(path.resolve(process.cwd(), constants_1.extensionsFile));
        exports.validateExtensionsFileSchema(this.ext).report();
    }
    apply(compiler) {
        compiler.hooks.thisCompilation.tap(ConsoleAssetPlugin.name, (compilation) => {
            // Generate additional assets
            compilation.hooks.processAssets.tap({
                name: ConsoleAssetPlugin.name,
                stage: webpack.Compilation.PROCESS_ASSETS_STAGE_ADDITIONAL,
            }, () => {
                compilation.emitAsset(constants_1.pluginManifestFile, new webpack.sources.RawSource(Buffer.from(JSON.stringify(getPluginManifest(this.pkg, this.ext), null, 2))));
            });
            // Post-process assets already present in the compilation
            compilation.hooks.processAssets.tap({
                name: ConsoleAssetPlugin.name,
                stage: webpack.Compilation.PROCESS_ASSETS_STAGE_ADDITIONS,
            }, () => {
                compilation.updateAsset(constants_1.remoteEntryFile, (source) => {
                    const newSource = new webpack.sources.ReplaceSource(source);
                    const fromIndex = source
                        .source()
                        .toString()
                        .indexOf(`${this.remoteEntryCallback}(`);
                    if (fromIndex < 0) {
                        const error = new webpack.WebpackError(`Missing call to ${this.remoteEntryCallback}`);
                        error.file = constants_1.remoteEntryFile;
                        compilation.errors.push(error);
                    }
                    else {
                        newSource.insert(fromIndex + this.remoteEntryCallback.length + 1, `'${this.pkg.consolePlugin.name}@${this.pkg.consolePlugin.version}', `);
                    }
                    return newSource;
                });
            });
        });
        if (!this.skipExtensionValidator) {
            compiler.hooks.emit.tap(ConsoleAssetPlugin.name, (compilation) => {
                const result = new ExtensionValidator_1.ExtensionValidator(constants_1.extensionsFile).validate(compilation, this.ext, this.pkg.consolePlugin.exposedModules || {});
                if (result.hasErrors()) {
                    const error = new webpack.WebpackError('ExtensionValidator has reported errors');
                    error.details = result.formatErrors();
                    error.file = constants_1.extensionsFile;
                    compilation.errors.push(error);
                }
            });
        }
    }
}
exports.ConsoleAssetPlugin = ConsoleAssetPlugin;
//# sourceMappingURL=ConsoleAssetPlugin.js.map