#!/usr/bin/env bash

set -euo pipefail

[[ -n "$SCRATCH_DIR" && -n "$SYSDIG_DIR" && -n "$OUTPUT_DIR" ]] || {
	echo >&2 "SCRATCH_DIR, SYSDIG_DIR and OUTPUT_DIR all need to be set"
	exit 1
}

SYSDIG_SCRATCH="${SCRATCH_DIR}/sysdig-src"
BUILD_SCRATCH="${SCRATCH_DIR}/cmake-build"

modfiles0() (
	local dir="$1"
	cd "$dir"
	find . -type f \( -name 'Makefile' -o -name '*.c' -o -name '*.h' \) -print0 | \
		LC_ALL=C sort -z
)

get_module_version() (
	modfiles0 "$@" | \
		xargs -0 sha256sum | \
		awk '{print$1 " " $2}' | \
		sha256sum | \
		awk '{print$1}'
)

rm -rf "$SYSDIG_SCRATCH" "$BUILD_SCRATCH" 2>/dev/null
mkdir -p "$SYSDIG_SCRATCH" "$BUILD_SCRATCH"
cp -r "${SYSDIG_DIR}/." "${SYSDIG_SCRATCH}"

cd "$BUILD_SCRATCH"
cmake \
	-DCMAKE_BUILD_TYPE=Release \
	-DCMAKE_C_FLAGS="-fno-pie" \
	-DPROBE_NAME=collector \
	-DBUILD_USERSPACE=OFF \
	-DBUILD_DRIVER=ON \
	-DENABLE_DKMS=OFF \
	-DBUILD_BPF=ON \
	"$SYSDIG_SCRATCH"

# Copy a few files generated by cmake to the source directory.
cp "${BUILD_SCRATCH}/driver/src/Makefile" "${SYSDIG_SCRATCH}/driver/Makefile"
cp "${BUILD_SCRATCH}/driver/src/driver_config.h" "${SYSDIG_SCRATCH}/driver/driver_config.h"

cd "${SYSDIG_SCRATCH}/driver"
module_version="$(get_module_version .)"
echo "Configured driver version: $module_version"

source_archive="${OUTPUT_DIR}/${module_version}/"

if [[ ! -d "$source_archive" ]]; then
	mkdir -p "$source_archive" "$source_archive"/bpf

	cp ./*.c ./*.h Makefile "$source_archive"
	cp bpf/*.c bpf/*.h bpf/Makefile "$source_archive"/bpf
else
	echo "Duplicate version '${module_version}' detected, skipping..."
fi
