/*
Copyright 2019 The hostpath provisioner operator Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package v1beta1

import (
	conditions "github.com/openshift/custom-resource-status/conditions/v1"
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// HostPathProvisionerSpec defines the desired state of HostPathProvisioner
// +k8s:openapi-gen=true
type HostPathProvisionerSpec struct {
	// ImagePullPolicy is the container pull policy for the host path provisioner containers
	ImagePullPolicy corev1.PullPolicy `json:"imagePullPolicy,omitempty" valid:"required"`
	// PathConfig describes the location and layout of PV storage on nodes. Deprecated
	PathConfig *PathConfig `json:"pathConfig,omitempty" optional:"true"`
	// Restrict on which nodes HPP workload pods will be scheduled
	Workload NodePlacement `json:"workload,omitempty"`
	// FeatureGates are a list of specific enabled feature gates
	// +listType=set
	FeatureGates []string `json:"featureGates,omitempty"`
	// StoragePools are a list of storage pools
	// +listType=atomic
	StoragePools []StoragePool `json:"storagePools,omitempty" optional:"true"`
}

// HostPathProvisionerStatus defines the observed state of HostPathProvisioner
// +k8s:openapi-gen=true
type HostPathProvisionerStatus struct {
	// Conditions contains the current conditions observed by the operator
	// +listType=atomic
	Conditions []conditions.Condition `json:"conditions,omitempty" optional:"true"`
	// OperatorVersion The version of the HostPathProvisioner Operator
	OperatorVersion string `json:"operatorVersion,omitempty" optional:"true"`
	// TargetVersion The targeted version of the HostPathProvisioner deployment
	TargetVersion string `json:"targetVersion,omitempty" optional:"true"`
	// ObservedVersion The observed version of the HostPathProvisioner deployment
	ObservedVersion string `json:"observedVersion,omitempty" optional:"true"`
	// +listType=atomic
	StoragePoolStatuses []StoragePoolStatus `json:"storagePoolStatuses,omitempty" optional:"true"`
}

// StoragePool defines how and where hostpath provisioner can use storage to create volumes.
// +k8s:openapi-gen=true
type StoragePool struct {
	// Name specifies an identifier that is used in the storage class arguments to identify the source to use.
	Name string `json:"name" valid:"required"`
	// PVCTemplate is the template of the PVC to create as the source volume
	PVCTemplate *corev1.PersistentVolumeClaimSpec `json:"pvcTemplate,omitempty" optional:"true"`
	// path the path to use on the host, this is a required field
	Path string `json:"path" valid:"required"`
}

// StoragePoolStatus is the status of the named storage pool
type StoragePoolStatus struct {
	// Name is the name of the storage pool
	Name string `json:"name" valid:"required"`
	// StoragePoolPhase indicates which phase the storage pool is in.
	Phase StoragePoolPhase `json:"phase" valid:"required"`
	// DesiredReady is the number of desired ready replicasets.
	DesiredReady int `json:"desiredReady,omitempty" optional:"true"`
	// CurrentReady is the number of currently ready replicasets.
	CurrentReady int `json:"currentReady,omitempty" optional:"true"`
	// The status of all the claims.
	// +listType=atomic
	ClaimStatuses []ClaimStatus `json:"claimStatuses,omitempty" optional:"true"`
}

// ClaimStatus defines the storage claim status for each PVC in a storage pool
type ClaimStatus struct {
	// Name of the PersistentVolumeClaim
	Name string `json:"name" valid:"required"`
	// Status of the PersistentVolumeClaim
	Status corev1.PersistentVolumeClaimStatus `json:"status" valid:"required"`
}

// StoragePoolPhase is the current phase of the storage pool.
type StoragePoolPhase string

const (
	// StoragePoolPreparing indicates the storage pool is preparing one or more volumes for use.
	StoragePoolPreparing StoragePoolPhase = "Preparing"
	//StoragePoolMounting indicates one or more pool are in the process of mounting.
	StoragePoolMounting StoragePoolPhase = "Mounting"
	// StoragePoolReady indicates all the volumes are ready for use.
	StoragePoolReady StoragePoolPhase = "Ready"
)

// this has to be here otherwise informer-gen doesn't recognize it
// see https://github.com/kubernetes/code-generator/issues/59
// +genclient:nonNamespaced

// HostPathProvisioner is the Schema for the hostpathprovisioners API
// +genclient
// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object
// +k8s:openapi-gen=true
// +kubebuilder:resource:path=hostpathprovisioners,scope=Cluster
// +kubebuilder:storageversion
type HostPathProvisioner struct {
	metav1.TypeMeta   `json:",inline"`
	metav1.ObjectMeta `json:"metadata,omitempty"`

	Spec   HostPathProvisionerSpec   `json:"spec,omitempty"`
	Status HostPathProvisionerStatus `json:"status,omitempty"`
}

// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// HostPathProvisionerList contains a list of HostPathProvisioner
type HostPathProvisionerList struct {
	metav1.TypeMeta `json:",inline"`
	metav1.ListMeta `json:"metadata,omitempty"`
	Items           []HostPathProvisioner `json:"items"`
}

// PathConfig contains the information needed to build the path where the PVs will be created. Deprecated
// +k8s:openapi-gen=true
type PathConfig struct {
	// Path The path the directories for the PVs are created under
	Path string `json:"path,omitempty" valid:"required"`
	// UseNamingPrefix Use the name of the PVC requesting the PV as part of the directory created
	UseNamingPrefix bool `json:"useNamingPrefix,omitempty"`
}

// NodePlacement describes node scheduling configuration.
// +k8s:openapi-gen=true
type NodePlacement struct {
	// nodeSelector is the node selector applied to the relevant kind of pods
	// It specifies a map of key-value pairs: for the pod to be eligible to run on a node,
	// the node must have each of the indicated key-value pairs as labels
	// (it can have additional labels as well).
	// See https://kubernetes.io/docs/concepts/configuration/assign-pod-node/#nodeselector
	// +kubebuilder:validation:Optional
	// +optional
	NodeSelector map[string]string `json:"nodeSelector,omitempty"`

	// affinity enables pod affinity/anti-affinity placement expanding the types of constraints
	// that can be expressed with nodeSelector.
	// affinity is going to be applied to the relevant kind of pods in parallel with nodeSelector
	// See https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#affinity-and-anti-affinity
	// +kubebuilder:validation:Optional
	// +optional
	Affinity *corev1.Affinity `json:"affinity,omitempty"`

	// tolerations is a list of tolerations applied to the relevant kind of pods
	// See https://kubernetes.io/docs/concepts/configuration/taint-and-toleration/ for more info.
	// These are additional tolerations other than default ones.
	// +kubebuilder:validation:Optional
	// +optional
	Tolerations []corev1.Toleration `json:"tolerations,omitempty"`
}
