#!/usr/bin/env python
import argparse
import json
import pycurl
import sys

from io import BytesIO
from jinja2 import Environment, FileSystemLoader, Undefined


DEFAULT_ERRATA = '45581'
DEFAULT_CHANNELS = ["RHEL-7-CNV-2.2", "RHEL-8-CNV-2.2"]
DEV_REGISTRY = 'brew-pulp-docker01.web.prod.ext.phx2.redhat.com:8888'
STAGING_REGISTRY = 'registry.stage.redhat.io'
PROD_REGISTRY = 'registry.redhat.io'
DEFAULT_REPOSITORY = 'container-native-virtualization'
DEFAULT_OUTPUTF = 'vars'


class IgnoreUndefined(Undefined):
    def _fail_with_undefined_error(self, *args, **kwargs):
        return ''


def fetch(url, secure=True):
    buffer = BytesIO()
    c = pycurl.Curl()
    c.setopt(c.URL, url)
    c.setopt(c.WRITEFUNCTION, buffer.write)
    c.setopt(c.HTTPAUTH, c.HTTPAUTH_GSSNEGOTIATE)
    c.setopt(c.USERPWD, ':')
    c.setopt(c.USERAGENT, 'Python Client')
    if not secure:
        c.setopt(c.SSL_VERIFYPEER, False)
        c.setopt(c.SSL_VERIFYHOST, False)
    c.perform()
    response_code = c.getinfo(c.RESPONSE_CODE)
    c.close()

    if response_code == 401:
        raise RuntimeError(
            'auth error: please run kinit to get a valid kerberos ticket'
        )
    elif response_code != 200:
        raise RuntimeError(
            'pycurl error {rc}'.format(rc=response_code)
        )

    data = buffer.getvalue()
    buffer.close()
    jdata = json.loads(data)
    return jdata


def get_errata_images(errata, verbose, secure, channels=DEFAULT_CHANNELS):
    url = 'https://errata.devel.redhat.com/api/v1/erratum/{e}/builds'.format(
        e=errata,
    )
    print("Fetching errata info from '{u}'".format(u=url))
    images = {}

    errata_out = fetch(url, secure=secure)
    for c in channels:
        builds = [i.keys()[0] for i in errata_out[c]["builds"]]
        for i in builds:
            s = i.rsplit('-', 3)
            images[s[-4]] = '-'.join(s[-2:])
    images = {key.replace('-', '_'): val for key, val in images.items()}
    return images


def update(
    errata=DEFAULT_ERRATA,
    verbose=False,
    secure=True,
    channels=DEFAULT_CHANNELS,
    of=DEFAULT_OUTPUTF,
    registry=DEV_REGISTRY,
):
    errata_images = get_errata_images(errata, verbose, secure, channels)
    if verbose:
        print("Builds in the errata:")
        for b in errata_images:
            print("\t{b}:{v}".format(b=b, v=errata_images[b]))

    env = Environment(
        loader=FileSystemLoader('templates/'),
        undefined=IgnoreUndefined,
    )
    template = env.get_template('vars.j2')
    rendered = template.render(
        v=errata_images,
        registry=registry,
        repository=DEFAULT_REPOSITORY,
    )
    if verbose:
        print("Rendered:")
        print(rendered)

    with open(of, 'w') as outfile:
        outfile.write(rendered)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description='helper to align versions in the CSV file at build time'
    )
    parser.add_argument(
        '--verbose',
        help='increase output verbosity',
        action='store_true'
    )
    parser.add_argument(
        '--insecure',
        help='Allow insecure server connections when using SSL',
        action='store_false',
        dest='secure'
    )
    parser.add_argument(
        '--errata',
        help='fetch version info from a specific errata'
             ' [Default: {d}]'.format(d=DEFAULT_ERRATA),
        default=DEFAULT_ERRATA
    )
    parser.add_argument(
        '--channels',
        help='errata channels'
             ' [Default: {d}]'.format(d=DEFAULT_CHANNELS),
        default=DEFAULT_CHANNELS,
    )
    parser.add_argument(
        '--of',
        help='Output file'
             ' [Default: {d}]'.format(d=DEFAULT_OUTPUTF),
        default=DEFAULT_OUTPUTF,
    )
    parser.add_argument(
        '--registry',
        help='registry address'
             ' [Default: {d}]'.format(d=DEV_REGISTRY),
        choices=[DEV_REGISTRY, STAGING_REGISTRY, PROD_REGISTRY],
        default=DEV_REGISTRY,
    )

    args = parser.parse_args()

    update(
        errata=args.errata,
        verbose=args.verbose,
        secure=args.secure,
        channels=args.channels,
        of=args.of,
        registry=args.registry,
    )
    sys.exit(0)
