/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.prestosql.plugin.hive.parquet;

import com.google.common.collect.ImmutableMap;
import io.airlift.units.DataSize;
import org.testng.annotations.Test;

import java.util.Map;

import static io.airlift.configuration.testing.ConfigAssertions.assertFullMapping;
import static io.airlift.configuration.testing.ConfigAssertions.assertRecordedDefaults;
import static io.airlift.configuration.testing.ConfigAssertions.recordDefaults;
import static io.airlift.units.DataSize.Unit.KILOBYTE;
import static io.airlift.units.DataSize.Unit.MEGABYTE;

public class TestParquetReaderConfig
{
    @Test
    public void testDefaults()
    {
        assertRecordedDefaults(recordDefaults(ParquetReaderConfig.class)
                .setFailOnCorruptedStatistics(true)
                .setMaxReadBlockSize(new DataSize(16, MEGABYTE))
                .setMaxMergeDistance(new DataSize(1, MEGABYTE))
                .setMaxBufferSize(new DataSize(8, MEGABYTE)));
    }

    @Test
    public void testExplicitPropertyMappings()
    {
        Map<String, String> properties = new ImmutableMap.Builder<String, String>()
                .put("parquet.fail-on-corrupted-statistics", "false")
                .put("parquet.max-read-block-size", "66kB")
                .put("parquet.max-buffer-size", "1431kB")
                .put("parquet.max-merge-distance", "342kB")
                .build();

        ParquetReaderConfig expected = new ParquetReaderConfig()
                .setFailOnCorruptedStatistics(false)
                .setMaxReadBlockSize(new DataSize(66, KILOBYTE))
                .setMaxBufferSize(new DataSize(1431, KILOBYTE))
                .setMaxMergeDistance(new DataSize(342, KILOBYTE));

        assertFullMapping(properties, expected);
    }
}
