// Copyright 2016 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// +build example
//
// This build tag means that "go install golang.org/x/exp/shiny/..." doesn't
// install this example program. Use "go run main.go" to run it or "go install
// -tags=example" to install it.

// Icongallery exhibits the Material Design icon set.
package main

// TODO: key events to navigate through the icons. Also add a search-by-name
// drop-down combo box?

import (
	"image"
	"image/draw"
	"log"

	"golang.org/x/exp/shiny/driver"
	"golang.org/x/exp/shiny/gesture"
	"golang.org/x/exp/shiny/iconvg"
	"golang.org/x/exp/shiny/materialdesign/icons"
	"golang.org/x/exp/shiny/screen"
	"golang.org/x/exp/shiny/unit"
	"golang.org/x/exp/shiny/widget"
	"golang.org/x/exp/shiny/widget/node"
	"golang.org/x/exp/shiny/widget/theme"
)

func stretch(n node.Node, alongWeight int) node.Node {
	return widget.WithLayoutData(n, widget.FlowLayoutData{
		AlongWeight:  alongWeight,
		ExpandAlong:  true,
		ShrinkAlong:  true,
		ExpandAcross: true,
		ShrinkAcross: true,
	})
}

type Body struct {
	node.LeafEmbed
	index int
	z     iconvg.Rasterizer
}

func NewBody() *Body {
	w := &Body{}
	w.Wrapper = w
	return w
}

func (w *Body) PaintBase(ctx *node.PaintBaseContext, origin image.Point) error {
	w.Marks.UnmarkNeedsPaintBase()
	wr := w.Rect.Add(origin)
	for y := wr.Min.Y; y < wr.Max.Y; y += checkersLen {
		for x := wr.Min.X; x < wr.Max.X; x += checkersLen {
			r := wr.Intersect(image.Rectangle{
				Min: image.Point{x + 0*checkersLen, y + 0*checkersLen},
				Max: image.Point{x + 1*checkersLen, y + 1*checkersLen},
			})
			draw.Draw(ctx.Dst, r, checkers, image.Point{}, draw.Src)
		}
	}

	wr = wr.Inset(16)
	if wr.Empty() {
		return nil
	}
	sz := wr.Size()
	if d := sz.X - sz.Y; d > 0 {
		wr.Min.X += d / 2
		wr.Max.X = wr.Min.X + sz.Y
	} else if d < 0 {
		wr.Min.Y -= d / 2
		wr.Max.Y = wr.Min.Y + sz.X
	}
	w.z.SetDstImage(ctx.Dst, wr, draw.Over)
	return iconvg.Decode(&w.z, iconList[w.index].data, nil)
}

// TODO: make a standard widget.Button.

type Button struct {
	node.LeafEmbed
	icon    []byte
	onClick func()
	z       iconvg.Rasterizer
}

func NewButton(icon []byte, onClick func()) *Button {
	w := &Button{
		icon:    icon,
		onClick: onClick,
	}
	w.Wrapper = w
	return w
}

func (w *Button) Measure(t *theme.Theme, widthHint, heightHint int) {
	px := t.Pixels(unit.Ems(1)).Ceil()
	w.MeasuredSize = image.Point{px, px}
}

func (w *Button) PaintBase(ctx *node.PaintBaseContext, origin image.Point) error {
	w.Marks.UnmarkNeedsPaintBase()
	w.z.SetDstImage(ctx.Dst, w.Rect.Add(origin), draw.Over)
	return iconvg.Decode(&w.z, w.icon, nil)
}

func (w *Button) OnInputEvent(e interface{}, origin image.Point) node.EventHandled {
	switch e := e.(type) {
	case gesture.Event:
		if e.Type != gesture.TypeTap {
			break
		}
		if w.onClick != nil {
			w.onClick()
		}
		return node.Handled
	}
	return node.NotHandled
}

func main() {
	log.SetFlags(0)
	driver.Main(func(s screen.Screen) {
		body := NewBody()
		iconName := widget.NewLabel(iconList[0].name)
		inc := func(delta int) {
			body.index += delta
			if body.index < 0 {
				body.index += len(iconList)
			} else if body.index >= len(iconList) {
				body.index -= len(iconList)
			}
			body.Mark(node.MarkNeedsPaintBase)
			iconName.Text = iconList[body.index].name
			iconName.Mark(node.MarkNeedsMeasureLayout | node.MarkNeedsPaintBase)
		}

		header := widget.NewUniform(theme.Neutral,
			widget.NewPadder(widget.AxisBoth, unit.Ems(0.5),
				widget.NewFlow(widget.AxisHorizontal,
					NewButton(icons.NavigationChevronLeft, func() { inc(-1) }),
					widget.NewSizer(unit.Ems(0.5), unit.Value{}, nil),
					NewButton(icons.NavigationChevronRight, func() { inc(+1) }),
					widget.NewSizer(unit.Ems(0.5), unit.Value{}, nil),
					stretch(iconName, 1),
				),
			),
		)
		divider := widget.NewSizer(unit.Value{}, unit.DIPs(2),
			widget.NewUniform(theme.Foreground, nil),
		)

		w := widget.NewFlow(widget.AxisVertical,
			stretch(widget.NewSheet(header), 0),
			stretch(widget.NewSheet(divider), 0),
			stretch(widget.NewSheet(body), 1),
		)

		if err := widget.RunWindow(s, w, &widget.RunWindowOptions{
			NewWindowOptions: screen.NewWindowOptions{
				Title: "IconGallery Shiny Example",
			},
		}); err != nil {
			log.Fatal(err)
		}
	})
}

const checkersLen = 256

var checkers = func() image.Image {
	m := image.NewRGBA(image.Rect(0, 0, checkersLen, checkersLen))
	pix := m.Pix
	for y := 0; y < checkersLen; y++ {
		for x := 0; x < checkersLen; x++ {
			g, b := uint8(0xc0), uint8(0xff)
			if x&16 == y&16 {
				g, b = b, g
			}
			pix[(y*checkersLen+x)*4+0] = 0xc0
			pix[(y*checkersLen+x)*4+1] = g
			pix[(y*checkersLen+x)*4+2] = b
			pix[(y*checkersLen+x)*4+3] = 0xff
		}
	}
	return m
}()

var iconList = []struct {
	name string
	data []byte
}{
	{"Action3DRotation", icons.Action3DRotation},
	{"ActionAccessibility", icons.ActionAccessibility},
	{"ActionAccessible", icons.ActionAccessible},
	{"ActionAccountBalance", icons.ActionAccountBalance},
	{"ActionAccountBalanceWallet", icons.ActionAccountBalanceWallet},
	{"ActionAccountBox", icons.ActionAccountBox},
	{"ActionAccountCircle", icons.ActionAccountCircle},
	{"ActionAddShoppingCart", icons.ActionAddShoppingCart},
	{"ActionAlarm", icons.ActionAlarm},
	{"ActionAlarmAdd", icons.ActionAlarmAdd},
	{"ActionAlarmOff", icons.ActionAlarmOff},
	{"ActionAlarmOn", icons.ActionAlarmOn},
	{"ActionAllOut", icons.ActionAllOut},
	{"ActionAndroid", icons.ActionAndroid},
	{"ActionAnnouncement", icons.ActionAnnouncement},
	{"ActionAspectRatio", icons.ActionAspectRatio},
	{"ActionAssessment", icons.ActionAssessment},
	{"ActionAssignment", icons.ActionAssignment},
	{"ActionAssignmentInd", icons.ActionAssignmentInd},
	{"ActionAssignmentLate", icons.ActionAssignmentLate},
	{"ActionAssignmentReturn", icons.ActionAssignmentReturn},
	{"ActionAssignmentReturned", icons.ActionAssignmentReturned},
	{"ActionAssignmentTurnedIn", icons.ActionAssignmentTurnedIn},
	{"ActionAutorenew", icons.ActionAutorenew},
	{"ActionBackup", icons.ActionBackup},
	{"ActionBook", icons.ActionBook},
	{"ActionBookmark", icons.ActionBookmark},
	{"ActionBookmarkBorder", icons.ActionBookmarkBorder},
	{"ActionBugReport", icons.ActionBugReport},
	{"ActionBuild", icons.ActionBuild},
	{"ActionCached", icons.ActionCached},
	{"ActionCameraEnhance", icons.ActionCameraEnhance},
	{"ActionCardGiftcard", icons.ActionCardGiftcard},
	{"ActionCardMembership", icons.ActionCardMembership},
	{"ActionCardTravel", icons.ActionCardTravel},
	{"ActionChangeHistory", icons.ActionChangeHistory},
	{"ActionCheckCircle", icons.ActionCheckCircle},
	{"ActionChromeReaderMode", icons.ActionChromeReaderMode},
	{"ActionClass", icons.ActionClass},
	{"ActionCode", icons.ActionCode},
	{"ActionCompareArrows", icons.ActionCompareArrows},
	{"ActionCopyright", icons.ActionCopyright},
	{"ActionCreditCard", icons.ActionCreditCard},
	{"ActionDashboard", icons.ActionDashboard},
	{"ActionDateRange", icons.ActionDateRange},
	{"ActionDelete", icons.ActionDelete},
	{"ActionDeleteForever", icons.ActionDeleteForever},
	{"ActionDescription", icons.ActionDescription},
	{"ActionDNS", icons.ActionDNS},
	{"ActionDone", icons.ActionDone},
	{"ActionDoneAll", icons.ActionDoneAll},
	{"ActionDonutLarge", icons.ActionDonutLarge},
	{"ActionDonutSmall", icons.ActionDonutSmall},
	{"ActionEject", icons.ActionEject},
	{"ActionEuroSymbol", icons.ActionEuroSymbol},
	{"ActionEvent", icons.ActionEvent},
	{"ActionEventSeat", icons.ActionEventSeat},
	{"ActionExitToApp", icons.ActionExitToApp},
	{"ActionExplore", icons.ActionExplore},
	{"ActionExtension", icons.ActionExtension},
	{"ActionFace", icons.ActionFace},
	{"ActionFavorite", icons.ActionFavorite},
	{"ActionFavoriteBorder", icons.ActionFavoriteBorder},
	{"ActionFeedback", icons.ActionFeedback},
	{"ActionFindInPage", icons.ActionFindInPage},
	{"ActionFindReplace", icons.ActionFindReplace},
	{"ActionFingerprint", icons.ActionFingerprint},
	{"ActionFlightLand", icons.ActionFlightLand},
	{"ActionFlightTakeoff", icons.ActionFlightTakeoff},
	{"ActionFlipToBack", icons.ActionFlipToBack},
	{"ActionFlipToFront", icons.ActionFlipToFront},
	{"ActionGTranslate", icons.ActionGTranslate},
	{"ActionGavel", icons.ActionGavel},
	{"ActionGetApp", icons.ActionGetApp},
	{"ActionGIF", icons.ActionGIF},
	{"ActionGrade", icons.ActionGrade},
	{"ActionGroupWork", icons.ActionGroupWork},
	{"ActionHelp", icons.ActionHelp},
	{"ActionHelpOutline", icons.ActionHelpOutline},
	{"ActionHighlightOff", icons.ActionHighlightOff},
	{"ActionHistory", icons.ActionHistory},
	{"ActionHome", icons.ActionHome},
	{"ActionHourglassEmpty", icons.ActionHourglassEmpty},
	{"ActionHourglassFull", icons.ActionHourglassFull},
	{"ActionHTTP", icons.ActionHTTP},
	{"ActionHTTPS", icons.ActionHTTPS},
	{"ActionImportantDevices", icons.ActionImportantDevices},
	{"ActionInfo", icons.ActionInfo},
	{"ActionInfoOutline", icons.ActionInfoOutline},
	{"ActionInput", icons.ActionInput},
	{"ActionInvertColors", icons.ActionInvertColors},
	{"ActionLabel", icons.ActionLabel},
	{"ActionLabelOutline", icons.ActionLabelOutline},
	{"ActionLanguage", icons.ActionLanguage},
	{"ActionLaunch", icons.ActionLaunch},
	{"ActionLightbulbOutline", icons.ActionLightbulbOutline},
	{"ActionLineStyle", icons.ActionLineStyle},
	{"ActionLineWeight", icons.ActionLineWeight},
	{"ActionList", icons.ActionList},
	{"ActionLock", icons.ActionLock},
	{"ActionLockOpen", icons.ActionLockOpen},
	{"ActionLockOutline", icons.ActionLockOutline},
	{"ActionLoyalty", icons.ActionLoyalty},
	{"ActionMarkUnreadMailbox", icons.ActionMarkUnreadMailbox},
	{"ActionMotorcycle", icons.ActionMotorcycle},
	{"ActionNoteAdd", icons.ActionNoteAdd},
	{"ActionOfflinePin", icons.ActionOfflinePin},
	{"ActionOpacity", icons.ActionOpacity},
	{"ActionOpenInBrowser", icons.ActionOpenInBrowser},
	{"ActionOpenInNew", icons.ActionOpenInNew},
	{"ActionOpenWith", icons.ActionOpenWith},
	{"ActionPageview", icons.ActionPageview},
	{"ActionPanTool", icons.ActionPanTool},
	{"ActionPayment", icons.ActionPayment},
	{"ActionPermCameraMic", icons.ActionPermCameraMic},
	{"ActionPermContactCalendar", icons.ActionPermContactCalendar},
	{"ActionPermDataSetting", icons.ActionPermDataSetting},
	{"ActionPermDeviceInformation", icons.ActionPermDeviceInformation},
	{"ActionPermIdentity", icons.ActionPermIdentity},
	{"ActionPermMedia", icons.ActionPermMedia},
	{"ActionPermPhoneMsg", icons.ActionPermPhoneMsg},
	{"ActionPermScanWiFi", icons.ActionPermScanWiFi},
	{"ActionPets", icons.ActionPets},
	{"ActionPictureInPicture", icons.ActionPictureInPicture},
	{"ActionPictureInPictureAlt", icons.ActionPictureInPictureAlt},
	{"ActionPlayForWork", icons.ActionPlayForWork},
	{"ActionPolymer", icons.ActionPolymer},
	{"ActionPowerSettingsNew", icons.ActionPowerSettingsNew},
	{"ActionPregnantWoman", icons.ActionPregnantWoman},
	{"ActionPrint", icons.ActionPrint},
	{"ActionQueryBuilder", icons.ActionQueryBuilder},
	{"ActionQuestionAnswer", icons.ActionQuestionAnswer},
	{"ActionReceipt", icons.ActionReceipt},
	{"ActionRecordVoiceOver", icons.ActionRecordVoiceOver},
	{"ActionRedeem", icons.ActionRedeem},
	{"ActionRemoveShoppingCart", icons.ActionRemoveShoppingCart},
	{"ActionReorder", icons.ActionReorder},
	{"ActionReportProblem", icons.ActionReportProblem},
	{"ActionRestore", icons.ActionRestore},
	{"ActionRestorePage", icons.ActionRestorePage},
	{"ActionRoom", icons.ActionRoom},
	{"ActionRoundedCorner", icons.ActionRoundedCorner},
	{"ActionRowing", icons.ActionRowing},
	{"ActionSchedule", icons.ActionSchedule},
	{"ActionSearch", icons.ActionSearch},
	{"ActionSettings", icons.ActionSettings},
	{"ActionSettingsApplications", icons.ActionSettingsApplications},
	{"ActionSettingsBackupRestore", icons.ActionSettingsBackupRestore},
	{"ActionSettingsBluetooth", icons.ActionSettingsBluetooth},
	{"ActionSettingsBrightness", icons.ActionSettingsBrightness},
	{"ActionSettingsCell", icons.ActionSettingsCell},
	{"ActionSettingsEthernet", icons.ActionSettingsEthernet},
	{"ActionSettingsInputAntenna", icons.ActionSettingsInputAntenna},
	{"ActionSettingsInputComponent", icons.ActionSettingsInputComponent},
	{"ActionSettingsInputComposite", icons.ActionSettingsInputComposite},
	{"ActionSettingsInputHDMI", icons.ActionSettingsInputHDMI},
	{"ActionSettingsInputSVideo", icons.ActionSettingsInputSVideo},
	{"ActionSettingsOverscan", icons.ActionSettingsOverscan},
	{"ActionSettingsPhone", icons.ActionSettingsPhone},
	{"ActionSettingsPower", icons.ActionSettingsPower},
	{"ActionSettingsRemote", icons.ActionSettingsRemote},
	{"ActionSettingsVoice", icons.ActionSettingsVoice},
	{"ActionShop", icons.ActionShop},
	{"ActionShopTwo", icons.ActionShopTwo},
	{"ActionShoppingBasket", icons.ActionShoppingBasket},
	{"ActionShoppingCart", icons.ActionShoppingCart},
	{"ActionSpeakerNotes", icons.ActionSpeakerNotes},
	{"ActionSpeakerNotesOff", icons.ActionSpeakerNotesOff},
	{"ActionSpellcheck", icons.ActionSpellcheck},
	{"ActionStarRate", icons.ActionStarRate},
	{"ActionStars", icons.ActionStars},
	{"ActionStore", icons.ActionStore},
	{"ActionSubject", icons.ActionSubject},
	{"ActionSupervisorAccount", icons.ActionSupervisorAccount},
	{"ActionSwapHoriz", icons.ActionSwapHoriz},
	{"ActionSwapVert", icons.ActionSwapVert},
	{"ActionSwapVerticalCircle", icons.ActionSwapVerticalCircle},
	{"ActionSystemUpdateAlt", icons.ActionSystemUpdateAlt},
	{"ActionTab", icons.ActionTab},
	{"ActionTabUnselected", icons.ActionTabUnselected},
	{"ActionTheaters", icons.ActionTheaters},
	{"ActionThumbDown", icons.ActionThumbDown},
	{"ActionThumbUp", icons.ActionThumbUp},
	{"ActionThumbsUpDown", icons.ActionThumbsUpDown},
	{"ActionTimeline", icons.ActionTimeline},
	{"ActionTOC", icons.ActionTOC},
	{"ActionToday", icons.ActionToday},
	{"ActionToll", icons.ActionToll},
	{"ActionTouchApp", icons.ActionTouchApp},
	{"ActionTrackChanges", icons.ActionTrackChanges},
	{"ActionTranslate", icons.ActionTranslate},
	{"ActionTrendingDown", icons.ActionTrendingDown},
	{"ActionTrendingFlat", icons.ActionTrendingFlat},
	{"ActionTrendingUp", icons.ActionTrendingUp},
	{"ActionTurnedIn", icons.ActionTurnedIn},
	{"ActionTurnedInNot", icons.ActionTurnedInNot},
	{"ActionUpdate", icons.ActionUpdate},
	{"ActionVerifiedUser", icons.ActionVerifiedUser},
	{"ActionViewAgenda", icons.ActionViewAgenda},
	{"ActionViewArray", icons.ActionViewArray},
	{"ActionViewCarousel", icons.ActionViewCarousel},
	{"ActionViewColumn", icons.ActionViewColumn},
	{"ActionViewDay", icons.ActionViewDay},
	{"ActionViewHeadline", icons.ActionViewHeadline},
	{"ActionViewList", icons.ActionViewList},
	{"ActionViewModule", icons.ActionViewModule},
	{"ActionViewQuilt", icons.ActionViewQuilt},
	{"ActionViewStream", icons.ActionViewStream},
	{"ActionViewWeek", icons.ActionViewWeek},
	{"ActionVisibility", icons.ActionVisibility},
	{"ActionVisibilityOff", icons.ActionVisibilityOff},
	{"ActionWatchLater", icons.ActionWatchLater},
	{"ActionWork", icons.ActionWork},
	{"ActionYoutubeSearchedFor", icons.ActionYoutubeSearchedFor},
	{"ActionZoomIn", icons.ActionZoomIn},
	{"ActionZoomOut", icons.ActionZoomOut},
	{"AlertAddAlert", icons.AlertAddAlert},
	{"AlertError", icons.AlertError},
	{"AlertErrorOutline", icons.AlertErrorOutline},
	{"AlertWarning", icons.AlertWarning},
	{"AVAddToQueue", icons.AVAddToQueue},
	{"AVAirplay", icons.AVAirplay},
	{"AVAlbum", icons.AVAlbum},
	{"AVArtTrack", icons.AVArtTrack},
	{"AVAVTimer", icons.AVAVTimer},
	{"AVBrandingWatermark", icons.AVBrandingWatermark},
	{"AVCallToAction", icons.AVCallToAction},
	{"AVClosedCaption", icons.AVClosedCaption},
	{"AVEqualizer", icons.AVEqualizer},
	{"AVExplicit", icons.AVExplicit},
	{"AVFastForward", icons.AVFastForward},
	{"AVFastRewind", icons.AVFastRewind},
	{"AVFeaturedPlayList", icons.AVFeaturedPlayList},
	{"AVFeaturedVideo", icons.AVFeaturedVideo},
	{"AVFiberDVR", icons.AVFiberDVR},
	{"AVFiberManualRecord", icons.AVFiberManualRecord},
	{"AVFiberNew", icons.AVFiberNew},
	{"AVFiberPin", icons.AVFiberPin},
	{"AVFiberSmartRecord", icons.AVFiberSmartRecord},
	{"AVForward10", icons.AVForward10},
	{"AVForward30", icons.AVForward30},
	{"AVForward5", icons.AVForward5},
	{"AVGames", icons.AVGames},
	{"AVHD", icons.AVHD},
	{"AVHearing", icons.AVHearing},
	{"AVHighQuality", icons.AVHighQuality},
	{"AVLibraryAdd", icons.AVLibraryAdd},
	{"AVLibraryBooks", icons.AVLibraryBooks},
	{"AVLibraryMusic", icons.AVLibraryMusic},
	{"AVLoop", icons.AVLoop},
	{"AVMic", icons.AVMic},
	{"AVMicNone", icons.AVMicNone},
	{"AVMicOff", icons.AVMicOff},
	{"AVMovie", icons.AVMovie},
	{"AVMusicVideo", icons.AVMusicVideo},
	{"AVNewReleases", icons.AVNewReleases},
	{"AVNotInterested", icons.AVNotInterested},
	{"AVNote", icons.AVNote},
	{"AVPause", icons.AVPause},
	{"AVPauseCircleFilled", icons.AVPauseCircleFilled},
	{"AVPauseCircleOutline", icons.AVPauseCircleOutline},
	{"AVPlayArrow", icons.AVPlayArrow},
	{"AVPlayCircleFilled", icons.AVPlayCircleFilled},
	{"AVPlayCircleOutline", icons.AVPlayCircleOutline},
	{"AVPlaylistAdd", icons.AVPlaylistAdd},
	{"AVPlaylistAddCheck", icons.AVPlaylistAddCheck},
	{"AVPlaylistPlay", icons.AVPlaylistPlay},
	{"AVQueue", icons.AVQueue},
	{"AVQueueMusic", icons.AVQueueMusic},
	{"AVQueuePlayNext", icons.AVQueuePlayNext},
	{"AVRadio", icons.AVRadio},
	{"AVRecentActors", icons.AVRecentActors},
	{"AVRemoveFromQueue", icons.AVRemoveFromQueue},
	{"AVRepeat", icons.AVRepeat},
	{"AVRepeatOne", icons.AVRepeatOne},
	{"AVReplay", icons.AVReplay},
	{"AVReplay10", icons.AVReplay10},
	{"AVReplay30", icons.AVReplay30},
	{"AVReplay5", icons.AVReplay5},
	{"AVShuffle", icons.AVShuffle},
	{"AVSkipNext", icons.AVSkipNext},
	{"AVSkipPrevious", icons.AVSkipPrevious},
	{"AVSlowMotionVideo", icons.AVSlowMotionVideo},
	{"AVSnooze", icons.AVSnooze},
	{"AVSortByAlpha", icons.AVSortByAlpha},
	{"AVStop", icons.AVStop},
	{"AVSubscriptions", icons.AVSubscriptions},
	{"AVSubtitles", icons.AVSubtitles},
	{"AVSurroundSound", icons.AVSurroundSound},
	{"AVVideoCall", icons.AVVideoCall},
	{"AVVideoLabel", icons.AVVideoLabel},
	{"AVVideoLibrary", icons.AVVideoLibrary},
	{"AVVideocam", icons.AVVideocam},
	{"AVVideocamOff", icons.AVVideocamOff},
	{"AVVolumeDown", icons.AVVolumeDown},
	{"AVVolumeMute", icons.AVVolumeMute},
	{"AVVolumeOff", icons.AVVolumeOff},
	{"AVVolumeUp", icons.AVVolumeUp},
	{"AVWeb", icons.AVWeb},
	{"AVWebAsset", icons.AVWebAsset},
	{"CommunicationBusiness", icons.CommunicationBusiness},
	{"CommunicationCall", icons.CommunicationCall},
	{"CommunicationCallEnd", icons.CommunicationCallEnd},
	{"CommunicationCallMade", icons.CommunicationCallMade},
	{"CommunicationCallMerge", icons.CommunicationCallMerge},
	{"CommunicationCallMissed", icons.CommunicationCallMissed},
	{"CommunicationCallMissedOutgoing", icons.CommunicationCallMissedOutgoing},
	{"CommunicationCallReceived", icons.CommunicationCallReceived},
	{"CommunicationCallSplit", icons.CommunicationCallSplit},
	{"CommunicationChat", icons.CommunicationChat},
	{"CommunicationChatBubble", icons.CommunicationChatBubble},
	{"CommunicationChatBubbleOutline", icons.CommunicationChatBubbleOutline},
	{"CommunicationClearAll", icons.CommunicationClearAll},
	{"CommunicationComment", icons.CommunicationComment},
	{"CommunicationContactMail", icons.CommunicationContactMail},
	{"CommunicationContactPhone", icons.CommunicationContactPhone},
	{"CommunicationContacts", icons.CommunicationContacts},
	{"CommunicationDialerSIP", icons.CommunicationDialerSIP},
	{"CommunicationDialpad", icons.CommunicationDialpad},
	{"CommunicationEmail", icons.CommunicationEmail},
	{"CommunicationForum", icons.CommunicationForum},
	{"CommunicationImportContacts", icons.CommunicationImportContacts},
	{"CommunicationImportExport", icons.CommunicationImportExport},
	{"CommunicationInvertColorsOff", icons.CommunicationInvertColorsOff},
	{"CommunicationLiveHelp", icons.CommunicationLiveHelp},
	{"CommunicationLocationOff", icons.CommunicationLocationOff},
	{"CommunicationLocationOn", icons.CommunicationLocationOn},
	{"CommunicationMailOutline", icons.CommunicationMailOutline},
	{"CommunicationMessage", icons.CommunicationMessage},
	{"CommunicationNoSIM", icons.CommunicationNoSIM},
	{"CommunicationPhone", icons.CommunicationPhone},
	{"CommunicationPhoneLinkErase", icons.CommunicationPhoneLinkErase},
	{"CommunicationPhoneLinkLock", icons.CommunicationPhoneLinkLock},
	{"CommunicationPhoneLinkRing", icons.CommunicationPhoneLinkRing},
	{"CommunicationPhoneLinkSetup", icons.CommunicationPhoneLinkSetup},
	{"CommunicationPortableWiFiOff", icons.CommunicationPortableWiFiOff},
	{"CommunicationPresentToAll", icons.CommunicationPresentToAll},
	{"CommunicationRingVolume", icons.CommunicationRingVolume},
	{"CommunicationRSSFeed", icons.CommunicationRSSFeed},
	{"CommunicationScreenShare", icons.CommunicationScreenShare},
	{"CommunicationSpeakerPhone", icons.CommunicationSpeakerPhone},
	{"CommunicationStayCurrentLandscape", icons.CommunicationStayCurrentLandscape},
	{"CommunicationStayCurrentPortrait", icons.CommunicationStayCurrentPortrait},
	{"CommunicationStayPrimaryLandscape", icons.CommunicationStayPrimaryLandscape},
	{"CommunicationStayPrimaryPortrait", icons.CommunicationStayPrimaryPortrait},
	{"CommunicationStopScreenShare", icons.CommunicationStopScreenShare},
	{"CommunicationSwapCalls", icons.CommunicationSwapCalls},
	{"CommunicationTextSMS", icons.CommunicationTextSMS},
	{"CommunicationVoicemail", icons.CommunicationVoicemail},
	{"CommunicationVPNKey", icons.CommunicationVPNKey},
	{"ContentAdd", icons.ContentAdd},
	{"ContentAddBox", icons.ContentAddBox},
	{"ContentAddCircle", icons.ContentAddCircle},
	{"ContentAddCircleOutline", icons.ContentAddCircleOutline},
	{"ContentArchive", icons.ContentArchive},
	{"ContentBackspace", icons.ContentBackspace},
	{"ContentBlock", icons.ContentBlock},
	{"ContentClear", icons.ContentClear},
	{"ContentContentCopy", icons.ContentContentCopy},
	{"ContentContentCut", icons.ContentContentCut},
	{"ContentContentPaste", icons.ContentContentPaste},
	{"ContentCreate", icons.ContentCreate},
	{"ContentDeleteSweep", icons.ContentDeleteSweep},
	{"ContentDrafts", icons.ContentDrafts},
	{"ContentFilterList", icons.ContentFilterList},
	{"ContentFlag", icons.ContentFlag},
	{"ContentFontDownload", icons.ContentFontDownload},
	{"ContentForward", icons.ContentForward},
	{"ContentGesture", icons.ContentGesture},
	{"ContentInbox", icons.ContentInbox},
	{"ContentLink", icons.ContentLink},
	{"ContentLowPriority", icons.ContentLowPriority},
	{"ContentMail", icons.ContentMail},
	{"ContentMarkUnread", icons.ContentMarkUnread},
	{"ContentMoveToInbox", icons.ContentMoveToInbox},
	{"ContentNextWeek", icons.ContentNextWeek},
	{"ContentRedo", icons.ContentRedo},
	{"ContentRemove", icons.ContentRemove},
	{"ContentRemoveCircle", icons.ContentRemoveCircle},
	{"ContentRemoveCircleOutline", icons.ContentRemoveCircleOutline},
	{"ContentReply", icons.ContentReply},
	{"ContentReplyAll", icons.ContentReplyAll},
	{"ContentReport", icons.ContentReport},
	{"ContentSave", icons.ContentSave},
	{"ContentSelectAll", icons.ContentSelectAll},
	{"ContentSend", icons.ContentSend},
	{"ContentSort", icons.ContentSort},
	{"ContentTextFormat", icons.ContentTextFormat},
	{"ContentUnarchive", icons.ContentUnarchive},
	{"ContentUndo", icons.ContentUndo},
	{"ContentWeekend", icons.ContentWeekend},
	{"DeviceAccessAlarm", icons.DeviceAccessAlarm},
	{"DeviceAccessAlarms", icons.DeviceAccessAlarms},
	{"DeviceAccessTime", icons.DeviceAccessTime},
	{"DeviceAddAlarm", icons.DeviceAddAlarm},
	{"DeviceAirplaneModeActive", icons.DeviceAirplaneModeActive},
	{"DeviceAirplaneModeInactive", icons.DeviceAirplaneModeInactive},
	{"DeviceBattery20", icons.DeviceBattery20},
	{"DeviceBattery30", icons.DeviceBattery30},
	{"DeviceBattery50", icons.DeviceBattery50},
	{"DeviceBattery60", icons.DeviceBattery60},
	{"DeviceBattery80", icons.DeviceBattery80},
	{"DeviceBattery90", icons.DeviceBattery90},
	{"DeviceBatteryAlert", icons.DeviceBatteryAlert},
	{"DeviceBatteryCharging20", icons.DeviceBatteryCharging20},
	{"DeviceBatteryCharging30", icons.DeviceBatteryCharging30},
	{"DeviceBatteryCharging50", icons.DeviceBatteryCharging50},
	{"DeviceBatteryCharging60", icons.DeviceBatteryCharging60},
	{"DeviceBatteryCharging80", icons.DeviceBatteryCharging80},
	{"DeviceBatteryCharging90", icons.DeviceBatteryCharging90},
	{"DeviceBatteryChargingFull", icons.DeviceBatteryChargingFull},
	{"DeviceBatteryFull", icons.DeviceBatteryFull},
	{"DeviceBatteryStd", icons.DeviceBatteryStd},
	{"DeviceBatteryUnknown", icons.DeviceBatteryUnknown},
	{"DeviceBluetooth", icons.DeviceBluetooth},
	{"DeviceBluetoothConnected", icons.DeviceBluetoothConnected},
	{"DeviceBluetoothDisabled", icons.DeviceBluetoothDisabled},
	{"DeviceBluetoothSearching", icons.DeviceBluetoothSearching},
	{"DeviceBrightnessAuto", icons.DeviceBrightnessAuto},
	{"DeviceBrightnessHigh", icons.DeviceBrightnessHigh},
	{"DeviceBrightnessLow", icons.DeviceBrightnessLow},
	{"DeviceBrightnessMedium", icons.DeviceBrightnessMedium},
	{"DeviceDataUsage", icons.DeviceDataUsage},
	{"DeviceDeveloperMode", icons.DeviceDeveloperMode},
	{"DeviceDevices", icons.DeviceDevices},
	{"DeviceDVR", icons.DeviceDVR},
	{"DeviceGPSFixed", icons.DeviceGPSFixed},
	{"DeviceGPSNotFixed", icons.DeviceGPSNotFixed},
	{"DeviceGPSOff", icons.DeviceGPSOff},
	{"DeviceGraphicEq", icons.DeviceGraphicEq},
	{"DeviceLocationDisabled", icons.DeviceLocationDisabled},
	{"DeviceLocationSearching", icons.DeviceLocationSearching},
	{"DeviceNetworkCell", icons.DeviceNetworkCell},
	{"DeviceNetworkWiFi", icons.DeviceNetworkWiFi},
	{"DeviceNFC", icons.DeviceNFC},
	{"DeviceScreenLockLandscape", icons.DeviceScreenLockLandscape},
	{"DeviceScreenLockPortrait", icons.DeviceScreenLockPortrait},
	{"DeviceScreenLockRotation", icons.DeviceScreenLockRotation},
	{"DeviceScreenRotation", icons.DeviceScreenRotation},
	{"DeviceSDStorage", icons.DeviceSDStorage},
	{"DeviceSettingsSystemDaydream", icons.DeviceSettingsSystemDaydream},
	{"DeviceSignalCellular0Bar", icons.DeviceSignalCellular0Bar},
	{"DeviceSignalCellular1Bar", icons.DeviceSignalCellular1Bar},
	{"DeviceSignalCellular2Bar", icons.DeviceSignalCellular2Bar},
	{"DeviceSignalCellular3Bar", icons.DeviceSignalCellular3Bar},
	{"DeviceSignalCellular4Bar", icons.DeviceSignalCellular4Bar},
	{"DeviceSignalCellularConnectedNoInternet0Bar", icons.DeviceSignalCellularConnectedNoInternet0Bar},
	{"DeviceSignalCellularConnectedNoInternet1Bar", icons.DeviceSignalCellularConnectedNoInternet1Bar},
	{"DeviceSignalCellularConnectedNoInternet2Bar", icons.DeviceSignalCellularConnectedNoInternet2Bar},
	{"DeviceSignalCellularConnectedNoInternet3Bar", icons.DeviceSignalCellularConnectedNoInternet3Bar},
	{"DeviceSignalCellularConnectedNoInternet4Bar", icons.DeviceSignalCellularConnectedNoInternet4Bar},
	{"DeviceSignalCellularNoSIM", icons.DeviceSignalCellularNoSIM},
	{"DeviceSignalCellularNull", icons.DeviceSignalCellularNull},
	{"DeviceSignalCellularOff", icons.DeviceSignalCellularOff},
	{"DeviceSignalWiFi0Bar", icons.DeviceSignalWiFi0Bar},
	{"DeviceSignalWiFi1Bar", icons.DeviceSignalWiFi1Bar},
	{"DeviceSignalWiFi1BarLock", icons.DeviceSignalWiFi1BarLock},
	{"DeviceSignalWiFi2Bar", icons.DeviceSignalWiFi2Bar},
	{"DeviceSignalWiFi2BarLock", icons.DeviceSignalWiFi2BarLock},
	{"DeviceSignalWiFi3Bar", icons.DeviceSignalWiFi3Bar},
	{"DeviceSignalWiFi3BarLock", icons.DeviceSignalWiFi3BarLock},
	{"DeviceSignalWiFi4Bar", icons.DeviceSignalWiFi4Bar},
	{"DeviceSignalWiFi4BarLock", icons.DeviceSignalWiFi4BarLock},
	{"DeviceSignalWiFiOff", icons.DeviceSignalWiFiOff},
	{"DeviceStorage", icons.DeviceStorage},
	{"DeviceUSB", icons.DeviceUSB},
	{"DeviceWallpaper", icons.DeviceWallpaper},
	{"DeviceWidgets", icons.DeviceWidgets},
	{"DeviceWiFiLock", icons.DeviceWiFiLock},
	{"DeviceWiFiTethering", icons.DeviceWiFiTethering},
	{"EditorAttachFile", icons.EditorAttachFile},
	{"EditorAttachMoney", icons.EditorAttachMoney},
	{"EditorBorderAll", icons.EditorBorderAll},
	{"EditorBorderBottom", icons.EditorBorderBottom},
	{"EditorBorderClear", icons.EditorBorderClear},
	{"EditorBorderColor", icons.EditorBorderColor},
	{"EditorBorderHorizontal", icons.EditorBorderHorizontal},
	{"EditorBorderInner", icons.EditorBorderInner},
	{"EditorBorderLeft", icons.EditorBorderLeft},
	{"EditorBorderOuter", icons.EditorBorderOuter},
	{"EditorBorderRight", icons.EditorBorderRight},
	{"EditorBorderStyle", icons.EditorBorderStyle},
	{"EditorBorderTop", icons.EditorBorderTop},
	{"EditorBorderVertical", icons.EditorBorderVertical},
	{"EditorBubbleChart", icons.EditorBubbleChart},
	{"EditorDragHandle", icons.EditorDragHandle},
	{"EditorFormatAlignCenter", icons.EditorFormatAlignCenter},
	{"EditorFormatAlignJustify", icons.EditorFormatAlignJustify},
	{"EditorFormatAlignLeft", icons.EditorFormatAlignLeft},
	{"EditorFormatAlignRight", icons.EditorFormatAlignRight},
	{"EditorFormatBold", icons.EditorFormatBold},
	{"EditorFormatClear", icons.EditorFormatClear},
	{"EditorFormatColorFill", icons.EditorFormatColorFill},
	{"EditorFormatColorReset", icons.EditorFormatColorReset},
	{"EditorFormatColorText", icons.EditorFormatColorText},
	{"EditorFormatIndentDecrease", icons.EditorFormatIndentDecrease},
	{"EditorFormatIndentIncrease", icons.EditorFormatIndentIncrease},
	{"EditorFormatItalic", icons.EditorFormatItalic},
	{"EditorFormatLineSpacing", icons.EditorFormatLineSpacing},
	{"EditorFormatListBulleted", icons.EditorFormatListBulleted},
	{"EditorFormatListNumbered", icons.EditorFormatListNumbered},
	{"EditorFormatPaint", icons.EditorFormatPaint},
	{"EditorFormatQuote", icons.EditorFormatQuote},
	{"EditorFormatShapes", icons.EditorFormatShapes},
	{"EditorFormatSize", icons.EditorFormatSize},
	{"EditorFormatStrikethrough", icons.EditorFormatStrikethrough},
	{"EditorFormatTextDirectionLToR", icons.EditorFormatTextDirectionLToR},
	{"EditorFormatTextDirectionRToL", icons.EditorFormatTextDirectionRToL},
	{"EditorFormatUnderlined", icons.EditorFormatUnderlined},
	{"EditorFunctions", icons.EditorFunctions},
	{"EditorHighlight", icons.EditorHighlight},
	{"EditorInsertChart", icons.EditorInsertChart},
	{"EditorInsertComment", icons.EditorInsertComment},
	{"EditorInsertDriveFile", icons.EditorInsertDriveFile},
	{"EditorInsertEmoticon", icons.EditorInsertEmoticon},
	{"EditorInsertInvitation", icons.EditorInsertInvitation},
	{"EditorInsertLink", icons.EditorInsertLink},
	{"EditorInsertPhoto", icons.EditorInsertPhoto},
	{"EditorLinearScale", icons.EditorLinearScale},
	{"EditorMergeType", icons.EditorMergeType},
	{"EditorModeComment", icons.EditorModeComment},
	{"EditorModeEdit", icons.EditorModeEdit},
	{"EditorMonetizationOn", icons.EditorMonetizationOn},
	{"EditorMoneyOff", icons.EditorMoneyOff},
	{"EditorMultilineChart", icons.EditorMultilineChart},
	{"EditorPieChart", icons.EditorPieChart},
	{"EditorPieChartOutlined", icons.EditorPieChartOutlined},
	{"EditorPublish", icons.EditorPublish},
	{"EditorShortText", icons.EditorShortText},
	{"EditorShowChart", icons.EditorShowChart},
	{"EditorSpaceBar", icons.EditorSpaceBar},
	{"EditorStrikethroughS", icons.EditorStrikethroughS},
	{"EditorTextFields", icons.EditorTextFields},
	{"EditorTitle", icons.EditorTitle},
	{"EditorVerticalAlignBottom", icons.EditorVerticalAlignBottom},
	{"EditorVerticalAlignCenter", icons.EditorVerticalAlignCenter},
	{"EditorVerticalAlignTop", icons.EditorVerticalAlignTop},
	{"EditorWrapText", icons.EditorWrapText},
	{"FileAttachment", icons.FileAttachment},
	{"FileCloud", icons.FileCloud},
	{"FileCloudCircle", icons.FileCloudCircle},
	{"FileCloudDone", icons.FileCloudDone},
	{"FileCloudDownload", icons.FileCloudDownload},
	{"FileCloudOff", icons.FileCloudOff},
	{"FileCloudQueue", icons.FileCloudQueue},
	{"FileCloudUpload", icons.FileCloudUpload},
	{"FileCreateNewFolder", icons.FileCreateNewFolder},
	{"FileFileDownload", icons.FileFileDownload},
	{"FileFileUpload", icons.FileFileUpload},
	{"FileFolder", icons.FileFolder},
	{"FileFolderOpen", icons.FileFolderOpen},
	{"FileFolderShared", icons.FileFolderShared},
	{"HardwareCast", icons.HardwareCast},
	{"HardwareCastConnected", icons.HardwareCastConnected},
	{"HardwareComputer", icons.HardwareComputer},
	{"HardwareDesktopMac", icons.HardwareDesktopMac},
	{"HardwareDesktopWindows", icons.HardwareDesktopWindows},
	{"HardwareDeveloperBoard", icons.HardwareDeveloperBoard},
	{"HardwareDeviceHub", icons.HardwareDeviceHub},
	{"HardwareDevicesOther", icons.HardwareDevicesOther},
	{"HardwareDock", icons.HardwareDock},
	{"HardwareGamepad", icons.HardwareGamepad},
	{"HardwareHeadset", icons.HardwareHeadset},
	{"HardwareHeadsetMic", icons.HardwareHeadsetMic},
	{"HardwareKeyboard", icons.HardwareKeyboard},
	{"HardwareKeyboardArrowDown", icons.HardwareKeyboardArrowDown},
	{"HardwareKeyboardArrowLeft", icons.HardwareKeyboardArrowLeft},
	{"HardwareKeyboardArrowRight", icons.HardwareKeyboardArrowRight},
	{"HardwareKeyboardArrowUp", icons.HardwareKeyboardArrowUp},
	{"HardwareKeyboardBackspace", icons.HardwareKeyboardBackspace},
	{"HardwareKeyboardCapslock", icons.HardwareKeyboardCapslock},
	{"HardwareKeyboardHide", icons.HardwareKeyboardHide},
	{"HardwareKeyboardReturn", icons.HardwareKeyboardReturn},
	{"HardwareKeyboardTab", icons.HardwareKeyboardTab},
	{"HardwareKeyboardVoice", icons.HardwareKeyboardVoice},
	{"HardwareLaptop", icons.HardwareLaptop},
	{"HardwareLaptopChromebook", icons.HardwareLaptopChromebook},
	{"HardwareLaptopMac", icons.HardwareLaptopMac},
	{"HardwareLaptopWindows", icons.HardwareLaptopWindows},
	{"HardwareMemory", icons.HardwareMemory},
	{"HardwareMouse", icons.HardwareMouse},
	{"HardwarePhoneAndroid", icons.HardwarePhoneAndroid},
	{"HardwarePhoneIPhone", icons.HardwarePhoneIPhone},
	{"HardwarePhoneLink", icons.HardwarePhoneLink},
	{"HardwarePhoneLinkOff", icons.HardwarePhoneLinkOff},
	{"HardwarePowerInput", icons.HardwarePowerInput},
	{"HardwareRouter", icons.HardwareRouter},
	{"HardwareScanner", icons.HardwareScanner},
	{"HardwareSecurity", icons.HardwareSecurity},
	{"HardwareSIMCard", icons.HardwareSIMCard},
	{"HardwareSmartphone", icons.HardwareSmartphone},
	{"HardwareSpeaker", icons.HardwareSpeaker},
	{"HardwareSpeakerGroup", icons.HardwareSpeakerGroup},
	{"HardwareTablet", icons.HardwareTablet},
	{"HardwareTabletAndroid", icons.HardwareTabletAndroid},
	{"HardwareTabletMac", icons.HardwareTabletMac},
	{"HardwareToys", icons.HardwareToys},
	{"HardwareTV", icons.HardwareTV},
	{"HardwareVideogameAsset", icons.HardwareVideogameAsset},
	{"HardwareWatch", icons.HardwareWatch},
	{"ImageAddAPhoto", icons.ImageAddAPhoto},
	{"ImageAddToPhotos", icons.ImageAddToPhotos},
	{"ImageAdjust", icons.ImageAdjust},
	{"ImageAssistant", icons.ImageAssistant},
	{"ImageAssistantPhoto", icons.ImageAssistantPhoto},
	{"ImageAudiotrack", icons.ImageAudiotrack},
	{"ImageBlurCircular", icons.ImageBlurCircular},
	{"ImageBlurLinear", icons.ImageBlurLinear},
	{"ImageBlurOff", icons.ImageBlurOff},
	{"ImageBlurOn", icons.ImageBlurOn},
	{"ImageBrightness1", icons.ImageBrightness1},
	{"ImageBrightness2", icons.ImageBrightness2},
	{"ImageBrightness3", icons.ImageBrightness3},
	{"ImageBrightness4", icons.ImageBrightness4},
	{"ImageBrightness5", icons.ImageBrightness5},
	{"ImageBrightness6", icons.ImageBrightness6},
	{"ImageBrightness7", icons.ImageBrightness7},
	{"ImageBrokenImage", icons.ImageBrokenImage},
	{"ImageBrush", icons.ImageBrush},
	{"ImageBurstMode", icons.ImageBurstMode},
	{"ImageCamera", icons.ImageCamera},
	{"ImageCameraAlt", icons.ImageCameraAlt},
	{"ImageCameraFront", icons.ImageCameraFront},
	{"ImageCameraRear", icons.ImageCameraRear},
	{"ImageCameraRoll", icons.ImageCameraRoll},
	{"ImageCenterFocusStrong", icons.ImageCenterFocusStrong},
	{"ImageCenterFocusWeak", icons.ImageCenterFocusWeak},
	{"ImageCollections", icons.ImageCollections},
	{"ImageCollectionsBookmark", icons.ImageCollectionsBookmark},
	{"ImageColorLens", icons.ImageColorLens},
	{"ImageColorize", icons.ImageColorize},
	{"ImageCompare", icons.ImageCompare},
	{"ImageControlPoint", icons.ImageControlPoint},
	{"ImageControlPointDuplicate", icons.ImageControlPointDuplicate},
	{"ImageCrop", icons.ImageCrop},
	{"ImageCrop169", icons.ImageCrop169},
	{"ImageCrop32", icons.ImageCrop32},
	{"ImageCrop54", icons.ImageCrop54},
	{"ImageCrop75", icons.ImageCrop75},
	{"ImageCropDIN", icons.ImageCropDIN},
	{"ImageCropFree", icons.ImageCropFree},
	{"ImageCropLandscape", icons.ImageCropLandscape},
	{"ImageCropOriginal", icons.ImageCropOriginal},
	{"ImageCropPortrait", icons.ImageCropPortrait},
	{"ImageCropRotate", icons.ImageCropRotate},
	{"ImageCropSquare", icons.ImageCropSquare},
	{"ImageDehaze", icons.ImageDehaze},
	{"ImageDetails", icons.ImageDetails},
	{"ImageEdit", icons.ImageEdit},
	{"ImageExposure", icons.ImageExposure},
	{"ImageExposureNeg1", icons.ImageExposureNeg1},
	{"ImageExposureNeg2", icons.ImageExposureNeg2},
	{"ImageExposurePlus1", icons.ImageExposurePlus1},
	{"ImageExposurePlus2", icons.ImageExposurePlus2},
	{"ImageExposureZero", icons.ImageExposureZero},
	{"ImageFilter", icons.ImageFilter},
	{"ImageFilter1", icons.ImageFilter1},
	{"ImageFilter2", icons.ImageFilter2},
	{"ImageFilter3", icons.ImageFilter3},
	{"ImageFilter4", icons.ImageFilter4},
	{"ImageFilter5", icons.ImageFilter5},
	{"ImageFilter6", icons.ImageFilter6},
	{"ImageFilter7", icons.ImageFilter7},
	{"ImageFilter8", icons.ImageFilter8},
	{"ImageFilter9", icons.ImageFilter9},
	{"ImageFilter9Plus", icons.ImageFilter9Plus},
	{"ImageFilterBAndW", icons.ImageFilterBAndW},
	{"ImageFilterCenterFocus", icons.ImageFilterCenterFocus},
	{"ImageFilterDrama", icons.ImageFilterDrama},
	{"ImageFilterFrames", icons.ImageFilterFrames},
	{"ImageFilterHDR", icons.ImageFilterHDR},
	{"ImageFilterNone", icons.ImageFilterNone},
	{"ImageFilterTiltShift", icons.ImageFilterTiltShift},
	{"ImageFilterVintage", icons.ImageFilterVintage},
	{"ImageFlare", icons.ImageFlare},
	{"ImageFlashAuto", icons.ImageFlashAuto},
	{"ImageFlashOff", icons.ImageFlashOff},
	{"ImageFlashOn", icons.ImageFlashOn},
	{"ImageFlip", icons.ImageFlip},
	{"ImageGradient", icons.ImageGradient},
	{"ImageGrain", icons.ImageGrain},
	{"ImageGridOff", icons.ImageGridOff},
	{"ImageGridOn", icons.ImageGridOn},
	{"ImageHDROff", icons.ImageHDROff},
	{"ImageHDROn", icons.ImageHDROn},
	{"ImageHDRStrong", icons.ImageHDRStrong},
	{"ImageHDRWeak", icons.ImageHDRWeak},
	{"ImageHealing", icons.ImageHealing},
	{"ImageImage", icons.ImageImage},
	{"ImageImageAspectRatio", icons.ImageImageAspectRatio},
	{"ImageISO", icons.ImageISO},
	{"ImageLandscape", icons.ImageLandscape},
	{"ImageLeakAdd", icons.ImageLeakAdd},
	{"ImageLeakRemove", icons.ImageLeakRemove},
	{"ImageLens", icons.ImageLens},
	{"ImageLinkedCamera", icons.ImageLinkedCamera},
	{"ImageLooks", icons.ImageLooks},
	{"ImageLooks3", icons.ImageLooks3},
	{"ImageLooks4", icons.ImageLooks4},
	{"ImageLooks5", icons.ImageLooks5},
	{"ImageLooks6", icons.ImageLooks6},
	{"ImageLooksOne", icons.ImageLooksOne},
	{"ImageLooksTwo", icons.ImageLooksTwo},
	{"ImageLoupe", icons.ImageLoupe},
	{"ImageMonochromePhotos", icons.ImageMonochromePhotos},
	{"ImageMovieCreation", icons.ImageMovieCreation},
	{"ImageMovieFilter", icons.ImageMovieFilter},
	{"ImageMusicNote", icons.ImageMusicNote},
	{"ImageNature", icons.ImageNature},
	{"ImageNaturePeople", icons.ImageNaturePeople},
	{"ImageNavigateBefore", icons.ImageNavigateBefore},
	{"ImageNavigateNext", icons.ImageNavigateNext},
	{"ImagePalette", icons.ImagePalette},
	{"ImagePanorama", icons.ImagePanorama},
	{"ImagePanoramaFishEye", icons.ImagePanoramaFishEye},
	{"ImagePanoramaHorizontal", icons.ImagePanoramaHorizontal},
	{"ImagePanoramaVertical", icons.ImagePanoramaVertical},
	{"ImagePanoramaWideAngle", icons.ImagePanoramaWideAngle},
	{"ImagePhoto", icons.ImagePhoto},
	{"ImagePhotoAlbum", icons.ImagePhotoAlbum},
	{"ImagePhotoCamera", icons.ImagePhotoCamera},
	{"ImagePhotoFilter", icons.ImagePhotoFilter},
	{"ImagePhotoLibrary", icons.ImagePhotoLibrary},
	{"ImagePhotoSizeSelectActual", icons.ImagePhotoSizeSelectActual},
	{"ImagePhotoSizeSelectLarge", icons.ImagePhotoSizeSelectLarge},
	{"ImagePhotoSizeSelectSmall", icons.ImagePhotoSizeSelectSmall},
	{"ImagePictureAsPDF", icons.ImagePictureAsPDF},
	{"ImagePortrait", icons.ImagePortrait},
	{"ImageRemoveRedEye", icons.ImageRemoveRedEye},
	{"ImageRotate90DegreesCCW", icons.ImageRotate90DegreesCCW},
	{"ImageRotateLeft", icons.ImageRotateLeft},
	{"ImageRotateRight", icons.ImageRotateRight},
	{"ImageSlideshow", icons.ImageSlideshow},
	{"ImageStraighten", icons.ImageStraighten},
	{"ImageStyle", icons.ImageStyle},
	{"ImageSwitchCamera", icons.ImageSwitchCamera},
	{"ImageSwitchVideo", icons.ImageSwitchVideo},
	{"ImageTagFaces", icons.ImageTagFaces},
	{"ImageTexture", icons.ImageTexture},
	{"ImageTimeLapse", icons.ImageTimeLapse},
	{"ImageTimer", icons.ImageTimer},
	{"ImageTimer10", icons.ImageTimer10},
	{"ImageTimer3", icons.ImageTimer3},
	{"ImageTimerOff", icons.ImageTimerOff},
	{"ImageTonality", icons.ImageTonality},
	{"ImageTransform", icons.ImageTransform},
	{"ImageTune", icons.ImageTune},
	{"ImageViewComfy", icons.ImageViewComfy},
	{"ImageViewCompact", icons.ImageViewCompact},
	{"ImageVignette", icons.ImageVignette},
	{"ImageWBAuto", icons.ImageWBAuto},
	{"ImageWBCloudy", icons.ImageWBCloudy},
	{"ImageWBIncandescent", icons.ImageWBIncandescent},
	{"ImageWBIridescent", icons.ImageWBIridescent},
	{"ImageWBSunny", icons.ImageWBSunny},
	{"MapsAddLocation", icons.MapsAddLocation},
	{"MapsBeenhere", icons.MapsBeenhere},
	{"MapsDirections", icons.MapsDirections},
	{"MapsDirectionsBike", icons.MapsDirectionsBike},
	{"MapsDirectionsBoat", icons.MapsDirectionsBoat},
	{"MapsDirectionsBus", icons.MapsDirectionsBus},
	{"MapsDirectionsCar", icons.MapsDirectionsCar},
	{"MapsDirectionsRailway", icons.MapsDirectionsRailway},
	{"MapsDirectionsRun", icons.MapsDirectionsRun},
	{"MapsDirectionsSubway", icons.MapsDirectionsSubway},
	{"MapsDirectionsTransit", icons.MapsDirectionsTransit},
	{"MapsDirectionsWalk", icons.MapsDirectionsWalk},
	{"MapsEditLocation", icons.MapsEditLocation},
	{"MapsEVStation", icons.MapsEVStation},
	{"MapsFlight", icons.MapsFlight},
	{"MapsHotel", icons.MapsHotel},
	{"MapsLayers", icons.MapsLayers},
	{"MapsLayersClear", icons.MapsLayersClear},
	{"MapsLocalActivity", icons.MapsLocalActivity},
	{"MapsLocalAirport", icons.MapsLocalAirport},
	{"MapsLocalATM", icons.MapsLocalATM},
	{"MapsLocalBar", icons.MapsLocalBar},
	{"MapsLocalCafe", icons.MapsLocalCafe},
	{"MapsLocalCarWash", icons.MapsLocalCarWash},
	{"MapsLocalConvenienceStore", icons.MapsLocalConvenienceStore},
	{"MapsLocalDining", icons.MapsLocalDining},
	{"MapsLocalDrink", icons.MapsLocalDrink},
	{"MapsLocalFlorist", icons.MapsLocalFlorist},
	{"MapsLocalGasStation", icons.MapsLocalGasStation},
	{"MapsLocalGroceryStore", icons.MapsLocalGroceryStore},
	{"MapsLocalHospital", icons.MapsLocalHospital},
	{"MapsLocalHotel", icons.MapsLocalHotel},
	{"MapsLocalLaundryService", icons.MapsLocalLaundryService},
	{"MapsLocalLibrary", icons.MapsLocalLibrary},
	{"MapsLocalMall", icons.MapsLocalMall},
	{"MapsLocalMovies", icons.MapsLocalMovies},
	{"MapsLocalOffer", icons.MapsLocalOffer},
	{"MapsLocalParking", icons.MapsLocalParking},
	{"MapsLocalPharmacy", icons.MapsLocalPharmacy},
	{"MapsLocalPhone", icons.MapsLocalPhone},
	{"MapsLocalPizza", icons.MapsLocalPizza},
	{"MapsLocalPlay", icons.MapsLocalPlay},
	{"MapsLocalPostOffice", icons.MapsLocalPostOffice},
	{"MapsLocalPrintshop", icons.MapsLocalPrintshop},
	{"MapsLocalSee", icons.MapsLocalSee},
	{"MapsLocalShipping", icons.MapsLocalShipping},
	{"MapsLocalTaxi", icons.MapsLocalTaxi},
	{"MapsMap", icons.MapsMap},
	{"MapsMyLocation", icons.MapsMyLocation},
	{"MapsNavigation", icons.MapsNavigation},
	{"MapsNearMe", icons.MapsNearMe},
	{"MapsPersonPin", icons.MapsPersonPin},
	{"MapsPersonPinCircle", icons.MapsPersonPinCircle},
	{"MapsPinDrop", icons.MapsPinDrop},
	{"MapsPlace", icons.MapsPlace},
	{"MapsRateReview", icons.MapsRateReview},
	{"MapsRestaurant", icons.MapsRestaurant},
	{"MapsRestaurantMenu", icons.MapsRestaurantMenu},
	{"MapsSatellite", icons.MapsSatellite},
	{"MapsStoreMallDirectory", icons.MapsStoreMallDirectory},
	{"MapsStreetView", icons.MapsStreetView},
	{"MapsSubway", icons.MapsSubway},
	{"MapsTerrain", icons.MapsTerrain},
	{"MapsTraffic", icons.MapsTraffic},
	{"MapsTrain", icons.MapsTrain},
	{"MapsTram", icons.MapsTram},
	{"MapsTransferWithinAStation", icons.MapsTransferWithinAStation},
	{"MapsZoomOutMap", icons.MapsZoomOutMap},
	{"NavigationApps", icons.NavigationApps},
	{"NavigationArrowBack", icons.NavigationArrowBack},
	{"NavigationArrowDownward", icons.NavigationArrowDownward},
	{"NavigationArrowDropDown", icons.NavigationArrowDropDown},
	{"NavigationArrowDropDownCircle", icons.NavigationArrowDropDownCircle},
	{"NavigationArrowDropUp", icons.NavigationArrowDropUp},
	{"NavigationArrowForward", icons.NavigationArrowForward},
	{"NavigationArrowUpward", icons.NavigationArrowUpward},
	{"NavigationCancel", icons.NavigationCancel},
	{"NavigationCheck", icons.NavigationCheck},
	{"NavigationChevronLeft", icons.NavigationChevronLeft},
	{"NavigationChevronRight", icons.NavigationChevronRight},
	{"NavigationClose", icons.NavigationClose},
	{"NavigationExpandLess", icons.NavigationExpandLess},
	{"NavigationExpandMore", icons.NavigationExpandMore},
	{"NavigationFirstPage", icons.NavigationFirstPage},
	{"NavigationFullscreen", icons.NavigationFullscreen},
	{"NavigationFullscreenExit", icons.NavigationFullscreenExit},
	{"NavigationLastPage", icons.NavigationLastPage},
	{"NavigationMenu", icons.NavigationMenu},
	{"NavigationMoreHoriz", icons.NavigationMoreHoriz},
	{"NavigationMoreVert", icons.NavigationMoreVert},
	{"NavigationRefresh", icons.NavigationRefresh},
	{"NavigationSubdirectoryArrowLeft", icons.NavigationSubdirectoryArrowLeft},
	{"NavigationSubdirectoryArrowRight", icons.NavigationSubdirectoryArrowRight},
	{"NavigationUnfoldLess", icons.NavigationUnfoldLess},
	{"NavigationUnfoldMore", icons.NavigationUnfoldMore},
	{"NotificationADB", icons.NotificationADB},
	{"NotificationAirlineSeatFlat", icons.NotificationAirlineSeatFlat},
	{"NotificationAirlineSeatFlatAngled", icons.NotificationAirlineSeatFlatAngled},
	{"NotificationAirlineSeatIndividualSuite", icons.NotificationAirlineSeatIndividualSuite},
	{"NotificationAirlineSeatLegroomExtra", icons.NotificationAirlineSeatLegroomExtra},
	{"NotificationAirlineSeatLegroomNormal", icons.NotificationAirlineSeatLegroomNormal},
	{"NotificationAirlineSeatLegroomReduced", icons.NotificationAirlineSeatLegroomReduced},
	{"NotificationAirlineSeatReclineExtra", icons.NotificationAirlineSeatReclineExtra},
	{"NotificationAirlineSeatReclineNormal", icons.NotificationAirlineSeatReclineNormal},
	{"NotificationBluetoothAudio", icons.NotificationBluetoothAudio},
	{"NotificationConfirmationNumber", icons.NotificationConfirmationNumber},
	{"NotificationDiscFull", icons.NotificationDiscFull},
	{"NotificationDoNotDisturb", icons.NotificationDoNotDisturb},
	{"NotificationDoNotDisturbAlt", icons.NotificationDoNotDisturbAlt},
	{"NotificationDoNotDisturbOff", icons.NotificationDoNotDisturbOff},
	{"NotificationDoNotDisturbOn", icons.NotificationDoNotDisturbOn},
	{"NotificationDriveETA", icons.NotificationDriveETA},
	{"NotificationEnhancedEncryption", icons.NotificationEnhancedEncryption},
	{"NotificationEventAvailable", icons.NotificationEventAvailable},
	{"NotificationEventBusy", icons.NotificationEventBusy},
	{"NotificationEventNote", icons.NotificationEventNote},
	{"NotificationFolderSpecial", icons.NotificationFolderSpecial},
	{"NotificationLiveTV", icons.NotificationLiveTV},
	{"NotificationMMS", icons.NotificationMMS},
	{"NotificationMore", icons.NotificationMore},
	{"NotificationNetworkCheck", icons.NotificationNetworkCheck},
	{"NotificationNetworkLocked", icons.NotificationNetworkLocked},
	{"NotificationNoEncryption", icons.NotificationNoEncryption},
	{"NotificationOnDemandVideo", icons.NotificationOnDemandVideo},
	{"NotificationPersonalVideo", icons.NotificationPersonalVideo},
	{"NotificationPhoneBluetoothSpeaker", icons.NotificationPhoneBluetoothSpeaker},
	{"NotificationPhoneForwarded", icons.NotificationPhoneForwarded},
	{"NotificationPhoneInTalk", icons.NotificationPhoneInTalk},
	{"NotificationPhoneLocked", icons.NotificationPhoneLocked},
	{"NotificationPhoneMissed", icons.NotificationPhoneMissed},
	{"NotificationPhonePaused", icons.NotificationPhonePaused},
	{"NotificationPower", icons.NotificationPower},
	{"NotificationPriorityHigh", icons.NotificationPriorityHigh},
	{"NotificationRVHookup", icons.NotificationRVHookup},
	{"NotificationSDCard", icons.NotificationSDCard},
	{"NotificationSIMCardAlert", icons.NotificationSIMCardAlert},
	{"NotificationSMS", icons.NotificationSMS},
	{"NotificationSMSFailed", icons.NotificationSMSFailed},
	{"NotificationSync", icons.NotificationSync},
	{"NotificationSyncDisabled", icons.NotificationSyncDisabled},
	{"NotificationSyncProblem", icons.NotificationSyncProblem},
	{"NotificationSystemUpdate", icons.NotificationSystemUpdate},
	{"NotificationTapAndPlay", icons.NotificationTapAndPlay},
	{"NotificationTimeToLeave", icons.NotificationTimeToLeave},
	{"NotificationVibration", icons.NotificationVibration},
	{"NotificationVoiceChat", icons.NotificationVoiceChat},
	{"NotificationVPNLock", icons.NotificationVPNLock},
	{"NotificationWC", icons.NotificationWC},
	{"NotificationWiFi", icons.NotificationWiFi},
	{"PlacesACUnit", icons.PlacesACUnit},
	{"PlacesAirportShuttle", icons.PlacesAirportShuttle},
	{"PlacesAllInclusive", icons.PlacesAllInclusive},
	{"PlacesBeachAccess", icons.PlacesBeachAccess},
	{"PlacesBusinessCenter", icons.PlacesBusinessCenter},
	{"PlacesCasino", icons.PlacesCasino},
	{"PlacesChildCare", icons.PlacesChildCare},
	{"PlacesChildFriendly", icons.PlacesChildFriendly},
	{"PlacesFitnessCenter", icons.PlacesFitnessCenter},
	{"PlacesFreeBreakfast", icons.PlacesFreeBreakfast},
	{"PlacesGolfCourse", icons.PlacesGolfCourse},
	{"PlacesHotTub", icons.PlacesHotTub},
	{"PlacesKitchen", icons.PlacesKitchen},
	{"PlacesPool", icons.PlacesPool},
	{"PlacesRoomService", icons.PlacesRoomService},
	{"PlacesRVHookup", icons.PlacesRVHookup},
	{"PlacesSmokeFree", icons.PlacesSmokeFree},
	{"PlacesSmokingRooms", icons.PlacesSmokingRooms},
	{"PlacesSpa", icons.PlacesSpa},
	{"SocialCake", icons.SocialCake},
	{"SocialDomain", icons.SocialDomain},
	{"SocialGroup", icons.SocialGroup},
	{"SocialGroupAdd", icons.SocialGroupAdd},
	{"SocialLocationCity", icons.SocialLocationCity},
	{"SocialMood", icons.SocialMood},
	{"SocialMoodBad", icons.SocialMoodBad},
	{"SocialNotifications", icons.SocialNotifications},
	{"SocialNotificationsActive", icons.SocialNotificationsActive},
	{"SocialNotificationsNone", icons.SocialNotificationsNone},
	{"SocialNotificationsOff", icons.SocialNotificationsOff},
	{"SocialNotificationsPaused", icons.SocialNotificationsPaused},
	{"SocialPages", icons.SocialPages},
	{"SocialPartyMode", icons.SocialPartyMode},
	{"SocialPeople", icons.SocialPeople},
	{"SocialPeopleOutline", icons.SocialPeopleOutline},
	{"SocialPerson", icons.SocialPerson},
	{"SocialPersonAdd", icons.SocialPersonAdd},
	{"SocialPersonOutline", icons.SocialPersonOutline},
	{"SocialPlusOne", icons.SocialPlusOne},
	{"SocialPoll", icons.SocialPoll},
	{"SocialPublic", icons.SocialPublic},
	{"SocialSchool", icons.SocialSchool},
	{"SocialSentimentDissatisfied", icons.SocialSentimentDissatisfied},
	{"SocialSentimentNeutral", icons.SocialSentimentNeutral},
	{"SocialSentimentSatisfied", icons.SocialSentimentSatisfied},
	{"SocialSentimentVeryDissatisfied", icons.SocialSentimentVeryDissatisfied},
	{"SocialSentimentVerySatisfied", icons.SocialSentimentVerySatisfied},
	{"SocialShare", icons.SocialShare},
	{"SocialWhatsHot", icons.SocialWhatsHot},
	{"ToggleCheckBox", icons.ToggleCheckBox},
	{"ToggleCheckBoxOutlineBlank", icons.ToggleCheckBoxOutlineBlank},
	{"ToggleIndeterminateCheckBox", icons.ToggleIndeterminateCheckBox},
	{"ToggleRadioButtonChecked", icons.ToggleRadioButtonChecked},
	{"ToggleRadioButtonUnchecked", icons.ToggleRadioButtonUnchecked},
	{"ToggleStar", icons.ToggleStar},
	{"ToggleStarBorder", icons.ToggleStarBorder},
	{"ToggleStarHalf", icons.ToggleStarHalf},
}
