/**
 * (C) Copyright IBM Corp. 2020.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Package transitgatewayapisv1 : Operations and models for the TransitGatewayApisV1 service
package transitgatewayapisv1

import (
	"encoding/json"
	"fmt"
	"github.com/IBM/go-sdk-core/v4/core"
	"github.com/go-openapi/strfmt"
	common "github.com/IBM/networking-go-sdk/common"
	"reflect"
)

// TransitGatewayApisV1 : No description provided (generated by Openapi Generator
// https://github.com/openapitools/openapi-generator)
//
// Version: __VERSION__
type TransitGatewayApisV1 struct {
	Service *core.BaseService

	// Requests the version of the API as of a date in the format `YYYY-MM-DD`. Any date up to the current date may be
	// provided. Specify the current date to request the latest version.
	Version *string
}

// DefaultServiceURL is the default URL to make service requests to.
const DefaultServiceURL = "https://transit.cloud.ibm.com/v1"

// DefaultServiceName is the default key used to find external configuration information.
const DefaultServiceName = "transit_gateway_apis"

// TransitGatewayApisV1Options : Service options
type TransitGatewayApisV1Options struct {
	ServiceName   string
	URL           string
	Authenticator core.Authenticator

	// Requests the version of the API as of a date in the format `YYYY-MM-DD`. Any date up to the current date may be
	// provided. Specify the current date to request the latest version.
	Version *string `validate:"required"`
}

// NewTransitGatewayApisV1UsingExternalConfig : constructs an instance of TransitGatewayApisV1 with passed in options and external configuration.
func NewTransitGatewayApisV1UsingExternalConfig(options *TransitGatewayApisV1Options) (transitGatewayApis *TransitGatewayApisV1, err error) {
	if options.ServiceName == "" {
		options.ServiceName = DefaultServiceName
	}

	if options.Authenticator == nil {
		options.Authenticator, err = core.GetAuthenticatorFromEnvironment(options.ServiceName)
		if err != nil {
			return
		}
	}

	transitGatewayApis, err = NewTransitGatewayApisV1(options)
	if err != nil {
		return
	}

	err = transitGatewayApis.Service.ConfigureService(options.ServiceName)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = transitGatewayApis.Service.SetServiceURL(options.URL)
	}
	return
}

// NewTransitGatewayApisV1 : constructs an instance of TransitGatewayApisV1 with passed in options.
func NewTransitGatewayApisV1(options *TransitGatewayApisV1Options) (service *TransitGatewayApisV1, err error) {
	serviceOptions := &core.ServiceOptions{
		URL:           DefaultServiceURL,
		Authenticator: options.Authenticator,
	}

	err = core.ValidateStruct(options, "options")
	if err != nil {
		return
	}

	baseService, err := core.NewBaseService(serviceOptions)
	if err != nil {
		return
	}

	if options.URL != "" {
		err = baseService.SetServiceURL(options.URL)
		if err != nil {
			return
		}
	}

	service = &TransitGatewayApisV1{
		Service: baseService,
		Version: options.Version,
	}

	return
}

// SetServiceURL sets the service URL
func (transitGatewayApis *TransitGatewayApisV1) SetServiceURL(url string) error {
	return transitGatewayApis.Service.SetServiceURL(url)
}

// ListTransitGateways : Retrieves all Transit Gateways
// List all the Transit Gateways in the account. User will get a list of Transit Gateways they have access to 'view'.
func (transitGatewayApis *TransitGatewayApisV1) ListTransitGateways(listTransitGatewaysOptions *ListTransitGatewaysOptions) (result *TransitGatewayCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listTransitGatewaysOptions, "listTransitGatewaysOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listTransitGatewaysOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "ListTransitGateways")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTransitGatewayCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateTransitGateway : Creates a Transit Gateway
// Create a Transit Gateway based on the supplied input template.
func (transitGatewayApis *TransitGatewayApisV1) CreateTransitGateway(createTransitGatewayOptions *CreateTransitGatewayOptions) (result *TransitGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createTransitGatewayOptions, "createTransitGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createTransitGatewayOptions, "createTransitGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.POST)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range createTransitGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "CreateTransitGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	body := make(map[string]interface{})
	if createTransitGatewayOptions.Location != nil {
		body["location"] = createTransitGatewayOptions.Location
	}
	if createTransitGatewayOptions.Name != nil {
		body["name"] = createTransitGatewayOptions.Name
	}
	if createTransitGatewayOptions.Global != nil {
		body["global"] = createTransitGatewayOptions.Global
	}
	if createTransitGatewayOptions.ResourceGroup != nil {
		body["resource_group"] = createTransitGatewayOptions.ResourceGroup
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTransitGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteTransitGateway : Deletes specified Transit Gateway
// This request deletes a Transit Gateway. This operation cannot be reversed. For this request to succeed, the Transit
// Gateway must not contain connections.
func (transitGatewayApis *TransitGatewayApisV1) DeleteTransitGateway(deleteTransitGatewayOptions *DeleteTransitGatewayOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteTransitGatewayOptions, "deleteTransitGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteTransitGatewayOptions, "deleteTransitGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways"}
	pathParameters := []string{*deleteTransitGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.DELETE)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteTransitGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "DeleteTransitGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = transitGatewayApis.Service.Request(request, nil)

	return
}

// GetTransitGateway : Retrieves specified Transit Gateway
// This request retrieves a single Transit Gateway specified by the identifier in the URL.
func (transitGatewayApis *TransitGatewayApisV1) GetTransitGateway(getTransitGatewayOptions *GetTransitGatewayOptions) (result *TransitGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getTransitGatewayOptions, "getTransitGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getTransitGatewayOptions, "getTransitGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways"}
	pathParameters := []string{*getTransitGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getTransitGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "GetTransitGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTransitGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateTransitGateway : Updates specified Transit Gateway
// This request updates a Transit Gateway's name and/or global flag.
func (transitGatewayApis *TransitGatewayApisV1) UpdateTransitGateway(updateTransitGatewayOptions *UpdateTransitGatewayOptions) (result *TransitGateway, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateTransitGatewayOptions, "updateTransitGatewayOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateTransitGatewayOptions, "updateTransitGatewayOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways"}
	pathParameters := []string{*updateTransitGatewayOptions.ID}

	builder := core.NewRequestBuilder(core.PATCH)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateTransitGatewayOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "UpdateTransitGateway")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	body := make(map[string]interface{})
	if updateTransitGatewayOptions.Global != nil {
		body["global"] = updateTransitGatewayOptions.Global
	}
	if updateTransitGatewayOptions.Name != nil {
		body["name"] = updateTransitGatewayOptions.Name
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTransitGateway)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// ListTransitGatewayConnections : Retrieves all connections in a Transit Gateway
// This request retrieves all connections in a Transit Gateway.
func (transitGatewayApis *TransitGatewayApisV1) ListTransitGatewayConnections(listTransitGatewayConnectionsOptions *ListTransitGatewayConnectionsOptions) (result *TransitGatewayConnectionCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(listTransitGatewayConnectionsOptions, "listTransitGatewayConnectionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(listTransitGatewayConnectionsOptions, "listTransitGatewayConnectionsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways", "connections"}
	pathParameters := []string{*listTransitGatewayConnectionsOptions.TransitGatewayID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listTransitGatewayConnectionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "ListTransitGatewayConnections")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTransitGatewayConnectionCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateTransitGatewayConnection : Add connection to a Transit Gateway
// Add a connection to Transit Gateway.
func (transitGatewayApis *TransitGatewayApisV1) CreateTransitGatewayConnection(createTransitGatewayConnectionOptions *CreateTransitGatewayConnectionOptions) (result *TransitGatewayConnectionCust, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createTransitGatewayConnectionOptions, "createTransitGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createTransitGatewayConnectionOptions, "createTransitGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways", "connections"}
	pathParameters := []string{*createTransitGatewayConnectionOptions.TransitGatewayID}

	builder := core.NewRequestBuilder(core.POST)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range createTransitGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "CreateTransitGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	body := make(map[string]interface{})
	if createTransitGatewayConnectionOptions.NetworkType != nil {
		body["network_type"] = createTransitGatewayConnectionOptions.NetworkType
	}
	if createTransitGatewayConnectionOptions.Name != nil {
		body["name"] = createTransitGatewayConnectionOptions.Name
	}
	if createTransitGatewayConnectionOptions.NetworkID != nil {
		body["network_id"] = createTransitGatewayConnectionOptions.NetworkID
	}
	if createTransitGatewayConnectionOptions.NetworkAccountID != nil {
		body["network_account_id"] = createTransitGatewayConnectionOptions.NetworkAccountID
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTransitGatewayConnectionCust)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// DeleteTransitGatewayConnection : Remove connection from Transit Gateway
// After the specified connection is detached, entities still within the Transit Gateway will no longer be able to
// communicate directly to it through the IBM Cloud private backbone.
func (transitGatewayApis *TransitGatewayApisV1) DeleteTransitGatewayConnection(deleteTransitGatewayConnectionOptions *DeleteTransitGatewayConnectionOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(deleteTransitGatewayConnectionOptions, "deleteTransitGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(deleteTransitGatewayConnectionOptions, "deleteTransitGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways", "connections"}
	pathParameters := []string{*deleteTransitGatewayConnectionOptions.TransitGatewayID, *deleteTransitGatewayConnectionOptions.ID}

	builder := core.NewRequestBuilder(core.DELETE)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range deleteTransitGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "DeleteTransitGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = transitGatewayApis.Service.Request(request, nil)

	return
}

// GetTransitGatewayConnection : Retrieves specified Transit Gateway connection
// This request retrieves a connection from the Transit Gateway.
func (transitGatewayApis *TransitGatewayApisV1) GetTransitGatewayConnection(getTransitGatewayConnectionOptions *GetTransitGatewayConnectionOptions) (result *TransitGatewayConnectionCust, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getTransitGatewayConnectionOptions, "getTransitGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getTransitGatewayConnectionOptions, "getTransitGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways", "connections"}
	pathParameters := []string{*getTransitGatewayConnectionOptions.TransitGatewayID, *getTransitGatewayConnectionOptions.ID}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getTransitGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "GetTransitGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTransitGatewayConnectionCust)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// UpdateTransitGatewayConnection : Updates specified Transit Gateway connection
// Update the name of a connection to a Transit Gateway.
func (transitGatewayApis *TransitGatewayApisV1) UpdateTransitGatewayConnection(updateTransitGatewayConnectionOptions *UpdateTransitGatewayConnectionOptions) (result *TransitGatewayConnectionCust, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(updateTransitGatewayConnectionOptions, "updateTransitGatewayConnectionOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(updateTransitGatewayConnectionOptions, "updateTransitGatewayConnectionOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways", "connections"}
	pathParameters := []string{*updateTransitGatewayConnectionOptions.TransitGatewayID, *updateTransitGatewayConnectionOptions.ID}

	builder := core.NewRequestBuilder(core.PATCH)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range updateTransitGatewayConnectionOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "UpdateTransitGatewayConnection")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	body := make(map[string]interface{})
	if updateTransitGatewayConnectionOptions.Name != nil {
		body["name"] = updateTransitGatewayConnectionOptions.Name
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTransitGatewayConnectionCust)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateTransitGatewayConnectionActions : Perform actions on a connection for a Transit Gateway
// Allow a network owner to approve or reject a cross-account connection request.
func (transitGatewayApis *TransitGatewayApisV1) CreateTransitGatewayConnectionActions(createTransitGatewayConnectionActionsOptions *CreateTransitGatewayConnectionActionsOptions) (response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(createTransitGatewayConnectionActionsOptions, "createTransitGatewayConnectionActionsOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(createTransitGatewayConnectionActionsOptions, "createTransitGatewayConnectionActionsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"transit_gateways", "connections", "actions"}
	pathParameters := []string{*createTransitGatewayConnectionActionsOptions.TransitGatewayID, *createTransitGatewayConnectionActionsOptions.ID}

	builder := core.NewRequestBuilder(core.POST)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range createTransitGatewayConnectionActionsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "CreateTransitGatewayConnectionActions")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Content-Type", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	body := make(map[string]interface{})
	if createTransitGatewayConnectionActionsOptions.Action != nil {
		body["action"] = createTransitGatewayConnectionActionsOptions.Action
	}
	_, err = builder.SetBodyContentJSON(body)
	if err != nil {
		return
	}

	request, err := builder.Build()
	if err != nil {
		return
	}

	response, err = transitGatewayApis.Service.Request(request, nil)

	return
}

// ListGatewayLocations : List all locations that support Transit Gateways
// List all locations that support Transit Gateways.
func (transitGatewayApis *TransitGatewayApisV1) ListGatewayLocations(listGatewayLocationsOptions *ListGatewayLocationsOptions) (result *TSCollection, response *core.DetailedResponse, err error) {
	err = core.ValidateStruct(listGatewayLocationsOptions, "listGatewayLocationsOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"locations"}
	pathParameters := []string{}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range listGatewayLocationsOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "ListGatewayLocations")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTSCollection)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// GetGatewayLocation : Show the details of a given Transit Gateway location
// Get the details of a Transit Gateway Location.
func (transitGatewayApis *TransitGatewayApisV1) GetGatewayLocation(getGatewayLocationOptions *GetGatewayLocationOptions) (result *TSLocation, response *core.DetailedResponse, err error) {
	err = core.ValidateNotNil(getGatewayLocationOptions, "getGatewayLocationOptions cannot be nil")
	if err != nil {
		return
	}
	err = core.ValidateStruct(getGatewayLocationOptions, "getGatewayLocationOptions")
	if err != nil {
		return
	}

	pathSegments := []string{"locations"}
	pathParameters := []string{*getGatewayLocationOptions.Name}

	builder := core.NewRequestBuilder(core.GET)
	_, err = builder.ConstructHTTPURL(transitGatewayApis.Service.Options.URL, pathSegments, pathParameters)
	if err != nil {
		return
	}

	for headerName, headerValue := range getGatewayLocationOptions.Headers {
		builder.AddHeader(headerName, headerValue)
	}

	sdkHeaders := common.GetSdkHeaders("transit_gateway_apis", "V1", "GetGatewayLocation")
	for headerName, headerValue := range sdkHeaders {
		builder.AddHeader(headerName, headerValue)
	}
	builder.AddHeader("Accept", "application/json")

	builder.AddQuery("version", fmt.Sprint(*transitGatewayApis.Version))

	request, err := builder.Build()
	if err != nil {
		return
	}

	var rawResponse map[string]json.RawMessage
	response, err = transitGatewayApis.Service.Request(request, &rawResponse)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(rawResponse, "", &result, UnmarshalTSLocation)
	if err != nil {
		return
	}
	response.Result = result

	return
}

// CreateTransitGatewayConnectionActionsOptions : The CreateTransitGatewayConnectionActions options.
type CreateTransitGatewayConnectionActionsOptions struct {
	// The Transit Gateway identifier.
	TransitGatewayID *string `json:"transit_gateway_id" validate:"required"`

	// The connection identifier.
	ID *string `json:"id" validate:"required"`

	// The action that is to be performed against the connection request.
	Action *string `json:"action" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateTransitGatewayConnectionActionsOptions.Action property.
// The action that is to be performed against the connection request.
const (
	CreateTransitGatewayConnectionActionsOptions_Action_Approve = "approve"
	CreateTransitGatewayConnectionActionsOptions_Action_Reject = "reject"
)

// NewCreateTransitGatewayConnectionActionsOptions : Instantiate CreateTransitGatewayConnectionActionsOptions
func (*TransitGatewayApisV1) NewCreateTransitGatewayConnectionActionsOptions(transitGatewayID string, id string, action string) *CreateTransitGatewayConnectionActionsOptions {
	return &CreateTransitGatewayConnectionActionsOptions{
		TransitGatewayID: core.StringPtr(transitGatewayID),
		ID: core.StringPtr(id),
		Action: core.StringPtr(action),
	}
}

// SetTransitGatewayID : Allow user to set TransitGatewayID
func (options *CreateTransitGatewayConnectionActionsOptions) SetTransitGatewayID(transitGatewayID string) *CreateTransitGatewayConnectionActionsOptions {
	options.TransitGatewayID = core.StringPtr(transitGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *CreateTransitGatewayConnectionActionsOptions) SetID(id string) *CreateTransitGatewayConnectionActionsOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetAction : Allow user to set Action
func (options *CreateTransitGatewayConnectionActionsOptions) SetAction(action string) *CreateTransitGatewayConnectionActionsOptions {
	options.Action = core.StringPtr(action)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateTransitGatewayConnectionActionsOptions) SetHeaders(param map[string]string) *CreateTransitGatewayConnectionActionsOptions {
	options.Headers = param
	return options
}

// CreateTransitGatewayConnectionOptions : The CreateTransitGatewayConnection options.
type CreateTransitGatewayConnectionOptions struct {
	// The Transit Gateway identifier.
	TransitGatewayID *string `json:"transit_gateway_id" validate:"required"`

	// Defines what type of network is connected via this connection.
	NetworkType *string `json:"network_type" validate:"required"`

	// The user-defined name for this transit gateway. If unspecified, the name will be the network name (the name of the
	// VPC in the case of network type 'vpc', and the word Classic, in the case of network type 'classic').
	Name *string `json:"name,omitempty"`

	// The ID of the network being connected via this connection. This field is required for some types, such as 'vpc'. For
	// network type 'vpc' this is the CRN of the VPC to be connected. This field is required to be unspecified for network
	// type 'classic'.
	NetworkID *string `json:"network_id,omitempty"`

	// The ID of the account which owns the network that is being connected. Generally only used if the network is in a
	// different account than the gateway.
	NetworkAccountID *string `json:"network_account_id,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// Constants associated with the CreateTransitGatewayConnectionOptions.NetworkType property.
// Defines what type of network is connected via this connection.
const (
	CreateTransitGatewayConnectionOptions_NetworkType_Classic = "classic"
	CreateTransitGatewayConnectionOptions_NetworkType_Vpc = "vpc"
)

// NewCreateTransitGatewayConnectionOptions : Instantiate CreateTransitGatewayConnectionOptions
func (*TransitGatewayApisV1) NewCreateTransitGatewayConnectionOptions(transitGatewayID string, networkType string) *CreateTransitGatewayConnectionOptions {
	return &CreateTransitGatewayConnectionOptions{
		TransitGatewayID: core.StringPtr(transitGatewayID),
		NetworkType: core.StringPtr(networkType),
	}
}

// SetTransitGatewayID : Allow user to set TransitGatewayID
func (options *CreateTransitGatewayConnectionOptions) SetTransitGatewayID(transitGatewayID string) *CreateTransitGatewayConnectionOptions {
	options.TransitGatewayID = core.StringPtr(transitGatewayID)
	return options
}

// SetNetworkType : Allow user to set NetworkType
func (options *CreateTransitGatewayConnectionOptions) SetNetworkType(networkType string) *CreateTransitGatewayConnectionOptions {
	options.NetworkType = core.StringPtr(networkType)
	return options
}

// SetName : Allow user to set Name
func (options *CreateTransitGatewayConnectionOptions) SetName(name string) *CreateTransitGatewayConnectionOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetNetworkID : Allow user to set NetworkID
func (options *CreateTransitGatewayConnectionOptions) SetNetworkID(networkID string) *CreateTransitGatewayConnectionOptions {
	options.NetworkID = core.StringPtr(networkID)
	return options
}

// SetNetworkAccountID : Allow user to set NetworkAccountID
func (options *CreateTransitGatewayConnectionOptions) SetNetworkAccountID(networkAccountID string) *CreateTransitGatewayConnectionOptions {
	options.NetworkAccountID = core.StringPtr(networkAccountID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateTransitGatewayConnectionOptions) SetHeaders(param map[string]string) *CreateTransitGatewayConnectionOptions {
	options.Headers = param
	return options
}

// CreateTransitGatewayOptions : The CreateTransitGateway options.
type CreateTransitGatewayOptions struct {
	// Location of Transit Gateway Services.
	Location *string `json:"location" validate:"required"`

	// Name Transit Gateway Services.
	Name *string `json:"name" validate:"required"`

	// Allow global routing for a Transit Gateway. If unspecified, the default value is false.
	Global *bool `json:"global,omitempty"`

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://console.bluemix.net/apidocs/resource-manager#introduction) is used.
	ResourceGroup *ResourceGroupIdentity `json:"resource_group,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewCreateTransitGatewayOptions : Instantiate CreateTransitGatewayOptions
func (*TransitGatewayApisV1) NewCreateTransitGatewayOptions(location string, name string) *CreateTransitGatewayOptions {
	return &CreateTransitGatewayOptions{
		Location: core.StringPtr(location),
		Name: core.StringPtr(name),
	}
}

// SetLocation : Allow user to set Location
func (options *CreateTransitGatewayOptions) SetLocation(location string) *CreateTransitGatewayOptions {
	options.Location = core.StringPtr(location)
	return options
}

// SetName : Allow user to set Name
func (options *CreateTransitGatewayOptions) SetName(name string) *CreateTransitGatewayOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetGlobal : Allow user to set Global
func (options *CreateTransitGatewayOptions) SetGlobal(global bool) *CreateTransitGatewayOptions {
	options.Global = core.BoolPtr(global)
	return options
}

// SetResourceGroup : Allow user to set ResourceGroup
func (options *CreateTransitGatewayOptions) SetResourceGroup(resourceGroup *ResourceGroupIdentity) *CreateTransitGatewayOptions {
	options.ResourceGroup = resourceGroup
	return options
}

// SetHeaders : Allow user to set Headers
func (options *CreateTransitGatewayOptions) SetHeaders(param map[string]string) *CreateTransitGatewayOptions {
	options.Headers = param
	return options
}

// DeleteTransitGatewayConnectionOptions : The DeleteTransitGatewayConnection options.
type DeleteTransitGatewayConnectionOptions struct {
	// The Transit Gateway identifier.
	TransitGatewayID *string `json:"transit_gateway_id" validate:"required"`

	// The connection identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteTransitGatewayConnectionOptions : Instantiate DeleteTransitGatewayConnectionOptions
func (*TransitGatewayApisV1) NewDeleteTransitGatewayConnectionOptions(transitGatewayID string, id string) *DeleteTransitGatewayConnectionOptions {
	return &DeleteTransitGatewayConnectionOptions{
		TransitGatewayID: core.StringPtr(transitGatewayID),
		ID: core.StringPtr(id),
	}
}

// SetTransitGatewayID : Allow user to set TransitGatewayID
func (options *DeleteTransitGatewayConnectionOptions) SetTransitGatewayID(transitGatewayID string) *DeleteTransitGatewayConnectionOptions {
	options.TransitGatewayID = core.StringPtr(transitGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *DeleteTransitGatewayConnectionOptions) SetID(id string) *DeleteTransitGatewayConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteTransitGatewayConnectionOptions) SetHeaders(param map[string]string) *DeleteTransitGatewayConnectionOptions {
	options.Headers = param
	return options
}

// DeleteTransitGatewayOptions : The DeleteTransitGateway options.
type DeleteTransitGatewayOptions struct {
	// The Transit Gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewDeleteTransitGatewayOptions : Instantiate DeleteTransitGatewayOptions
func (*TransitGatewayApisV1) NewDeleteTransitGatewayOptions(id string) *DeleteTransitGatewayOptions {
	return &DeleteTransitGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *DeleteTransitGatewayOptions) SetID(id string) *DeleteTransitGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *DeleteTransitGatewayOptions) SetHeaders(param map[string]string) *DeleteTransitGatewayOptions {
	options.Headers = param
	return options
}

// GetGatewayLocationOptions : The GetGatewayLocation options.
type GetGatewayLocationOptions struct {
	// The Transit Gateway location Name.
	Name *string `json:"name" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetGatewayLocationOptions : Instantiate GetGatewayLocationOptions
func (*TransitGatewayApisV1) NewGetGatewayLocationOptions(name string) *GetGatewayLocationOptions {
	return &GetGatewayLocationOptions{
		Name: core.StringPtr(name),
	}
}

// SetName : Allow user to set Name
func (options *GetGatewayLocationOptions) SetName(name string) *GetGatewayLocationOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetGatewayLocationOptions) SetHeaders(param map[string]string) *GetGatewayLocationOptions {
	options.Headers = param
	return options
}

// GetTransitGatewayConnectionOptions : The GetTransitGatewayConnection options.
type GetTransitGatewayConnectionOptions struct {
	// The Transit Gateway identifier.
	TransitGatewayID *string `json:"transit_gateway_id" validate:"required"`

	// The connection identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetTransitGatewayConnectionOptions : Instantiate GetTransitGatewayConnectionOptions
func (*TransitGatewayApisV1) NewGetTransitGatewayConnectionOptions(transitGatewayID string, id string) *GetTransitGatewayConnectionOptions {
	return &GetTransitGatewayConnectionOptions{
		TransitGatewayID: core.StringPtr(transitGatewayID),
		ID: core.StringPtr(id),
	}
}

// SetTransitGatewayID : Allow user to set TransitGatewayID
func (options *GetTransitGatewayConnectionOptions) SetTransitGatewayID(transitGatewayID string) *GetTransitGatewayConnectionOptions {
	options.TransitGatewayID = core.StringPtr(transitGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *GetTransitGatewayConnectionOptions) SetID(id string) *GetTransitGatewayConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetTransitGatewayConnectionOptions) SetHeaders(param map[string]string) *GetTransitGatewayConnectionOptions {
	options.Headers = param
	return options
}

// GetTransitGatewayOptions : The GetTransitGateway options.
type GetTransitGatewayOptions struct {
	// The Transit Gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewGetTransitGatewayOptions : Instantiate GetTransitGatewayOptions
func (*TransitGatewayApisV1) NewGetTransitGatewayOptions(id string) *GetTransitGatewayOptions {
	return &GetTransitGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *GetTransitGatewayOptions) SetID(id string) *GetTransitGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *GetTransitGatewayOptions) SetHeaders(param map[string]string) *GetTransitGatewayOptions {
	options.Headers = param
	return options
}

// ListGatewayLocationsOptions : The ListGatewayLocations options.
type ListGatewayLocationsOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListGatewayLocationsOptions : Instantiate ListGatewayLocationsOptions
func (*TransitGatewayApisV1) NewListGatewayLocationsOptions() *ListGatewayLocationsOptions {
	return &ListGatewayLocationsOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ListGatewayLocationsOptions) SetHeaders(param map[string]string) *ListGatewayLocationsOptions {
	options.Headers = param
	return options
}

// ListTransitGatewayConnectionsOptions : The ListTransitGatewayConnections options.
type ListTransitGatewayConnectionsOptions struct {
	// The Transit Gateway identifier.
	TransitGatewayID *string `json:"transit_gateway_id" validate:"required"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListTransitGatewayConnectionsOptions : Instantiate ListTransitGatewayConnectionsOptions
func (*TransitGatewayApisV1) NewListTransitGatewayConnectionsOptions(transitGatewayID string) *ListTransitGatewayConnectionsOptions {
	return &ListTransitGatewayConnectionsOptions{
		TransitGatewayID: core.StringPtr(transitGatewayID),
	}
}

// SetTransitGatewayID : Allow user to set TransitGatewayID
func (options *ListTransitGatewayConnectionsOptions) SetTransitGatewayID(transitGatewayID string) *ListTransitGatewayConnectionsOptions {
	options.TransitGatewayID = core.StringPtr(transitGatewayID)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *ListTransitGatewayConnectionsOptions) SetHeaders(param map[string]string) *ListTransitGatewayConnectionsOptions {
	options.Headers = param
	return options
}

// ListTransitGatewaysOptions : The ListTransitGateways options.
type ListTransitGatewaysOptions struct {

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewListTransitGatewaysOptions : Instantiate ListTransitGatewaysOptions
func (*TransitGatewayApisV1) NewListTransitGatewaysOptions() *ListTransitGatewaysOptions {
	return &ListTransitGatewaysOptions{}
}

// SetHeaders : Allow user to set Headers
func (options *ListTransitGatewaysOptions) SetHeaders(param map[string]string) *ListTransitGatewaysOptions {
	options.Headers = param
	return options
}

// ResourceGroupIdentity : The resource group to use. If unspecified, the account's [default resource
// group](https://console.bluemix.net/apidocs/resource-manager#introduction) is used.
type ResourceGroupIdentity struct {
	// The unique identifier for this resource group.
	ID *string `json:"id" validate:"required"`
}


// NewResourceGroupIdentity : Instantiate ResourceGroupIdentity (Generic Model Constructor)
func (*TransitGatewayApisV1) NewResourceGroupIdentity(id string) (model *ResourceGroupIdentity, err error) {
	model = &ResourceGroupIdentity{
		ID: core.StringPtr(id),
	}
	err = core.ValidateStruct(model, "required parameters")
	return
}

// UnmarshalResourceGroupIdentity unmarshals an instance of ResourceGroupIdentity from the specified map of raw messages.
func UnmarshalResourceGroupIdentity(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ResourceGroupIdentity)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// ResourceGroupReference : The resource group to use. If unspecified, the account's [default resource
// group](https://console.bluemix.net/apidocs/resource-manager#introduction) is used.
type ResourceGroupReference struct {
	// The unique identifier for this resource group.
	ID *string `json:"id" validate:"required"`

	// The URL for this resource group.
	Href *string `json:"href" validate:"required"`
}


// UnmarshalResourceGroupReference unmarshals an instance of ResourceGroupReference from the specified map of raw messages.
func UnmarshalResourceGroupReference(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(ResourceGroupReference)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "href", &obj.Href)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// TSCollection : A list of Transit Gateway locations.
type TSCollection struct {
	// Collection of Transit Gateway locations.
	Locations []TSLocationBasic `json:"locations" validate:"required"`
}


// UnmarshalTSCollection unmarshals an instance of TSCollection from the specified map of raw messages.
func UnmarshalTSCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(TSCollection)
	err = core.UnmarshalModel(m, "locations", &obj.Locations, UnmarshalTSLocationBasic)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// TSLocalLocation : Details of a local connection location.
type TSLocalLocation struct {
	// A descriptive display name for the location.
	DisplayName *string `json:"display_name" validate:"required"`

	// The name of the location.
	Name *string `json:"name" validate:"required"`

	// The type of the location, determining is this a multi-zone region, a single data center, or a point of presence.
	Type *string `json:"type" validate:"required"`
}

// Constants associated with the TSLocalLocation.Type property.
// The type of the location, determining is this a multi-zone region, a single data center, or a point of presence.
const (
	TSLocalLocation_Type_Region = "region"
)


// UnmarshalTSLocalLocation unmarshals an instance of TSLocalLocation from the specified map of raw messages.
func UnmarshalTSLocalLocation(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(TSLocalLocation)
	err = core.UnmarshalPrimitive(m, "display_name", &obj.DisplayName)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// TSLocation : Details of a Transit Gateway location.
type TSLocation struct {
	// The geographical location of this location, used for billing purposes.
	BillingLocation *string `json:"billing_location" validate:"required"`

	// Name of the Location.
	Name *string `json:"name" validate:"required"`

	// The type of the location, determining is this a multi-zone region, a single data center, or a point of presence.
	Type *string `json:"type" validate:"required"`

	// The set of network locations that are considered local for this Transit Gateway location.
	LocalConnectionLocations []TSLocalLocation `json:"local_connection_locations" validate:"required"`
}


// UnmarshalTSLocation unmarshals an instance of TSLocation from the specified map of raw messages.
func UnmarshalTSLocation(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(TSLocation)
	err = core.UnmarshalPrimitive(m, "billing_location", &obj.BillingLocation)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "local_connection_locations", &obj.LocalConnectionLocations, UnmarshalTSLocalLocation)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// TSLocationBasic : Details of a Transit Gateway location.
type TSLocationBasic struct {
	// The geographical location of this location, used for billing purposes.
	BillingLocation *string `json:"billing_location" validate:"required"`

	// Name of the Location.
	Name *string `json:"name" validate:"required"`

	// The type of the location, determining is this a multi-zone region, a single data center, or a point of presence.
	Type *string `json:"type" validate:"required"`
}


// UnmarshalTSLocationBasic unmarshals an instance of TSLocationBasic from the specified map of raw messages.
func UnmarshalTSLocationBasic(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(TSLocationBasic)
	err = core.UnmarshalPrimitive(m, "billing_location", &obj.BillingLocation)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "type", &obj.Type)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// TransitGateway : Details of a Transit Gateway.
type TransitGateway struct {
	// The unique identifier for this Transit Gateway.
	ID *string `json:"id" validate:"required"`

	// The CRN for this Transit Gateway.
	Crn *string `json:"crn" validate:"required"`

	// A human readable name for the transit gateway.
	Name *string `json:"name" validate:"required"`

	// Location of Transit Gateway Services.
	Location *string `json:"location" validate:"required"`

	// The date and time that this gateway was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// Allow global routing for a Transit Gateway.
	Global *bool `json:"global" validate:"required"`

	// The resource group to use. If unspecified, the account's [default resource
	// group](https://console.bluemix.net/apidocs/resource-manager#introduction) is used.
	ResourceGroup *ResourceGroupReference `json:"resource_group,omitempty"`

	// The status of the Transit Gateway.
	Status *string `json:"status" validate:"required"`

	// The date and time that this gateway was last updated.
	UpdatedAt *strfmt.DateTime `json:"updated_at,omitempty"`
}

// Constants associated with the TransitGateway.Status property.
// The status of the Transit Gateway.
const (
	TransitGateway_Status_Available = "available"
	TransitGateway_Status_Deleting = "deleting"
	TransitGateway_Status_Failed = "failed"
	TransitGateway_Status_Pending = "pending"
)


// UnmarshalTransitGateway unmarshals an instance of TransitGateway from the specified map of raw messages.
func UnmarshalTransitGateway(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(TransitGateway)
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "crn", &obj.Crn)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "location", &obj.Location)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "global", &obj.Global)
	if err != nil {
		return
	}
	err = core.UnmarshalModel(m, "resource_group", &obj.ResourceGroup, UnmarshalResourceGroupReference)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "updated_at", &obj.UpdatedAt)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// TransitGatewayCollection : A list of Transit Gateways.
type TransitGatewayCollection struct {
	// Collection of Transit Services gateways.
	TransitGateways []TransitGateway `json:"transit_gateways" validate:"required"`
}


// UnmarshalTransitGatewayCollection unmarshals an instance of TransitGatewayCollection from the specified map of raw messages.
func UnmarshalTransitGatewayCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(TransitGatewayCollection)
	err = core.UnmarshalModel(m, "transit_gateways", &obj.TransitGateways, UnmarshalTransitGateway)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// TransitGatewayConnectionCollection : A set of Transit Gateway network connections.
type TransitGatewayConnectionCollection struct {
	// Array of transit gateways network Connections.
	Connections []TransitGatewayConnectionCust `json:"connections" validate:"required"`
}


// UnmarshalTransitGatewayConnectionCollection unmarshals an instance of TransitGatewayConnectionCollection from the specified map of raw messages.
func UnmarshalTransitGatewayConnectionCollection(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(TransitGatewayConnectionCollection)
	err = core.UnmarshalModel(m, "connections", &obj.Connections, UnmarshalTransitGatewayConnectionCust)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// TransitGatewayConnectionCust : Connection included in transit gateway.
type TransitGatewayConnectionCust struct {
	// The user-defined name for this transit gateway. If unspecified, the name will be the network name (the name of the
	// VPC in the case of network type 'vpc', and the word Classic, in the case of network type 'classic').
	Name *string `json:"name,omitempty"`

	// The ID of the network being connected via this connection. This field is required for some types, such as 'vpc'. For
	// network type 'vpc' this is the CRN of the VPC to be connected. This field is required to be unspecified for network
	// type 'classic'.
	NetworkID *string `json:"network_id,omitempty"`

	// Defines what type of network is connected via this connection.
	NetworkType *string `json:"network_type" validate:"required"`

	// The ID of the account which owns the network that is being connected. Generally only used if the network is in a
	// different account than the gateway.
	NetworkAccountID *string `json:"network_account_id,omitempty"`

	// The unique identifier for this Transit Gateway Connection to Network (vpc/classic).
	ID *string `json:"id" validate:"required"`

	// The date and time that this connection was created.
	CreatedAt *strfmt.DateTime `json:"created_at" validate:"required"`

	// Only visible for cross account connections, this field represents the status of the request to connect the given
	// network between accounts.
	RequestStatus *string `json:"request_status,omitempty"`

	// What is the current configuration state of this connection.
	Status *string `json:"status,omitempty"`

	// The date and time that this connection was last updated.
	UpdatedAt *strfmt.DateTime `json:"updated_at,omitempty"`
}

// Constants associated with the TransitGatewayConnectionCust.NetworkType property.
// Defines what type of network is connected via this connection.
const (
	TransitGatewayConnectionCust_NetworkType_Classic = "classic"
	TransitGatewayConnectionCust_NetworkType_Vpc = "vpc"
)

// Constants associated with the TransitGatewayConnectionCust.RequestStatus property.
// Only visible for cross account connections, this field represents the status of the request to connect the given
// network between accounts.
const (
	TransitGatewayConnectionCust_RequestStatus_Approved = "approved"
	TransitGatewayConnectionCust_RequestStatus_Detached = "detached"
	TransitGatewayConnectionCust_RequestStatus_Expired = "expired"
	TransitGatewayConnectionCust_RequestStatus_Pending = "pending"
	TransitGatewayConnectionCust_RequestStatus_Rejected = "rejected"
)

// Constants associated with the TransitGatewayConnectionCust.Status property.
// What is the current configuration state of this connection.
const (
	TransitGatewayConnectionCust_Status_Attached = "attached"
	TransitGatewayConnectionCust_Status_Deleting = "deleting"
	TransitGatewayConnectionCust_Status_Detached = "detached"
	TransitGatewayConnectionCust_Status_Detaching = "detaching"
	TransitGatewayConnectionCust_Status_Failed = "failed"
	TransitGatewayConnectionCust_Status_Pending = "pending"
)


// UnmarshalTransitGatewayConnectionCust unmarshals an instance of TransitGatewayConnectionCust from the specified map of raw messages.
func UnmarshalTransitGatewayConnectionCust(m map[string]json.RawMessage, result interface{}) (err error) {
	obj := new(TransitGatewayConnectionCust)
	err = core.UnmarshalPrimitive(m, "name", &obj.Name)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "network_id", &obj.NetworkID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "network_type", &obj.NetworkType)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "network_account_id", &obj.NetworkAccountID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "id", &obj.ID)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "created_at", &obj.CreatedAt)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "request_status", &obj.RequestStatus)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "status", &obj.Status)
	if err != nil {
		return
	}
	err = core.UnmarshalPrimitive(m, "updated_at", &obj.UpdatedAt)
	if err != nil {
		return
	}
	reflect.ValueOf(result).Elem().Set(reflect.ValueOf(obj))
	return
}

// UpdateTransitGatewayConnectionOptions : The UpdateTransitGatewayConnection options.
type UpdateTransitGatewayConnectionOptions struct {
	// The Transit Gateway identifier.
	TransitGatewayID *string `json:"transit_gateway_id" validate:"required"`

	// The connection identifier.
	ID *string `json:"id" validate:"required"`

	// The user-defined name for this transit gateway. If specified as empty string or nil,  the name will be the network
	// name (the name of the VPC in the case of network type 'vpc',  and the word Classic, in the case of network type
	// 'classic').
	Name *string `json:"name,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateTransitGatewayConnectionOptions : Instantiate UpdateTransitGatewayConnectionOptions
func (*TransitGatewayApisV1) NewUpdateTransitGatewayConnectionOptions(transitGatewayID string, id string) *UpdateTransitGatewayConnectionOptions {
	return &UpdateTransitGatewayConnectionOptions{
		TransitGatewayID: core.StringPtr(transitGatewayID),
		ID: core.StringPtr(id),
	}
}

// SetTransitGatewayID : Allow user to set TransitGatewayID
func (options *UpdateTransitGatewayConnectionOptions) SetTransitGatewayID(transitGatewayID string) *UpdateTransitGatewayConnectionOptions {
	options.TransitGatewayID = core.StringPtr(transitGatewayID)
	return options
}

// SetID : Allow user to set ID
func (options *UpdateTransitGatewayConnectionOptions) SetID(id string) *UpdateTransitGatewayConnectionOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetName : Allow user to set Name
func (options *UpdateTransitGatewayConnectionOptions) SetName(name string) *UpdateTransitGatewayConnectionOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateTransitGatewayConnectionOptions) SetHeaders(param map[string]string) *UpdateTransitGatewayConnectionOptions {
	options.Headers = param
	return options
}

// UpdateTransitGatewayOptions : The UpdateTransitGateway options.
type UpdateTransitGatewayOptions struct {
	// The Transit Gateway identifier.
	ID *string `json:"id" validate:"required"`

	// Allow global routing for a Transit Gateway.
	Global *bool `json:"global,omitempty"`

	// The user-defined name for this transit gateway.
	Name *string `json:"name,omitempty"`

	// Allows users to set headers on API requests
	Headers map[string]string
}

// NewUpdateTransitGatewayOptions : Instantiate UpdateTransitGatewayOptions
func (*TransitGatewayApisV1) NewUpdateTransitGatewayOptions(id string) *UpdateTransitGatewayOptions {
	return &UpdateTransitGatewayOptions{
		ID: core.StringPtr(id),
	}
}

// SetID : Allow user to set ID
func (options *UpdateTransitGatewayOptions) SetID(id string) *UpdateTransitGatewayOptions {
	options.ID = core.StringPtr(id)
	return options
}

// SetGlobal : Allow user to set Global
func (options *UpdateTransitGatewayOptions) SetGlobal(global bool) *UpdateTransitGatewayOptions {
	options.Global = core.BoolPtr(global)
	return options
}

// SetName : Allow user to set Name
func (options *UpdateTransitGatewayOptions) SetName(name string) *UpdateTransitGatewayOptions {
	options.Name = core.StringPtr(name)
	return options
}

// SetHeaders : Allow user to set Headers
func (options *UpdateTransitGatewayOptions) SetHeaders(param map[string]string) *UpdateTransitGatewayOptions {
	options.Headers = param
	return options
}
