package openstack

import (
	b64 "encoding/base64"
	"encoding/json"
	"fmt"
	"strings"

	ignition "github.com/coreos/ignition/config/v2_2/types"
	"github.com/gophercloud/gophercloud/openstack/objectstorage/v1/containers"
	"github.com/gophercloud/gophercloud/openstack/objectstorage/v1/objects"
	"github.com/gophercloud/utils/openstack/clientconfig"
	"github.com/sirupsen/logrus"
	"github.com/vincent-petithory/dataurl"
	"k8s.io/apimachinery/pkg/util/rand"
)

// createBootstrapSwiftObject creates a container and object in swift with the bootstrap ignition config.
func createBootstrapSwiftObject(cloud string, bootstrapIgn string, clusterID string) (string, error) {
	logrus.Debugln("Creating a Swift container for your bootstrap ignition...")
	opts := clientconfig.ClientOpts{
		Cloud: cloud,
	}

	conn, err := clientconfig.NewServiceClient("object-store", &opts)
	if err != nil {
		return "", err
	}

	containerCreateOpts := containers.CreateOpts{
		ContainerRead: ".r:*",

		// "kubernetes.io/cluster/${var.cluster_id}" = "owned"
		Metadata: map[string]string{
			"Name":               fmt.Sprintf("%s-ignition", clusterID),
			"openshiftClusterID": clusterID,
		},
	}

	_, err = containers.Create(conn, clusterID, containerCreateOpts).Extract()
	if err != nil {
		return "", err
	}
	logrus.Debugf("Container %s was created.", clusterID)

	logrus.Debugf("Creating a Swift object in container %s containing your bootstrap ignition...", clusterID)
	objectCreateOpts := objects.CreateOpts{
		ContentType: "text/plain",
		Content:     strings.NewReader(bootstrapIgn),
		DeleteAfter: 3600,
	}

	objID := rand.String(16)

	_, err = objects.Create(conn, clusterID, objID, objectCreateOpts).Extract()
	if err != nil {
		return "", err
	}
	logrus.Debugf("The object was created.")

	return objID, nil
}

// To allow Ignition to download its config on the bootstrap machine from a location secured by a
// self-signed certificate, we have to provide it a valid custom ca bundle.
// To do so we generate a small ignition config that contains just Security section with the bundle
// and later append it to the main ignition config.
// We can't do it directly in Terraform, because Ignition provider suppors only 2.1 version, but
// Security section was added in 2.2 only.

// generateIgnitionShim is used to generate an ignition file that contains a user ca bundle
// in its Security section.
func generateIgnitionShim(userCA string, clusterID string, swiftObject string) (string, error) {
	fileMode := 420

	// Hostname Config
	contents := fmt.Sprintf("%s-bootstrap", clusterID)

	hostnameConfigFile := ignition.File{
		Node: ignition.Node{
			Filesystem: "root",
			Path:       "/etc/hostname",
		},
		FileEmbedded1: ignition.FileEmbedded1{
			Mode: &fileMode,
			Contents: ignition.FileContents{
				Source: fmt.Sprintf("data:text/plain;base64,%s", b64.StdEncoding.EncodeToString([]byte(contents))),
			},
		},
	}

	security := ignition.Security{}
	if userCA != "" {
		security = ignition.Security{
			TLS: ignition.TLS{
				CertificateAuthorities: []ignition.CaReference{{
					Source: dataurl.EncodeBytes([]byte(userCA)),
				}},
			},
		}
	}

	ign := ignition.Config{
		Ignition: ignition.Ignition{
			Version:  ignition.MaxVersion.String(),
			Security: security,
			Config: ignition.IgnitionConfig{
				Append: []ignition.ConfigReference{
					{
						Source: swiftObject,
					},
				},
			},
		},
		Storage: ignition.Storage{
			Files: []ignition.File{
				hostnameConfigFile,
			},
		},
	}

	data, err := json.Marshal(ign)
	if err != nil {
		return "", err
	}

	return string(data), nil
}
