package keyvault

import (
	"reflect"
	"testing"

	"github.com/hashicorp/terraform-plugin-sdk/terraform"
)

func TestKeyVaultMigrateState(t *testing.T) {
	cases := map[string]struct {
		StateVersion int
		ID           string
		Attributes   map[string]string
		Expected     map[string]string
		Meta         interface{}
	}{
		"v0_1_empty": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes:   map[string]string{},
			Expected:     map[string]string{},
		},
		"v0_1_read_only": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.application_id":            "",
				"access_policy.0.certificate_permissions.#": "1",
				"access_policy.0.certificate_permissions.0": "Get",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "Get",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "Get",
				"access_policy.0.storage_permissions.#":     "0",
			},
			Expected: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.application_id":            "",
				"access_policy.0.certificate_permissions.#": "1",
				"access_policy.0.certificate_permissions.0": "Get",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "Get",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "Get",
				"access_policy.0.storage_permissions.#":     "0",
			},
		},
		"v0_1_certificates": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "1",
				"access_policy.0.certificate_permissions.0": "All",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "Get",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "Get",
				"access_policy.0.storage_permissions.#":     "0",
			},
			Expected: map[string]string{
				"access_policy.#":                            "1",
				"access_policy.0.application_id":             "",
				"access_policy.0.tenant_id":                  "abc123",
				"access_policy.0.object_id":                  "bcd234",
				"access_policy.0.certificate_permissions.#":  "12",
				"access_policy.0.certificate_permissions.0":  "create",
				"access_policy.0.certificate_permissions.1":  "delete",
				"access_policy.0.certificate_permissions.2":  "deleteissuers",
				"access_policy.0.certificate_permissions.3":  "get",
				"access_policy.0.certificate_permissions.4":  "getissuers",
				"access_policy.0.certificate_permissions.5":  "import",
				"access_policy.0.certificate_permissions.6":  "list",
				"access_policy.0.certificate_permissions.7":  "listissuers",
				"access_policy.0.certificate_permissions.8":  "managecontacts",
				"access_policy.0.certificate_permissions.9":  "manageissuers",
				"access_policy.0.certificate_permissions.10": "setissuers",
				"access_policy.0.certificate_permissions.11": "update",
				"access_policy.0.key_permissions.#":          "1",
				"access_policy.0.key_permissions.0":          "Get",
				"access_policy.0.secret_permissions.#":       "1",
				"access_policy.0.secret_permissions.0":       "Get",
				"access_policy.0.storage_permissions.#":      "0",
			},
		},
		"v0_1_certificates_multiple": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes: map[string]string{
				// an All and a Get should be replaced by all of the values
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "2",
				"access_policy.0.certificate_permissions.0": "Get",
				"access_policy.0.certificate_permissions.1": "All",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "Get",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "Get",
				"access_policy.0.storage_permissions.#":     "0",
			},
			Expected: map[string]string{
				"access_policy.#":                            "1",
				"access_policy.0.application_id":             "",
				"access_policy.0.tenant_id":                  "abc123",
				"access_policy.0.object_id":                  "bcd234",
				"access_policy.0.certificate_permissions.#":  "12",
				"access_policy.0.certificate_permissions.0":  "create",
				"access_policy.0.certificate_permissions.1":  "delete",
				"access_policy.0.certificate_permissions.2":  "deleteissuers",
				"access_policy.0.certificate_permissions.3":  "get",
				"access_policy.0.certificate_permissions.4":  "getissuers",
				"access_policy.0.certificate_permissions.5":  "import",
				"access_policy.0.certificate_permissions.6":  "list",
				"access_policy.0.certificate_permissions.7":  "listissuers",
				"access_policy.0.certificate_permissions.8":  "managecontacts",
				"access_policy.0.certificate_permissions.9":  "manageissuers",
				"access_policy.0.certificate_permissions.10": "setissuers",
				"access_policy.0.certificate_permissions.11": "update",
				"access_policy.0.key_permissions.#":          "1",
				"access_policy.0.key_permissions.0":          "Get",
				"access_policy.0.secret_permissions.#":       "1",
				"access_policy.0.secret_permissions.0":       "Get",
				"access_policy.0.storage_permissions.#":      "0",
			},
		},
		"v0_1_keys": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "0",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "All",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "Get",
				"access_policy.0.storage_permissions.#":     "0",
			},
			Expected: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "0",
				"access_policy.0.key_permissions.#":         "16",
				"access_policy.0.key_permissions.0":         "backup",
				"access_policy.0.key_permissions.1":         "create",
				"access_policy.0.key_permissions.2":         "decrypt",
				"access_policy.0.key_permissions.3":         "delete",
				"access_policy.0.key_permissions.4":         "encrypt",
				"access_policy.0.key_permissions.5":         "get",
				"access_policy.0.key_permissions.6":         "import",
				"access_policy.0.key_permissions.7":         "list",
				"access_policy.0.key_permissions.8":         "purge",
				"access_policy.0.key_permissions.9":         "recover",
				"access_policy.0.key_permissions.10":        "restore",
				"access_policy.0.key_permissions.11":        "sign",
				"access_policy.0.key_permissions.12":        "unwrapKey",
				"access_policy.0.key_permissions.13":        "update",
				"access_policy.0.key_permissions.14":        "verify",
				"access_policy.0.key_permissions.15":        "wrapKey",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "Get",
				"access_policy.0.storage_permissions.#":     "0",
			},
		},
		"v0_1_keys_multiple": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes: map[string]string{
				// an All and a Get should be replaced by all of the values
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "0",
				"access_policy.0.key_permissions.#":         "2",
				"access_policy.0.key_permissions.0":         "all",
				"access_policy.0.key_permissions.1":         "create",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "Get",
				"access_policy.0.storage_permissions.#":     "0",
			},
			Expected: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "0",
				"access_policy.0.key_permissions.#":         "16",
				"access_policy.0.key_permissions.0":         "backup",
				"access_policy.0.key_permissions.1":         "create",
				"access_policy.0.key_permissions.2":         "decrypt",
				"access_policy.0.key_permissions.3":         "delete",
				"access_policy.0.key_permissions.4":         "encrypt",
				"access_policy.0.key_permissions.5":         "get",
				"access_policy.0.key_permissions.6":         "import",
				"access_policy.0.key_permissions.7":         "list",
				"access_policy.0.key_permissions.8":         "purge",
				"access_policy.0.key_permissions.9":         "recover",
				"access_policy.0.key_permissions.10":        "restore",
				"access_policy.0.key_permissions.11":        "sign",
				"access_policy.0.key_permissions.12":        "unwrapKey",
				"access_policy.0.key_permissions.13":        "update",
				"access_policy.0.key_permissions.14":        "verify",
				"access_policy.0.key_permissions.15":        "wrapKey",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "Get",
				"access_policy.0.storage_permissions.#":     "0",
			},
		},
		"v0_1_secrets": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "0",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "create",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "All",
				"access_policy.0.storage_permissions.#":     "0",
			},
			Expected: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "0",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "create",
				"access_policy.0.secret_permissions.#":      "8",
				"access_policy.0.secret_permissions.0":      "backup",
				"access_policy.0.secret_permissions.1":      "delete",
				"access_policy.0.secret_permissions.2":      "get",
				"access_policy.0.secret_permissions.3":      "list",
				"access_policy.0.secret_permissions.4":      "purge",
				"access_policy.0.secret_permissions.5":      "recover",
				"access_policy.0.secret_permissions.6":      "restore",
				"access_policy.0.secret_permissions.7":      "set",
				"access_policy.0.storage_permissions.#":     "0",
			},
		},
		"v0_1_secrets_multiple": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes: map[string]string{
				// an All and a Get should be replaced by all of the values
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "0",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "create",
				"access_policy.0.secret_permissions.#":      "2",
				"access_policy.0.secret_permissions.0":      "backup",
				"access_policy.0.secret_permissions.1":      "all",
				"access_policy.0.storage_permissions.#":     "0",
			},
			Expected: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "0",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "create",
				"access_policy.0.secret_permissions.#":      "8",
				"access_policy.0.secret_permissions.0":      "backup",
				"access_policy.0.secret_permissions.1":      "delete",
				"access_policy.0.secret_permissions.2":      "get",
				"access_policy.0.secret_permissions.3":      "list",
				"access_policy.0.secret_permissions.4":      "purge",
				"access_policy.0.secret_permissions.5":      "recover",
				"access_policy.0.secret_permissions.6":      "restore",
				"access_policy.0.secret_permissions.7":      "set",
				"access_policy.0.storage_permissions.#":     "0",
			},
		},
		"v0_1_all": {
			StateVersion: 0,
			ID:           "some_id",
			Attributes: map[string]string{
				"access_policy.#":                           "1",
				"access_policy.0.application_id":            "",
				"access_policy.0.tenant_id":                 "abc123",
				"access_policy.0.object_id":                 "bcd234",
				"access_policy.0.certificate_permissions.#": "1",
				"access_policy.0.certificate_permissions.0": "all",
				"access_policy.0.key_permissions.#":         "1",
				"access_policy.0.key_permissions.0":         "all",
				"access_policy.0.secret_permissions.#":      "1",
				"access_policy.0.secret_permissions.0":      "all",
				"access_policy.0.storage_permissions.#":     "0",
			},
			Expected: map[string]string{
				"access_policy.#":                            "1",
				"access_policy.0.application_id":             "",
				"access_policy.0.tenant_id":                  "abc123",
				"access_policy.0.object_id":                  "bcd234",
				"access_policy.0.certificate_permissions.#":  "12",
				"access_policy.0.certificate_permissions.0":  "create",
				"access_policy.0.certificate_permissions.1":  "delete",
				"access_policy.0.certificate_permissions.2":  "deleteissuers",
				"access_policy.0.certificate_permissions.3":  "get",
				"access_policy.0.certificate_permissions.4":  "getissuers",
				"access_policy.0.certificate_permissions.5":  "import",
				"access_policy.0.certificate_permissions.6":  "list",
				"access_policy.0.certificate_permissions.7":  "listissuers",
				"access_policy.0.certificate_permissions.8":  "managecontacts",
				"access_policy.0.certificate_permissions.9":  "manageissuers",
				"access_policy.0.certificate_permissions.10": "setissuers",
				"access_policy.0.certificate_permissions.11": "update",
				"access_policy.0.key_permissions.#":          "16",
				"access_policy.0.key_permissions.0":          "backup",
				"access_policy.0.key_permissions.1":          "create",
				"access_policy.0.key_permissions.2":          "decrypt",
				"access_policy.0.key_permissions.3":          "delete",
				"access_policy.0.key_permissions.4":          "encrypt",
				"access_policy.0.key_permissions.5":          "get",
				"access_policy.0.key_permissions.6":          "import",
				"access_policy.0.key_permissions.7":          "list",
				"access_policy.0.key_permissions.8":          "purge",
				"access_policy.0.key_permissions.9":          "recover",
				"access_policy.0.key_permissions.10":         "restore",
				"access_policy.0.key_permissions.11":         "sign",
				"access_policy.0.key_permissions.12":         "unwrapKey",
				"access_policy.0.key_permissions.13":         "update",
				"access_policy.0.key_permissions.14":         "verify",
				"access_policy.0.key_permissions.15":         "wrapKey",
				"access_policy.0.secret_permissions.#":       "8",
				"access_policy.0.secret_permissions.0":       "backup",
				"access_policy.0.secret_permissions.1":       "delete",
				"access_policy.0.secret_permissions.2":       "get",
				"access_policy.0.secret_permissions.3":       "list",
				"access_policy.0.secret_permissions.4":       "purge",
				"access_policy.0.secret_permissions.5":       "recover",
				"access_policy.0.secret_permissions.6":       "restore",
				"access_policy.0.secret_permissions.7":       "set",
				"access_policy.0.storage_permissions.#":      "0",
			},
		},
	}

	for tn, tc := range cases {
		is := &terraform.InstanceState{
			ID:         tc.ID,
			Attributes: tc.Attributes,
		}
		is, err := resourceKeyVaultMigrateState(tc.StateVersion, is, tc.Meta)
		if err != nil {
			t.Fatalf("bad: %q, err: %+v", tn, err)
		}

		if !reflect.DeepEqual(tc.Expected, is.Attributes) {
			t.Fatalf("Bad Key Vault Migrate\n\n. Got: %+v\n\n expected: %+v", is.Attributes, tc.Expected)
		}
	}
}
