#include <linux/percpu_counter.h>
#include <lustre_debug.h>
#include <lprocfs_status.h>
#include "ptlrpc_internal.h"

static struct proc_dir_entry *sla_dir = NULL;
static struct proc_dir_entry *sla_req_stats = NULL;
/* global sla percpu counter for error request */
static int init_mask = 0;
static struct percpu_counter sla_ok_req_total;
static struct percpu_counter sla_failed_req_total;
static struct percpu_counter sla_lost_perf_req_total;
static struct percpu_counter sla_other_req_total;

#define SLA_OK_REQ_BIT        0x01
#define SLA_FAILED_REQ_BIT    0x02
#define SLA_LOST_PERF_REQ_BIT 0x04
#define SLA_OTHER_REQ_BIT     0x08

static int sla_req_stats_seq_show(struct seq_file *m, void *data)
{
    seq_printf(m, "ok:%lld\nlostperf:%lld\nfailed:%lld\nother:%lld",
        percpu_counter_sum(&sla_ok_req_total),
        percpu_counter_sum(&sla_lost_perf_req_total),
        percpu_counter_sum(&sla_failed_req_total),
        percpu_counter_sum(&sla_other_req_total));
    return 0;
}
LPROC_SEQ_FOPS_RO(sla_req_stats);

int sla_init()
{
    int rc = 0;
    sla_dir = proc_mkdir("cpfs_sla", NULL);
    if (sla_dir == NULL) {
        CERROR("create cpfs_sla proc dir fail\n");
        goto fail;
    }
    CWARN("create proc dir cpfs_sla succeed, sla_dir:%p\n", sla_dir);
    sla_req_stats = lprocfs_add_simple(sla_dir, "req_stats",
        NULL, &sla_req_stats_fops);
    if (sla_req_stats == NULL) {
        CERROR("create sla req stats proc dir fail\n");
        goto fail;
    }
    CWARN("create proc file cpfs_sla/req_stats succeed, sla_req_stats:%p\n", sla_req_stats);

#define DO_INIT_COUNTER(counter, bitmask, ...)           \
    do                                                   \
    {                                                    \
        rc = percpu_counter_init(&counter, __VA_ARGS__); \
        if (rc != 0)                                     \
        {                                                \
            CERROR("init %s fail\n", #counter);          \
            goto fail;                                   \
        }                                                \
        init_mask |= bitmask;                            \
        CWARN("init counter %s succeed\n", #counter);    \
    } while (0)

#ifdef HAVE_PERCPU_COUNTER_INIT_GFP_FLAG
    DO_INIT_COUNTER(sla_ok_req_total, SLA_OK_REQ_BIT, 0, GFP_KERNEL);
    DO_INIT_COUNTER(sla_lost_perf_req_total, SLA_LOST_PERF_REQ_BIT, 0, GFP_KERNEL);
    DO_INIT_COUNTER(sla_failed_req_total, SLA_FAILED_REQ_BIT, 0, GFP_KERNEL);
    DO_INIT_COUNTER(sla_other_req_total, SLA_OTHER_REQ_BIT, 0, GFP_KERNEL);
#else
    DO_INIT_COUNTER(sla_ok_req_total, SLA_OK_REQ_BIT, 0);
    DO_INIT_COUNTER(sla_lost_perf_req_total, SLA_LOST_PERF_REQ_BIT, 0);
    DO_INIT_COUNTER(sla_failed_req_total, SLA_FAILED_REQ_BIT, 0);
    DO_INIT_COUNTER(sla_other_req_total, SLA_OTHER_REQ_BIT, 0);
#endif
#undef DO_INIT_COUNTER
    return 0;
fail:
    sla_fini();
    return -1;
}

#ifndef HAVE_PROC_REMOVE
static void proc_remove(struct proc_dir_entry *de)
{
	if (de)
		remove_proc_subtree(de->name, de->parent);
}
#endif

void sla_fini()
{
    if (sla_req_stats != NULL) {
        CWARN("remove proc sla_req_stats:%p succeed\n", sla_req_stats);
        proc_remove(sla_req_stats);
        sla_req_stats = NULL;
    }
    if (sla_dir != NULL) {
        CWARN("remove proc sla_dir:%p succeed\n", sla_dir);
        proc_remove(sla_dir);
        sla_dir = NULL;
    }
#define DO_DESTROY_COUNTER(counter, bitmask)             \
    if (init_mask & (bitmask))                           \
    {                                                    \
        percpu_counter_destroy(&counter);                \
        CWARN("destroy counter %s succeed\n", #counter); \
    }
    
    DO_DESTROY_COUNTER(sla_ok_req_total, SLA_OK_REQ_BIT);
    DO_DESTROY_COUNTER(sla_lost_perf_req_total, SLA_LOST_PERF_REQ_BIT);
    DO_DESTROY_COUNTER(sla_failed_req_total, SLA_FAILED_REQ_BIT);
    DO_DESTROY_COUNTER(sla_other_req_total, SLA_OTHER_REQ_BIT);
#undef DO_DESTROY_COUNTER
    init_mask = 0;
}

void sla_audit_request(struct ptlrpc_request * req)
{
    if (req != NULL)
    {
        sla_audit_error(req->rq_status);
    }
}

void sla_audit_error(int error)
{
    // for better performance
    if (error >= 0)
    {
        percpu_counter_inc(&sla_ok_req_total);
        return;
    }
    // TODO: add more classified errors.
    switch (error)
    {
        case -EIO:
        case -ENOMEM:
        case -ETIMEDOUT:
            percpu_counter_inc(&sla_failed_req_total);
            break;
        case -ENODEV:
        case -ENOTCONN:
            percpu_counter_inc(&sla_lost_perf_req_total);
            break;
        default:
            percpu_counter_inc(&sla_other_req_total);
            break;
    }
}
