package filesystem

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"io"
	"net/http"
)

// Client is the creates an Azure Data Lake Store filesystem client.
type Client struct {
	BaseClient
}

// NewClient creates an instance of the Client client.
func NewClient() Client {
	return Client{New()}
}

// Append appends to the specified file. This method does not support multiple concurrent appends to the file. NOTE:
// Concurrent append and normal (serial) append CANNOT be used interchangeably. Once a file has been appended to using
// either append option, it can only be appended to using that append option. Use the ConcurrentAppend option if you
// would like support for concurrent appends.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// directFilePath - the Data Lake Store path (starting with '/') of the file to which to append.
// streamContents - the file contents to include when appending to the file.
// offset - the optional offset in the stream to begin the append operation. Default is to append at the end of
// the stream.
func (client Client) Append(ctx context.Context, accountName string, directFilePath string, streamContents io.ReadCloser, offset *int64) (result autorest.Response, err error) {
	req, err := client.AppendPreparer(ctx, accountName, directFilePath, streamContents, offset)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Append", nil, "Failure preparing request")
		return
	}

	resp, err := client.AppendSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Append", resp, "Failure sending request")
		return
	}

	result, err = client.AppendResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Append", resp, "Failure responding to request")
	}

	return
}

// AppendPreparer prepares the Append request.
func (client Client) AppendPreparer(ctx context.Context, accountName string, directFilePath string, streamContents io.ReadCloser, offset *int64) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"directFilePath": autorest.Encode("path", directFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"append":      autorest.Encode("query", "true"),
		"op":          autorest.Encode("query", "APPEND"),
	}
	if offset != nil {
		queryParameters["offset"] = autorest.Encode("query", *offset)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{directFilePath}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// AppendSender sends the Append request. The method will close the
// http.Response Body if it receives an error.
func (client Client) AppendSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AppendResponder handles the response to the Append request. The method always
// closes the http.Response Body.
func (client Client) AppendResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CheckAccess checks if the specified access is available at the given path.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// pathParameter - the Data Lake Store path (starting with '/') of the file or directory for which to check
// access.
// fsaction - file system operation read/write/execute in string form, matching regex pattern '[rwx-]{3}'
func (client Client) CheckAccess(ctx context.Context, accountName string, pathParameter string, fsaction string) (result autorest.Response, err error) {
	req, err := client.CheckAccessPreparer(ctx, accountName, pathParameter, fsaction)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "CheckAccess", nil, "Failure preparing request")
		return
	}

	resp, err := client.CheckAccessSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "CheckAccess", resp, "Failure sending request")
		return
	}

	result, err = client.CheckAccessResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "CheckAccess", resp, "Failure responding to request")
	}

	return
}

// CheckAccessPreparer prepares the CheckAccess request.
func (client Client) CheckAccessPreparer(ctx context.Context, accountName string, pathParameter string, fsaction string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "CHECKACCESS"),
	}
	if len(fsaction) > 0 {
		queryParameters["fsaction"] = autorest.Encode("query", fsaction)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CheckAccessSender sends the CheckAccess request. The method will close the
// http.Response Body if it receives an error.
func (client Client) CheckAccessSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CheckAccessResponder handles the response to the CheckAccess request. The method always
// closes the http.Response Body.
func (client Client) CheckAccessResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Concat concatenates the list of source files into the destination file, removing all source files upon success.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// destinationPath - the Data Lake Store path (starting with '/') of the destination file resulting from the
// concatenation.
// sources - a list of comma seperated Data Lake Store paths (starting with '/') of the files to concatenate,
// in the order in which they should be concatenated.
func (client Client) Concat(ctx context.Context, accountName string, destinationPath string, sources []string) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: sources,
			Constraints: []validation.Constraint{{Target: "sources", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("filesystem.Client", "Concat", err.Error())
	}

	req, err := client.ConcatPreparer(ctx, accountName, destinationPath, sources)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Concat", nil, "Failure preparing request")
		return
	}

	resp, err := client.ConcatSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Concat", resp, "Failure sending request")
		return
	}

	result, err = client.ConcatResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Concat", resp, "Failure responding to request")
	}

	return
}

// ConcatPreparer prepares the Concat request.
func (client Client) ConcatPreparer(ctx context.Context, accountName string, destinationPath string, sources []string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"destinationPath": autorest.Encode("path", destinationPath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "CONCAT"),
		"sources":     autorest.Encode("query", sources, ","),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{destinationPath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ConcatSender sends the Concat request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ConcatSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ConcatResponder handles the response to the Concat request. The method always
// closes the http.Response Body.
func (client Client) ConcatResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ConcurrentAppend appends to the specified file. This method supports multiple concurrent appends to the file. NOTE:
// ConcurrentAppend and normal (serial) Append CANNOT be used interchangeably; once a file has been appended to using
// either of these append options, it can only be appended to using that append option. ConcurrentAppend DOES NOT
// guarantee order and can result in duplicated data landing in the target file. In order to close a file after using
// ConcurrentAppend, call the Flush method.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// filePath - the Data Lake Store path (starting with '/') of the file to which to append using concurrent
// append.
// streamContents - the file contents to include when appending to the file.
// appendMode - indicates the concurrent append call should create the file if it doesn't exist or just open
// the existing file for append
func (client Client) ConcurrentAppend(ctx context.Context, accountName string, filePath string, streamContents io.ReadCloser, appendMode AppendModeType) (result autorest.Response, err error) {
	req, err := client.ConcurrentAppendPreparer(ctx, accountName, filePath, streamContents, appendMode)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ConcurrentAppend", nil, "Failure preparing request")
		return
	}

	resp, err := client.ConcurrentAppendSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ConcurrentAppend", resp, "Failure sending request")
		return
	}

	result, err = client.ConcurrentAppendResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ConcurrentAppend", resp, "Failure responding to request")
	}

	return
}

// ConcurrentAppendPreparer prepares the ConcurrentAppend request.
func (client Client) ConcurrentAppendPreparer(ctx context.Context, accountName string, filePath string, streamContents io.ReadCloser, appendMode AppendModeType) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"filePath": autorest.Encode("path", filePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "CONCURRENTAPPEND"),
	}
	if len(string(appendMode)) > 0 {
		queryParameters["appendMode"] = autorest.Encode("query", appendMode)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/WebHdfsExt/{filePath}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters),
		autorest.WithHeader("Transfer-Encoding", "chunked"))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ConcurrentAppendSender sends the ConcurrentAppend request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ConcurrentAppendSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ConcurrentAppendResponder handles the response to the ConcurrentAppend request. The method always
// closes the http.Response Body.
func (client Client) ConcurrentAppendResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Create creates a file with optionally specified content.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// directFilePath - the Data Lake Store path (starting with '/') of the file to create.
// streamContents - the file contents to include when creating the file. This parameter is optional, resulting
// in an empty file if not specified.
// overwrite - the indication of if the file should be overwritten.
func (client Client) Create(ctx context.Context, accountName string, directFilePath string, streamContents io.ReadCloser, overwrite *bool) (result autorest.Response, err error) {
	req, err := client.CreatePreparer(ctx, accountName, directFilePath, streamContents, overwrite)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Create", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Create", resp, "Failure sending request")
		return
	}

	result, err = client.CreateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Create", resp, "Failure responding to request")
	}

	return
}

// CreatePreparer prepares the Create request.
func (client Client) CreatePreparer(ctx context.Context, accountName string, directFilePath string, streamContents io.ReadCloser, overwrite *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"directFilePath": autorest.Encode("path", directFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "CREATE"),
		"write":       autorest.Encode("query", "true"),
	}
	if overwrite != nil {
		queryParameters["overwrite"] = autorest.Encode("query", *overwrite)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{directFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if streamContents != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithFile(streamContents))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateSender sends the Create request. The method will close the
// http.Response Body if it receives an error.
func (client Client) CreateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateResponder handles the response to the Create request. The method always
// closes the http.Response Body.
func (client Client) CreateResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusCreated),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Delete deletes the requested file or directory, optionally recursively.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// filePath - the Data Lake Store path (starting with '/') of the file or directory to delete.
// recursive - the optional switch indicating if the delete should be recursive
func (client Client) Delete(ctx context.Context, accountName string, filePath string, recursive *bool) (result FileOperationResult, err error) {
	req, err := client.DeletePreparer(ctx, accountName, filePath, recursive)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Delete", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Delete", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Delete", resp, "Failure responding to request")
	}

	return
}

// DeletePreparer prepares the Delete request.
func (client Client) DeletePreparer(ctx context.Context, accountName string, filePath string, recursive *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"filePath": autorest.Encode("path", filePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "DELETE"),
	}
	if recursive != nil {
		queryParameters["recursive"] = autorest.Encode("query", *recursive)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{filePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteSender sends the Delete request. The method will close the
// http.Response Body if it receives an error.
func (client Client) DeleteSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteResponder handles the response to the Delete request. The method always
// closes the http.Response Body.
func (client Client) DeleteResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Flush flushes the specified file to the store. This forces an update to the metadata of the file (returned from
// GetFileStatus), and is required by ConcurrentAppend once the file is done to populate finalized metadata.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// flushFilePath - the Data Lake Store path (starting with '/') of the file to which to flush.
func (client Client) Flush(ctx context.Context, accountName string, flushFilePath string) (result autorest.Response, err error) {
	req, err := client.FlushPreparer(ctx, accountName, flushFilePath)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Flush", nil, "Failure preparing request")
		return
	}

	resp, err := client.FlushSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Flush", resp, "Failure sending request")
		return
	}

	result, err = client.FlushResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Flush", resp, "Failure responding to request")
	}

	return
}

// FlushPreparer prepares the Flush request.
func (client Client) FlushPreparer(ctx context.Context, accountName string, flushFilePath string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"flushFilePath": autorest.Encode("path", flushFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"append":      autorest.Encode("query", "true"),
		"flush":       autorest.Encode("query", "true"),
		"op":          autorest.Encode("query", "APPEND"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{flushFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// FlushSender sends the Flush request. The method will close the
// http.Response Body if it receives an error.
func (client Client) FlushSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// FlushResponder handles the response to the Flush request. The method always
// closes the http.Response Body.
func (client Client) FlushResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// GetACLStatus gets Access Control List (ACL) entries for the specified file or directory.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// ACLFilePath - the Data Lake Store path (starting with '/') of the file or directory for which to get the
// ACL.
func (client Client) GetACLStatus(ctx context.Context, accountName string, ACLFilePath string) (result ACLStatusResult, err error) {
	req, err := client.GetACLStatusPreparer(ctx, accountName, ACLFilePath)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetACLStatus", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetACLStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetACLStatus", resp, "Failure sending request")
		return
	}

	result, err = client.GetACLStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetACLStatus", resp, "Failure responding to request")
	}

	return
}

// GetACLStatusPreparer prepares the GetACLStatus request.
func (client Client) GetACLStatusPreparer(ctx context.Context, accountName string, ACLFilePath string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"aclFilePath": autorest.Encode("path", ACLFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "MSGETACLSTATUS"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{aclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetACLStatusSender sends the GetACLStatus request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetACLStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetACLStatusResponder handles the response to the GetACLStatus request. The method always
// closes the http.Response Body.
func (client Client) GetACLStatusResponder(resp *http.Response) (result ACLStatusResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetContentSummary gets the file content summary object specified by the file path.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// getContentSummaryFilePath - the Data Lake Store path (starting with '/') of the file for which to retrieve
// the summary.
func (client Client) GetContentSummary(ctx context.Context, accountName string, getContentSummaryFilePath string) (result ContentSummaryResult, err error) {
	req, err := client.GetContentSummaryPreparer(ctx, accountName, getContentSummaryFilePath)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetContentSummary", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetContentSummarySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetContentSummary", resp, "Failure sending request")
		return
	}

	result, err = client.GetContentSummaryResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetContentSummary", resp, "Failure responding to request")
	}

	return
}

// GetContentSummaryPreparer prepares the GetContentSummary request.
func (client Client) GetContentSummaryPreparer(ctx context.Context, accountName string, getContentSummaryFilePath string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"getContentSummaryFilePath": autorest.Encode("path", getContentSummaryFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "GETCONTENTSUMMARY"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/va/{getContentSummaryFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetContentSummarySender sends the GetContentSummary request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetContentSummarySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetContentSummaryResponder handles the response to the GetContentSummary request. The method always
// closes the http.Response Body.
func (client Client) GetContentSummaryResponder(resp *http.Response) (result ContentSummaryResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetFileStatus get the file status object specified by the file path.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// getFilePath - the Data Lake Store path (starting with '/') of the file or directory for which to retrieve
// the status.
func (client Client) GetFileStatus(ctx context.Context, accountName string, getFilePath string) (result FileStatusResult, err error) {
	req, err := client.GetFileStatusPreparer(ctx, accountName, getFilePath)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetFileStatus", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetFileStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetFileStatus", resp, "Failure sending request")
		return
	}

	result, err = client.GetFileStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "GetFileStatus", resp, "Failure responding to request")
	}

	return
}

// GetFileStatusPreparer prepares the GetFileStatus request.
func (client Client) GetFileStatusPreparer(ctx context.Context, accountName string, getFilePath string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"getFilePath": autorest.Encode("path", getFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "MSGETFILESTATUS"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{getFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetFileStatusSender sends the GetFileStatus request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetFileStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetFileStatusResponder handles the response to the GetFileStatus request. The method always
// closes the http.Response Body.
func (client Client) GetFileStatusResponder(resp *http.Response) (result FileStatusResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ListFileStatus get the list of file status objects specified by the file path, with optional pagination parameters
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// listFilePath - the Data Lake Store path (starting with '/') of the directory to list.
// listSize - gets or sets the number of items to return. Optional.
// listAfter - gets or sets the item or lexographical index after which to begin returning results. For
// example, a file list of 'a','b','d' and listAfter='b' will return 'd', and a listAfter='c' will also return
// 'd'. Optional.
// listBefore - gets or sets the item or lexographical index before which to begin returning results. For
// example, a file list of 'a','b','d' and listBefore='d' will return 'a','b', and a listBefore='c' will also
// return 'a','b'. Optional.
func (client Client) ListFileStatus(ctx context.Context, accountName string, listFilePath string, listSize *int32, listAfter string, listBefore string) (result FileStatusesResult, err error) {
	req, err := client.ListFileStatusPreparer(ctx, accountName, listFilePath, listSize, listAfter, listBefore)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ListFileStatus", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListFileStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ListFileStatus", resp, "Failure sending request")
		return
	}

	result, err = client.ListFileStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ListFileStatus", resp, "Failure responding to request")
	}

	return
}

// ListFileStatusPreparer prepares the ListFileStatus request.
func (client Client) ListFileStatusPreparer(ctx context.Context, accountName string, listFilePath string, listSize *int32, listAfter string, listBefore string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"listFilePath": autorest.Encode("path", listFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "MSLISTSTATUS"),
	}
	if listSize != nil {
		queryParameters["listSize"] = autorest.Encode("query", *listSize)
	}
	if len(listAfter) > 0 {
		queryParameters["listAfter"] = autorest.Encode("query", listAfter)
	}
	if len(listBefore) > 0 {
		queryParameters["listBefore"] = autorest.Encode("query", listBefore)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{listFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListFileStatusSender sends the ListFileStatus request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListFileStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListFileStatusResponder handles the response to the ListFileStatus request. The method always
// closes the http.Response Body.
func (client Client) ListFileStatusResponder(resp *http.Response) (result FileStatusesResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Mkdirs creates a directory.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// pathParameter - the Data Lake Store path (starting with '/') of the directory to create.
func (client Client) Mkdirs(ctx context.Context, accountName string, pathParameter string) (result FileOperationResult, err error) {
	req, err := client.MkdirsPreparer(ctx, accountName, pathParameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Mkdirs", nil, "Failure preparing request")
		return
	}

	resp, err := client.MkdirsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Mkdirs", resp, "Failure sending request")
		return
	}

	result, err = client.MkdirsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Mkdirs", resp, "Failure responding to request")
	}

	return
}

// MkdirsPreparer prepares the Mkdirs request.
func (client Client) MkdirsPreparer(ctx context.Context, accountName string, pathParameter string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "MKDIRS"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// MkdirsSender sends the Mkdirs request. The method will close the
// http.Response Body if it receives an error.
func (client Client) MkdirsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// MkdirsResponder handles the response to the Mkdirs request. The method always
// closes the http.Response Body.
func (client Client) MkdirsResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ModifyACLEntries modifies existing Access Control List (ACL) entries on a file or folder.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// modifyACLFilePath - the Data Lake Store path (starting with '/') of the file or directory with the ACL being
// modified.
// aclspec - the ACL specification included in ACL modification operations in the format
// '[default:]user|group|other::r|-w|-x|-'
func (client Client) ModifyACLEntries(ctx context.Context, accountName string, modifyACLFilePath string, aclspec string) (result autorest.Response, err error) {
	req, err := client.ModifyACLEntriesPreparer(ctx, accountName, modifyACLFilePath, aclspec)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ModifyACLEntries", nil, "Failure preparing request")
		return
	}

	resp, err := client.ModifyACLEntriesSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ModifyACLEntries", resp, "Failure sending request")
		return
	}

	result, err = client.ModifyACLEntriesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "ModifyACLEntries", resp, "Failure responding to request")
	}

	return
}

// ModifyACLEntriesPreparer prepares the ModifyACLEntries request.
func (client Client) ModifyACLEntriesPreparer(ctx context.Context, accountName string, modifyACLFilePath string, aclspec string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"modifyAclFilePath": autorest.Encode("path", modifyACLFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "MODIFYACLENTRIES"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{modifyAclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ModifyACLEntriesSender sends the ModifyACLEntries request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ModifyACLEntriesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ModifyACLEntriesResponder handles the response to the ModifyACLEntries request. The method always
// closes the http.Response Body.
func (client Client) ModifyACLEntriesResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// MsConcat concatenates the list of source files into the destination file, deleting all source files upon success.
// This method accepts more source file paths than the Concat method. This method and the parameters it accepts are
// subject to change for usability in an upcoming version.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// msConcatDestinationPath - the Data Lake Store path (starting with '/') of the destination file resulting
// from the concatenation.
// streamContents - a list of Data Lake Store paths (starting with '/') of the source files. Must be in the
// format: sources=<comma separated list>
// deleteSourceDirectory - indicates that as an optimization instead of deleting each individual source stream,
// delete the source stream folder if all streams are in the same folder instead. This results in a substantial
// performance improvement when the only streams in the folder are part of the concatenation operation.
// WARNING: This includes the deletion of any other files that are not source files. Only set this to true when
// source files are the only files in the source directory.
func (client Client) MsConcat(ctx context.Context, accountName string, msConcatDestinationPath string, streamContents io.ReadCloser, deleteSourceDirectory *bool) (result autorest.Response, err error) {
	req, err := client.MsConcatPreparer(ctx, accountName, msConcatDestinationPath, streamContents, deleteSourceDirectory)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "MsConcat", nil, "Failure preparing request")
		return
	}

	resp, err := client.MsConcatSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "MsConcat", resp, "Failure sending request")
		return
	}

	result, err = client.MsConcatResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "MsConcat", resp, "Failure responding to request")
	}

	return
}

// MsConcatPreparer prepares the MsConcat request.
func (client Client) MsConcatPreparer(ctx context.Context, accountName string, msConcatDestinationPath string, streamContents io.ReadCloser, deleteSourceDirectory *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"msConcatDestinationPath": autorest.Encode("path", msConcatDestinationPath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "MSCONCAT"),
	}
	if deleteSourceDirectory != nil {
		queryParameters["deleteSourceDirectory"] = autorest.Encode("query", *deleteSourceDirectory)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/octet-stream"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{msConcatDestinationPath}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// MsConcatSender sends the MsConcat request. The method will close the
// http.Response Body if it receives an error.
func (client Client) MsConcatSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// MsConcatResponder handles the response to the MsConcat request. The method always
// closes the http.Response Body.
func (client Client) MsConcatResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Open opens and reads from the specified file.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// directFilePath - the Data Lake Store path (starting with '/') of the file to open.
func (client Client) Open(ctx context.Context, accountName string, directFilePath string, length *int64, offset *int64) (result ReadCloser, err error) {
	req, err := client.OpenPreparer(ctx, accountName, directFilePath, length, offset)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Open", nil, "Failure preparing request")
		return
	}

	resp, err := client.OpenSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Open", resp, "Failure sending request")
		return
	}

	result, err = client.OpenResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Open", resp, "Failure responding to request")
	}

	return
}

// OpenPreparer prepares the Open request.
func (client Client) OpenPreparer(ctx context.Context, accountName string, directFilePath string, length *int64, offset *int64) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"directFilePath": autorest.Encode("path", directFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "OPEN"),
		"read":        autorest.Encode("query", "true"),
	}
	if length != nil {
		queryParameters["length"] = autorest.Encode("query", *length)
	}
	if offset != nil {
		queryParameters["offset"] = autorest.Encode("query", *offset)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{directFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// OpenSender sends the Open request. The method will close the
// http.Response Body if it receives an error.
func (client Client) OpenSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// OpenResponder handles the response to the Open request. The method always
// closes the http.Response Body.
func (client Client) OpenResponder(resp *http.Response) (result ReadCloser, err error) {
	result.Value = &resp.Body
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK))
	result.Response = autorest.Response{Response: resp}
	return
}

// RemoveACL removes the existing Access Control List (ACL) of the specified file or directory.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// ACLFilePath - the Data Lake Store path (starting with '/') of the file or directory with the ACL being
// removed.
func (client Client) RemoveACL(ctx context.Context, accountName string, ACLFilePath string) (result autorest.Response, err error) {
	req, err := client.RemoveACLPreparer(ctx, accountName, ACLFilePath)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveACL", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveACLSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveACL", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveACLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveACL", resp, "Failure responding to request")
	}

	return
}

// RemoveACLPreparer prepares the RemoveACL request.
func (client Client) RemoveACLPreparer(ctx context.Context, accountName string, ACLFilePath string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"aclFilePath": autorest.Encode("path", ACLFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "REMOVEACL"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{aclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RemoveACLSender sends the RemoveACL request. The method will close the
// http.Response Body if it receives an error.
func (client Client) RemoveACLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveACLResponder handles the response to the RemoveACL request. The method always
// closes the http.Response Body.
func (client Client) RemoveACLResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RemoveACLEntries removes existing Access Control List (ACL) entries for a file or folder.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// removeACLFilePath - the Data Lake Store path (starting with '/') of the file or directory with the ACL being
// removed.
// aclspec - the ACL spec included in ACL removal operations in the format '[default:]user|group|other'
func (client Client) RemoveACLEntries(ctx context.Context, accountName string, removeACLFilePath string, aclspec string) (result autorest.Response, err error) {
	req, err := client.RemoveACLEntriesPreparer(ctx, accountName, removeACLFilePath, aclspec)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveACLEntries", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveACLEntriesSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveACLEntries", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveACLEntriesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveACLEntries", resp, "Failure responding to request")
	}

	return
}

// RemoveACLEntriesPreparer prepares the RemoveACLEntries request.
func (client Client) RemoveACLEntriesPreparer(ctx context.Context, accountName string, removeACLFilePath string, aclspec string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"removeAclFilePath": autorest.Encode("path", removeACLFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "REMOVEACLENTRIES"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{removeAclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RemoveACLEntriesSender sends the RemoveACLEntries request. The method will close the
// http.Response Body if it receives an error.
func (client Client) RemoveACLEntriesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveACLEntriesResponder handles the response to the RemoveACLEntries request. The method always
// closes the http.Response Body.
func (client Client) RemoveACLEntriesResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RemoveDefaultACL removes the existing Default Access Control List (ACL) of the specified directory.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// defaultACLFilePath - the Data Lake Store path (starting with '/') of the directory with the default ACL
// being removed.
func (client Client) RemoveDefaultACL(ctx context.Context, accountName string, defaultACLFilePath string) (result autorest.Response, err error) {
	req, err := client.RemoveDefaultACLPreparer(ctx, accountName, defaultACLFilePath)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveDefaultACL", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveDefaultACLSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveDefaultACL", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveDefaultACLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "RemoveDefaultACL", resp, "Failure responding to request")
	}

	return
}

// RemoveDefaultACLPreparer prepares the RemoveDefaultACL request.
func (client Client) RemoveDefaultACLPreparer(ctx context.Context, accountName string, defaultACLFilePath string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"defaultAclFilePath": autorest.Encode("path", defaultACLFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "REMOVEDEFAULTACL"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{defaultAclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RemoveDefaultACLSender sends the RemoveDefaultACL request. The method will close the
// http.Response Body if it receives an error.
func (client Client) RemoveDefaultACLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveDefaultACLResponder handles the response to the RemoveDefaultACL request. The method always
// closes the http.Response Body.
func (client Client) RemoveDefaultACLResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Rename rename a file or directory.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// renameFilePath - the Data Lake Store path (starting with '/') of the file or directory to move/rename.
// destination - the path to move/rename the file or folder to
func (client Client) Rename(ctx context.Context, accountName string, renameFilePath string, destination string) (result FileOperationResult, err error) {
	req, err := client.RenamePreparer(ctx, accountName, renameFilePath, destination)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Rename", nil, "Failure preparing request")
		return
	}

	resp, err := client.RenameSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Rename", resp, "Failure sending request")
		return
	}

	result, err = client.RenameResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "Rename", resp, "Failure responding to request")
	}

	return
}

// RenamePreparer prepares the Rename request.
func (client Client) RenamePreparer(ctx context.Context, accountName string, renameFilePath string, destination string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"renameFilePath": autorest.Encode("path", renameFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"destination": autorest.Encode("query", destination),
		"op":          autorest.Encode("query", "RENAME"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{renameFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RenameSender sends the Rename request. The method will close the
// http.Response Body if it receives an error.
func (client Client) RenameSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RenameResponder handles the response to the Rename request. The method always
// closes the http.Response Body.
func (client Client) RenameResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetACL sets the Access Control List (ACL) for a file or folder.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// setACLFilePath - the Data Lake Store path (starting with '/') of the file or directory on which to set the
// ACL.
// aclspec - the ACL spec included in ACL creation operations in the format
// '[default:]user|group|other::r|-w|-x|-'
func (client Client) SetACL(ctx context.Context, accountName string, setACLFilePath string, aclspec string) (result autorest.Response, err error) {
	req, err := client.SetACLPreparer(ctx, accountName, setACLFilePath, aclspec)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetACL", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetACLSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetACL", resp, "Failure sending request")
		return
	}

	result, err = client.SetACLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetACL", resp, "Failure responding to request")
	}

	return
}

// SetACLPreparer prepares the SetACL request.
func (client Client) SetACLPreparer(ctx context.Context, accountName string, setACLFilePath string, aclspec string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"setAclFilePath": autorest.Encode("path", setACLFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "SETACL"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{setAclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetACLSender sends the SetACL request. The method will close the
// http.Response Body if it receives an error.
func (client Client) SetACLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetACLResponder handles the response to the SetACL request. The method always
// closes the http.Response Body.
func (client Client) SetACLResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// SetFileExpiry sets or removes the expiration time on the specified file. This operation can only be executed against
// files. Folders are not supported.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// filePath - the Data Lake Store path (starting with '/') of the file on which to set or remove the expiration
// time.
// expiryOption - indicates the type of expiration to use for the file: 1. NeverExpire: ExpireTime is ignored.
// 2. RelativeToNow: ExpireTime is an integer in milliseconds representing the expiration date relative to when
// file expiration is updated. 3. RelativeToCreationDate: ExpireTime is an integer in milliseconds representing
// the expiration date relative to file creation. 4. Absolute: ExpireTime is an integer in milliseconds, as a
// Unix timestamp relative to 1/1/1970 00:00:00.
// expireTime - the time that the file will expire, corresponding to the ExpiryOption that was set.
func (client Client) SetFileExpiry(ctx context.Context, accountName string, filePath string, expiryOption ExpiryOptionType, expireTime *int64) (result autorest.Response, err error) {
	req, err := client.SetFileExpiryPreparer(ctx, accountName, filePath, expiryOption, expireTime)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetFileExpiry", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetFileExpirySender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetFileExpiry", resp, "Failure sending request")
		return
	}

	result, err = client.SetFileExpiryResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetFileExpiry", resp, "Failure responding to request")
	}

	return
}

// SetFileExpiryPreparer prepares the SetFileExpiry request.
func (client Client) SetFileExpiryPreparer(ctx context.Context, accountName string, filePath string, expiryOption ExpiryOptionType, expireTime *int64) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"filePath": autorest.Encode("path", filePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version":  APIVersion,
		"expiryOption": autorest.Encode("query", expiryOption),
		"op":           autorest.Encode("query", "SETEXPIRY"),
	}
	if expireTime != nil {
		queryParameters["expireTime"] = autorest.Encode("query", *expireTime)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/WebHdfsExt/{filePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetFileExpirySender sends the SetFileExpiry request. The method will close the
// http.Response Body if it receives an error.
func (client Client) SetFileExpirySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetFileExpiryResponder handles the response to the SetFileExpiry request. The method always
// closes the http.Response Body.
func (client Client) SetFileExpiryResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// SetOwner sets the owner of a file or directory.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// setOwnerFilePath - the Data Lake Store path (starting with '/') of the file or directory for which to set
// the owner.
// owner - the AAD Object ID of the user owner of the file or directory. If empty, the property will remain
// unchanged.
// group - the AAD Object ID of the group owner of the file or directory. If empty, the property will remain
// unchanged.
func (client Client) SetOwner(ctx context.Context, accountName string, setOwnerFilePath string, owner string, group string) (result autorest.Response, err error) {
	req, err := client.SetOwnerPreparer(ctx, accountName, setOwnerFilePath, owner, group)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetOwner", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetOwnerSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetOwner", resp, "Failure sending request")
		return
	}

	result, err = client.SetOwnerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetOwner", resp, "Failure responding to request")
	}

	return
}

// SetOwnerPreparer prepares the SetOwner request.
func (client Client) SetOwnerPreparer(ctx context.Context, accountName string, setOwnerFilePath string, owner string, group string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"setOwnerFilePath": autorest.Encode("path", setOwnerFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "SETOWNER"),
	}
	if len(owner) > 0 {
		queryParameters["owner"] = autorest.Encode("query", owner)
	}
	if len(group) > 0 {
		queryParameters["group"] = autorest.Encode("query", group)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{setOwnerFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetOwnerSender sends the SetOwner request. The method will close the
// http.Response Body if it receives an error.
func (client Client) SetOwnerSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetOwnerResponder handles the response to the SetOwner request. The method always
// closes the http.Response Body.
func (client Client) SetOwnerResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// SetPermission sets the permission of the file or folder.
// Parameters:
// accountName - the Azure Data Lake Store account to execute filesystem operations on.
// setPermissionFilePath - the Data Lake Store path (starting with '/') of the file or directory for which to
// set the permission.
// permission - a string representation of the permission (i.e 'rwx'). If empty, this property remains
// unchanged.
func (client Client) SetPermission(ctx context.Context, accountName string, setPermissionFilePath string, permission string) (result autorest.Response, err error) {
	req, err := client.SetPermissionPreparer(ctx, accountName, setPermissionFilePath, permission)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetPermission", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetPermissionSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetPermission", resp, "Failure sending request")
		return
	}

	result, err = client.SetPermissionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.Client", "SetPermission", resp, "Failure responding to request")
	}

	return
}

// SetPermissionPreparer prepares the SetPermission request.
func (client Client) SetPermissionPreparer(ctx context.Context, accountName string, setPermissionFilePath string, permission string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"setPermissionFilePath": autorest.Encode("path", setPermissionFilePath),
	}

	const APIVersion = "2015-10-01-preview"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "SETPERMISSION"),
	}
	if len(permission) > 0 {
		queryParameters["permission"] = autorest.Encode("query", permission)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{setPermissionFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetPermissionSender sends the SetPermission request. The method will close the
// http.Response Body if it receives an error.
func (client Client) SetPermissionSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetPermissionResponder handles the response to the SetPermission request. The method always
// closes the http.Response Body.
func (client Client) SetPermissionResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}
