package cmd

import (
	"bytes"
	"io"
	"io/ioutil"
	"os"
	"strings"
	"testing"
	"time"

	"github.com/openshift/origin/tools/depcheck/glide"
)

var yamlFileData = `package: github.com/openshift/origin
excludeDirs:
- dir/one
- dir/two
import:
- package: github.com/pkg/one
  version: v3.2.8
- package: github.com/pkg/two
  version: bde6d0fbd9dec5a997e906611fe0364001364c41

# this comment should be preserved
- package: github.com/pkg/three
  version: release-1.0
`

var lockFileData = `hash: 7308ac17a0294aa86998765c0fc8bd42c73a215584e73cffbedb0c38d7b31c1d
updated: 2017-12-14T09:07:15.649059681-05:00
imports:
- name: github.com/pkg/one
  version: 75cd24fc2f2c2a2088577d12123ddee5f54e0675
- name: github.com/pkg/two
  version: 3b1ae45394a234c385be014e9a488f2bb6eef821
- name: github.com/pkg/three
  version: fc380db5fc13c6dd71a5b0bf2af0d182865d1b1d
- name: github.com/pkg/four
  version: bbbad097214e2918d8543d5201d12bfd7bca254d
  repo: https://github.com/openshift/pkg-four
- name: github.com/pkg/five
  version: 593d64559f7600f29581a3ee42177f5dbded27a9
`

func setupAndRunPinOpts(out, errout io.Writer) (*PinImportsOpts, error) {
	yamlfile := &glide.YamlFile{
		Package: "test",
		Imports: []*glide.YamlFileImport{},
	}
	lockfile := &glide.LockFile{
		Hash:    "test",
		Updated: time.Now(),
		Imports: []*glide.LockFileImport{},
	}

	opts := &PinImportsOpts{
		LockFile: lockfile,
		YamlFile: yamlfile,

		ExistingGlideYamlContent: []byte(yamlFileData),

		Out:    out,
		ErrOut: errout,
	}

	if err := opts.Validate(); err != nil {
		return nil, err
	}

	return opts, nil
}

func TestCorrectlySetupOptsRunWithNoError(t *testing.T) {
	out := bytes.NewBuffer([]byte{})
	errout := out

	opts, err := setupAndRunPinOpts(out, errout)
	if err != nil {
		t.Fatal(err)
	}

	if err := opts.Run(); err != nil {
		t.Fatal(err)
	}
}

func TestOptsOutputsOriginalData(t *testing.T) {
	out := bytes.NewBuffer([]byte{})
	errout := out

	opts, err := setupAndRunPinOpts(out, errout)
	if err != nil {
		t.Fatal(err)
	}

	if err := opts.Run(); err != nil {
		t.Fatal(err)
	}

	if !strings.Contains(out.String(), yamlFileData) {
		t.Fatalf("expected options output to preserve original glide.yaml file data %v, but got %v", yamlFileData, out.String())
	}
}

func TestOptsOutputsGeneratedComment(t *testing.T) {
	out := bytes.NewBuffer([]byte{})
	errout := out

	opts, err := setupAndRunPinOpts(out, errout)
	if err != nil {
		t.Fatal(err)
	}

	if err := opts.Run(); err != nil {
		t.Fatal(err)
	}

	if !strings.Contains(out.String(), "# generated by") {
		t.Fatalf("expected options output to contain \"# generated by ...\" comment, but got %v", out.String())
	}
}

func TestOptionsRunErrorsOnNilLockFile(t *testing.T) {
	out := bytes.NewBuffer([]byte{})
	errout := out

	opts, err := setupAndRunPinOpts(out, errout)
	if err != nil {
		t.Fatal(err)
	}

	opts.LockFile = nil
	err = opts.Run()
	if err == nil {
		t.Fatalf("expected missing imports error but got no error")
	}

	needle := "unable to calculate missing imports"
	if !strings.Contains(err.Error(), needle) {
		t.Fatalf("expected opts error to contain %v, but got %v", needle, err.Error())
	}
}

func TestValidDependenciesArePinnedToOutput(t *testing.T) {
	out := bytes.NewBuffer([]byte{})
	errout := bytes.NewBuffer([]byte{})

	yamlfile := &glide.YamlFile{
		Package: "test",
		Imports: []*glide.YamlFileImport{
			{
				Package: "pkg/one",
				Version: "1",
			},
			{
				Package: "pkg/two",
				Version: "2",
			},
		},
	}
	lockfile := &glide.LockFile{
		Hash:    "test",
		Updated: time.Now(),
		Imports: []*glide.LockFileImport{
			{
				Name:    "pkg/one",
				Version: "1",
			},
			{
				Name:    "pkg/two",
				Version: "2",
			},
			{
				Name:    "pkg/three",
				Version: "3",
			},
			{
				Name:    "pkg/four",
				Repo:    "repo",
				Version: "4",
			},
		},
	}

	encodedYamlFile, err := yamlfile.Encode()
	if err != nil {
		t.Fatal(err)
	}

	opts := &PinImportsOpts{
		LockFile: lockfile,
		YamlFile: yamlfile,

		ExistingGlideYamlContent: encodedYamlFile,

		Out:    out,
		ErrOut: errout,
	}

	if err := opts.Validate(); err != nil {
		t.Fatal(err)
	}

	err = opts.Run()
	if err != nil {
		t.Fatal(err)
	}

	expected := `package: test
excludeDirs: []
import:
- package: pkg/one
  version: "1"
- package: pkg/two
  version: "2"


# generated by tools/depcheck
- package: pkg/three
  version: "3"
`

	if out.String() != expected {
		t.Fatalf("invalid command output: expected\n %v\n\n but got\n %v", expected, out.String())
	}
}

func TestFlagsToOptionsReturnsOptions(t *testing.T) {
	tests := []struct {
		name             string
		lockFileData     []byte
		yamlFileData     []byte
		expectErr        bool
		expectErrMessage string
	}{
		{
			name:             "error on empty LockFile",
			expectErr:        true,
			expectErrMessage: "contained an empty file",
		},
		{
			name:             "error on empty YamlFile",
			expectErr:        true,
			lockFileData:     []byte(lockFileData),
			expectErrMessage: "contained an empty file",
		},
		{
			name:             "error on invalid data format",
			lockFileData:     []byte(lockFileData),
			yamlFileData:     []byte("invalid"),
			expectErr:        true,
			expectErrMessage: "cannot unmarshal",
		},
		{
			name:         "success creating options with decoded yaml bytes",
			lockFileData: []byte(lockFileData),
			yamlFileData: []byte(yamlFileData),
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			out := bytes.NewBuffer([]byte{})
			errout := out

			lockfile, err := ioutil.TempFile(os.TempDir(), "lockfile")
			if err != nil {
				t.Fatal(err)
			}
			defer os.Remove(lockfile.Name())

			yamlfile, err := ioutil.TempFile(os.TempDir(), "yamlfile")
			if err != nil {
				t.Fatal(err)
			}
			defer os.Remove(yamlfile.Name())

			if len(tc.lockFileData) > 0 {
				if _, err := lockfile.Write(tc.lockFileData); err != nil {
					t.Fatal(err)
				}
			}
			if len(tc.yamlFileData) > 0 {
				if _, err := yamlfile.Write(tc.yamlFileData); err != nil {
					t.Fatal(err)
				}
			}

			flags := PinImportsFlags{
				LockFileName: lockfile.Name(),
				YamlFileName: yamlfile.Name(),
			}
			opts, err := flags.ToOptions(out, errout)
			switch {
			case tc.expectErr && err != nil:
				if !strings.Contains(err.Error(), tc.expectErrMessage) {
					t.Fatalf("expected error to contain %q, but got %q", tc.expectErrMessage, err.Error())
				}
			case tc.expectErr && err == nil:
				t.Fatalf("expected error with message %q, but got no error", tc.expectErrMessage)
			case !tc.expectErr && err != nil:
				t.Fatalf("unexpected error: %v", err)
			}

			if tc.expectErr {
				return
			}
			if opts == nil {
				t.Fatalf("expected non-nil options struct")
			}
		})
	}
}

func TestPinImportsAppendsNewImports(t *testing.T) {
	out := bytes.NewBuffer([]byte{})
	errout := bytes.NewBuffer([]byte{})

	lockfile := &glide.LockFile{}
	yamlfile := &glide.YamlFile{}

	if err := lockfile.Decode([]byte(lockFileData)); err != nil {
		t.Fatalf("%v", err)
	}

	if err := yamlfile.Decode([]byte(yamlFileData)); err != nil {
		t.Fatalf("%v", err)
	}

	opts := PinImportsOpts{
		LockFile: lockfile,
		YamlFile: yamlfile,

		Out:    out,
		ErrOut: errout,
	}

	err := opts.Run()
	if err != nil {
		t.Fatalf("%v", err)
	}

	needle := "github.com/pkg/five"
	if !strings.Contains(out.String(), needle) {
		t.Fatalf("expected command output to contain %q but got %s", needle, out.String())
	}
}
