/*
Copyright 2019 HAProxy Technologies

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package actions

import (
	"fmt"
	"strings"
)

// http-check send [meth <method>] [{ uri <uri> | uri-lf <fmt> }>] [ver <version>]
//                 [hdr <name> <fmt>]* [{ body <string> | body-lf <fmt> }]
//                 [comment <msg>]
type CheckSend struct {
	Method        string
	URI           string
	URILogFormat  string
	Version       string
	Header        []CheckSendHeader
	Body          string
	BodyLogFormat string
	CheckComment  string
	Comment       string
}

type CheckSendHeader struct {
	Name   string
	Format string
}

func (c *CheckSend) Parse(parts []string, comment string) error {
	if comment != "" {
		c.Comment = comment
	}

	if len(parts) < 3 {
		return fmt.Errorf("not enough params")
	}

	for i := 2; i < len(parts); i++ {
		el := parts[i]
		switch el {
		case "meth":
			checkParsePair(parts, &i, &c.Method)
		case "uri":
			checkParsePair(parts, &i, &c.URI)
		case "uri-lf":
			checkParsePair(parts, &i, &c.URILogFormat)
		case "ver":
			checkParsePair(parts, &i, &c.Version)
		// NOTE: HAProxy config supports header values containing spaces,
		// which config-parser normally would support with `\ `.
		// However, because parts is split by spaces and hdr can be provided
		// multiple times with other directives surrounding it, it's
		// impossible to read ahead to put the pieces together.
		// As such, header values with spaces are not supported.
		case "hdr":
			if (i+1) < len(parts) && (i+2) < len(parts) {
				c.Header = append(c.Header, CheckSendHeader{Name: parts[i+1], Format: parts[i+2]})
				i++
			}
		case "body":
			checkParsePair(parts, &i, &c.Body)
		case "body-lf":
			checkParsePair(parts, &i, &c.BodyLogFormat)
		case "comment":
			checkParsePair(parts, &i, &c.CheckComment)
		}
	}

	return nil
}

func (c *CheckSend) String() string {
	sb := &strings.Builder{}

	sb.WriteString("send")

	checkWritePair(sb, "meth", c.Method)
	checkWritePair(sb, "uri", c.URI)
	checkWritePair(sb, "uri-lf", c.URILogFormat)
	checkWritePair(sb, "ver", c.Version)
	for _, h := range c.Header {
		checkWritePair(sb, "hdr", h.Name)
		checkWritePair(sb, "", h.Format)
	}
	checkWritePair(sb, "body", c.Body)
	checkWritePair(sb, "body-lf", c.BodyLogFormat)
	checkWritePair(sb, "comment", c.CheckComment)

	return sb.String()
}

func (c *CheckSend) GetComment() string {
	return c.Comment
}
