import * as _ from 'lodash-es';
import * as React from 'react';
import { sortable } from '@patternfly/react-table';
import * as classNames from 'classnames';
import { MachineConfigKind, referenceForModel } from '../module/k8s';
import { MachineConfigModel } from '../models';
import { DetailsPage, ListPage, Table, TableRow, TableData, RowFunction } from './factory';
import {
  Kebab,
  navFactory,
  ResourceKebab,
  ResourceLink,
  ResourceSummary,
  SectionHeading,
  Timestamp,
} from './utils';
import { ResourceEventStream } from './events';

export const machineConfigReference = referenceForModel(MachineConfigModel);
const machineConfigMenuActions = [
  ...Kebab.getExtensionsActionsForKind(MachineConfigModel),
  ...Kebab.factory.common,
];

const MachineConfigSummary: React.SFC<MachineConfigSummaryProps> = ({ obj }) => (
  <ResourceSummary resource={obj}>
    <dt>OS Image URL</dt>
    <dd>{obj.spec.osImageURL || '-'}</dd>
  </ResourceSummary>
);

const MachineConfigDetails: React.SFC<MachineConfigDetailsProps> = ({ obj }) => (
  <div className="co-m-pane__body">
    <SectionHeading text="Machine Config Details" />
    <div className="row">
      <div className="col-xs-12">
        <MachineConfigSummary obj={obj} />
      </div>
    </div>
  </div>
);

const pages = [
  navFactory.details(MachineConfigDetails),
  navFactory.editYaml(),
  navFactory.events(ResourceEventStream),
];

export const MachineConfigDetailsPage: React.SFC<any> = (props) => {
  return (
    <DetailsPage
      {...props}
      kind={machineConfigReference}
      menuActions={machineConfigMenuActions}
      pages={pages}
    />
  );
};

const tableColumnClasses = [
  classNames('col-lg-2', 'col-md-3', 'col-sm-4', 'col-xs-6'),
  classNames('col-lg-3', 'col-md-4', 'col-sm-6', 'hidden-xs'),
  classNames('col-lg-3', 'col-md-3', 'hidden-sm', 'hidden-xs'),
  classNames('col-lg-2', 'hidden-md', 'hidden-sm', 'hidden-xs'),
  classNames('col-lg-2', 'col-md-2', 'col-sm-2', 'col-xs-6'),
  Kebab.columnClass,
];

const MachineConfigTableHeader = () => {
  return [
    {
      title: 'Name',
      sortField: 'metadata.name',
      transforms: [sortable],
      props: { className: tableColumnClasses[0] },
    },
    {
      title: 'Generated By Controller',
      sortField:
        "metadata.annotations['machineconfiguration.openshift.io/generated-by-controller-version']",
      transforms: [sortable],
      props: { className: tableColumnClasses[1] },
    },
    {
      title: 'Ignition Version',
      sortField: 'spec.config.ignition.version',
      transforms: [sortable],
      props: { className: tableColumnClasses[2] },
    },
    {
      title: 'OS Image URL',
      sortField: 'spec.osImageURL',
      transforms: [sortable],
      props: { className: tableColumnClasses[3] },
    },
    {
      title: 'Created',
      sortField: 'metadata.creationTimestamp',
      transforms: [sortable],
      props: { className: tableColumnClasses[4] },
    },
    {
      title: '',
      props: { className: tableColumnClasses[5] },
    },
  ];
};
MachineConfigTableHeader.displayName = 'MachineConfigTableHeader';

const MachineConfigTableRow: RowFunction<MachineConfigKind> = ({ obj, index, key, style }) => {
  return (
    <TableRow id={obj.metadata.uid} index={index} trKey={key} style={style}>
      <TableData className={tableColumnClasses[0]}>
        <ResourceLink
          kind={machineConfigReference}
          name={obj.metadata.name}
          title={obj.metadata.name}
        />
      </TableData>
      <TableData className={classNames(tableColumnClasses[1], 'co-break-word')}>
        {_.get(
          obj,
          [
            'metadata',
            'annotations',
            'machineconfiguration.openshift.io/generated-by-controller-version',
          ],
          '-',
        )}
      </TableData>
      <TableData className={tableColumnClasses[2]}>
        {_.get(obj, 'spec.config.ignition.version') || '-'}
      </TableData>
      <TableData className={classNames(tableColumnClasses[3], 'co-break-word')}>
        {_.get(obj, 'spec.osImageURL') || '-'}
      </TableData>
      <TableData className={tableColumnClasses[4]}>
        <Timestamp timestamp={obj.metadata.creationTimestamp} />
      </TableData>
      <TableData className={tableColumnClasses[5]}>
        <ResourceKebab
          actions={machineConfigMenuActions}
          kind={machineConfigReference}
          resource={obj}
        />
      </TableData>
    </TableRow>
  );
};

const MachineConfigList: React.SFC<any> = (props) => (
  <Table
    {...props}
    aria-label="Machine Configs"
    Header={MachineConfigTableHeader}
    Row={MachineConfigTableRow}
    virtualize
  />
);

export const MachineConfigPage: React.SFC<any> = ({ canCreate = true, ...rest }) => (
  <ListPage
    {...rest}
    canCreate={canCreate}
    ListComponent={MachineConfigList}
    kind={machineConfigReference}
  />
);

type MachineConfigDetailsProps = {
  obj: MachineConfigKind;
};

type MachineConfigSummaryProps = {
  obj: MachineConfigKind;
};
