/*
Copyright 2018 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package wrapper

import (
	"context"
	"errors"
	"flag"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strconv"
	"strings"
	"time"

	"github.com/fsnotify/fsnotify"
	"github.com/sirupsen/logrus"
)

// Options exposes the configuration options
// used when wrapping test execution
type Options struct {
	// Args is the process and args to run
	Args []string `json:"args,omitempty"`

	// ProcessLog will contain std{out,err} from the
	// wrapped test process
	ProcessLog string `json:"process_log"`

	// MarkerFile will be written with the exit code
	// of the test process or an internal error code
	// if the entrypoint fails.
	MarkerFile string `json:"marker_file"`

	// MetadataFile is a file generated by the job,
	// and contains job metadata info like node image
	// versions for rendering in other tools like
	// testgrid/gubernator.
	// Prow will parse the file and merge it into
	// the `metadata` field in finished.json
	MetadataFile string `json:"metadata_file"`
}

// AddFlags adds flags to the FlagSet that populate
// the wrapper options struct provided.
func (o *Options) AddFlags(fs *flag.FlagSet) {
	fs.StringVar(&o.ProcessLog, "process-log", "", "path to the log where stdout and stderr are streamed for the process we execute")
	fs.StringVar(&o.MarkerFile, "marker-file", "", "file we write the return code of the process we execute once it has finished running")
	fs.StringVar(&o.MetadataFile, "metadata-file", "", "path to the metadata file generated from the job")
}

// Validate ensures that the set of options are
// self-consistent and valid
func (o *Options) Validate() error {
	if o.ProcessLog == "" {
		return errors.New("no log file specified with --process-log")
	}

	if o.MarkerFile == "" {
		return errors.New("no marker file specified with --marker-file")
	}

	return nil
}

func WaitForMarker(ctx context.Context, path string) (int, error) {
	// Only start watching file events if the file doesn't exist
	// If the file exists, it means the main process already completed.
	if _, err := os.Stat(path); os.IsNotExist(err) {
		watcher, err := fsnotify.NewWatcher()
		if err != nil {
			return -1, fmt.Errorf("new fsnotify watch: %v", err)
		}
		defer watcher.Close()
		dir := filepath.Dir(path)
		if err := watcher.Add(dir); err != nil {
			return -1, fmt.Errorf("add %s to fsnotify watch: %v", dir, err)
		}

		ticker := time.NewTicker(10 * time.Second)
		defer ticker.Stop()
		missing := true
		for missing {
			select {
			case <-ctx.Done():
				return -1, fmt.Errorf("cancelled: %v", ctx.Err())
			case event := <-watcher.Events:
				if event.Name == path && event.Op&fsnotify.Create == fsnotify.Create {
					missing = false
				}
			case err := <-watcher.Errors:
				logrus.WithError(err).Info("fsnotify watch error")
			case <-ticker.C:
				switch _, err := os.Stat(path); {
				case err == nil, !os.IsNotExist(err):
					missing = false
				}
			}
		}
	}

	returnCodeData, err := ioutil.ReadFile(path)
	if err != nil {
		return -1, fmt.Errorf("bad read: %v", err)
	}
	returnCode, err := strconv.Atoi(strings.TrimSpace(string(returnCodeData)))
	if err != nil {
		return -1, fmt.Errorf("invalid return code: %v", err)
	}
	return returnCode, nil
}
