#!/usr/bin/env python3

import logging
from fnmatch import fnmatch
from pathlib import Path
from shutil import copytree, rmtree
from typing import Iterable, MutableMapping
from os import environ
import re
import json
import sys

from yaml import safe_dump, safe_load, Dumper

# Don't use yaml anchors
Dumper.ignore_aliases = lambda *args: True

logger = logging.getLogger()
logging.basicConfig(level=logging.INFO)


def main() -> None:
    logger.info("Add ocs-operator to the related images list")

    try:
        manifest_dir = Path(sys.argv[1])
    except IndexError:
        logger.error("Manifests dir should be provided")
        raise

    csv = next(manifest_dir.glob("*.clusterserviceversion.*"), None)
    if not csv:
        raise RuntimeError(f"Failed to find a csv file at {manifest_dir.absolute()}")

    ocs_image = environ.get("OCS_IMAGE")
    if not ocs_image:
        raise RuntimeError("Failed to get the OCS image pull spec from $OCS_IMAGE")

    patch_csv(csv, ocs_image)
    logger.info("Success")


def patch_csv(csv: Path, ocs_image: str) -> None:
    with csv.open(mode="r+") as fd:
        csv_struct = safe_load(fd)
        csv_struct["spec"].setdefault("relatedImages", []).append(
            {"image": ocs_image, "name": "ocs-operator"}
        )
        fd.truncate(0)
        fd.seek(0)
        safe_dump(csv_struct, fd, default_flow_style=False)


if __name__ == "__main__":
    main()
