#! /usr/bin/env bash

set -ex

# This map maps the image names that we use for our upstream testing
# (quay.io images) to the pullable image version (i.e. using DS testing
# registry) and final image version (using redhat.io registry) separated by
# '|' character . It is used to translate the current quay.io images and
# their tags to the latest available image identified by SHA
declare -A DS_MAP
DS_MAP=(
        [quay.io/rhceph-dev/cephcsi]="registry-proxy.engineering.redhat.com/rh-osbs/cephcsi|registry.redhat.io/ocs4/cephcsi-rhel8"
        [quay.io/rhceph-dev/mcg-core]="registry-proxy.engineering.redhat.com/rh-osbs/mcg-core|registry.redhat.io/ocs4/mcg-core-rhel8"
        [quay.io/rhceph-dev/mcg-operator]="registry-proxy.engineering.redhat.com/rh-osbs/mcg-operator|registry.redhat.io/ocs4/mcg-rhel8-operator"
        [quay.io/rhceph-dev/ocs-operator]="registry-proxy.engineering.redhat.com/rh-osbs/ocs-operator|registry.redhat.io/ocs4/ocs-rhel8-operator"
        [quay.io/rhceph-dev/rhceph]="registry-proxy.engineering.redhat.com/rh-osbs/rhceph|registry.redhat.io/rhceph/rhceph-4-rhel8"
        [quay.io/rhceph-dev/rook-ceph]="registry-proxy.engineering.redhat.com/rh-osbs/rook-ceph|registry.redhat.io/ocs4/rook-ceph-rhel8-operator"
        [registry.redhat.io/openshift4/ose-csi-external-attacher]="registry.redhat.io/openshift4/ose-csi-external-attacher|registry.redhat.io/openshift4/ose-csi-external-attacher"
        [registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel7]="registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel7|registry.redhat.io/openshift4/ose-csi-external-provisioner-rhel7"
        [registry.redhat.io/openshift4/ose-csi-driver-registrar]="registry.redhat.io/openshift4/ose-csi-driver-registrar|registry.redhat.io/openshift4/ose-csi-driver-registrar"
        [registry.redhat.io/rhscl/mongodb-36-rhel7]="registry.redhat.io/rhscl/mongodb-36-rhel7|registry.redhat.io/rhscl/mongodb-36-rhel7"
)

CSV_VERSION="$1"
RELEASE_CANDIDATE="$2"

if test -z "$CSV_VERSION" -o -z "$RELEASE_CANDIDATE"; then
        echo "Usage: $0 <CSV_VERSION> <RELEASE_CANDIDATE>"
        exit 0
fi

# Install proper yq yaml parser, first
echo "Downloading yq..."
wget -q --continue https://github.com/mikefarah/yq/releases/download/2.4.0/yq_linux_amd64 -O yq
export YQ_TOOL=$(readlink -f ./yq)
chmod +x "$YQ_TOOL"

# Install oc
echo "Downloading oc..."
wget -q --continue "https://mirror.openshift.com/pub/openshift-v4/clients/ocp/4.2.0/openshift-client-linux-4.2.0.tar.gz" -O oc.tar.gz
tar -xzvf oc.tar.gz
export OC_TOOL=$(readlink -f ./oc)
chmod +x "$OC_TOOL"

### Fix CSV version
# The main package file
PKG_FILE="manifests/ocs-operator.package.yaml"

# Get old versions
OLD_CSV=$("$YQ_TOOL" r "$PKG_FILE" 'channels[0].currentCSV')
OLD_FILE=$(echo manifests/*/$OLD_CSV.clusterserviceversion.yaml)
OLD_VERSION=$("$YQ_TOOL" r "$OLD_FILE" 'spec.version')

# Move the CSV versioned dir
git mv manifests/$OLD_VERSION manifests/$CSV_VERSION

# Move the main CSV file
OLD_FILE=$(echo manifests/*/$OLD_CSV.clusterserviceversion.yaml)
NEW_CSV="ocs-operator.v$CSV_VERSION"
NEW_FILE="$(dirname $OLD_FILE)/$NEW_CSV.clusterserviceversion.yaml"
git mv "$OLD_FILE" "$NEW_FILE"

# Write currentCSV
"$YQ_TOOL" w -i "$PKG_FILE" 'channels[0].currentCSV' "$NEW_CSV"
"$YQ_TOOL" w -i "$PKG_FILE" 'channels[0].name' "stable-4.2"

# Fix skipRange
"$YQ_TOOL" w -i "$NEW_FILE" 'metadata.annotations[olm.skipRange]' ">=4.2.0 <$CSV_VERSION"

# Fix name and version in main CSV file
"$YQ_TOOL" w -i "$NEW_FILE" 'metadata.name' "$NEW_CSV"
"$YQ_TOOL" w -i "$NEW_FILE" 'spec.version' "$CSV_VERSION"

# Fix the CSV images to DS location
# This also changes the tags to SHAs
# Loop over the repos/build jobs
echo "=== CSV VERSION -> $CSV_VERSION-$RELEASE_CANDIDATE ===" | tee -a BUILDLOG
for IMAGE in "${!DS_MAP[@]}"; do
        # Parse the map value to PULL_IMAGE/RELEASE_IMAGE
        PULL_IMAGE=$(echo ${DS_MAP[$IMAGE]} | cut -d '|' -f 1)
        RELEASE_IMAGE=$(echo ${DS_MAP[$IMAGE]} | cut -d '|' -f 2)

        # Get full pull spec -- image + tag
        IMAGE_SPEC=$(grep "$IMAGE" "$NEW_FILE" | awk '{print $2}' |head -1)
        # Translate it to pull spec
        PULL_SPEC=$(echo $IMAGE_SPEC|sed -e "s|$IMAGE|$PULL_IMAGE|")

        # Get the image digest
        IMAGE_SHA=$(sudo "$OC_TOOL" image info $PULL_SPEC --filter-by-os=linux/amd64 | egrep '^Digest:' | awk '{print $2}')
        # Make sure we could get the IMAGE SHA, fail otherwise (we use set -ex)
        test -n "$IMAGE_SHA"
        # Translate the image to release image for release spec
        RELEASE_SPEC="${RELEASE_IMAGE}@${IMAGE_SHA}"
        echo "$IMAGE_SPEC -> $RELEASE_SPEC" | tee -a BUILDLOG

        # Update CSV
        sed -i "s|$IMAGE_SPEC|$RELEASE_SPEC|g" "$NEW_FILE"
done
