/*********************************************************************
 * Copyright (c) 2018 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
import { AxiosError, AxiosRequestConfig, AxiosResponse } from 'axios';
import { IResources, WorkspaceSettings, Preferences, User, KubernetesNamespace, IResourceQueryParams, IResourceCreateParams, FactoryResolver } from './resources';
import { che } from '@eclipse-che/api';
export declare enum METHOD {
    getAll = 0,
    getAllByNamespace = 1,
    fetchById = 2,
    create = 3,
    update = 4,
    delete = 5,
    start = 6,
    startTemporary = 7,
    stop = 8,
    getSettings = 9,
}
export interface IRequestConfig extends AxiosRequestConfig {
}
export interface IResponse<T> extends AxiosResponse<T> {
    data: T;
    status: number;
    statusText: string;
    headers: any;
    config: IRequestConfig;
    request?: any;
}
export interface IRequestError extends Error {
    status?: number;
    config: AxiosRequestConfig;
    request?: any;
    response?: IResponse<any>;
}
export declare class RequestError implements IRequestError {
    status: number | undefined;
    name: string;
    message: string;
    config: AxiosRequestConfig;
    request: any;
    response: AxiosResponse | undefined;
    constructor(error: AxiosError);
}
export interface IRemoteAPI {
    /**
     * Returns list of all user's workspaces.
     */
    getAll<T = che.workspace.Workspace>(): Promise<T[]>;
    /**
     * Returns list of workspaces in the namespace.
     *
     * @param namespace
     */
    getAllByNamespace<T = che.workspace.Workspace>(namespace: string): Promise<T[]>;
    /**
     * Returns a workspace by ID or key.
     *
     * @param workspaceKey workspace ID or `namespace/workspaceName`
     */
    getById<T = che.workspace.Workspace>(workspaceKey: string): Promise<T>;
    /**
     * Creates a new workspace from devfile.
     *
     * @param devfile workspace devfile.
     * @param params optional creating params.
     */
    create<T = che.workspace.Workspace>(devfile: che.workspace.devfile.Devfile, params?: IResourceCreateParams): Promise<T>;
    /**
     * Updates the workspace.
     *
     * @param workspaceId a workspace ID to update
     * @param workspace a new workspace data
     */
    update<T = che.workspace.Workspace>(workspaceId: string, workspace: che.workspace.Workspace): Promise<T>;
    /**
     * Deletes the workspace.
     *
     * @param workspaceId a workspace ID to delete
     */
    delete(workspaceId: string): Promise<void>;
    /**
     * Starts the workspace.
     *
     * @param workspaceId a workspace ID.
     * @param params resource query params.
     */
    start<T = che.workspace.Workspace>(workspaceId: string, params?: IResourceQueryParams): Promise<T>;
    /**
     * Stops the workspace.
     *
     * @param workspaceId a workspace ID.
     */
    stop(workspaceId: string): Promise<void>;
    /**
     * Returns settings.
     */
    getSettings<T = WorkspaceSettings>(): Promise<T>;
    /**
     * Returns a factory resolver.
     */
    getFactoryResolver<T = FactoryResolver>(url: string, overrideParams?: {
        [params: string]: string;
    }): Promise<T>;
    generateSshKey<T = che.ssh.SshPair>(service: string, name: string): Promise<T>;
    createSshKey(sshKeyPair: che.ssh.SshPair): Promise<void>;
    getSshKey<T = che.ssh.SshPair>(service: string, name: string): Promise<T>;
    getAllSshKey<T = che.ssh.SshPair>(service: string): Promise<T[]>;
    deleteSshKey(service: string, name: string): Promise<void>;
    /**
     * Return the current authenticated user
     */
    getCurrentUser(): Promise<User>;
    getCurrentUserProfile(): Promise<che.user.Profile>;
    getUserPreferences(): Promise<Preferences>;
    getUserPreferences(filter: string | undefined): Promise<Preferences>;
    updateUserPreferences(update: Preferences): Promise<Preferences>;
    replaceUserPreferences(preferences: Preferences): Promise<Preferences>;
    deleteUserPreferences(): Promise<void>;
    deleteUserPreferences(list: string[] | undefined): Promise<void>;
    /**
     * Return registered oauth token.
     *
     * @param oAuthProvider oauth provider's name e.g. github.
     */
    getOAuthToken(oAuthProvider: string): Promise<string>;
    /**
     * Return list of registered oAuth providers.
     */
    getOAuthProviders(): Promise<string[]>;
    /**
     * Updates workspace activity timestamp to prevent stop by timeout when workspace is running and using.
     *
     * @param workspaceId a workspace ID to update activity timestamp
     */
    updateActivity(workspaceId: string): Promise<void>;
    /**
     * Returns list of kubernetes namespace.
     */
    getKubernetesNamespace<T = KubernetesNamespace[]>(): Promise<T>;
    /**
     * Provision k8s namespace where user is able to create workspaces. Returns the kubernetes namespace.
     */
    provisionKubernetesNamespace(): Promise<KubernetesNamespace>;
    /**
     * Returns the devfile JSON Schema.
     *
     * @param version a devfile version.
     */
    getDevfileSchema<T = Object>(version?: string): Promise<T>;
    /**
     * Returns the che server api information
     */
    getApiInfo<T = Object>(): Promise<T>;
}
export declare class RemoteAPI implements IRemoteAPI {
    private promises;
    private remoteAPI;
    constructor(remoteApi: IResources);
    getAll<T = che.workspace.Workspace>(): Promise<T[]>;
    getAllByNamespace<T = che.workspace.Workspace>(namespace: string): Promise<T[]>;
    getById<T = che.workspace.Workspace>(workspaceKey: string): Promise<T>;
    create<T = che.workspace.Workspace>(devfile: che.workspace.devfile.Devfile, params?: IResourceCreateParams): Promise<T>;
    update<T = che.workspace.Workspace>(workspaceId: string, workspace: che.workspace.Workspace): Promise<T>;
    delete(workspaceId: string): Promise<void>;
    start<T = che.workspace.Workspace>(workspaceId: string, params?: IResourceQueryParams): Promise<T>;
    stop(workspaceId: string): Promise<void>;
    getSettings<T = WorkspaceSettings>(): Promise<T>;
    getFactoryResolver<T = FactoryResolver>(url: string, overrideParams?: {
        [params: string]: string;
    }): Promise<T>;
    generateSshKey<T = che.ssh.SshPair>(service: string, name: string): Promise<T>;
    createSshKey(sshKeyPair: any): Promise<void>;
    getSshKey<T = che.ssh.SshPair>(service: string, name: string): Promise<T>;
    getAllSshKey<T = che.ssh.SshPair>(service: string): Promise<T[]>;
    deleteSshKey(service: string, name: string): Promise<void>;
    getCurrentUser(): Promise<User>;
    getCurrentUserProfile(): Promise<che.user.Profile>;
    getUserPreferences(filter?: string | undefined): Promise<Preferences>;
    updateUserPreferences(update: Preferences): Promise<Preferences>;
    replaceUserPreferences(preferences: Preferences): Promise<Preferences>;
    deleteUserPreferences(list?: string[] | undefined): Promise<void>;
    getOAuthToken(oAuthProvider: string): Promise<string>;
    getOAuthProviders(): Promise<string[]>;
    getKubernetesNamespace<T = KubernetesNamespace[]>(): Promise<T>;
    provisionKubernetesNamespace(): Promise<KubernetesNamespace>;
    getDevfileSchema<T = Object>(version?: string): Promise<T>;
    getApiInfo<T = Object>(): Promise<T>;
    updateActivity(workspaceId: string): Promise<void>;
    /**
     * Returns a string key to identify the request.
     *
     * @param {METHOD} method a method name
     * @param {string} args
     * @returns {string}
     */
    protected buildKey(method: METHOD, ...args: string[]): string;
    /**
     * Returns stored request promise by a string key.
     *
     * @param {string} key a key to identify the request promise
     * @returns {Promise<any> | undefined}
     */
    protected getRequestPromise(key: string): Promise<any> | undefined;
    /**
     * Save the request promise.
     *
     * @param {string} key a key to identify the request promise.
     * @param {Promise<any>} promise a request promise
     */
    protected saveRequestPromise(key: string, promise: Promise<any>): void;
}
