import * as ESTree from '../estree';
import { Parser } from '../types';
import { Context } from '../utilities';
/**
 * Parses statement list items
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-StatementListItem)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseStatementListItem(parser: Parser, context: Context): ESTree.Statement;
/**
 * Parses statements
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-Statement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseStatement(parser: Parser, context: Context): ESTree.Statement;
/**
 * Parses empty statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-EmptyStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseEmptyStatement(parser: Parser, context: Context): ESTree.EmptyStatement;
/**
 * Parses the continue statement production
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-ContinueStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseContinueStatement(parser: Parser, context: Context): ESTree.ContinueStatement;
/**
 * Parses the break statement production
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-BreakStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseBreakStatement(parser: Parser, context: Context): ESTree.BreakStatement;
/**
 * Parses the if statement production
 *
 * @see [Link](https://tc39.github.io/ecma262/#sec-if-statement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseIfStatement(parser: Parser, context: Context): ESTree.IfStatement;
/**
 * Parses the debugger statement production
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-DebuggerStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseDebuggerStatement(parser: Parser, context: Context): ESTree.DebuggerStatement;
/**
 * Parses try statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-TryStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseTryStatement(parser: Parser, context: Context): ESTree.TryStatement;
/**
 * Parses catch block
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-Catch)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseCatchBlock(parser: Parser, context: Context): ESTree.CatchClause;
/**
 * Parses throw statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-ThrowStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseThrowStatement(parser: Parser, context: Context): ESTree.ThrowStatement;
/**
 * Parses expression statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-ExpressionStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseExpressionStatement(parser: Parser, context: Context): ESTree.ExpressionStatement;
/**
 * Parse directive node
 *
 * * @see [Link](https://tc39.github.io/ecma262/#sec-directive-prologues-and-the-use-strict-directive)
 *
 * @param parser Parser object
 * @param context Context masks
 */
export declare function parseDirective(parser: Parser, context: Context): ESTree.ExpressionStatement;
/**
 * Parses either expression or labelled statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-ExpressionStatement)
 * @see [Link](https://tc39.github.io/ecma262/#prod-LabelledStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseExpressionOrLabelledStatement(parser: Parser, context: Context): ESTree.ExpressionStatement | ESTree.LabeledStatement;
/**
 * Parses do while statement
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseDoWhileStatement(parser: Parser, context: Context): ESTree.DoWhileStatement;
/**
 * Parses while statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-grammar-notation-WhileStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseWhileStatement(parser: Parser, context: Context): ESTree.WhileStatement;
/**
 * Parses block statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-BlockStatement)
 * @see [Link](https://tc39.github.io/ecma262/#prod-Block)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseBlockStatement(parser: Parser, context: Context): ESTree.BlockStatement;
/**
 * Parses return statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-ReturnStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseReturnStatement(parser: Parser, context: Context): ESTree.ReturnStatement;
/**
 * Sets the necessary mutable parser flags. The parser flags will
 * be unset after done parsing out the statements.
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-grammar-notation-IterationStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseIterationStatement(parser: Parser, context: Context): ESTree.Statement;
/**
 * Parses with statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-WithStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseWithStatement(parser: Parser, context: Context): ESTree.WithStatement;
/**
 * Parses switch statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-SwitchStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseSwitchStatement(parser: Parser, context: Context): ESTree.SwitchStatement;
/**
 * Parses either default clause or case clauses
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-CaseClauses)
 * @see [Link](https://tc39.github.io/ecma262/#prod-DefaultClause)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseCaseOrDefaultClauses(parser: Parser, context: Context): ESTree.SwitchCase;
/**
 * Parses variable statement
 *
 * @see [Link](https://tc39.github.io/ecma262/#prod-VariableStatement)
 *
 * @param parser  Parser object
 * @param context Context masks
 */
export declare function parseVariableStatement(parser: Parser, context: Context, shouldConsume?: boolean): ESTree.VariableDeclaration;
