package gateway

import (
	"context"

	"github.com/che-incubator/devworkspace-che-operator/apis/che-controller/v1alpha1"
	"github.com/che-incubator/devworkspace-che-operator/pkg/defaults"
	"github.com/che-incubator/devworkspace-che-operator/pkg/sync"
	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	routev1 "github.com/openshift/api/route/v1"
	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/api/errors"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/util/intstr"
	"sigs.k8s.io/controller-runtime/pkg/client"
)

const (
	generatedHostAnnotation = "openshift.io/host.generated"
)

var (
	// used when the che manager spec defines the host explicitly
	explicitHostRouteDiffOpts = cmp.Options{
		cmpopts.IgnoreFields(routev1.Route{}, "TypeMeta", "ObjectMeta", "Status"),
		cmpopts.IgnoreFields(routev1.RouteSpec{}, "WildcardPolicy"),
		cmpopts.IgnoreFields(routev1.RouteTargetReference{}, "Weight"),
	}

	generatedHostRouteDiffOpts = cmp.Options{
		cmpopts.IgnoreFields(routev1.Route{}, "TypeMeta", "ObjectMeta", "Status"),
		cmpopts.IgnoreFields(routev1.RouteSpec{}, "WildcardPolicy", "Host"),
		cmpopts.IgnoreFields(routev1.RouteTargetReference{}, "Weight"),
	}
)

func (g *CheGateway) reconcileRoute(syncer sync.Syncer, ctx context.Context, mgr *v1alpha1.CheManager) (bool, string, error) {
	route := getRouteSpec(mgr)
	var changed bool
	var err error
	var routeHost string

	if mgr.Spec.GatewayDisabled {
		changed, routeHost, err = true, "", syncer.Delete(ctx, route)
	} else {
		// The trouble with routes is that they don't support updating the host. Therefore they need to be
		// recreated everytime. The problem with that is that we might not record the host in the CR
		// (which means we let openshift decide on it). Therefore, we should ignore host in comparisons.
		// But when we declare an explicit host, we should make sure it is honored, therefore we should NOT
		// ignore the host in comparisons.
		// Of course, the real trouble starts when the user switches from explicit host to implicit host.
		// Fortunately, OpenShift records the fact whether the host has been autogenerated or not in an annotation.
		// We can take advantage of that and figure out what to do in all cases.

		// first try to get the route and see if openshift has generated the hostname for it.
		// existing = generated, now = generated -> sync without host
		// existing = generated, now = explicit -> re-create the route
		// existing = explicit, now = generated -> re-create the route
		// existing = explicit, now = explicit -> sync with host

		expectGeneratedHost := mgr.Spec.GatewayHost == ""

		key := client.ObjectKey{Name: route.Name, Namespace: route.Namespace}
		existing := &routev1.Route{}
		if err := g.client.Get(ctx, key, existing); err != nil {
			if !errors.IsNotFound(err) {
				return false, "", err
			}
		}

		existingGeneratedHostValue := existing.Annotations[generatedHostAnnotation]
		var existingGeneratedHost bool

		if existingGeneratedHostValue == "" || existingGeneratedHostValue == "false" {
			existingGeneratedHost = false
		} else {
			existingGeneratedHost = true
		}

		if existingGeneratedHost != expectGeneratedHost {
			syncer.Delete(ctx, route)
		}

		var diffOpts cmp.Options
		if expectGeneratedHost {
			diffOpts = generatedHostRouteDiffOpts
		} else {
			diffOpts = explicitHostRouteDiffOpts
		}

		var inCluster runtime.Object

		// ok, another complication with routes is that they inline their TLS key and secret - they cannot
		// use an external secret as ingresses do. We use a secret name in our config to be compatible with
		// the ingresses, which means that we need to read the secret and inline its contents into the route.
		// We only finalize the route config here, because might not have gotten here the first couple of
		// reconciliation rounds and don't want to read the secret repeatedly unnecessarily.
		err = g.updateTlsConfig(ctx, mgr, route)
		if err != nil {
			return changed, "", err
		}

		changed, inCluster, err = syncer.Sync(ctx, mgr, route, diffOpts)
		if err != nil {
			return changed, "", err
		}
		routeHost = inCluster.(*routev1.Route).Spec.Host
	}

	return changed, routeHost, err
}

// getRouteSpec gets the basic spec of the route. It DOESN'T include the TLS configuration. You should use
// updateTlsConfig function for that.
func getRouteSpec(manager *v1alpha1.CheManager) *routev1.Route {
	return &routev1.Route{
		ObjectMeta: metav1.ObjectMeta{
			Name:      manager.Name,
			Namespace: manager.Namespace,
			Labels:    defaults.GetLabelsForComponent(manager, "external-access"),
		},
		Spec: routev1.RouteSpec{
			Host: manager.Spec.GatewayHost,
			To: routev1.RouteTargetReference{
				Kind: "Service",
				Name: GetGatewayServiceName(manager),
			},
			Port: &routev1.RoutePort{
				TargetPort: intstr.FromInt(GatewayPort),
			},
			TLS: &routev1.TLSConfig{
				InsecureEdgeTerminationPolicy: routev1.InsecureEdgeTerminationPolicyRedirect,
				Termination:                   routev1.TLSTerminationEdge,
			},
		},
	}
}

func (g *CheGateway) updateTlsConfig(ctx context.Context, manager *v1alpha1.CheManager, route *routev1.Route) error {
	if manager.Spec.TlsSecretName == "" {
		return nil
	}

	secret := &corev1.Secret{}
	err := g.client.Get(ctx, client.ObjectKey{Name: manager.Spec.TlsSecretName, Namespace: manager.Namespace}, secret)
	if err != nil {
		return err
	}

	route.Spec.TLS.Key = string(secret.Data["tls.key"])
	route.Spec.TLS.Certificate = string(secret.Data["tls.crt"])

	return nil
}
