package util

import (
	"fmt"

	"github.com/pkg/errors"
	"k8s.io/apimachinery/pkg/api/meta"
	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/client-go/discovery"
	"k8s.io/client-go/kubernetes/scheme"
	"k8s.io/client-go/rest"
	"k8s.io/client-go/restmapper"
	"k8s.io/klog"
)

const (
	MetadataField = "metadata"
	LabelsField   = "labels"
)

func BuildRestMapper(restConfig *rest.Config) (meta.RESTMapper, error) {
	discoveryClient, err := discovery.NewDiscoveryClientForConfig(restConfig)
	if err != nil {
		return nil, fmt.Errorf("error creating discovery client: %v", err)
	}

	groupResources, err := restmapper.GetAPIGroupResources(discoveryClient)
	if err != nil {
		return nil, fmt.Errorf("error retrieving API group resources: %v", err)
	}

	return restmapper.NewDiscoveryRESTMapper(groupResources), nil
}

func ToUnstructuredResource(from runtime.Object, restMapper meta.RESTMapper) (*unstructured.Unstructured, *schema.GroupVersionResource,
	error) {
	to, err := ToUnstructured(from)
	if err != nil {
		return nil, nil, err
	}

	gvr, err := FindGroupVersionResource(to, restMapper)
	if err != nil {
		return nil, nil, err
	}

	return to, gvr, nil
}

func ToUnstructured(from runtime.Object) (*unstructured.Unstructured, error) {
	switch f := from.(type) {
	case *unstructured.Unstructured:
		return f.DeepCopy(), nil
	default:
		to := &unstructured.Unstructured{}
		err := scheme.Scheme.Convert(from, to, nil)
		if err != nil {
			return nil, errors.WithMessagef(err, "error converting %#v to unstructured.Unstructured", from)
		}

		return to, nil
	}
}

func FindGroupVersionResource(from *unstructured.Unstructured, restMapper meta.RESTMapper) (*schema.GroupVersionResource, error) {
	gvk := from.GroupVersionKind()
	mapping, err := restMapper.RESTMapping(gvk.GroupKind(), gvk.Version)
	if err != nil {
		return nil, errors.WithMessagef(err, "error getting REST mapper for %#v", gvk)
	}

	return &mapping.Resource, nil
}

func GetMetadata(from *unstructured.Unstructured) map[string]interface{} {
	value, _, _ := unstructured.NestedFieldNoCopy(from.Object, MetadataField)
	if value != nil {
		return value.(map[string]interface{})
	}

	return map[string]interface{}{}
}

func GetSpec(obj *unstructured.Unstructured) interface{} {
	return GetNestedField(obj, "spec")
}

func GetNestedField(obj *unstructured.Unstructured, fields ...string) interface{} {
	nested, _, err := unstructured.NestedFieldNoCopy(obj.Object, fields...)
	if err != nil {
		klog.Errorf("Error retrieving %v field for %#v: %v", fields, obj, err)
	}

	return nested
}
