import { __spreadArray, __read, __values, __assign, __awaiter, __generator } from 'tslib';
import DataLoader from 'dataloader';
import { Kind, visit } from 'graphql';
import { relocatedError, memoize2of4 } from '@graphql-tools/utils/es5';

// adapted from https://github.com/gatsbyjs/gatsby/blob/master/packages/gatsby-source-graphql/src/batching/merge-queries.js
function createPrefix(index) {
    return "graphqlTools" + index + "_";
}
function parseKey(prefixedKey) {
    var match = /^graphqlTools([\d]+)_(.*)$/.exec(prefixedKey);
    if (match && match.length === 3 && !isNaN(Number(match[1])) && match[2]) {
        return { index: Number(match[1]), originalKey: match[2] };
    }
    throw new Error("Key " + prefixedKey + " is not correctly prefixed");
}

// adapted from https://github.com/gatsbyjs/gatsby/blob/master/packages/gatsby-source-graphql/src/batching/merge-queries.js
/**
 * Merge multiple queries into a single query in such a way that query results
 * can be split and transformed as if they were obtained by running original queries.
 *
 * Merging algorithm involves several transformations:
 *  1. Replace top-level fragment spreads with inline fragments (... on Query {})
 *  2. Add unique aliases to all top-level query fields (including those on inline fragments)
 *  3. Prefix all variable definitions and variable usages
 *  4. Prefix names (and spreads) of fragments
 *
 * i.e transform:
 *   [
 *     `query Foo($id: ID!) { foo, bar(id: $id), ...FooQuery }
 *     fragment FooQuery on Query { baz }`,
 *
 *    `query Bar($id: ID!) { foo: baz, bar(id: $id), ... on Query { baz } }`
 *   ]
 * to:
 *   query (
 *     $graphqlTools1_id: ID!
 *     $graphqlTools2_id: ID!
 *   ) {
 *     graphqlTools1_foo: foo,
 *     graphqlTools1_bar: bar(id: $graphqlTools1_id)
 *     ... on Query {
 *       graphqlTools1__baz: baz
 *     }
 *     graphqlTools1__foo: baz
 *     graphqlTools1__bar: bar(id: $graphqlTools1__id)
 *     ... on Query {
 *       graphqlTools1__baz: baz
 *     }
 *   }
 */
function mergeRequests(requests, extensionsReducer) {
    var e_1, _a;
    var mergedVariables = Object.create(null);
    var mergedVariableDefinitions = [];
    var mergedSelections = [];
    var mergedFragmentDefinitions = [];
    var mergedExtensions = Object.create(null);
    for (var index in requests) {
        var request = requests[index];
        var prefixedRequests = prefixRequest(createPrefix(index), request);
        try {
            for (var _b = (e_1 = void 0, __values(prefixedRequests.document.definitions)), _c = _b.next(); !_c.done; _c = _b.next()) {
                var def = _c.value;
                if (isOperationDefinition(def)) {
                    mergedSelections.push.apply(mergedSelections, __spreadArray([], __read(def.selectionSet.selections), false));
                    if (def.variableDefinitions) {
                        mergedVariableDefinitions.push.apply(mergedVariableDefinitions, __spreadArray([], __read(def.variableDefinitions), false));
                    }
                }
                if (isFragmentDefinition(def)) {
                    mergedFragmentDefinitions.push(def);
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_1) throw e_1.error; }
        }
        Object.assign(mergedVariables, prefixedRequests.variables);
        mergedExtensions = extensionsReducer(mergedExtensions, request);
    }
    var mergedOperationDefinition = {
        kind: Kind.OPERATION_DEFINITION,
        operation: requests[0].operationType,
        variableDefinitions: mergedVariableDefinitions,
        selectionSet: {
            kind: Kind.SELECTION_SET,
            selections: mergedSelections,
        },
    };
    return {
        document: {
            kind: Kind.DOCUMENT,
            definitions: __spreadArray([mergedOperationDefinition], __read(mergedFragmentDefinitions), false),
        },
        variables: mergedVariables,
        extensions: mergedExtensions,
        context: requests[0].context,
        info: requests[0].info,
        operationType: requests[0].operationType,
    };
}
function prefixRequest(prefix, request) {
    var _a, e_2, _b;
    var _c;
    var executionVariables = (_c = request.variables) !== null && _c !== void 0 ? _c : {};
    function prefixNode(node) {
        return prefixNodeName(node, prefix);
    }
    var prefixedDocument = aliasTopLevelFields(prefix, request.document);
    var executionVariableNames = Object.keys(executionVariables);
    if (executionVariableNames.length > 0) {
        prefixedDocument = visit(prefixedDocument, (_a = {},
            _a[Kind.VARIABLE] = prefixNode,
            _a[Kind.FRAGMENT_DEFINITION] = prefixNode,
            _a[Kind.FRAGMENT_SPREAD] = prefixNode,
            _a));
    }
    var prefixedVariables = {};
    try {
        for (var executionVariableNames_1 = __values(executionVariableNames), executionVariableNames_1_1 = executionVariableNames_1.next(); !executionVariableNames_1_1.done; executionVariableNames_1_1 = executionVariableNames_1.next()) {
            var variableName = executionVariableNames_1_1.value;
            prefixedVariables[prefix + variableName] = executionVariables[variableName];
        }
    }
    catch (e_2_1) { e_2 = { error: e_2_1 }; }
    finally {
        try {
            if (executionVariableNames_1_1 && !executionVariableNames_1_1.done && (_b = executionVariableNames_1.return)) _b.call(executionVariableNames_1);
        }
        finally { if (e_2) throw e_2.error; }
    }
    return {
        document: prefixedDocument,
        variables: prefixedVariables,
        operationType: request.operationType,
    };
}
/**
 * Adds prefixed aliases to top-level fields of the query.
 *
 * @see aliasFieldsInSelection for implementation details
 */
function aliasTopLevelFields(prefix, document) {
    var _a, _b;
    var transformer = (_a = {},
        _a[Kind.OPERATION_DEFINITION] = function (def) {
            var selections = def.selectionSet.selections;
            return __assign(__assign({}, def), { selectionSet: __assign(__assign({}, def.selectionSet), { selections: aliasFieldsInSelection(prefix, selections, document) }) });
        },
        _a);
    return visit(document, transformer, (_b = {},
        _b[Kind.DOCUMENT] = ["definitions"],
        _b));
}
/**
 * Add aliases to fields of the selection, including top-level fields of inline fragments.
 * Fragment spreads are converted to inline fragments and their top-level fields are also aliased.
 *
 * Note that this method is shallow. It adds aliases only to the top-level fields and doesn't
 * descend to field sub-selections.
 *
 * For example, transforms:
 *   {
 *     foo
 *     ... on Query { foo }
 *     ...FragmentWithBarField
 *   }
 * To:
 *   {
 *     graphqlTools1_foo: foo
 *     ... on Query { graphqlTools1_foo: foo }
 *     ... on Query { graphqlTools1_bar: bar }
 *   }
 */
function aliasFieldsInSelection(prefix, selections, document) {
    return selections.map(function (selection) {
        switch (selection.kind) {
            case Kind.INLINE_FRAGMENT:
                return aliasFieldsInInlineFragment(prefix, selection, document);
            case Kind.FRAGMENT_SPREAD: {
                var inlineFragment = inlineFragmentSpread(selection, document);
                return aliasFieldsInInlineFragment(prefix, inlineFragment, document);
            }
            case Kind.FIELD:
            default:
                return aliasField(selection, prefix);
        }
    });
}
/**
 * Add aliases to top-level fields of the inline fragment.
 * Returns new inline fragment node.
 *
 * For Example, transforms:
 *   ... on Query { foo, ... on Query { bar: foo } }
 * To
 *   ... on Query { graphqlTools1_foo: foo, ... on Query { graphqlTools1_bar: foo } }
 */
function aliasFieldsInInlineFragment(prefix, fragment, document) {
    var selections = fragment.selectionSet.selections;
    return __assign(__assign({}, fragment), { selectionSet: __assign(__assign({}, fragment.selectionSet), { selections: aliasFieldsInSelection(prefix, selections, document) }) });
}
/**
 * Replaces fragment spread with inline fragment
 *
 * Example:
 *   query { ...Spread }
 *   fragment Spread on Query { bar }
 *
 * Transforms to:
 *   query { ... on Query { bar } }
 */
function inlineFragmentSpread(spread, document) {
    var fragment = document.definitions.find(function (def) { return isFragmentDefinition(def) && def.name.value === spread.name.value; });
    if (!fragment) {
        throw new Error("Fragment " + spread.name.value + " does not exist");
    }
    var typeCondition = fragment.typeCondition, selectionSet = fragment.selectionSet;
    return {
        kind: Kind.INLINE_FRAGMENT,
        typeCondition: typeCondition,
        selectionSet: selectionSet,
        directives: spread.directives,
    };
}
function prefixNodeName(namedNode, prefix) {
    return __assign(__assign({}, namedNode), { name: __assign(__assign({}, namedNode.name), { value: prefix + namedNode.name.value }) });
}
/**
 * Returns a new FieldNode with prefixed alias
 *
 * Example. Given prefix === "graphqlTools1_" transforms:
 *   { foo } -> { graphqlTools1_foo: foo }
 *   { foo: bar } -> { graphqlTools1_foo: bar }
 */
function aliasField(field, aliasPrefix) {
    var aliasNode = field.alias ? field.alias : field.name;
    return __assign(__assign({}, field), { alias: __assign(__assign({}, aliasNode), { value: aliasPrefix + aliasNode.value }) });
}
function isOperationDefinition(def) {
    return def.kind === Kind.OPERATION_DEFINITION;
}
function isFragmentDefinition(def) {
    return def.kind === Kind.FRAGMENT_DEFINITION;
}

// adapted from https://github.com/gatsbyjs/gatsby/blob/master/packages/gatsby-source-graphql/src/batching/merge-queries.js
/**
 * Split and transform result of the query produced by the `merge` function
 */
function splitResult(_a, numResults) {
    var _b, e_1, _c;
    var data = _a.data, errors = _a.errors;
    var splitResults = [];
    for (var i = 0; i < numResults; i++) {
        splitResults.push({});
    }
    if (data) {
        for (var prefixedKey in data) {
            var _d = parseKey(prefixedKey), index = _d.index, originalKey = _d.originalKey;
            var result = splitResults[index];
            if (result == null) {
                continue;
            }
            if (result.data == null) {
                result.data = (_b = {}, _b[originalKey] = data[prefixedKey], _b);
            }
            else {
                result.data[originalKey] = data[prefixedKey];
            }
        }
    }
    if (errors) {
        try {
            for (var errors_1 = __values(errors), errors_1_1 = errors_1.next(); !errors_1_1.done; errors_1_1 = errors_1.next()) {
                var error = errors_1_1.value;
                if (error.path) {
                    var parsedKey = parseKey(error.path[0]);
                    var index = parsedKey.index, originalKey = parsedKey.originalKey;
                    var newError = relocatedError(error, __spreadArray([originalKey], __read(error.path.slice(1)), false));
                    var errors_2 = (splitResults[index].errors = (splitResults[index].errors || []));
                    errors_2.push(newError);
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (errors_1_1 && !errors_1_1.done && (_c = errors_1.return)) _c.call(errors_1);
            }
            finally { if (e_1) throw e_1.error; }
        }
    }
    return splitResults;
}

function createBatchingExecutor(executor, dataLoaderOptions, extensionsReducer) {
    if (extensionsReducer === void 0) { extensionsReducer = defaultExtensionsReducer; }
    var loadFn = createLoadFn(executor, extensionsReducer);
    var loader = new DataLoader(loadFn, dataLoaderOptions);
    return function batchingExecutor(request) {
        return request.operationType === 'subscription' ? executor(request) : loader.load(request);
    };
}
function createLoadFn(executor, extensionsReducer) {
    return function batchExecuteLoadFn(requests) {
        return __awaiter(this, void 0, void 0, function () {
            var execBatches, index, request, currentBatch, operationType, currentOperationType, results;
            var _this = this;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        execBatches = [];
                        index = 0;
                        request = requests[index];
                        currentBatch = [request];
                        execBatches.push(currentBatch);
                        operationType = request.operationType;
                        while (++index < requests.length) {
                            currentOperationType = requests[index].operationType;
                            if (operationType == null) {
                                throw new Error('Could not identify operation type of document.');
                            }
                            if (operationType === currentOperationType) {
                                currentBatch.push(requests[index]);
                            }
                            else {
                                currentBatch = [requests[index]];
                                execBatches.push(currentBatch);
                            }
                        }
                        return [4 /*yield*/, Promise.all(execBatches.map(function (execBatch) { return __awaiter(_this, void 0, void 0, function () {
                                var mergedRequests, resultBatches;
                                return __generator(this, function (_a) {
                                    switch (_a.label) {
                                        case 0:
                                            mergedRequests = mergeRequests(execBatch, extensionsReducer);
                                            return [4 /*yield*/, executor(mergedRequests)];
                                        case 1:
                                            resultBatches = (_a.sent());
                                            return [2 /*return*/, splitResult(resultBatches, execBatch.length)];
                                    }
                                });
                            }); }))];
                    case 1:
                        results = _a.sent();
                        return [2 /*return*/, results.flat()];
                }
            });
        });
    };
}
function defaultExtensionsReducer(mergedExtensions, request) {
    var newExtensions = request.extensions;
    if (newExtensions != null) {
        Object.assign(mergedExtensions, newExtensions);
    }
    return mergedExtensions;
}

var getBatchingExecutor = memoize2of4(function getBatchingExecutor(_context, executor, dataLoaderOptions, extensionsReducer) {
    return createBatchingExecutor(executor, dataLoaderOptions, extensionsReducer);
});

export { createBatchingExecutor, getBatchingExecutor };
