import { IntrospectionOptions, DocumentNode } from 'graphql';
import { SchemaPointerSingle, Source, DocumentLoader, SingleFileOptions } from '@graphql-tools/utils';
import { AsyncExecutor, Subscriber, SyncExecutor } from '@graphql-tools/delegate';
import WebSocket from 'isomorphic-ws';
import 'isomorphic-form-data';
export declare type AsyncFetchFn = typeof import('cross-fetch').fetch;
export declare type SyncFetchFn = (input: RequestInfo, init?: RequestInit) => SyncResponse;
export declare type SyncResponse = Omit<Response, 'json' | 'text'> & {
    json: () => any;
    text: () => string;
};
export declare type FetchFn = AsyncFetchFn | SyncFetchFn;
declare type Headers = Record<string, string> | Array<Record<string, string>>;
declare type BuildExecutorOptions<TFetchFn = FetchFn> = {
    pointer: string;
    fetch: TFetchFn;
    extraHeaders: any;
    defaultMethod: 'GET' | 'POST';
    useGETForQueries: boolean;
    multipart?: boolean;
};
/**
 * Additional options for loading from a URL
 */
export interface LoadFromUrlOptions extends SingleFileOptions, Partial<IntrospectionOptions> {
    /**
     * Additional headers to include when querying the original schema
     */
    headers?: Headers;
    /**
     * A custom `fetch` implementation to use when querying the original schema.
     * Defaults to `cross-fetch`
     */
    customFetch?: FetchFn | string;
    /**
     * HTTP method to use when querying the original schema.
     */
    method?: 'GET' | 'POST';
    /**
     * Whether to enable subscriptions on the loaded schema
     */
    enableSubscriptions?: boolean;
    /**
     * Custom WebSocket implementation used by the loaded schema if subscriptions
     * are enabled
     */
    webSocketImpl?: typeof WebSocket | string;
    /**
     * Whether to use the GET HTTP method for queries when querying the original schema
     */
    useGETForQueries?: boolean;
    /**
     * Use multipart for POST requests
     */
    multipart?: boolean;
}
/**
 * This loader loads a schema from a URL. The loaded schema is a fully-executable,
 * remote schema since it's created using [@graphql-tools/wrap](/docs/remote-schemas).
 *
 * ```
 * const schema = await loadSchema('http://localhost:3000/graphql', {
 *   loaders: [
 *     new UrlLoader(),
 *   ]
 * });
 * ```
 */
export declare class UrlLoader implements DocumentLoader<LoadFromUrlOptions> {
    loaderId(): string;
    canLoad(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<boolean>;
    canLoadSync(pointer: SchemaPointerSingle, _options: LoadFromUrlOptions): boolean;
    createFormDataFromVariables<TVariables>({ query, variables }: {
        query: string;
        variables: TVariables;
    }): Promise<FormData>;
    buildExecutor(options: BuildExecutorOptions<SyncFetchFn>): SyncExecutor;
    buildExecutor(options: BuildExecutorOptions<AsyncFetchFn>): AsyncExecutor;
    buildSubscriber(pointer: string, webSocketImpl: typeof WebSocket): Subscriber;
    getFetch(customFetch: LoadFromUrlOptions['customFetch'], importFn: (moduleName: string) => PromiseLike<unknown>, async: true): PromiseLike<AsyncFetchFn>;
    getFetch(customFetch: LoadFromUrlOptions['customFetch'], importFn: (moduleName: string) => unknown, async: false): SyncFetchFn;
    private getHeadersFromOptions;
    private getDefaultMethodFromOptions;
    getWebSocketImpl(options: LoadFromUrlOptions, importFn: (moduleName: string) => PromiseLike<unknown>): PromiseLike<typeof WebSocket>;
    getWebSocketImpl(options: LoadFromUrlOptions, importFn: (moduleName: string) => unknown): typeof WebSocket;
    getExecutorAndSubscriberAsync(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<{
        executor: AsyncExecutor;
        subscriber: Subscriber;
    }>;
    getExecutorAndSubscriberSync(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): {
        executor: SyncExecutor;
        subscriber: Subscriber;
    };
    getSubschemaConfigAsync(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<{
        schema: import("graphql").GraphQLSchema;
        executor: AsyncExecutor;
        subscriber: Subscriber;
    }>;
    getSubschemaConfigSync(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): {
        schema: import("graphql").GraphQLSchema;
        executor: SyncExecutor;
        subscriber: Subscriber;
    };
    handleSDLAsync(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<{
        document: DocumentNode;
        schema: import("graphql").GraphQLSchema;
    }>;
    handleSDLSync(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): {
        document: DocumentNode;
        schema: import("graphql").GraphQLSchema;
    };
    load(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<Source>;
    loadSync(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Source;
}
export {};
