/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
// flowlint ambiguous-object-type:error
'use strict';

var RelayConcreteNode = require('../util/RelayConcreteNode');

var RelayModernRecord = require('./RelayModernRecord');

var RelayStoreUtils = require('./RelayStoreUtils');

var cloneRelayHandleSourceField = require('./cloneRelayHandleSourceField');

var invariant = require("fbjs/lib/invariant");

var CONDITION = RelayConcreteNode.CONDITION,
    CLIENT_EXTENSION = RelayConcreteNode.CLIENT_EXTENSION,
    DEFER = RelayConcreteNode.DEFER,
    FRAGMENT_SPREAD = RelayConcreteNode.FRAGMENT_SPREAD,
    INLINE_FRAGMENT = RelayConcreteNode.INLINE_FRAGMENT,
    LINKED_FIELD = RelayConcreteNode.LINKED_FIELD,
    MODULE_IMPORT = RelayConcreteNode.MODULE_IMPORT,
    LINKED_HANDLE = RelayConcreteNode.LINKED_HANDLE,
    SCALAR_FIELD = RelayConcreteNode.SCALAR_FIELD,
    SCALAR_HANDLE = RelayConcreteNode.SCALAR_HANDLE,
    STREAM = RelayConcreteNode.STREAM;
var getStorageKey = RelayStoreUtils.getStorageKey,
    getModuleOperationKey = RelayStoreUtils.getModuleOperationKey;

function mark(recordSource, selector, references, operationLoader) {
  var dataID = selector.dataID,
      node = selector.node,
      variables = selector.variables;
  var marker = new RelayReferenceMarker(recordSource, variables, references, operationLoader);
  marker.mark(node, dataID);
}
/**
 * @private
 */


var RelayReferenceMarker = /*#__PURE__*/function () {
  function RelayReferenceMarker(recordSource, variables, references, operationLoader) {
    this._operationLoader = operationLoader !== null && operationLoader !== void 0 ? operationLoader : null;
    this._recordSource = recordSource;
    this._references = references;
    this._variables = variables;
  }

  var _proto = RelayReferenceMarker.prototype;

  _proto.mark = function mark(node, dataID) {
    this._traverse(node, dataID);
  };

  _proto._traverse = function _traverse(node, dataID) {
    this._references.add(dataID);

    var record = this._recordSource.get(dataID);

    if (record == null) {
      return;
    }

    this._traverseSelections(node.selections, record);
  };

  _proto._getVariableValue = function _getVariableValue(name) {
    !this._variables.hasOwnProperty(name) ? process.env.NODE_ENV !== "production" ? invariant(false, 'RelayReferenceMarker(): Undefined variable `%s`.', name) : invariant(false) : void 0;
    return this._variables[name];
  };

  _proto._traverseSelections = function _traverseSelections(selections, record) {
    var _this = this;

    selections.forEach(function (selection) {
      /* eslint-disable no-fallthrough */
      switch (selection.kind) {
        case LINKED_FIELD:
          if (selection.plural) {
            _this._traversePluralLink(selection, record);
          } else {
            _this._traverseLink(selection, record);
          }

          break;

        case CONDITION:
          var conditionValue = _this._getVariableValue(selection.condition);

          if (conditionValue === selection.passingValue) {
            _this._traverseSelections(selection.selections, record);
          }

          break;

        case INLINE_FRAGMENT:
          var typeName = RelayModernRecord.getType(record);

          if (typeName != null && typeName === selection.type) {
            _this._traverseSelections(selection.selections, record);
          }

          break;

        case FRAGMENT_SPREAD:
          !false ? process.env.NODE_ENV !== "production" ? invariant(false, 'RelayReferenceMarker(): Unexpected fragment spread `...%s`, ' + 'expected all fragments to be inlined.', selection.name) : invariant(false) : void 0;

        case LINKED_HANDLE:
          // The selections for a "handle" field are the same as those of the
          // original linked field where the handle was applied. Reference marking
          // therefore requires traversing the original field selections against
          // the synthesized client field.
          //
          // TODO: Instead of finding the source field in `selections`, change
          // the concrete structure to allow shared subtrees, and have the linked
          // handle directly refer to the same selections as the LinkedField that
          // it was split from.
          var handleField = cloneRelayHandleSourceField(selection, selections, _this._variables);

          if (handleField.plural) {
            _this._traversePluralLink(handleField, record);
          } else {
            _this._traverseLink(handleField, record);
          }

          break;

        case DEFER:
        case STREAM:
          _this._traverseSelections(selection.selections, record);

          break;

        case SCALAR_FIELD:
        case SCALAR_HANDLE:
          break;

        case MODULE_IMPORT:
          _this._traverseModuleImport(selection, record);

          break;

        case CLIENT_EXTENSION:
          _this._traverseSelections(selection.selections, record);

          break;

        default:
          selection;
          !false ? process.env.NODE_ENV !== "production" ? invariant(false, 'RelayReferenceMarker: Unknown AST node `%s`.', selection) : invariant(false) : void 0;
      }
    });
  };

  _proto._traverseModuleImport = function _traverseModuleImport(moduleImport, record) {
    var operationLoader = this._operationLoader;
    !(operationLoader !== null) ? process.env.NODE_ENV !== "production" ? invariant(false, 'RelayReferenceMarker: Expected an operationLoader to be configured when using `@module`.') : invariant(false) : void 0;
    var operationKey = getModuleOperationKey(moduleImport.documentName);
    var operationReference = RelayModernRecord.getValue(record, operationKey);

    if (operationReference == null) {
      return;
    }

    var operation = operationLoader.get(operationReference);

    if (operation != null) {
      this._traverseSelections(operation.selections, record);
    } // Otherwise, if the operation is not available, we assume that the data
    // cannot have been processed yet and therefore isn't in the store to
    // begin with.

  };

  _proto._traverseLink = function _traverseLink(field, record) {
    var storageKey = getStorageKey(field, this._variables);
    var linkedID = RelayModernRecord.getLinkedRecordID(record, storageKey);

    if (linkedID == null) {
      return;
    }

    this._traverse(field, linkedID);
  };

  _proto._traversePluralLink = function _traversePluralLink(field, record) {
    var _this2 = this;

    var storageKey = getStorageKey(field, this._variables);
    var linkedIDs = RelayModernRecord.getLinkedRecordIDs(record, storageKey);

    if (linkedIDs == null) {
      return;
    }

    linkedIDs.forEach(function (linkedID) {
      if (linkedID != null) {
        _this2._traverse(field, linkedID);
      }
    });
  };

  return RelayReferenceMarker;
}();

module.exports = {
  mark: mark
};