/* Prototypes and definition for malloc implementation.
   Copyright (C) 1996-2015 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

#ifndef _MALLOC_H
#define _MALLOC_H 1

#include <features.h>
#include <stddef.h>
#include <stdio.h>

#ifdef _LIBC
# define __MALLOC_HOOK_VOLATILE
# define __MALLOC_DEPRECATED
#else
# define __MALLOC_HOOK_VOLATILE volatile
# define __MALLOC_DEPRECATED __attribute_deprecated__
#endif

__BEGIN_DECLS
/* Allocate SIZE bytes of memory.  */
extern void *malloc(size_t __size)
__THROW __attribute_malloc__ __wur;

/* Allocate NMEMB elements of SIZE bytes each, all initialized to 0.  */
extern void *calloc(size_t __nmemb, size_t __size)
__THROW __attribute_malloc__ __wur;

/* Re-allocate the previously allocated block in __ptr, making the new
   block SIZE bytes long.  */
/* __attribute_malloc__ is not used, because if realloc returns
   the same pointer that was passed to it, aliasing needs to be allowed
   between objects pointed by the old and new pointers.  */
extern void *realloc(void *__ptr, size_t __size)
__THROW __attribute_warn_unused_result__;

/* Free a block allocated by `malloc', `realloc' or `calloc'.  */
extern void free(void *__ptr) __THROW;

/* Free a block allocated by `calloc'. */
extern void cfree(void *__ptr) __THROW;

/* Allocate SIZE bytes allocated to ALIGNMENT bytes.  */
extern void *memalign(size_t __alignment, size_t __size)
__THROW __attribute_malloc__ __wur;

/* Allocate SIZE bytes on a page boundary.  */
extern void *valloc(size_t __size)
__THROW __attribute_malloc__ __wur;

/* Equivalent to valloc(minimum-page-that-holds(n)), that is, round up
   __size to nearest pagesize. */
extern void *pvalloc(size_t __size)
__THROW __attribute_malloc__ __wur;

/* Underlying allocation function; successive calls should return
   contiguous pieces of memory.  */
extern void *(*__morecore) (ptrdiff_t __size);

/* Default value of `__morecore'.  */
extern void *__default_morecore(ptrdiff_t __size)
__THROW __attribute_malloc__;

/* SVID2/XPG mallinfo structure */

struct mallinfo {
	int arena;		/* non-mmapped space allocated from system */
	int ordblks;		/* number of free chunks */
	int smblks;		/* number of fastbin blocks */
	int hblks;		/* number of mmapped regions */
	int hblkhd;		/* space in mmapped regions */
	int usmblks;		/* maximum total allocated space */
	int fsmblks;		/* space available in freed fastbin blocks */
	int uordblks;		/* total allocated space */
	int fordblks;		/* total free space */
	int keepcost;		/* top-most, releasable (via malloc_trim) space */
};

/* Returns a copy of the updated current mallinfo. */
extern struct mallinfo mallinfo(void) __THROW;

/* SVID2/XPG mallopt options */
#ifndef M_MXFAST
# define M_MXFAST  1		/* maximum request size for "fastbins" */
#endif
#ifndef M_NLBLKS
# define M_NLBLKS  2		/* UNUSED in this malloc */
#endif
#ifndef M_GRAIN
# define M_GRAIN   3		/* UNUSED in this malloc */
#endif
#ifndef M_KEEP
# define M_KEEP    4		/* UNUSED in this malloc */
#endif

/* mallopt options that actually do something */
#define M_TRIM_THRESHOLD    -1
#define M_TOP_PAD           -2
#define M_MMAP_THRESHOLD    -3
#define M_MMAP_MAX          -4
#define M_CHECK_ACTION      -5
#define M_PERTURB           -6
#define M_ARENA_TEST        -7
#define M_ARENA_MAX         -8

/* General SVID/XPG interface to tunable parameters. */
extern int mallopt(int __param, int __val) __THROW;

/* Release all but __pad bytes of freed top-most memory back to the
   system. Return 1 if successful, else 0. */
extern int malloc_trim(size_t __pad) __THROW;

/* Report the number of usable allocated bytes associated with allocated
   chunk __ptr. */
extern size_t malloc_usable_size(void *__ptr) __THROW;

/* Prints brief summary statistics on stderr. */
extern void malloc_stats(void) __THROW;

/* Output information about state of allocator to stream FP.  */
extern int malloc_info(int __options, FILE * __fp) __THROW;

/* Record the state of all malloc variables in an opaque data structure. */
extern void *malloc_get_state(void) __THROW;

/* Restore the state of all malloc variables from data obtained with
   malloc_get_state(). */
extern int malloc_set_state(void *__ptr) __THROW;

/* Called once when malloc is initialized; redefining this variable in
   the application provides the preferred way to set up the hook
   pointers. */
extern void (*__MALLOC_HOOK_VOLATILE __malloc_initialize_hook) (void)__MALLOC_DEPRECATED;
/* Hooks for debugging and user-defined versions. */
extern void (*__MALLOC_HOOK_VOLATILE __free_hook) (void *__ptr, const void *)__MALLOC_DEPRECATED;
extern void *(*__MALLOC_HOOK_VOLATILE __malloc_hook) (size_t __size, const void *)__MALLOC_DEPRECATED;
extern void *(*__MALLOC_HOOK_VOLATILE __realloc_hook) (void *__ptr, size_t __size, const void *)__MALLOC_DEPRECATED;
extern void *(*__MALLOC_HOOK_VOLATILE __memalign_hook) (size_t __alignment, size_t __size, const void *)__MALLOC_DEPRECATED;
extern void (*__MALLOC_HOOK_VOLATILE __after_morecore_hook) (void);

/* Activate a standard set of debugging hooks. */
extern void __malloc_check_init(void)
__THROW __MALLOC_DEPRECATED;

__END_DECLS
#endif				/* malloc.h */
