// +build ignore

// gen downloads an updated version of the PSL list and compiles it into go code.
//
// It is meant to be used by maintainers in conjunction with the go generate tool
// to update the list.
package main

import (
	"bytes"
	"context"
	"fmt"
	"go/format"
	"io/ioutil"
	"log"
	"net/http"
	"strings"
	"text/template"
	"time"

	"github.com/google/go-github/github"
	"github.com/weppos/publicsuffix-go/publicsuffix"
)

const (
	// where the rules will be written
	filename = "rules.go"

	list = `// This file is automatically generated
// Run "go run cmd/gen/gen.go" to update the list.

package publicsuffix

const defaultListVersion = "PSL version {{.VersionSHA}} ({{.VersionDate}})"

func DefaultRules() [{{len .Rules}}]Rule {
	return r
}

var r = [{{len .Rules}}]Rule{
	{{range $r := .Rules}} \
	{ {{$r.Type}}, "{{$r.Value}}", {{$r.Length}}, {{$r.Private}} },
	{{end}}
}

func init() {
	for i := range r {
		DefaultList.AddRule(&r[i])
	}
}

`
)

var (
	listTmpl = template.Must(template.New("list").Parse(cont(list)))
)

// https://github.com/golang/go/issues/9969
// Requires go1.6
func cont(s string) string {
	return strings.Replace(s, "\\\n", "", -1)
}

func main() {
	log.Printf("Fetching PSL version...\n")
	sha, datetime := extractHeadInfo()

	log.Printf("Downloading PSL %s...\n", sha[:6])
	resp, err := http.Get(fmt.Sprintf("https://raw.githubusercontent.com/publicsuffix/list/%s/public_suffix_list.dat", sha))
	if err != nil {
		log.Fatal(err)
	}
	defer resp.Body.Close()

	list := publicsuffix.NewList()
	rules, err := list.Load(resp.Body, nil)
	if err != nil {
		log.Fatal(err)
	}

	data := struct {
		VersionSHA  string
		VersionDate string
		Rules       []publicsuffix.Rule
	}{
		sha[:6],
		datetime.Format(time.ANSIC),
		rules,
	}

	log.Printf("Parsing PSL...\n")
	buf := new(bytes.Buffer)
	err = listTmpl.Execute(buf, &data)
	if err != nil {
		log.Fatal(err)
	}

	formatted, err := format.Source(buf.Bytes())
	if err != nil {
		log.Fatal(err)
	}

	log.Printf("Writing %v...\n", filename)
	err = ioutil.WriteFile(filename, formatted, 0644)
	if err != nil {
		log.Fatal(err)
	}
}

func extractHeadInfo() (sha string, datetime time.Time) {
	client := github.NewClient(nil)

	commits, _, err := client.Repositories.ListCommits(context.Background(), "publicsuffix", "list", nil)
	if err != nil {
		log.Fatal(err)
	}

	lastCommit := commits[0]
	return lastCommit.GetSHA(), lastCommit.GetCommit().GetCommitter().GetDate()
}
