/** *****************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018. All Rights Reserved.
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 ****************************************************************************** */
/* Copyright (c) 2020 Red Hat, Inc. */
import { gql } from 'apollo-server-express';
import _ from 'lodash';
import getClusterElements from './clusterHelper';
import getWeaveScopeElements from './relationshipHelper';
import getApplicationElements from './applicationHelper';
import getPolicyElements from './policyHelper';

export const typeDef = gql`
type Resource {
  cluster: String
  clusterName: String
  labels: [Label]
  name: String
  shortName: String
  namespace: String
  relationships: [Relationship]
  topology: String
  type: String
  specs: JSON
  uid: String
  id: String
}

type Relationship {
  from: Resource
  to: Resource
  type: String
  specs: JSON
}

type Topology {
  resources: [Resource]
  relationships: [Relationship]
}

type Label {
  name: String
  value: String
}

type TopologyDetails {
  pods: [JSON]
}

input TopologyDetailsFilter {
  clusterNames: [String]
}
`;

export const resolver = {
  Query: {
    // objects in topology view
    topology: async (root, { topologyResourceType, filter },
      {
        mongoModel, clusterModel, applicationModel, complianceModel,
      }) => {
      let resources = [];
      let relationships = [];
      switch (topologyResourceType) {
        case 'HCMCluster': {
          ({ resources } = await getClusterElements(clusterModel, complianceModel));
          break;
        }
        case 'HCMTopology': {
          if (Object.keys(filter).length > 0) {
            ({ resources, relationships } = await getWeaveScopeElements(clusterModel, mongoModel, filter));
          }
          break;
        }
        case 'HCMApplication': {
          const { name, namespace, channel } = filter.application[0];
          const model = await applicationModel.getApplication(name, namespace, channel);
          if (model) {
            ({ resources, relationships } = await getApplicationElements(model, clusterModel));
          }
          break;
        }
        case 'HCMPolicy': {
          const { name, namespace } = filter.policy[0];
          const model = await complianceModel.getCompliances(name, namespace);
          if (model) {
            ({ resources, relationships } = await getPolicyElements(clusterModel, complianceModel, model));
          }
          break;
        }

        default:
          break;
      }
      return { resources, relationships };
    },

    // second pass--get topology details
    topologyDetails: async (root, args) => {
      let pods = await this.kubeConnector.getResources(
        (ns) => `/apis/apps.open-cluster-management.io/v1/namespaces/${ns}/pods`,
        {
          kind: 'PlacementRule',
          namespaces: _.get(args, 'filter.clusterNames'),
        },
      );
      pods = pods.map((pod) => {
        const {
          metadata, cluster, containers: cntrs, status, hostIP, podIP, restarts, startedAt,
        } = pod;
        const {
          name, namespace, creationTimestamp, labels,
        } = metadata;
        const containers = cntrs.map(({ name: n, image }) => ({
          name: n,
          image,
        }));
        return {
          name,
          namespace,
          status,
          cluster,
          containers,
          creationTimestamp,
          labels,
          hostIP,
          podIP,
          restarts,
          startedAt,
        };
      });
      return { pods };
    },
  },
};
