/** *****************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 ****************************************************************************** */
/* Copyright (c) 2020 Red Hat, Inc. */
import _ from 'lodash';
import logger from '../lib/logger';

const metadataNsStr = 'metadata.namespace';
const metadataNameStr = 'metadata.name';
const statusStatusStr = 'status.status';
const lCompliantStr = 'status.compliant';
const uCompliantStr = 'status.Compliant';
const validityStr = 'status.Validity';
const validValidityStr = 'status.Validity.valid';
const rtRulesStr = 'runtime-rules';
const specRtRulesStr = 'spec.runtime-rules';
const specRemActionStr = 'spec.remediationAction';
const policyTempStr = 'policy-templates';
const objTempStr = 'object-templates';
const roleTempStr = 'role-templates';
const rbTempStr = 'roleBinding-templates';

function getTemplates(policy = {}, templateType = '') {
  const templates = [];
  Object.entries(policy.spec || []).forEach(([key, value]) => {
    if (key.endsWith(`${templateType}-templates`)) {
      value.forEach((item) => templates.push({ ...item, templateType: key }));
    }
  });
  return templates;
}

export default class ComplianceModel {
  constructor({ kubeConnector }) {
    if (!kubeConnector) {
      throw new Error('kubeConnector is a required parameter');
    }

    this.kubeConnector = kubeConnector;
  }

  async getCompliances(name, namespace) {
    let policies = [];

    if (namespace) {
      if (name) {
        // get single policy with a specific name and a specific namespace
        const URL = `/apis/policy.open-cluster-management.io/v1/namespaces/${namespace}/policies/${name}`;
        const policyResponse = await this.kubeConnector.get(URL);
        if (policyResponse.code || policyResponse.message) {
          logger.error(`GRC ERROR ${policyResponse.code} - ${policyResponse.message} - URL : ${URL}`);
        } else {
          policies.push(policyResponse);
        }
      } else {
        // for getting policy list with a specific namespace
        const URL = `/apis/policy.open-cluster-management.io/v1/namespaces/${namespace}/policies`;
        const policyResponse = await this.kubeConnector.get(URL);
        if (policyResponse.code || policyResponse.message) {
          logger.error(`GRC ERROR ${policyResponse.code} - ${policyResponse.message} - URL : ${URL}`);
        }
        policies = policyResponse.items || [];
      }
    } else {
      // all possible namespaces
      const allNameSpace = this.kubeConnector.namespaces;
      // remove cluster namespaces
      const nsPromises = allNameSpace.map(async (ns) => {
      // check ns one by one, if got normal response then it's cluster namespace
        const URL = `/apis/internal.open-cluster-management.io/v1beta1/namespaces/${ns}/managedclusterinfos/`;
        const checkClusterNameSpace = await this.kubeConnector.get(URL);
        if (checkClusterNameSpace.items && checkClusterNameSpace.items.length > 0) {
          return null; // cluster namespaces
        }
        return ns; // non cluster namespaces
      });

      // here need to await all async check cluster namespace calls completed
      let allNonClusterNameSpace = await Promise.all(nsPromises);
      // remove cluster namespaces which already set to null
      allNonClusterNameSpace = allNonClusterNameSpace.filter((ns) => ns !== null);

      if (name) {
        // get single policy with a specific name and all non-clusters namespaces
        const promises = allNonClusterNameSpace.map(async (ns) => {
          const URL = `/apis/policy.open-cluster-management.io/v1/namespaces/${ns}/policies/${name}`;
          const policyResponse = await this.kubeConnector.get(URL);
          if (policyResponse.code || policyResponse.message) {
            logger.error(`GRC ERROR ${policyResponse.code} - ${policyResponse.message} - URL : ${URL}`);
            return null;// 404 or not found
          }
          return policyResponse;// found policy
        });
        // here need to await all async calls completed then combine their results together
        const policyResponses = await Promise.all(promises);
        // remove no found policies
        policies = policyResponses.filter((policyResponse) => policyResponse !== null);
      } else { // most general case for all policies
        // for getting policy list with all non-clusters namespaces
        const promises = allNonClusterNameSpace.map(async (ns) => {
          const URL = `/apis/policy.open-cluster-management.io/v1/namespaces/${ns}/policies`;
          const policyResponse = await this.kubeConnector.get(URL);
          if (policyResponse.code || policyResponse.message) {
            logger.error(`GRC ERROR ${policyResponse.code} - ${policyResponse.message} - URL : ${URL}`);
          }
          return policyResponse.items;
        });
        // here need to await all async calls completed then combine their results together
        const policyResponses = await Promise.all(promises);
        // remove empty policies namespaces
        policies = policyResponses.filter((policyResponse) => policyResponse.length > 0);
        // flatten 'array of array of object' to 'array of object'
        policies = _.flatten(policies);
      }
    }

    return policies.map((entry) => ({
      ...entry,
      raw: entry,
      name: _.get(entry, metadataNameStr, ''),
      namespace: _.get(entry, metadataNsStr, ''),
      remediation: _.get(entry, specRemActionStr, ''),
      clusters: _.keys(_.get(entry, statusStatusStr), ''),
    }));
  }

  static resolveCompliancePolicies(parent) {
    const aggregatedStatus = _.get(parent, statusStatusStr);
    // compliance that has aggregatedStatus
    if (aggregatedStatus) {
      return this.resolvePolicyFromStatus(aggregatedStatus, parent);
    }
    // in this case, a compliance doesn't connect with a
    // placementPolicy may not have aggregatedStatus
    return this.resolvePolicyFromSpec(parent);
  }

  static resolveCompliancePolicy(parent) {
    const aggregatedStatus = _.get(parent, statusStatusStr);
    return this.resolveCompliancePoliciesFromSpec(parent, aggregatedStatus);
  }

  static resolveCompliancePoliciesFromSpec(parent, aggregatedStatus) {
    const compliancePolicies = {};
    const policies = _.get(parent, specRtRulesStr) ? _.get(parent, specRtRulesStr) : [parent];
    policies.forEach((policy) => {
      const key = _.get(policy, metadataNameStr);
      const value = {
        name: _.get(policy, metadataNameStr),
        complianceName: _.get(parent, metadataNameStr),
        complianceNamespace: _.get(parent, metadataNsStr),
        complianceSelfLink: _.get(parent, 'metadata.selfLink'),
        roleTemplates: this.resolvePolicyTemplates(policy, roleTempStr),
        roleBindingTemplates: this.resolvePolicyTemplates(policy, rbTempStr),
        objectTemplates: this.resolvePolicyTemplates(policy, objTempStr),
        policyTemplates: this.resolvePolicyTemplates(policy, policyTempStr),
        detail: this.resolvePolicyDetails(policy),
        raw: policy,
      };
      compliancePolicies[key] = value;
    });

    if (aggregatedStatus) {
      Object.values(aggregatedStatus).forEach((cluster) => {
        Object.entries(_.get(cluster, 'aggregatePoliciesStatus', {})).forEach(([key, value]) => {
          let policy;
          if (parent.spec[rtRulesStr]) {
            policy = parent.spec[rtRulesStr].find((p) => p.metadata.name === key);
          }
          const policyObject = {
            compliant: this.resolveStatus(value),
            enforcement: _.get(policy, specRemActionStr, 'unknown'),
            message: _.get(value, 'message', '-'),
            rules: this.resolvePolicyRules(policy), // TODO: Use resolver.
            status: this.resolveStatus(value),
            violations: this.resolvePolicyViolations(policy, cluster), // TODO: Use resolver.
            metadata: {
              ...parent.metadata,
              name: key,
            },
          };

          compliancePolicies[key] = { ...compliancePolicies[key], ...policyObject };
        });
      });
    }

    return Object.values(compliancePolicies);
  }

  static resolvePolicyFromStatus(aggregatedStatus, parent) {
    const compliancePolicies = [];
    Object.values(aggregatedStatus).forEach((cluster) => {
      Object.entries(_.get(cluster, 'aggregatePoliciesStatus', {})).forEach(([key, value]) => {
        let policy;
        if (parent.spec[rtRulesStr]) {
          policy = parent.spec[rtRulesStr].find((p) => p.metadata.name === key);
        }
        const policyObject = {
          cluster: _.get(cluster, 'clustername', parent.metadata.namespace),
          complianceName: parent.metadata.name,
          complianceNamespace: parent.metadata.namespace,
          compliant: this.resolveStatus(value),
          enforcement: _.get(policy, specRemActionStr, 'unknown'),
          message: _.get(value, 'message', '-'),
          name: key,
          rules: this.resolvePolicyRules(policy), // TODO: Use resolver.
          status: this.resolveStatus(value),
          valid: this.resolveValid(value),
          violations: this.resolvePolicyViolations(policy, cluster), // TODO: Use resolver.
          roleTemplates: this.resolvePolicyTemplates(policy, roleTempStr),
          roleBindingTemplates: this.resolvePolicyTemplates(policy, rbTempStr),
          objectTemplates: this.resolvePolicyTemplates(policy, objTempStr),
          detail: this.resolvePolicyDetails(policy),
          raw: policy,
          metadata: {
            ...parent.metadata,
            name: key,
          },
        };

        compliancePolicies.push(policyObject);
      });
    });

    const tempResult = {};
    Object.values(compliancePolicies).forEach((policy) => {
      if (!tempResult[policy.name]) {
        tempResult[policy.name] = {
          name: _.get(policy, 'name'),
          complianceName: _.get(policy, 'complianceName'),
          complianceNamespace: _.get(policy, 'complianceNamespace'),
          clusterCompliant: [],
          clusterNotCompliant: [],
          policies: [],
        };
      }
      tempResult[policy.name].policies.push(policy);
      if (_.get(policy, 'compliant', '').toLowerCase() === 'compliant') {
        tempResult[policy.name].clusterCompliant.push(_.get(policy, 'cluster'));
      } else {
        tempResult[policy.name].clusterNotCompliant.push(_.get(policy, 'cluster'));
      }
    });
    return Object.values(tempResult);
  }

  static resolvePolicyFromSpec(parent) {
    const compliancePolicies = [];
    const policies = _.get(parent, specRtRulesStr, []);
    policies.forEach((policy) => {
      compliancePolicies.push({
        name: _.get(policy, metadataNameStr),
        complianceName: _.get(parent, metadataNameStr),
        complianceNamespace: _.get(parent, metadataNsStr),
      });
    });
    return Object.values(compliancePolicies);
  }

  static resolveStatus(parent) {
    return _.get(parent, 'Compliant') || _.get(parent, 'compliant', 'unknown');
  }

  static resolveValid(parent) {
    if (_.get(parent, 'Valid') !== undefined) {
      return _.get(parent, 'Valid') ? true : 'invalid';
    }
    if (_.get(parent, 'valid') !== undefined) {
      return _.get(parent, 'valid') ? true : 'invalid';
    }
    return 'unknown';
  }

  static resolveComplianceStatus(parent) {
    const complianceStatus = [];
    Object.entries(_.get(parent, statusStatusStr, {}))
      .forEach(([clusterName, perClusterStatus]) => {
        const aggregatedStatus = _.get(perClusterStatus, 'aggregatePoliciesStatus', {});

        // get compliant status per cluster
        if (aggregatedStatus) {
          let validNum = 0;
          let compliantNum = 0;
          let policyNum = 0;
          Object.values(aggregatedStatus).forEach((object) => {
            if (this.resolveStatus(object) === 'Compliant') {
              compliantNum += 1;
            }
            if (this.resolveValid(object)) {
              validNum += 1;
            }
            policyNum += 1;
          });
          complianceStatus.push({
            clusterNamespace: clusterName,
            localCompliantStatus: `${compliantNum}/${policyNum}`,
            localValidStatus: `${validNum}/${policyNum}`,
            compliant: _.get(perClusterStatus, 'compliant', '-'),
          });
        }
      });

    return complianceStatus;
  }

  static resolvePolicyCompliant({ status = {} }) {
    let totalPolicies = 0;
    let compliantPolicies = 0;

    Object.values(status.status || []).forEach((cluster) => {
      Object.values(cluster.aggregatePoliciesStatus || {}).forEach((policyValue) => {
        totalPolicies += 1;
        if (this.resolveStatus(policyValue).toLowerCase() === 'compliant') {
          compliantPolicies += 1;
        }
      });
    });

    return `${totalPolicies - compliantPolicies}/${totalPolicies}`;
  }

  static resolveClusterCompliant({ status = {} }) {
    if (status && status.status) {
      const totalClusters = Object.keys(status.status).length;
      const compliantClusters = Object.values(status.status || [])
        .filter((cluster) => (_.get(cluster, 'compliant', '').toLowerCase() === 'compliant'));
      return `${totalClusters - compliantClusters.length}/${totalClusters}`;
    }
    return '0/0';
  }

  static resolveAnnotations(parent) {
    const rawAnnotations = _.get(parent, 'metadata.annotations', {});
    return {
      categories: _.get(rawAnnotations, 'policy.open-cluster-management.io/categories', '-'),
      controls: _.get(rawAnnotations, 'policy.open-cluster-management.io/controls', '-'),
      standards: _.get(rawAnnotations, 'policy.open-cluster-management.io/standards', '-'),
    };
  }

  async getPlacementRules(parent = {}) {
    const placements = _.get(parent, 'status.placement', []);
    const policies = [];
    placements.forEach((placement) => {
      policies.push(placement.placementRule);
    });
    const response = await this.kubeConnector.getResources(
      (ns) => `/apis/apps.open-cluster-management.io/v1/namespaces/${ns}/placementrules`,
      { kind: 'PlacementRule' },
    );
    const map = new Map();
    if (response) {
      response.forEach((item) => map.set(item.metadata.name, item));
    }
    const placementPolicies = [];
    policies.forEach((policy) => {
      const pp = map.get(policy);
      if (pp) {
        const spec = pp.spec || {};
        placementPolicies.push({
          clusterLabels: spec.clusterSelector,
          metadata: pp.metadata,
          raw: pp,
          clusterReplicas: spec.clusterReplicas,
          resourceSelector: spec.resourceHint,
          status: pp.status,
        });
      }
    });
    return placementPolicies;
  }

  async getPlacementBindings(parent = {}) {
    const bindings = _.get(parent, 'status.placementBindings', []);
    const response = await this.kubeConnector.getResources(
      (ns) => `/apis/policy.open-cluster-management.io/v1/namespaces/${ns}/placementbindings`,
      { kind: 'PlacementBinding' },
    );
    const map = new Map();
    if (response) {
      response.forEach((item) => map.set(item.metadata.name, item));
    }
    const placementBindings = [];

    bindings.forEach((binding) => {
      const pb = map.get(binding);
      if (pb) {
        placementBindings.push({
          metadata: pb.metadata,
          raw: pb,
          placementRef: pb.placementRef,
          subjects: pb.subjects,
        });
      }
    });
    return placementBindings;
  }

  static resolvePolicyDetails(parent) {
    return {
      exclude_namespace: _.get(parent, 'spec.namespaces.exclude', ['*']),
      include_namespace: _.get(parent, 'spec.namespaces.include', ['*']),
    };
  }

  static resolvePolicyEnforcement(parent) {
    return _.get(parent, specRemActionStr, 'unknown');
  }

  static resolvePolicyRules(parent) {
    const rules = [];
    getTemplates(parent).forEach((res) => {
      if (res.rules) {
        Object.entries(res.rules).forEach(([key, rul]) => {
          const complianceType = _.get(rul, 'complianceType');
          if (complianceType) {
            const rule = {
              complianceType,
              apiGroups: _.get(rul, 'policyRule.apiGroups', ['-']),
              resources: _.get(rul, 'policyRule.resources', ['-']),
              verbs: _.get(rul, 'policyRule.verbs', ['-']),
              templateType: _.get(res, 'templateType', ''),
              ruleUID: `${_.get(res, metadataNameStr, '-')}-rule-${key}`,
            };
            rules.push(rule);
          }
        });
      }
    });
    return rules;
  }

  static resolveRoleSubjects(parent) {
    let roleSubjects = [];
    getTemplates(parent).forEach((res) => {
      if (_.get(res, 'templateType') === rbTempStr) {
        roleSubjects = [..._.get(res, 'roleBinding.subjects', [])];
      }
    });
    return roleSubjects;
  }

  static resolveRoleRef(parent) {
    const roleRef = [];
    getTemplates(parent).forEach((res) => {
      if (_.get(res, 'templateType') === rbTempStr) {
        roleRef.push(_.get(res, 'roleBinding.roleRef', {}));
      }
    });
    return roleRef;
  }

  static resolvePolicyStatus(parent) {
    if (_.get(parent, uCompliantStr) || _.get(parent, lCompliantStr)) {
      return _.get(parent, uCompliantStr) || _.get(parent, lCompliantStr);
    }
    if (_.get(parent, 'status.Valid') !== undefined) {
      return _.get(parent, 'status.Valid') ? 'valid' : 'invalid';
    }
    if (_.get(parent, 'status.valid') !== undefined) {
      return _.get(parent, 'status.valid') ? 'valid' : 'invalid';
    }
    return 'unknown';
  }

  static resolvePolicyMessage(parent) {
    return _.get(parent, 'status.message', '-');
  }

  static resolvePolicyTemplates(parent, type) {
    const tempArray = [];
    getTemplates(parent).forEach((res) => {
      if (_.get(res, 'templateType') === type) {
        if (type === rbTempStr) {
          tempArray.push({
            name: _.get(res, 'roleBinding.metadata.name', '-'),
            lastTransition: _.get(res, 'status.conditions[0].lastTransitionTime', ''),
            complianceType: _.get(res, 'complianceType', ''),
            apiVersion: _.get(res, 'roleBinding.apiVersion', ''),
            compliant: _.get(res, uCompliantStr, ''),
            validity: _.get(res, validValidityStr) || _.get(res, validityStr, ''),
            raw: res,
          });
        } else if (type === objTempStr || type === policyTempStr) {
          tempArray.push({
            name: _.get(res, 'objectDefinition.metadata.name', '-'),
            lastTransition: _.get(res, 'status.conditions[0].lastTransitionTime', ''),
            complianceType: _.get(res, 'complianceType', ''),
            apiVersion: _.get(res, 'objectDefinition.apiVersion', ''),
            kind: _.get(res, 'objectDefinition.kind', ''),
            compliant: _.get(res, uCompliantStr, ''),
            status: _.get(res, uCompliantStr, ''),
            validity: _.get(res, validValidityStr) || _.get(res, validityStr, ''),
            raw: res,
          });
        } else {
          tempArray.push({
            name: _.get(res, metadataNameStr, '-'),
            lastTransition: _.get(res, 'status.conditions[0].lastTransitionTime', ''),
            complianceType: _.get(res, 'complianceType', ''),
            apiVersion: _.get(res, 'apiVersion', ''),
            compliant: _.get(res, uCompliantStr, ''),
            status: _.get(res, uCompliantStr, ''),
            validity: _.get(res, validValidityStr) || _.get(res, validityStr, ''),
            raw: res,
          });
        }
      }
    });
    return tempArray;
  }

  static resolvePolicyViolations(parent, cluster) {
    const violationArray = [];
    getTemplates(parent).forEach((res) => {
      const templateCondition = _.get(res, 'status.conditions[0]');
      if (_.get(res, 'templateType') === roleTempStr) {
        violationArray.push({
          name: _.get(res, metadataNameStr, '-'),
          cluster: _.get(cluster, 'clustername', '-'),
          status: this.resolvePolicyStatus(res),
          message: (templateCondition && _.get(templateCondition, 'message', '-')) || '-',
          reason: (templateCondition && _.get(templateCondition, 'reason', '-')) || '-',
          selector: _.get(res, 'selector', ''),
        });
      } else if (_.get(res, 'templateType') === objTempStr
        || _.get(res, 'templateType') === policyTempStr) {
        violationArray.push({
          name: _.get(res, 'objectDefinition.metadata.name', '-'),
          cluster: _.get(cluster, 'clustername', '-'),
          status: this.resolvePolicyStatus(res),
          message: (templateCondition && _.get(templateCondition, 'message', '-')) || '-',
          reason: (templateCondition && _.get(templateCondition, 'reason', '-')) || '-',
          selector: _.get(res, 'selector', ''),
        });
      }
    });
    return violationArray;
  }
}
