/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect, Selectors, SidecarExpect, Util } from '@kui-shell/test';
import { splitViaButton, focus } from '../core-support2/split-helpers';
import { dirname, join } from 'path';
const ROOT = dirname(require.resolve('@kui-shell/plugin-kubectl/tests/package.json'));
const base64Input = 'hi';
const base64Output = Buffer.from(base64Input).toString('base64');
describe(`snapshot and replay with title ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    const file = Util.uniqueFileForSnapshot();
    const testTabLabel1 = 'test1';
    const testTabLabel2 = 'test2';
    const doReplay = (testTabLabel) => {
        it('should replay', () => __awaiter(this, void 0, void 0, function* () {
            try {
                yield CLI.command(`replay ${file}`, this.app);
                yield this.app.client
                    .$(Selectors.TOP_TAB_WITH_TITLE(testTabLabel))
                    .then(_ => _.waitForExist({ timeout: CLI.waitTimeout }));
                let idx = 0;
                yield this.app.client.waitUntil(() => __awaiter(this, void 0, void 0, function* () {
                    const txt = yield this.app.client.$(Selectors.OUTPUT_LAST).then(_ => _.getText());
                    if (++idx > 5) {
                        console.error(`still waiting for expected=${base64Output}; actual=${txt}`);
                    }
                    return txt === base64Output;
                }), { timeout: CLI.waitTimeout });
            }
            catch (err) {
                yield Common.oops(this, true)(err);
            }
        }));
    };
    it(`should open new tab with title ${testTabLabel1}`, () => CLI.command(`tab new --title ${testTabLabel1}`, this.app)
        .then(() => CLI.waitForSession(this))
        .catch(Common.oops(this)));
    it(`should base64 ${base64Input}`, () => CLI.command(`base64 ${base64Input}`, this.app)
        .then(ReplExpect.okWithString(base64Output))
        .catch(Common.oops(this, true)));
    it('should snapshot without specifying title', () => CLI.command(`snapshot ${file}`, this.app)
        .then(ReplExpect.justOK)
        .catch(Common.oops(this, true)));
    // clear the tabs so there isn't a tab naming `testTabLabel1`
    it('should refresh', () => Common.refresh(this));
    doReplay(testTabLabel1);
    it(`should snapshot with title ${testTabLabel2}`, () => CLI.command(`snapshot ${file} --title ${testTabLabel2}`, this.app)
        .then(ReplExpect.justOK)
        .catch(Common.oops(this, true)));
    // clear the tabs so there isn't a tab naming `testTabLabel2`
    it('should refresh', () => Common.refresh(this));
    doReplay(testTabLabel2);
    it('should snapshot again but without specifying title', () => CLI.command(`snapshot ${file}`, this.app)
        .then(ReplExpect.justOK)
        .catch(Common.oops(this, true)));
    // Currently, The replay command will display the notebook in a new tab.
    // Without refreshing, the second tab can't be opened due to tab name duplication.
    it('should refresh', () => Common.refresh(this));
    doReplay(testTabLabel2);
});
describe(`snapshot and replay ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    const file = Util.uniqueFileForSnapshot();
    it(`should base64 ${base64Input}`, () => CLI.command(`base64 ${base64Input}`, this.app)
        .then(ReplExpect.okWithString(base64Output))
        .catch(Common.oops(this, true)));
    // do something to open the sidecar, so we can verify it's not open on replay
    it('should open sidecar', () => CLI.command(`open ${(join(ROOT), 'package.json')}`, this.app)
        .then(ReplExpect.ok)
        .then(SidecarExpect.open)
        .catch(Common.oops(this, true)));
    it('should snapshot', () => CLI.command(`snapshot ${file}`, this.app)
        .then(ReplExpect.justOK)
        .catch(Common.oops(this, true)));
    it('should refresh', () => Common.refresh(this));
    it('should replay', () => __awaiter(this, void 0, void 0, function* () {
        try {
            yield CLI.command(`replay ${file} --status-stripe blue`, this.app);
            yield this.app.client
                .$(Selectors.STATUS_STRIPE_TYPE('blue'))
                .then(_ => _.waitForExist({ timeout: CLI.waitTimeout }));
            const res = yield CLI.command('version', this.app);
            // verify the base64 command replay
            let idx = 0;
            yield this.app.client.waitUntil(() => __awaiter(this, void 0, void 0, function* () {
                const txt = yield this.app.client.$(Selectors.OUTPUT_N(res.count - 2)).then(_ => _.getText());
                if (++idx > 5) {
                    console.error(`still waiting for expected=${base64Output}; actual=${txt}`);
                }
                return txt === base64Output;
            }), { timeout: CLI.waitTimeout });
            // verify the about replay
            yield SidecarExpect.notOpen(res);
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
    it('should clear and show default status stripe', () => __awaiter(this, void 0, void 0, function* () {
        yield CLI.command('clear', this.app).then(() => ReplExpect.consoleToBeClear(this.app));
        yield this.app.client.$(Selectors.STATUS_STRIPE_TYPE('default')).then(_ => _.waitForExist());
    }));
});
describe(`split-snapshot-replay ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    return __awaiter(this, void 0, void 0, function* () {
        before(Common.before(this));
        after(Common.after(this));
        const file = Util.uniqueFileForSnapshot();
        const splitTheTerminalViaButton = splitViaButton.bind(this);
        const clickToFocus = focus.bind(this);
        const doBase64 = (splitIdx) => {
            clickToFocus(splitIdx);
            it(`should base64 ${base64Input}`, () => CLI.commandInSplit(`base64 ${base64Input}`, this.app, splitIdx)
                .then(ReplExpect.okWithString(base64Output))
                .catch(Common.oops(this, true)));
        };
        const doSnapShot = (splitIdx) => {
            it('should snapshot with exec', () => CLI.commandInSplit(`snapshot ${file} --exec`, this.app, splitIdx)
                .then(ReplExpect.justOK)
                .catch(Common.oops(this, true)));
        };
        doBase64(1);
        splitTheTerminalViaButton(2);
        clickToFocus(2);
        doBase64(2);
        splitTheTerminalViaButton(3);
        doBase64(3);
        doSnapShot(3);
        it('should refresh', () => Common.refresh(this));
        it('should replay', () => CLI.command(`replay ${file}`, this.app).catch(Common.oops(this, true)));
        it('should validate replay', () => __awaiter(this, void 0, void 0, function* () {
            try {
                let idx = 0;
                yield this.app.client.waitUntil(() => __awaiter(this, void 0, void 0, function* () {
                    // commands in split1: [session connect in browser],base64, split, version
                    const base64InSplit1 = yield this.app.client.$(Selectors.OUTPUT_LAST_FOR_SPLIT(1)).then(_ => _.getText());
                    // commands in split2: base64,version
                    const base64InSplit2 = yield this.app.client.$(Selectors.OUTPUT_LAST_FOR_SPLIT(2)).then(_ => _.getText());
                    const base64InSplit3 = yield this.app.client.$(Selectors.OUTPUT_LAST_FOR_SPLIT(3)).then(_ => _.getText());
                    if (++idx > 5) {
                        console.error(`still waiting for expected=${base64Output}; actual=${base64InSplit1}, ${base64InSplit2}, ${base64InSplit3}`);
                    }
                    return base64InSplit1 === base64Output && base64InSplit2 === base64Output && base64InSplit3 === base64Output;
                }), { timeout: CLI.waitTimeout });
            }
            catch (err) {
                yield Common.oops(this, true)(err);
            }
        }));
    });
});
Common.proxyDescribe(`core snapshot and replay by query ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    const file = Util.uniqueFileForSnapshot();
    it(`should base64 ${base64Input}`, () => CLI.command(`base64 ${base64Input}`, this.app)
        .then(ReplExpect.okWithString(base64Output))
        .catch(Common.oops(this, true)));
    const title = 'replay-by-query';
    it('should snapshot', () => CLI.command(`snapshot ${file} --title ${title}`, this.app)
        .then(ReplExpect.justOK)
        .catch(Common.oops(this, true)));
    it('should replay by query', () => __awaiter(this, void 0, void 0, function* () {
        try {
            yield this.app.client.url(`${process.env.WEBPACK_CLIENT_URL}?command=replay ${file}`);
            yield this.app.client
                .$(Selectors.TOP_TAB_WITH_TITLE(title))
                .then(_ => _.waitForExist({ timeout: CLI.waitTimeout }));
            // verify the base64 command replay
            let idx = 0;
            yield this.app.client.waitUntil(() => __awaiter(this, void 0, void 0, function* () {
                const txt = yield this.app.client.$(Selectors.OUTPUT_LAST).then(_ => _.getText());
                if (++idx > 5) {
                    console.error(`still waiting for expected=${base64Output}; actual=${txt}`);
                }
                return txt === base64Output;
            }), { timeout: CLI.waitTimeout });
            // back to the original url, without this line, the following tests will fail at `before` state
            yield this.app.client.url(process.env.WEBPACK_CLIENT_URL);
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
});
//# sourceMappingURL=replay.js.map