/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import React from 'react';
import { eventBus, eventChannelUnsafe } from '@kui-shell/core';
import Input from './Input';
import Output from './Output';
import { isActive, isBeingRerun, isEmpty, isFinished, isOutputOnly, isProcessing, isAnnouncement, hasUUID } from './BlockModel';
export default class Block extends React.PureComponent {
    constructor(props) {
        super(props);
        this._willChangeSize = this.willChangeSize.bind(this);
        this._onOutputRender = this.onOutputRender.bind(this);
        this.state = {
            isFocused: false,
            isMaximized: false
        };
    }
    /** Owner wants us to focus on the current prompt */
    doFocus() {
        if (this._input) {
            this._input.doFocus();
        }
    }
    inputValue() {
        return this._input && this._input.value();
    }
    /** Child wants to maximize/restore */
    willChangeSize(width) {
        this.setState({
            isMaximized: width === "100%" /* Maximized */
        });
        setTimeout(() => {
            eventBus.emitTabLayoutChange(this.props.tab.uuid);
            if (this.state._block) {
                this.state._block.scrollIntoView(true);
            }
        });
    }
    onOutputRender() {
        if (this.props.onOutputRender) {
            this.props.onOutputRender();
        }
        if (this.props.noActiveInput && this.state._block) {
            this.state._block.scrollIntoView();
        }
    }
    output() {
        if (isFinished(this.props.model) || isProcessing(this.props.model)) {
            return (React.createElement(Output, { uuid: this.props.uuid, tab: this.props.tab, idx: this.props.idx, model: this.props.model, isBeingRerun: isBeingRerun(this.props.model), willRemove: this.props.willRemove, willChangeSize: this._willChangeSize, onRender: this._onOutputRender, willUpdateCommand: this.props.willUpdateCommand, isPartOfMiniSplit: this.props.isPartOfMiniSplit, isWidthConstrained: this.props.isWidthConstrained, willFocusBlock: this.props.willFocusBlock }));
        }
    }
    willScreenshot() {
        setTimeout(() => {
            const element = this.state._block.querySelector('.kui--screenshotable') || this.state._block;
            eventChannelUnsafe.emit('/screenshot/element', element);
        });
    }
    customInput() {
        if (this.props.children && React.isValidElement(this.props.children)) {
            return React.cloneElement(this.props.children, {
                idx: this.props.idx,
                tab: this.props.tab,
                uuid: this.props.uuid,
                block: this.state._block
            });
        }
    }
    input() {
        return (this.customInput() ||
            (this.state._block && (React.createElement(Input, Object.assign({ key: this.props.idx, uuid: this.props.uuid, tab: this.props.tab, model: this.props.model, isExperimental: this.props.isExperimental }, this.props, { willScreenshot: this.state._block && this.props.willRemove ? () => this.willScreenshot() : undefined, willFocusBlock: this.props.willFocusBlock, _block: this.state._block, ref: c => (this._input = c) }), this.props.children))));
    }
    /**
     * For Active or Empty blocks, just show the <Input/>, otherwise
     * wrap the <Input/>-<Output/> pair.
     *
     */
    render() {
        return ((!this.props.noActiveInput || !isActive(this.props.model)) && (React.createElement("li", { className: 'repl-block kui--maximize-candidate ' + this.props.model.state.toString(), "data-is-maximized": this.state.isMaximized || undefined, "data-is-output-only": isOutputOnly(this.props.model) || undefined, "data-is-empty": isEmpty(this.props.model) || undefined, "data-announcement": isAnnouncement(this.props.model) || undefined, "data-uuid": hasUUID(this.props.model) && this.props.model.execUUID, "data-scrollback-uuid": this.props.uuid, "data-input-count": this.props.idx, "data-is-focused": this.props.isFocused || undefined, "data-is-visible-in-minisplit": this.props.isVisibleInMiniSplit || undefined, ref: c => this.setState({ _block: c }), tabIndex: isActive(this.props.model) ? -1 : 1, onClick: this.props.willFocusBlock, onFocus: this.props.onFocus }, isAnnouncement(this.props.model) || isOutputOnly(this.props.model) ? (this.output()) : isActive(this.props.model) || isEmpty(this.props.model) ? (this.input()) : (React.createElement(React.Fragment, null,
            this.input(),
            this.output())))));
    }
}
//# sourceMappingURL=index.js.map