/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import React from 'react';
import prettyPrintDuration from 'pretty-ms';
import { Td } from '@patternfly/react-table';
import { split, isPopup, eventBus, pexecInCurrentTab, isHeadless } from '@kui-shell/core';
import Icons from '../../spi/Icons';
import Tooltip from '../../spi/Tooltip';
import tooltipContent, { tooltipProps } from './Tooltip';
const Markdown = React.lazy(() => import('../Markdown'));
import ErrorCell from './ErrorCell';
import whenNothingIsSelected from '../../../util/selection';
function XOR(a, b) {
    return (a || b) && !(a && b);
}
/**
 * Generate an onclick handler for a cell
 *
 */
export function onClickForCell(row, tab, repl, cell, opts) {
    const { drilldownTo = 'side-split', selectRow = () => undefined } = opts || {};
    const handler = cell && cell.onclick ? cell.onclick : row.onclick;
    if (handler === false) {
        return () => handler;
    }
    else if (typeof handler === 'function') {
        return whenNothingIsSelected((evt) => {
            evt.stopPropagation();
            selectRow();
            handler();
            return false;
        });
    }
    else if (handler && handler.startEvent && handler.completeEvent) {
        return whenNothingIsSelected((evt) => {
            evt.stopPropagation();
            selectRow();
            eventBus.emitCommandStart(handler.startEvent);
            eventBus.emitCommandComplete(handler.completeEvent);
            return false;
        });
    }
    else if (handler) {
        const opts = { tab };
        if (!row.onclickExec || row.onclickExec === 'pexec') {
            return whenNothingIsSelected((evt) => __awaiter(this, void 0, void 0, function* () {
                evt.stopPropagation();
                selectRow();
                if (!isPopup() && drilldownTo === 'side-split' && !XOR(evt.metaKey, !!process.env.KUI_SPLIT_DRILLDOWN)) {
                    pexecInCurrentTab(`split --ifnot is-split --cmdline "${handler}"`, undefined, false, true);
                }
                else if (!isHeadless() && drilldownTo === 'new-window') {
                    const { ipcRenderer } = yield import('electron');
                    ipcRenderer.send('synchronous-message', JSON.stringify({
                        operation: 'new-window',
                        argv: split(handler)
                    }));
                }
                else {
                    repl.pexec(handler, opts);
                }
                return false;
            }));
        }
        else {
            return whenNothingIsSelected((evt) => {
                evt.stopPropagation();
                selectRow();
                repl.qexec(handler, undefined, undefined, { tab });
                return false;
            });
        }
    }
}
/**
 * Render a TableCell part
 *
 */
export default function renderCell(table, kuiRow, justUpdated, tab, repl) {
    return function KuiTableCell(key, value, tag, outerCSS, css, onclick, cidx) {
        // className for the td
        const cellClassName = cidx === 0
            ? 'entity-name ' + (outerCSS || '')
            : (/NAME/i.test(key) ? 'kui--entity-name-secondary ' : /STATUS/i.test(key) ? 'kui--status-cell' : '') +
                (outerCSS || '');
        const outerClassName = 'display-inline-block cell-inner ' + (css || '') + (onclick ? ' clickable' : '');
        // the text value of the cell
        const valueDom = cidx > 0 && kuiRow.attributes[cidx - 1] && kuiRow.attributes[cidx - 1].valueDom;
        const title = (cidx - 1 === table.durationColumnIdx || cidx - 1 === table.coldStartColumnIdx) && value
            ? prettyPrintDuration(parseInt(value, 10))
            : value;
        const innerText = valueDom || title;
        const innerSpan = (React.createElement("span", { className: "kui--cell-inner-text" }, cidx === 0 && kuiRow.fontawesome === 'fas fa-check' ? (React.createElement(Icons, { icon: "Checkmark" })) : table.markdown ? (React.createElement(Markdown, { nested: true, source: title })) : (innerText)));
        const innerSpanWithTooltip = cidx === 0 && table.title ? (React.createElement(Tooltip, Object.assign({ markdown: tooltipContent(table.title, kuiRow.name) }, tooltipProps), innerSpan)) : (innerSpan);
        const { attributes = [] } = kuiRow;
        // re: OBJECT, see https://github.com/IBM/kui/issues/6831
        return (React.createElement(Td, { key: cidx, className: cellClassName, modifier: /OBJECT/i.test(key) || /MESSAGE/i.test(key) ? 'wrap' : !/NAME|NAMESPACE/i.test(key) ? 'fitContent' : undefined },
            React.createElement("div", { "data-key": key, "data-value": value, "data-tag": tag, className: outerClassName, onClick: onclick ? onClickForCell(kuiRow, tab, repl, attributes[cidx - 1], table) : undefined },
                tag === 'badge' && (React.createElement("span", { key: css /* force restart of animation if color changes */, title: title, className: css || 'kui--status-unknown', "data-tag": "badge-circle", "data-just-updated": justUpdated || undefined }, /red-background/.test(css) ? React.createElement(ErrorCell, null) : undefined)),
                innerSpanWithTooltip)));
    };
}
//# sourceMappingURL=TableCell.js.map