/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import React from 'react';
import { isFile } from '@kui-shell/plugin-bash-like/fs';
import { encodeComponent, isHTML, isRadioTable, isReactProvider, isStringWithOptionalContentType, isTable, isCommandStringContent, isFunctionContent, isScalarContent, isStringDiffContent } from '@kui-shell/core';
import Eval from './Eval';
import Editor from './Editor';
import DiffEditor from './Editor/DiffEditor';
import renderTable from './Table';
import Markdown from './Markdown';
import HTMLString from './HTMLString';
import HTMLDom from './Scalar/HTMLDom';
import { KuiContext } from '../../';
import RadioTableSpi from '../spi/RadioTable';
export default class KuiContent extends React.PureComponent {
    constructor(props) {
        super(props);
        this.state = KuiContent.getDerivedStateFromProps(props);
    }
    static getDerivedStateFromProps(props, state) {
        if (!state || state.mode !== props.mode || state.execUUID !== props.execUUID) {
            return Object.assign(state || {}, { mode: props.mode, execUUID: props.execUUID });
        }
        else {
            return state;
        }
    }
    render() {
        if (!this.props.isActive) {
            return React.createElement(React.Fragment, null);
        }
        const { mode } = this.state;
        const { tab, response, willUpdateToolbar } = this.props;
        if (isStringWithOptionalContentType(mode)) {
            if (mode.contentType === 'text/html') {
                return React.createElement(HTMLString, { content: mode.content });
            }
            else if (mode.contentType === 'text/markdown') {
                if (!mode.content && isFile(response)) {
                    // then we need to fetch the file content
                    return (React.createElement(Eval, Object.assign({}, this.props, { command: `vfs fslice ${encodeComponent(response.spec.fullpath)} 0 8192`, contentType: mode.contentType })));
                }
                else {
                    return (React.createElement(Markdown, { tab: tab, repl: tab.REPL, fullpath: isFile(response) ? response.spec.fullpath : undefined, source: mode.content }));
                }
            }
            else {
                return (React.createElement(Editor, { content: mode, readOnly: false, willUpdateToolbar: willUpdateToolbar, response: response, repl: tab.REPL, tabUUID: tab.uuid }));
            }
        }
        else if (isStringDiffContent(mode)) {
            return (React.createElement(DiffEditor, { contentType: mode.contentType, originalContent: mode.content.a, modifiedContent: mode.content.b, response: response, renderSideBySide: true, tabUUID: tab.uuid }));
        }
        else if (isCommandStringContent(mode)) {
            const key = `${mode.contentFrom} ${mode.contentType}`;
            return React.createElement(Eval, Object.assign({}, this.props, { key: key, command: mode.contentFrom, contentType: mode.contentType }));
        }
        else if (isFunctionContent(mode)) {
            const command = mode.content;
            return React.createElement(Eval, Object.assign({}, this.props, { key: command.toString(), command: command }));
        }
        else if (isScalarContent(mode)) {
            if (isReactProvider(mode)) {
                return mode.react({ willUpdateToolbar });
            }
            else if (isRadioTable(mode.content)) {
                const radioTable = mode.content;
                // ^^^ Notes: Even though isRadioTable(mode.content) checks the type of mode.content,
                // RadioTableSpi in KuiContext.Consumer doesn't know the type of mode.content is RadioTable and throws error
                // so we have to re-assign mode.content to work around this typescript compile error
                return (React.createElement(KuiContext.Consumer, null, config => React.createElement(RadioTableSpi, { table: radioTable, title: !config.disableTableTitle, repl: tab.REPL })));
            }
            else if (isTable(mode.content)) {
                return renderTable(tab, tab.REPL, mode.content, false);
                // ^^^ Notes: typescript doesn't like this, and i don't know why:
                // "is not assignable to type IntrinsicAttributes..."
                // <PaginatedTable {...props} />
            }
            else if (isHTML(mode.content)) {
                return React.createElement(HTMLDom, { content: mode.content });
            }
            else {
                console.error('Unsupported scalar content', mode);
            }
        }
        return React.createElement("div", { className: "oops" }, "Unsupported content");
    }
}
export function isFocusable(node) {
    return typeof node.doFocus === 'function';
}
//# sourceMappingURL=KuiContent.js.map