"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.doExec = void 0;

var _debug = _interopRequireDefault(require("debug"));

var _child_process = require("child_process");

var _core = require("@kui-shell/core");

var _exec = require("../util/exec");

var _json = require("../util/json");

var _catchall = require("./catchall");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2017 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};
/**
 * This plugin introduces commands that dispatch to a local bash-like
 * shell
 *
 */


const debug = (0, _debug.default)('plugins/bash-like/cmds/general');

function doSpawn(argv, execOptions // eslint-disable-next-line @typescript-eslint/no-explicit-any
) {
  // eslint-disable-next-line no-async-promise-executor
  return new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
    try {
      const options = {
        env: Object.assign({}, process.env, execOptions['env'] || {})
      };

      if (process.env.SHELL && argv.find(_ => />|\||`|\$/.test(_))) {
        // if the argv has a | or $ or a backtick, then we need to use
        // a shell to evaluate the argv
        options.shell = process.env.SHELL;
      }

      if (!execOptions.onInit && !execOptions.onExit && (0, _core.isHeadless)() && !(0, _core.inProxy)()) {
        options.stdio = 'inherit';
      }

      const proc = (0, _child_process.spawn)(argv[0], argv.slice(1), options);
      proc.on('error', reject);
      proc.on('close', exitCode => __awaiter(this, void 0, void 0, function* () {
        if (exitCode === 0) {
          resolve(true);
        } else {
          reject(new Error(`non-zero exit code: ${exitCode}`));
        }
      }));
    } catch (err) {
      reject(err);
    }
  }));
}

const doExec = (cmdLine, argv, execOptions // eslint-disable-next-line @typescript-eslint/no-explicit-any
) => {
  if (!execOptions.onInit && !execOptions.onExit && (0, _core.isHeadless)() && !(0, _core.inProxy)()) {
    return doSpawn(argv, execOptions);
  } // eslint-disable-next-line no-async-promise-executor


  return new Promise((resolve, reject) => __awaiter(void 0, void 0, void 0, function* () {
    try {
      const options = {
        maxBuffer: 1 * 1024 * 1024,
        env: Object.assign({}, process.env, execOptions['env'] || {})
      };

      if (process.env.SHELL) {
        options.shell = process.env.SHELL;
      }

      const proc = (0, _child_process.exec)(cmdLine, options); // accumulate doms from the output of the subcommand

      let rawOut = '';
      let rawErr = '';
      let stdout; // caller has requested a callback to set up the stdout stream

      if (execOptions.onInit) {
        debug('setting up stdout via onInit', cmdLine.slice(0, 10));
        stdout = Promise.resolve(execOptions.onInit({
          abort: () => proc.kill(),
          xon: () => {},
          xoff: () => {},
          write: () => {} // eslint-disable-line @typescript-eslint/no-empty-function

        }));
      }

      proc.stdout.on('data', data => __awaiter(void 0, void 0, void 0, function* () {
        const out = data.toString();

        if (stdout) {
          ;
          (yield stdout)(data);
        } else {
          rawOut += out;
        }
      }));
      proc.stderr.on('data', data => __awaiter(void 0, void 0, void 0, function* () {
        rawErr += data;

        if (execOptions.stderr) {
          execOptions.stderr(data.toString()); // stderrLines += data.toString()
        } else if (stdout) {
          ;
          (yield stdout)(data.toString());
        } else {
          debug(data.toString());
        }
      }));
      proc.on('error', reject);
      proc.on('close', exitCode => __awaiter(void 0, void 0, void 0, function* () {
        if (execOptions.onExit) {
          debug('onExit', exitCode, cmdLine.slice(0, 10));
          execOptions.onExit(exitCode);
          return;
        }

        if (exitCode === 0) {
          // great, the process exited normally. resolve!
          if (execOptions && execOptions['json']) {
            // caller expects JSON back
            try {
              resolve(JSON.parse(rawOut));
            } catch (err) {
              const error = new Error('unexpected non-JSON');
              error['value'] = rawOut;
              reject(error);
            }
          } else if (execOptions && execOptions.raw) {
            // caller just wants the raw textual output
            resolve(rawOut);
          } else if (!rawOut && !rawErr) {
            // in this case, the command produced nothing, but it did exit
            // with a 0 exit code
            resolve(true);
          } else {
            // else, we pass back a formatted form of the output
            const json = (0, _json.extractJSON)(rawOut);

            if (json && typeof json === 'object') {
              json['type'] = 'shell';
              json['verb'] = 'get';
              resolve(json);
            } else {
              resolve(rawOut);
            }
          }
        } else {
          // oops, non-zero exit code. reject!
          debug('non-zero exit code', exitCode); // strip off e.g. /bin/sh: line 0:

          const cleanErr = rawErr.replace(/(^\/[^/]+\/[^:]+: )(line \d+: )?/, '');

          try {
            (0, _exec.handleNonZeroExitCode)(cmdLine, exitCode, rawOut, cleanErr, execOptions);
          } catch (err) {
            reject(err);
          }
        }
      }));
    } catch (err) {
      reject(err);
    }
  }));
};

exports.doExec = doExec;

const specialHandler = args => {
  if (args.execOptions.type === _core.ExecType.TopLevel) {
    throw new Error('this command is intended for internal consumption only');
  }

  return (0, _catchall.dispatchToShell)(args);
};
/**
 * Register command handlers
 *
 */


var _default = commandTree => {
  commandTree.listen('/!', _catchall.dispatchToShell, {
    docs: 'Execute a UNIX shell command',
    requiresLocal: true
  });
  commandTree.listen('/sendtopty', specialHandler, {
    docs: 'Execute a UNIX shell command with a PTY',
    hidden: true
  });
  commandTree.listen('/pwd', () => process.env.PWD || process.env.HOME || '/');
};

exports.default = _default;