/*
 * Copyright 2021 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import colors from 'colors';
import wrap from 'word-wrap';
export function command(cmdline) {
    return `${colors.cyan(cmdline)}`;
}
export function intro(paragraph) {
    return wrap(paragraph, { width: 100, indent: '  ' }) + '\n';
}
export function option(opts) {
    return typeof opts === 'string' ? colors.bold(opts) : opts.map(_ => colors.bold(_)).join(', ');
}
export function title(_) {
    return `${colors.bold.yellow(_.command)} ${_.doc}
`;
}
/** @return pretty-printed version of the given bash pipeline */
export function pipeline(code) {
    return code
        .replace(/(\s+\|\s+)(\w+)/g, (_, p1, p2) => `${p1}${colors.cyan(p2)}`)
        .replace(/(\s+--\s+)(\w+)/g, (_, p1, p2) => `${p1}${colors.cyan(p2)}`)
        .replace(/\[[^\]]*options[^\]]*\]/, _ => colors.gray(_))
        .replace(/\[[^\]]*name[^\]]*\]/, _ => colors.gray(_))
        .replace(/\/s3\/aws\/commoncrawl\/[./\-{},\w]+/, _ => colors.dim(_))
        .replace(/ \|/g, _ => colors.magenta(_))
        .replace(/ \\/g, _ => colors.dim(_))
        .replace(/kiwi super/, _ => colors.yellow(_));
}
// \u2063 is "invisible separator" https://www.fileformat.info/info/unicode/char/2063/index.htm
// needed for columnify, which deletes whitespace
const indent = '  ';
const indentedLines = '\n  ';
export function usage(usages) {
    return `Usage:
  ${usages.map(pipeline).join(indentedLines)}
`;
}
export function examples(examples, sectionTitle = 'Examples') {
    const data = examples.map(_ => ({
        command: `${indent}${command(_.command)}`,
        doc: _.doc
    }));
    return `${sectionTitle}:
${require('columnify')(data, { showHeaders: false })} 
`;
}
export function related(related) {
    return `Related:
  ${colors.dim(related.map(command).join(', '))}
`;
}
export function options(options) {
    const data = options.map(_ => ({
        option: `${indent}${option(_.flags)}`,
        doc: `${_.doc}${_.default ? ` ${colors.gray('default: ' + _.default)}` : ''}`
    }));
    return `Options:
${require('columnify')(data, { showHeaders: false })}
`;
}
//# sourceMappingURL=pretty-code.js.map