/*
 * Copyright 2018 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import Debug from 'debug';
import { join } from 'path';
import needle from 'needle';
import { isHeadless, inBrowser, hasProxy, encodeComponent, i18n } from '@kui-shell/core';
import JSONStream from './json';
import getProxyState from '../../controller/client/proxy';
import { isRecursive } from '../../controller/kubectl/options';
import { getCurrentDefaultContextName } from '../../controller/kubectl/contexts';
const strings = i18n('plugin-kubectl');
const debug = Debug('plugin-kubectl/util/fetch-file');
/** Maximum number of times to retry on ECONNREFUSED */
const MAX_ECONNREFUSED_RETRIES = isHeadless() ? 0 : 10;
const httpScheme = /http(s)?:\/\//;
const openshiftScheme = /^openshift:\/\//;
const kubernetesScheme = /^kubernetes:\/\//;
/** Instantiate a kubernetes:// scheme with the current kubectl proxy state */
function rescheme(url, args) {
    return __awaiter(this, void 0, void 0, function* () {
        const context = yield getCurrentDefaultContextName(args);
        if (kubernetesScheme.test(url)) {
            const { baseUrl } = yield getProxyState('kubectl', context);
            return url.replace(kubernetesScheme, baseUrl);
        }
        else if (openshiftScheme.test(url)) {
            const { baseUrl } = yield getProxyState('oc', context);
            return url.replace(openshiftScheme, baseUrl);
        }
        else {
            return url;
        }
    });
}
export function openStream(args, url, mgmt, headers) {
    return __awaiter(this, void 0, void 0, function* () {
        if (inBrowser() && hasProxy()) {
            debug('routing openStream request to proxy', url);
            yield args.REPL.rexec(`_openstream ${encodeComponent(url)}`, Object.assign({}, {
                onInit: mgmt.onInit,
                onReady: mgmt.onReady,
                onExit: mgmt.onExit,
                data: { headers }
            }));
            debug('routed openStream to proxy: done', url);
        }
        else {
            // we need to set JSON to false to disable needle's parsing, which
            // seems not to be compatible with streaming
            const uri = yield rescheme(url, args);
            debug('routing openStream request to endpoint', uri);
            const stream = needle.get(uri, { headers, parse: false });
            const onData = mgmt.onInit({
                abort: () => {
                    debug('abort requested', typeof stream['destroy']);
                    if (typeof stream['destroy'] === 'function') {
                        stream['destroy']();
                    }
                    if (typeof mgmt.onExit === 'function') {
                        debug('sending onExit');
                        mgmt.onExit(0);
                    }
                },
                xon: () => stream.resume(),
                xoff: () => stream.pause(),
                resize: () => {
                    console.error('Unsupported Operation: resize');
                },
                write: () => {
                    console.error('Unsupported Operation: write');
                }
            });
            stream.on('error', err => {
                //
                // It's highly likely to be ok if the stream went away. This is
                // usually a sign that we are in the Kui proxy, and the Kui browser
                // client simply went away.
                //
                // !!DANGER!!: do not throw an exception here, as it will currently
                //             percolate all the way up and result in a Kui proxy
                //             failure due to an uncaughtException.
                //
                if (err.code !== 'ERR_STREAM_DESTROYED') {
                    console.error('stream suddenly died', err);
                }
            });
            JSONStream(stream, yield onData, mgmt.onExit);
        }
    });
}
export function _needle(repl, url, opts, retryCount = 0) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!inBrowser()) {
            const method = (opts && opts.method) || 'get';
            const headers = Object.assign({ connection: 'keep-alive' }, opts.headers);
            debug('fetch via needle', method, headers);
            // internal usage: test kui's error handling of apiServer
            if (process.env.TRAVIS_CHAOS_TESTING) {
                // simulate a condition where kuiproxy is not installed on the apiserver
                const error = new Error('404 page not found');
                error.code = 404;
                error.statusCode = 404;
                throw error;
            }
            try {
                const { statusCode, body } = yield needle(method, yield rescheme(url, { REPL: repl }), opts.data, {
                    json: true,
                    follow_max: 10,
                    headers
                });
                debug('fetch via needle done', statusCode);
                return { statusCode, body };
            }
            catch (err) {
                if (err.code === 'ECONNREFUSED' && retryCount < MAX_ECONNREFUSED_RETRIES) {
                    // then the proxy is probably transiently offline, e.g. due to
                    // switching context or namespace
                    debug('retrying due to ECONNREFUSED');
                    return new Promise((resolve, reject) => {
                        setTimeout(() => __awaiter(this, void 0, void 0, function* () {
                            try {
                                _needle(repl, url, opts, retryCount + 1).then(resolve, reject);
                            }
                            catch (err) {
                                reject(err);
                            }
                        }), 50);
                    });
                }
                else {
                    throw err;
                }
            }
        }
        else if (inBrowser()) {
            // Unfortunately, we cannot rely on being able to fetch files
            // directly from a browser. For one, if the remote site does not
            // offer an Access-Control-Allow-Origin, then well behaving
            // browsers will refuse to load their content;
            // e.g. https://k8s.io/examples/controllers/nginx-deployment.yaml
            // Solution: have the kui proxy do this
            if (!hasProxy()) {
                throw new Error(strings('Unable to fetch remote file'));
            }
            else {
                debug('fetch via proxy');
                const body = (yield repl.rexec(`_fetchfile ${encodeComponent(url)}`, { data: opts }))
                    .content[0];
                return {
                    statusCode: 200,
                    body
                };
            }
        }
    });
}
function fetchRemote(repl, url, opts) {
    return __awaiter(this, void 0, void 0, function* () {
        debug('fetchRemote', url, opts);
        const fetchOnce = () => _needle(repl, url, opts).then(_ => {
            if (_.statusCode === 0 || _.statusCode === 200 || typeof _.body !== 'string') {
                return _.body;
            }
            else {
                const err = new Error(_.body.toString());
                err.code = _.statusCode;
                throw err;
            }
        });
        const retry = (delay) => (err) => __awaiter(this, void 0, void 0, function* () {
            if (/timeout/.test(err.message) || /hang up/.test(err.message) || /hangup/.test(err.message)) {
                debug('retrying', err);
                yield new Promise(resolve => setTimeout(resolve, delay));
                return fetchOnce();
            }
            else {
                throw err;
            }
        });
        // fetch with three retries
        return fetchOnce()
            .catch(retry(500))
            .catch(retry(1000))
            .catch(retry(5000));
    });
}
export function isReturnedError(file) {
    const err = file;
    return typeof file !== 'string' && !Buffer.isBuffer(file) && typeof err.error === 'string';
}
/**
 * Either fetch a remote file or read a local one
 *
 */
export function fetchFile(repl, url, opts) {
    return __awaiter(this, void 0, void 0, function* () {
        const urls = url.split(/,/);
        debug('fetchFile', urls);
        const start = Date.now();
        const responses = yield Promise.all(urls.map((url, idx) => __awaiter(this, void 0, void 0, function* () {
            if (httpScheme.test(url) || kubernetesScheme.test(url) || openshiftScheme.test(url)) {
                debug('fetch remote', url);
                return fetchRemote(repl, url, Object.assign({}, opts, { data: Array.isArray(opts.data) ? opts.data[idx] : opts.data })).catch(err => {
                    if (opts && opts.returnErrors) {
                        return {
                            code: err.code,
                            error: err.message || JSON.stringify(err)
                        };
                    }
                    else
                        throw err;
                });
            }
            else {
                const filepath = url;
                debug('fetch local', filepath);
                const stats = (yield repl.rexec(`vfs fstat ${repl.encodeComponent(filepath)} --with-data`))
                    .content;
                return stats.data;
            }
        })));
        debug('fetchFile done', Date.now() - start);
        return responses;
    });
}
/** same as fetchFile, but returning a string rather than a Buffer */
export function fetchFileString(repl, url, headers) {
    return __awaiter(this, void 0, void 0, function* () {
        const files = yield fetchFile(repl, url, { headers });
        return files.map(_ => {
            try {
                return _ ? (typeof _ === 'string' ? _ : Buffer.isBuffer(_) ? _.toString() : JSON.stringify(_)) : undefined;
            }
            catch (err) {
                console.error('Unable to convert fetched file to string', err, _);
                return '';
            }
        });
    });
}
export function fetchFileKustomize(repl, url) {
    return __awaiter(this, void 0, void 0, function* () {
        const stats = (yield repl.rexec(`_fetchfile ${repl.encodeComponent(url)} --kustomize`)).content;
        return stats;
    });
}
/**
 * fetch raw files from `filepath`
 */
export function fetchRawFiles(args, filepath) {
    return __awaiter(this, void 0, void 0, function* () {
        if (filepath.match(/http(s)?:\/\//)) {
            return fetchRemote(args.REPL, filepath);
        }
        else {
            const path = args.REPL.encodeComponent(filepath);
            const resourceStats = (yield args.REPL.rexec(`vfs fstat ${path} --with-data`)).content;
            if (resourceStats.isDirectory) {
                return args.REPL.rexec(`vfs ls ${join(path, isRecursive(args) ? '/**/*.yaml' : '/*.yaml')} --with-data`)
                    .then(_ => _.content)
                    .then(filenames => Promise.all(filenames.map(({ path }) => args.REPL.rexec(`vfs fstat ${path} --with-data`).then(_ => _.content.data))))
                    .then(_ => _.join('---\n'));
            }
            else {
                return resourceStats.data;
            }
        }
    });
}
export default fetchFileString;
export function fetchFilesVFS(args, filepath, fetchNestedYamls) {
    return __awaiter(this, void 0, void 0, function* () {
        const path = fetchNestedYamls ? `${encodeComponent(filepath)} ${encodeComponent(filepath)}/**/*.{yaml,yml}` : filepath;
        const paths = new Set((yield args.REPL.rexec(`vfs ls ${path}`)).content.map(_ => _.path));
        const raw = yield fetchFileString(args.REPL, paths.size === 0 ? filepath : Array.from(paths).join(','));
        return raw.map((data, idx) => {
            if (data) {
                return { filepath: paths.size === 0 ? filepath : Array.from(paths)[idx], data };
            }
        });
    });
}
export function fetchKusto(args, kusto) {
    return __awaiter(this, void 0, void 0, function* () {
        const [{ load }, { join }, raw] = yield Promise.all([
            import('js-yaml'),
            import('path'),
            fetchFileKustomize(args.REPL, kusto)
        ]);
        const kustomization = load(raw.data);
        if (kustomization.resources) {
            const resources = kustomization.resources
                .map(resource => encodeComponent(raw.dir ? join(raw.dir, resource) : resource))
                .join(' ');
            const files = yield fetchFilesVFS(args, resources);
            return {
                customization: { filepath: kusto, data: raw.data, contentType: 'yaml', isFor: 'k' },
                templates: files.map(({ filepath, data }) => ({ filepath, data, contentType: 'yaml', isFor: 'k' }))
            };
        }
    });
}
//# sourceMappingURL=fetch-file.js.map