/*
 * Copyright 2018 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import kubeuiApiVersion from '../../controller/kubectl/apiVersion';
export class DefaultKubeStatus {
    constructor() {
        this.message = undefined;
    }
}
export function hasResourceVersion(resource) {
    const withVersion = resource;
    return typeof withVersion.metadata.resourceVersion === 'string';
}
export function sameResourceVersion(a, b) {
    return (a.apiVersion === b.apiVersion &&
        a.kind === b.kind &&
        a.metadata.name === b.metadata.name &&
        a.metadata.namespace === b.metadata.namespace &&
        hasResourceVersion(a) &&
        hasResourceVersion(b) &&
        a.metadata.resourceVersion === b.metadata.resourceVersion);
}
export function hasSingleOwnerReference(resource) {
    if (!resource.metadata) {
        return false;
    }
    const { ownerReferences } = resource.metadata;
    return (ownerReferences &&
        Array.isArray(ownerReferences) &&
        ownerReferences.length === 1 &&
        typeof ownerReferences[0].apiVersion === 'string' &&
        typeof ownerReferences[0].kind === 'string' &&
        typeof ownerReferences[0].name === 'string');
}
export class DefaultKubeMetadata {
    constructor() {
        this.kind = undefined;
        this.name = undefined;
    }
}
export function hasRawData(resource) {
    const withData = resource;
    return typeof withData.kuiRawData === 'string';
}
/** is the resource Namespaced? */
export function isNamespaced(resource) {
    return resource.metadata !== undefined && resource.metadata.namespace !== undefined;
}
/** is the command response a Kubernetes resource? note: excluding any ones we simulate in kubeui */
export function isKubeResource(entity) {
    const kube = entity;
    return (kube !== undefined &&
        kube.isKubeResource === true &&
        kube.apiVersion !== undefined &&
        kube.apiVersion !== kubeuiApiVersion &&
        kube.kind !== undefined);
}
export function isKubeResourceWithItsOwnSummary(resource) {
    return resource !== undefined && resource.summary !== undefined;
}
/**
 * This allows us to exclude certain resource kinds from auto-summarization
 *
 */
export function isSummarizableKubeResource(resource) {
    return (isKubeResource(resource) &&
        (isKubeResourceWithItsOwnSummary(resource) ||
            (resource.kind !== undefined && resource.kind !== 'List' && resource.kind !== 'CustomResourceDefinition')));
}
export function isRole(resource) {
    const role = resource;
    return role.rules !== undefined;
}
export function isRoleBinding(resource) {
    const rb = resource;
    return rb.roleRef !== undefined && rb.subjects !== undefined;
}
export function isServiceAccount(resource) {
    const sa = resource;
    return isKubeResource(resource) && sa.secrets !== undefined;
}
/**
 * @return whether the given resource is an instance of a Pod
 *
 */
export function isPod(resource) {
    return isKubeResource(resource) && resource.apiVersion === 'v1' && resource.kind === 'Pod';
}
/**
 * @return whether the given resource is an instance of a Namespace
 *
 */
export function isNamespace(resource) {
    return isKubeResource(resource) && resource.apiVersion === 'v1' && resource.kind === 'Namespace';
}
/**
 * @return whether the given resource is an instance of a Deploymemt
 *
 */
export function isJob(resource) {
    return resource.apiVersion === 'batch/v1' && resource.kind === 'Job';
}
/**
 * @return whether the given resource is an instance of a Deployment
 *
 */
export function isDeployment(resource) {
    return (isKubeResource(resource) &&
        resource.kind === 'Deployment' &&
        (resource.apiVersion === 'extensions/v1beta1' || resource.apiVersion === 'apps/v1'));
}
/**
 * @return whether the given resource is an instance of a ReplicaSet
 *
 */
export function isReplicaSet(resource) {
    return (isKubeResource(resource) &&
        (resource.apiVersion === 'extensions/v1beta1' || resource.apiVersion === 'apps/v1') &&
        resource.kind === 'ReplicaSet');
}
export function hasInvolvedObject(resource) {
    const io = resource;
    return (io.involvedObject !== undefined &&
        typeof io.involvedObject.apiVersion === 'string' &&
        typeof io.involvedObject.kind === 'string' &&
        typeof io.involvedObject.name === 'string');
}
/**
 * @return whether the given resource is an instance of an Event
 *
 */
export function isEvent(resource) {
    return isKubeResource(resource) && resource.apiVersion === 'v1' && resource.kind === 'Event';
}
/** is the command response a kube resource that can responds to "kubectl delete", etc.? */
export function isCrudableKubeResource(entity) {
    return isKubeResource(entity) && !isEvent(entity) && !entity.isSimulacrum;
}
export function isKubeItems(resource) {
    return isKubeResource(resource) && resource.apiVersion === 'v1' && /List$/.test(resource.kind);
}
export function isPodList(resource) {
    return isKubeResource(resource) && resource.apiVersion === 'v1' && resource.kind === 'PodList';
}
export function isKubeItemsOfKind(resource, isOfKind) {
    return isKubeItems(resource) && resource.items.length > 0 && isOfKind(resource.items[0]);
}
/**
 * @return whether the given resource is an instance of a CustomResourceDefinition
 *
 */
export function isCustomResourceDefinition(resource) {
    return (isKubeResource(resource) &&
        (resource.apiVersion === 'apiextensions.k8s.io/v1' || resource.apiVersion === 'apiextensions.k8s.io/v1beta1') &&
        resource.kind === 'CustomResourceDefinition');
}
/**
 * @return whether the given resource is an instance of a CustomResourceDefinition
 *
 */
export function isConfigMap(resource) {
    return isKubeResource(resource) && resource.apiVersion === 'v1' && resource.kind === 'ConfigMap';
}
/**
 * Is the given resource kind cluster scoped (as opposed to namespace scoped)?
 * FIXME: apiVersion
 */
export function isClusterScoped(kind) {
    return kind === 'CustomResourceDefinition' || kind === 'Namespace' || kind === 'Node';
}
export function isNode(resource) {
    return resource.apiVersion === 'v1' && resource.kind === 'Node';
}
/**
 * @return whether the given resource might possibly have events;
 * since Events never have Events, we can exclude those always
 *
 */
export function hasEvents(resource) {
    return isCrudableKubeResource(resource) && !isEvent(resource) && isNamespaced(resource);
}
export function isMetaTable(response) {
    const table = response;
    return table.apiVersion === 'meta.k8s.io/v1' && table.kind === 'Table';
}
export function isStatus(resource) {
    if (Buffer.isBuffer(resource) || typeof resource === 'string') {
        return false;
    }
    else {
        const status = resource;
        return status.apiVersion === 'v1' && status.kind === 'Status';
    }
}
export function isSecret(resource) {
    return resource.apiVersion === 'v1' && resource.kind === 'Secret';
}
//# sourceMappingURL=resource.js.map