/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import React from 'react';
import { Icons, Select, TextWithIconWidget } from '@kui-shell/plugin-client-common';
import { eventBus, i18n, eventChannelUnsafe, getTab, pexecInCurrentTab } from '@kui-shell/core';
import { kubectl, getTabState, getCurrentDefaultNamespace, onKubectlConfigChangeEvents, offKubectlConfigChangeEvents } from '@kui-shell/plugin-kubectl';
import { ready } from './CurrentContext';
import { isInternalNamespace } from '@kui-shell/plugin-kubectl/heuristics';
const strings = i18n('plugin-kubectl');
export default class CurrentNamespace extends React.PureComponent {
    constructor(props) {
        super(props);
        this.handler = this.reportCurrentNamespace.bind(this);
        this.handlerNotCallingKubectl = this.getCurrentNamespaceFromTab.bind(this);
        this.state = {
            currentNamespace: strings('Loading...'),
            allNamespaces: [],
            viewLevel: 'info'
        };
    }
    /** @return a short string that we can fit into a context UI widget */
    renderNamespace(context) {
        return context.metadata.namespace;
    }
    debounce() {
        const now = Date.now();
        const last = this.last;
        this.last = now;
        return last && now - last < 250;
    }
    reportCurrentNamespace(idx) {
        return __awaiter(this, void 0, void 0, function* () {
            const tab = getTab(typeof idx === 'string' ? undefined : idx);
            if (!tab || !tab.REPL) {
                if (tab && !tab.REPL) {
                    eventChannelUnsafe.once(`/tab/new/${tab.uuid}`, () => this.reportCurrentNamespace());
                }
                return;
            }
            else if (this.debounce()) {
                return;
            }
            try {
                const [currentNamespace, allNamespaces] = yield Promise.all([
                    getCurrentDefaultNamespace(tab),
                    tab.REPL.qexec(`${kubectl} get ns -o name`).then(_ => _.split(/\n/).map(_ => _.replace(/^namespace\//, '')))
                ]);
                if (currentNamespace) {
                    this.setState({
                        currentNamespace,
                        allNamespaces,
                        viewLevel: 'normal' // only show normally if we succeed; see https://github.com/IBM/kui/issues/3537
                    });
                }
            }
            catch (err) {
                console.error(err);
                this.last = undefined;
                this.setState({
                    currentNamespace: '',
                    allNamespaces: [],
                    viewLevel: 'hidden' // only show normally if we succeed; see https://github.com/IBM/kui/issues/3537
                });
            }
        });
    }
    getCurrentNamespaceFromTab(args) {
        const { tab } = args;
        if (tab) {
            const currentNamespace = getTabState(tab, 'namespace');
            if (currentNamespace) {
                this.setState({
                    currentNamespace,
                    viewLevel: 'normal'
                });
            }
        }
    }
    /**
     * Once we have mounted, we immediately check the current branch,
     * and schedule an update based on standard REPL events.
     *
     */
    componentDidMount() {
        if (!ready) {
            eventBus.once('/tab/new', this.handler);
        }
        else {
            this.handler();
        }
        eventBus.on('/tab/switch/request/done', this.handlerNotCallingKubectl);
        eventBus.onAnyCommandComplete(this.handler);
        onKubectlConfigChangeEvents(this.handler);
    }
    /** Bye! */
    componentWillUnmount() {
        eventBus.off('/tab/new', this.handler);
        eventBus.off('/tab/switch/request/done', this.handlerNotCallingKubectl);
        eventBus.offAnyCommandComplete(this.handler);
        offKubectlConfigChangeEvents(this.handler);
    }
    listNamespace() {
        return (React.createElement("a", { href: "#", onClick: () => pexecInCurrentTab(`${kubectl} get namespace`) }, strings('Show Full Details')));
    }
    popoverHeader() {
        return (React.createElement(React.Fragment, null,
            React.createElement("div", null, strings('Kubernetes Namespace')),
            React.createElement("div", { className: "do-not-overflow" },
                React.createElement("strong", null, this.state.currentNamespace)),
            React.createElement("div", { className: "sub-text even-smaller-text" }, this.listNamespace())));
    }
    switchNamespaceDescription() {
        const key = this.state.allNamespaces.length === 0
            ? 'Please wait, while we find your namespaces'
            : 'To change, select from the following list of all known namespaces.';
        return React.createElement("span", { className: "sub-text bottom-pad" }, strings(key));
    }
    /** @return the options model for the given namespace named `ns` */
    optionFor(ns) {
        const isSelected = ns === this.state.currentNamespace;
        return {
            label: ns,
            isSelected,
            description: isSelected ? strings('This is your current namespace') : undefined,
            command: `${kubectl} config set-context --current --namespace=${ns}`
        };
    }
    switchNamespace() {
        if (this.state.allNamespaces.length === 0) {
            return;
        }
        const internalNs = this.state.allNamespaces.filter(_ => isInternalNamespace(_)).map(_ => this.optionFor(_));
        const regularNs = this.state.allNamespaces.filter(_ => !isInternalNamespace(_)).map(_ => this.optionFor(_));
        const options = internalNs.length > 0 ? undefined : regularNs;
        const groups = internalNs.length === 0
            ? undefined
            : [
                { label: '', options: regularNs },
                { divider: true },
                { label: strings('System Namespaces'), options: internalNs }
            ];
        return (React.createElement(React.Suspense, { fallback: React.createElement("div", null) },
            React.createElement(Select, { variant: "typeahead", maxHeight: "11rem", className: "small-top-pad", selected: this.state.currentNamespace, options: options, groups: groups, isOpen: true, isClosable: false })));
    }
    popoverBody() {
        return (React.createElement("div", { className: "top-pad bottom-pad" },
            this.switchNamespaceDescription(),
            this.switchNamespace()));
    }
    popover() {
        return {
            className: 'kui--popover-select',
            bodyContent: this.popoverBody(),
            headerContent: this.popoverHeader()
        };
    }
    render() {
        return (React.createElement(TextWithIconWidget, { className: this.props.className, text: this.state.currentNamespace, viewLevel: this.state.viewLevel, id: "kui--plugin-kubeui--current-namespace", title: strings('Kubernetes namespace'), popover: this.popover() },
            React.createElement(Icons, { icon: "At" })));
    }
}
//# sourceMappingURL=CurrentNamespace.js.map