# Upgrading `operator-sdk`

The following are steps to perform when upgrading the `operator-sdk` version
used by the operator. This guide is not intended to be all-encompassing, but a
good starting point.

As a general guideline to upgrading the `operator-sdk` version, you may want to
consider upgrading one minor version at a time. This can be time consuming, but
it may help better deduce the incremental changes that were made in each
version.

## 1. Download planned upgrade version of `operator-sdk`

See https://sdk.operatorframework.io/docs/installation/ for installation
instructions for your development host.

## 2. Update version of `operator-sdk` in Makefile

Update the `OPERATOR_SDK_VERSION` environment variable in the
[Makefile](../Makefile) to the desired version and commit the change.

```shell
export OPERATOR_SDK_VERSION=v1.3.2
sed -i "s/OPERATOR_SDK_VERSION ?= .*/OPERATOR_SDK_VERSION ?= ${OPERATOR_SDK_VERSION}/" Makefile
git commit -m "Update operator-sdk to ${OPERATOR_SDK_VERSION}" Makefile
```

## 3. View guide to upgrading `operator-sdk` version

View the official `operator-sdk` [version upgrade
guide](https://sdk.operatorframework.io/docs/upgrading-sdk-version/) to see
what changes were introduced in each version of the operator. This will help
you build an understanding of what changes will need to be made. It's possible
there may not be significant changes. If that's the case, you may be able to
make the changes directly in the source code without proceeding to the next
steps.

## 4. Initialize a new temporary operator

This step involves initializing a new temporary operator so that you can see
all the scaffolding that is generated by the `operator-sdk`. You should use the
same original command that was used for this operator as shown below:

```shell
# Execute from the root of this repository
mkdir ../gatekeeper-operator-sdk-${OPERATOR_SDK_VERSION//./-}
cd ../gatekeeper-operator-sdk-${OPERATOR_SDK_VERSION//./-}
operator-sdk init --domain=gatekeeper.sh --repo=github.com/gatekeeper/gatekeeper-operator
```

Feel free to examine the generated source files to assess differences and see
if they align with your expectations and those documented in step 3 above.

**NOTE: At this point it may be helpful to use `git init` to track and commit
files that are generated and modified by these `operator-sdk` commands.
However, when using the `cp` command in a below step, you'll want to make sure
to avoid the `.git` directory so that it doesn't clobber your current `.git`
directory in this repo.**

## 5. Generate scaffolding for the Gatekeeper Custom Resource

This step involves creating the Gatekeeper API in the temporary operator
created in step 4 above. You should use the same original command that was used
for this operator as shown below, albiet with any changes to the versions as
necessary:

```shell
operator-sdk create api --group operator --version v1alpha1 --kind Gatekeeper --crd-version=v1
```

## 6. Copy over newly generated files into this operator repo

After you're done generating files, you can change back to the original
`gatekeeper-operator` repository directory and simply copy over the files using
the below command.

**WARNING: If you've used `git init` to initialize a git
repository for this temporary operator, DO NOT copy over the .git repository.
Make sure to delete the `.git` directory first before running the below command.**

```shell
cd ../gatekeeper-operator/
cp -r --preserve ../gatekeeper-operator-sdk-${OPERATOR_SDK_VERSION//./-} .
```

## 7. Compare differences

Use your favorite diff editor to compare differences to files. Bring in any new
changes that were introduced by the newer version of the `operator-sdk` and
preserve any changes that were made to the existing files. The following are
some noteworthy areas to keep an eye out for as they have been modified from
the original scaffolding:

- The `./config/rbac` directory contents have been moved to
  `./config/rbac/base` so that we can track an `overlay` for OpenShift specific
  modifications.
