package idn

const (
	propertyUnknown    property = iota // unknown character property
	propertyPVALID                     // allowed to be used in IDNs
	propertyCONTEXTJ                   // invisible or problematic characters (join controls)
	propertyCONTEXTO                   // invisible or problematic characters (others)
	propertyDISALLOWED                 // should not be included in IDNs
	propertyUNASSIGNED                 // code points that are not designated in the Unicode Standard
)

// property stores the property of a code point, as described in RFC 5892,
// section 1
type property int

// codePoints list all code points in Unicode Character Database (UCD) Format
// according to RFC 5892, appendix B.1. Thanks to libidn2 (GNU) -
// http://www.gnu.org/software/libidn/libidn2/
var codePoints = []struct {
	start rune
	end   rune
	state property
}{
	{0x0000, 0x002C, propertyDISALLOWED},    // <control>..COMMA
	{0x002D, 0x0, propertyPVALID},           // HYPHEN-MINUS
	{0x002E, 0x002F, propertyDISALLOWED},    // FULL STOP..SOLIDUS
	{0x0030, 0x0039, propertyPVALID},        // DIGIT ZERO..DIGIT NINE
	{0x003A, 0x0060, propertyDISALLOWED},    // COLON..GRAVE ACCENT
	{0x0041, 0x005A, propertyPVALID},        // LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
	{0x0061, 0x007A, propertyPVALID},        // LATIN SMALL LETTER A..LATIN SMALL LETTER Z
	{0x007B, 0x00B6, propertyDISALLOWED},    // LEFT CURLY BRACKET..PILCROW SIGN
	{0x00B7, 0x0, propertyCONTEXTO},         // MIDDLE DOT
	{0x00B8, 0x00DE, propertyDISALLOWED},    // CEDILLA..LATIN CAPITAL LETTER THORN
	{0x00DF, 0x00F6, propertyPVALID},        // LATIN SMALL LETTER SHARP S..LATIN SMALL LETT
	{0x00F7, 0x0, propertyDISALLOWED},       // DIVISION SIGN
	{0x00F8, 0x00FF, propertyPVALID},        // LATIN SMALL LETTER O WITH STROKE..LATIN SMAL
	{0x0100, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH MACRON
	{0x0101, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH MACRON
	{0x0102, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH BREVE
	{0x0103, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH BREVE
	{0x0104, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH OGONEK
	{0x0105, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH OGONEK
	{0x0106, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER C WITH ACUTE
	{0x0107, 0x0, propertyPVALID},           // LATIN SMALL LETTER C WITH ACUTE
	{0x0108, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER C WITH CIRCUMFLEX
	{0x0109, 0x0, propertyPVALID},           // LATIN SMALL LETTER C WITH CIRCUMFLEX
	{0x010A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER C WITH DOT ABOVE
	{0x010B, 0x0, propertyPVALID},           // LATIN SMALL LETTER C WITH DOT ABOVE
	{0x010C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER C WITH CARON
	{0x010D, 0x0, propertyPVALID},           // LATIN SMALL LETTER C WITH CARON
	{0x010E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER D WITH CARON
	{0x010F, 0x0, propertyPVALID},           // LATIN SMALL LETTER D WITH CARON
	{0x0110, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER D WITH STROKE
	{0x0111, 0x0, propertyPVALID},           // LATIN SMALL LETTER D WITH STROKE
	{0x0112, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH MACRON
	{0x0113, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH MACRON
	{0x0114, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH BREVE
	{0x0115, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH BREVE
	{0x0116, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH DOT ABOVE
	{0x0117, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH DOT ABOVE
	{0x0118, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH OGONEK
	{0x0119, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH OGONEK
	{0x011A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CARON
	{0x011B, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CARON
	{0x011C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER G WITH CIRCUMFLEX
	{0x011D, 0x0, propertyPVALID},           // LATIN SMALL LETTER G WITH CIRCUMFLEX
	{0x011E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER G WITH BREVE
	{0x011F, 0x0, propertyPVALID},           // LATIN SMALL LETTER G WITH BREVE
	{0x0120, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER G WITH DOT ABOVE
	{0x0121, 0x0, propertyPVALID},           // LATIN SMALL LETTER G WITH DOT ABOVE
	{0x0122, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER G WITH CEDILLA
	{0x0123, 0x0, propertyPVALID},           // LATIN SMALL LETTER G WITH CEDILLA
	{0x0124, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH CIRCUMFLEX
	{0x0125, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH CIRCUMFLEX
	{0x0126, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH STROKE
	{0x0127, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH STROKE
	{0x0128, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH TILDE
	{0x0129, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH TILDE
	{0x012A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH MACRON
	{0x012B, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH MACRON
	{0x012C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH BREVE
	{0x012D, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH BREVE
	{0x012E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH OGONEK
	{0x012F, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH OGONEK
	{0x0130, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH DOT ABOVE
	{0x0131, 0x0, propertyPVALID},           // LATIN SMALL LETTER DOTLESS I
	{0x0132, 0x0134, propertyDISALLOWED},    // LATIN CAPITAL LIGATURE IJ..LATIN CAPITAL LET
	{0x0135, 0x0, propertyPVALID},           // LATIN SMALL LETTER J WITH CIRCUMFLEX
	{0x0136, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH CEDILLA
	{0x0137, 0x0138, propertyPVALID},        // LATIN SMALL LETTER K WITH CEDILLA..LATIN SMA
	{0x0139, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH ACUTE
	{0x013A, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH ACUTE
	{0x013B, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH CEDILLA
	{0x013C, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH CEDILLA
	{0x013D, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH CARON
	{0x013E, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH CARON
	{0x013F, 0x0141, propertyDISALLOWED},    // LATIN CAPITAL LETTER L WITH MIDDLE DOT..LATI
	{0x0142, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH STROKE
	{0x0143, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER N WITH ACUTE
	{0x0144, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH ACUTE
	{0x0145, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER N WITH CEDILLA
	{0x0146, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH CEDILLA
	{0x0147, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER N WITH CARON
	{0x0148, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH CARON
	{0x0149, 0x014A, propertyDISALLOWED},    // LATIN SMALL LETTER N PRECEDED BY APOSTROPHE.
	{0x014B, 0x0, propertyPVALID},           // LATIN SMALL LETTER ENG
	{0x014C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH MACRON
	{0x014D, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH MACRON
	{0x014E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH BREVE
	{0x014F, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH BREVE
	{0x0150, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
	{0x0151, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH DOUBLE ACUTE
	{0x0152, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LIGATURE OE
	{0x0153, 0x0, propertyPVALID},           // LATIN SMALL LIGATURE OE
	{0x0154, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH ACUTE
	{0x0155, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH ACUTE
	{0x0156, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH CEDILLA
	{0x0157, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH CEDILLA
	{0x0158, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH CARON
	{0x0159, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH CARON
	{0x015A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH ACUTE
	{0x015B, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH ACUTE
	{0x015C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH CIRCUMFLEX
	{0x015D, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH CIRCUMFLEX
	{0x015E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH CEDILLA
	{0x015F, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH CEDILLA
	{0x0160, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH CARON
	{0x0161, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH CARON
	{0x0162, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH CEDILLA
	{0x0163, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH CEDILLA
	{0x0164, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH CARON
	{0x0165, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH CARON
	{0x0166, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH STROKE
	{0x0167, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH STROKE
	{0x0168, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH TILDE
	{0x0169, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH TILDE
	{0x016A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH MACRON
	{0x016B, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH MACRON
	{0x016C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH BREVE
	{0x016D, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH BREVE
	{0x016E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH RING ABOVE
	{0x016F, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH RING ABOVE
	{0x0170, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
	{0x0171, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH DOUBLE ACUTE
	{0x0172, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH OGONEK
	{0x0173, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH OGONEK
	{0x0174, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER W WITH CIRCUMFLEX
	{0x0175, 0x0, propertyPVALID},           // LATIN SMALL LETTER W WITH CIRCUMFLEX
	{0x0176, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
	{0x0177, 0x0, propertyPVALID},           // LATIN SMALL LETTER Y WITH CIRCUMFLEX
	{0x0178, 0x0179, propertyDISALLOWED},    // LATIN CAPITAL LETTER Y WITH DIAERESIS..LATIN
	{0x017A, 0x0, propertyPVALID},           // LATIN SMALL LETTER Z WITH ACUTE
	{0x017B, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Z WITH DOT ABOVE
	{0x017C, 0x0, propertyPVALID},           // LATIN SMALL LETTER Z WITH DOT ABOVE
	{0x017D, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Z WITH CARON
	{0x017E, 0x0, propertyPVALID},           // LATIN SMALL LETTER Z WITH CARON
	{0x017F, 0x0, propertyDISALLOWED},       // LATIN SMALL LETTER LONG S
	{0x0180, 0x0, propertyPVALID},           // LATIN SMALL LETTER B WITH STROKE
	{0x0181, 0x0182, propertyDISALLOWED},    // LATIN CAPITAL LETTER B WITH HOOK..LATIN CAPI
	{0x0183, 0x0, propertyPVALID},           // LATIN SMALL LETTER B WITH TOPBAR
	{0x0184, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER TONE SIX
	{0x0185, 0x0, propertyPVALID},           // LATIN SMALL LETTER TONE SIX
	{0x0186, 0x0187, propertyDISALLOWED},    // LATIN CAPITAL LETTER OPEN O..LATIN CAPITAL L
	{0x0188, 0x0, propertyPVALID},           // LATIN SMALL LETTER C WITH HOOK
	{0x0189, 0x018B, propertyDISALLOWED},    // LATIN CAPITAL LETTER AFRICAN D..LATIN CAPITA
	{0x018C, 0x018D, propertyPVALID},        // LATIN SMALL LETTER D WITH TOPBAR..LATIN SMAL
	{0x018E, 0x0191, propertyDISALLOWED},    // LATIN CAPITAL LETTER REVERSED E..LATIN CAPIT
	{0x0192, 0x0, propertyPVALID},           // LATIN SMALL LETTER F WITH HOOK
	{0x0193, 0x0194, propertyDISALLOWED},    // LATIN CAPITAL LETTER G WITH HOOK..LATIN CAPI
	{0x0195, 0x0, propertyPVALID},           // LATIN SMALL LETTER HV
	{0x0196, 0x0198, propertyDISALLOWED},    // LATIN CAPITAL LETTER IOTA..LATIN CAPITAL LET
	{0x0199, 0x019B, propertyPVALID},        // LATIN SMALL LETTER K WITH HOOK..LATIN SMALL
	{0x019C, 0x019D, propertyDISALLOWED},    // LATIN CAPITAL LETTER TURNED M..LATIN CAPITAL
	{0x019E, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH LONG RIGHT LEG
	{0x019F, 0x01A0, propertyDISALLOWED},    // LATIN CAPITAL LETTER O WITH MIDDLE TILDE..LA
	{0x01A1, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH HORN
	{0x01A2, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER OI
	{0x01A3, 0x0, propertyPVALID},           // LATIN SMALL LETTER OI
	{0x01A4, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER P WITH HOOK
	{0x01A5, 0x0, propertyPVALID},           // LATIN SMALL LETTER P WITH HOOK
	{0x01A6, 0x01A7, propertyDISALLOWED},    // LATIN LETTER YR..LATIN CAPITAL LETTER TONE T
	{0x01A8, 0x0, propertyPVALID},           // LATIN SMALL LETTER TONE TWO
	{0x01A9, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER ESH
	{0x01AA, 0x01AB, propertyPVALID},        // LATIN LETTER REVERSED ESH LOOP..LATIN SMALL
	{0x01AC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH HOOK
	{0x01AD, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH HOOK
	{0x01AE, 0x01AF, propertyDISALLOWED},    // LATIN CAPITAL LETTER T WITH RETROFLEX HOOK..
	{0x01B0, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH HORN
	{0x01B1, 0x01B3, propertyDISALLOWED},    // LATIN CAPITAL LETTER UPSILON..LATIN CAPITAL
	{0x01B4, 0x0, propertyPVALID},           // LATIN SMALL LETTER Y WITH HOOK
	{0x01B5, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Z WITH STROKE
	{0x01B6, 0x0, propertyPVALID},           // LATIN SMALL LETTER Z WITH STROKE
	{0x01B7, 0x01B8, propertyDISALLOWED},    // LATIN CAPITAL LETTER EZH..LATIN CAPITAL LETT
	{0x01B9, 0x01BB, propertyPVALID},        // LATIN SMALL LETTER EZH REVERSED..LATIN LETTE
	{0x01BC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER TONE FIVE
	{0x01BD, 0x01C3, propertyPVALID},        // LATIN SMALL LETTER TONE FIVE..LATIN LETTER R
	{0x01C4, 0x01CD, propertyDISALLOWED},    // LATIN CAPITAL LETTER DZ WITH CARON..LATIN CA
	{0x01CE, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH CARON
	{0x01CF, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH CARON
	{0x01D0, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH CARON
	{0x01D1, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH CARON
	{0x01D2, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH CARON
	{0x01D3, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH CARON
	{0x01D4, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH CARON
	{0x01D5, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH DIAERESIS AND MA
	{0x01D6, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH DIAERESIS AND MACR
	{0x01D7, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH DIAERESIS AND AC
	{0x01D8, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH DIAERESIS AND ACUT
	{0x01D9, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH DIAERESIS AND CA
	{0x01DA, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH DIAERESIS AND CARO
	{0x01DB, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH DIAERESIS AND GR
	{0x01DC, 0x01DD, propertyPVALID},        // LATIN SMALL LETTER U WITH DIAERESIS AND GRAV
	{0x01DE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH DIAERESIS AND MA
	{0x01DF, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH DIAERESIS AND MACR
	{0x01E0, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH DOT ABOVE AND MA
	{0x01E1, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH DOT ABOVE AND MACR
	{0x01E2, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER AE WITH MACRON
	{0x01E3, 0x0, propertyPVALID},           // LATIN SMALL LETTER AE WITH MACRON
	{0x01E4, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER G WITH STROKE
	{0x01E5, 0x0, propertyPVALID},           // LATIN SMALL LETTER G WITH STROKE
	{0x01E6, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER G WITH CARON
	{0x01E7, 0x0, propertyPVALID},           // LATIN SMALL LETTER G WITH CARON
	{0x01E8, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH CARON
	{0x01E9, 0x0, propertyPVALID},           // LATIN SMALL LETTER K WITH CARON
	{0x01EA, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH OGONEK
	{0x01EB, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH OGONEK
	{0x01EC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH OGONEK AND MACRO
	{0x01ED, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH OGONEK AND MACRON
	{0x01EE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER EZH WITH CARON
	{0x01EF, 0x01F0, propertyPVALID},        // LATIN SMALL LETTER EZH WITH CARON..LATIN SMA
	{0x01F1, 0x01F4, propertyDISALLOWED},    // LATIN CAPITAL LETTER DZ..LATIN CAPITAL LETTE
	{0x01F5, 0x0, propertyPVALID},           // LATIN SMALL LETTER G WITH ACUTE
	{0x01F6, 0x01F8, propertyDISALLOWED},    // LATIN CAPITAL LETTER HWAIR..LATIN CAPITAL LE
	{0x01F9, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH GRAVE
	{0x01FA, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH RING ABOVE AND A
	{0x01FB, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH RING ABOVE AND ACU
	{0x01FC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER AE WITH ACUTE
	{0x01FD, 0x0, propertyPVALID},           // LATIN SMALL LETTER AE WITH ACUTE
	{0x01FE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
	{0x01FF, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH STROKE AND ACUTE
	{0x0200, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
	{0x0201, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH DOUBLE GRAVE
	{0x0202, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH INVERTED BREVE
	{0x0203, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH INVERTED BREVE
	{0x0204, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
	{0x0205, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH DOUBLE GRAVE
	{0x0206, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH INVERTED BREVE
	{0x0207, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH INVERTED BREVE
	{0x0208, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
	{0x0209, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH DOUBLE GRAVE
	{0x020A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH INVERTED BREVE
	{0x020B, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH INVERTED BREVE
	{0x020C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
	{0x020D, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH DOUBLE GRAVE
	{0x020E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH INVERTED BREVE
	{0x020F, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH INVERTED BREVE
	{0x0210, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
	{0x0211, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH DOUBLE GRAVE
	{0x0212, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH INVERTED BREVE
	{0x0213, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH INVERTED BREVE
	{0x0214, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
	{0x0215, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH DOUBLE GRAVE
	{0x0216, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH INVERTED BREVE
	{0x0217, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH INVERTED BREVE
	{0x0218, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH COMMA BELOW
	{0x0219, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH COMMA BELOW
	{0x021A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH COMMA BELOW
	{0x021B, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH COMMA BELOW
	{0x021C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER YOGH
	{0x021D, 0x0, propertyPVALID},           // LATIN SMALL LETTER YOGH
	{0x021E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH CARON
	{0x021F, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH CARON
	{0x0220, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
	{0x0221, 0x0, propertyPVALID},           // LATIN SMALL LETTER D WITH CURL
	{0x0222, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER OU
	{0x0223, 0x0, propertyPVALID},           // LATIN SMALL LETTER OU
	{0x0224, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Z WITH HOOK
	{0x0225, 0x0, propertyPVALID},           // LATIN SMALL LETTER Z WITH HOOK
	{0x0226, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH DOT ABOVE
	{0x0227, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH DOT ABOVE
	{0x0228, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CEDILLA
	{0x0229, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CEDILLA
	{0x022A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH DIAERESIS AND MA
	{0x022B, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH DIAERESIS AND MACR
	{0x022C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH TILDE AND MACRON
	{0x022D, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH TILDE AND MACRON
	{0x022E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH DOT ABOVE
	{0x022F, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH DOT ABOVE
	{0x0230, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH DOT ABOVE AND MA
	{0x0231, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH DOT ABOVE AND MACR
	{0x0232, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH MACRON
	{0x0233, 0x0239, propertyPVALID},        // LATIN SMALL LETTER Y WITH MACRON..LATIN SMAL
	{0x023A, 0x023B, propertyDISALLOWED},    // LATIN CAPITAL LETTER A WITH STROKE..LATIN CA
	{0x023C, 0x0, propertyPVALID},           // LATIN SMALL LETTER C WITH STROKE
	{0x023D, 0x023E, propertyDISALLOWED},    // LATIN CAPITAL LETTER L WITH BAR..LATIN CAPIT
	{0x023F, 0x0240, propertyPVALID},        // LATIN SMALL LETTER S WITH SWASH TAIL..LATIN
	{0x0241, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER GLOTTAL STOP
	{0x0242, 0x0, propertyPVALID},           // LATIN SMALL LETTER GLOTTAL STOP
	{0x0243, 0x0246, propertyDISALLOWED},    // LATIN CAPITAL LETTER B WITH STROKE..LATIN CA
	{0x0247, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH STROKE
	{0x0248, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER J WITH STROKE
	{0x0249, 0x0, propertyPVALID},           // LATIN SMALL LETTER J WITH STROKE
	{0x024A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
	{0x024B, 0x0, propertyPVALID},           // LATIN SMALL LETTER Q WITH HOOK TAIL
	{0x024C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH STROKE
	{0x024D, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH STROKE
	{0x024E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH STROKE
	{0x024F, 0x02AF, propertyPVALID},        // LATIN SMALL LETTER Y WITH STROKE..LATIN SMAL
	{0x02B0, 0x02B8, propertyDISALLOWED},    // MODIFIER LETTER SMALL H..MODIFIER LETTER SMA
	{0x02B9, 0x02C1, propertyPVALID},        // MODIFIER LETTER PRIME..MODIFIER LETTER REVER
	{0x02C2, 0x02C5, propertyDISALLOWED},    // MODIFIER LETTER LEFT ARROWHEAD..MODIFIER LET
	{0x02C6, 0x02D1, propertyPVALID},        // MODIFIER LETTER CIRCUMFLEX ACCENT..MODIFIER
	{0x02D2, 0x02EB, propertyDISALLOWED},    // MODIFIER LETTER CENTRED RIGHT HALF RING..MOD
	{0x02EC, 0x0, propertyPVALID},           // MODIFIER LETTER VOICING
	{0x02ED, 0x0, propertyDISALLOWED},       // MODIFIER LETTER UNASPIRATED
	{0x02EE, 0x0, propertyPVALID},           // MODIFIER LETTER DOUBLE APOSTROPHE
	{0x02EF, 0x02FF, propertyDISALLOWED},    // MODIFIER LETTER LOW DOWN ARROWHEAD..MODIFIER
	{0x0300, 0x033F, propertyPVALID},        // COMBINING GRAVE ACCENT..COMBINING DOUBLE OVE
	{0x0340, 0x0341, propertyDISALLOWED},    // COMBINING GRAVE TONE MARK..COMBINING ACUTE T
	{0x0342, 0x0, propertyPVALID},           // COMBINING GREEK PERISPOMENI
	{0x0343, 0x0345, propertyDISALLOWED},    // COMBINING GREEK KORONIS..COMBINING GREEK YPO
	{0x0346, 0x034E, propertyPVALID},        // COMBINING BRIDGE ABOVE..COMBINING UPWARDS AR
	{0x034F, 0x0, propertyDISALLOWED},       // COMBINING GRAPHEME JOINER
	{0x0350, 0x036F, propertyPVALID},        // COMBINING RIGHT ARROWHEAD ABOVE..COMBINING L
	{0x0370, 0x0, propertyDISALLOWED},       // GREEK CAPITAL LETTER HETA
	{0x0371, 0x0, propertyPVALID},           // GREEK SMALL LETTER HETA
	{0x0372, 0x0, propertyDISALLOWED},       // GREEK CAPITAL LETTER ARCHAIC SAMPI
	{0x0373, 0x0, propertyPVALID},           // GREEK SMALL LETTER ARCHAIC SAMPI
	{0x0374, 0x0, propertyDISALLOWED},       // GREEK NUMERAL SIGN
	{0x0375, 0x0, propertyCONTEXTO},         // GREEK LOWER NUMERAL SIGN
	{0x0376, 0x0, propertyDISALLOWED},       // GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
	{0x0377, 0x0, propertyPVALID},           // GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
	{0x0378, 0x0379, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x037A, 0x0, propertyDISALLOWED},       // GREEK YPOGEGRAMMENI
	{0x037B, 0x037D, propertyPVALID},        // GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GR
	{0x037E, 0x0, propertyDISALLOWED},       // GREEK QUESTION MARK
	{0x037F, 0x0383, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0384, 0x038A, propertyDISALLOWED},    // GREEK TONOS..GREEK CAPITAL LETTER IOTA WITH
	{0x038B, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x038C, 0x0, propertyDISALLOWED},       // GREEK CAPITAL LETTER OMICRON WITH TONOS
	{0x038D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x038E, 0x038F, propertyDISALLOWED},    // GREEK CAPITAL LETTER UPSILON WITH TONOS..GRE
	{0x0390, 0x0, propertyPVALID},           // GREEK SMALL LETTER IOTA WITH DIALYTIKA AND T
	{0x0391, 0x03A1, propertyDISALLOWED},    // GREEK CAPITAL LETTER ALPHA..GREEK CAPITAL LE
	{0x03A2, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x03A3, 0x03AB, propertyDISALLOWED},    // GREEK CAPITAL LETTER SIGMA..GREEK CAPITAL LE
	{0x03AC, 0x03CE, propertyPVALID},        // GREEK SMALL LETTER ALPHA WITH TONOS..GREEK S
	{0x03CF, 0x03D6, propertyDISALLOWED},    // GREEK CAPITAL KAI SYMBOL..GREEK PI SYMBOL
	{0x03D7, 0x0, propertyPVALID},           // GREEK KAI SYMBOL
	{0x03D8, 0x0, propertyDISALLOWED},       // GREEK LETTER ARCHAIC KOPPA
	{0x03D9, 0x0, propertyPVALID},           // GREEK SMALL LETTER ARCHAIC KOPPA
	{0x03DA, 0x0, propertyDISALLOWED},       // GREEK LETTER STIGMA
	{0x03DB, 0x0, propertyPVALID},           // GREEK SMALL LETTER STIGMA
	{0x03DC, 0x0, propertyDISALLOWED},       // GREEK LETTER DIGAMMA
	{0x03DD, 0x0, propertyPVALID},           // GREEK SMALL LETTER DIGAMMA
	{0x03DE, 0x0, propertyDISALLOWED},       // GREEK LETTER KOPPA
	{0x03DF, 0x0, propertyPVALID},           // GREEK SMALL LETTER KOPPA
	{0x03E0, 0x0, propertyDISALLOWED},       // GREEK LETTER SAMPI
	{0x03E1, 0x0, propertyPVALID},           // GREEK SMALL LETTER SAMPI
	{0x03E2, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER SHEI
	{0x03E3, 0x0, propertyPVALID},           // COPTIC SMALL LETTER SHEI
	{0x03E4, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER FEI
	{0x03E5, 0x0, propertyPVALID},           // COPTIC SMALL LETTER FEI
	{0x03E6, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER KHEI
	{0x03E7, 0x0, propertyPVALID},           // COPTIC SMALL LETTER KHEI
	{0x03E8, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER HORI
	{0x03E9, 0x0, propertyPVALID},           // COPTIC SMALL LETTER HORI
	{0x03EA, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER GANGIA
	{0x03EB, 0x0, propertyPVALID},           // COPTIC SMALL LETTER GANGIA
	{0x03EC, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER SHIMA
	{0x03ED, 0x0, propertyPVALID},           // COPTIC SMALL LETTER SHIMA
	{0x03EE, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER DEI
	{0x03EF, 0x0, propertyPVALID},           // COPTIC SMALL LETTER DEI
	{0x03F0, 0x03F2, propertyDISALLOWED},    // GREEK KAPPA SYMBOL..GREEK LUNATE SIGMA SYMBO
	{0x03F3, 0x0, propertyPVALID},           // GREEK LETTER YOT
	{0x03F4, 0x03F7, propertyDISALLOWED},    // GREEK CAPITAL THETA SYMBOL..GREEK CAPITAL LE
	{0x03F8, 0x0, propertyPVALID},           // GREEK SMALL LETTER SHO
	{0x03F9, 0x03FA, propertyDISALLOWED},    // GREEK CAPITAL LUNATE SIGMA SYMBOL..GREEK CAP
	{0x03FB, 0x03FC, propertyPVALID},        // GREEK SMALL LETTER SAN..GREEK RHO WITH STROK
	{0x03FD, 0x042F, propertyDISALLOWED},    // GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL..
	{0x0430, 0x045F, propertyPVALID},        // CYRILLIC SMALL LETTER A..CYRILLIC SMALL LETT
	{0x0460, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER OMEGA
	{0x0461, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER OMEGA
	{0x0462, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER YAT
	{0x0463, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER YAT
	{0x0464, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IOTIFIED E
	{0x0465, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IOTIFIED E
	{0x0466, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER LITTLE YUS
	{0x0467, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER LITTLE YUS
	{0x0468, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
	{0x0469, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS
	{0x046A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER BIG YUS
	{0x046B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER BIG YUS
	{0x046C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
	{0x046D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IOTIFIED BIG YUS
	{0x046E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KSI
	{0x046F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KSI
	{0x0470, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER PSI
	{0x0471, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER PSI
	{0x0472, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER FITA
	{0x0473, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER FITA
	{0x0474, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IZHITSA
	{0x0475, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IZHITSA
	{0x0476, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE
	{0x0477, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GR
	{0x0478, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER UK
	{0x0479, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER UK
	{0x047A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ROUND OMEGA
	{0x047B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ROUND OMEGA
	{0x047C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
	{0x047D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER OMEGA WITH TITLO
	{0x047E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER OT
	{0x047F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER OT
	{0x0480, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOPPA
	{0x0481, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOPPA
	{0x0482, 0x0, propertyDISALLOWED},       // CYRILLIC THOUSANDS SIGN
	{0x0483, 0x0487, propertyPVALID},        // COMBINING CYRILLIC TITLO..COMBINING CYRILLIC
	{0x0488, 0x048A, propertyDISALLOWED},    // COMBINING CYRILLIC HUNDRED THOUSANDS SIGN..C
	{0x048B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SHORT I WITH TAIL
	{0x048C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER SEMISOFT SIGN
	{0x048D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SEMISOFT SIGN
	{0x048E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ER WITH TICK
	{0x048F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ER WITH TICK
	{0x0490, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER GHE WITH UPTURN
	{0x0491, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER GHE WITH UPTURN
	{0x0492, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER GHE WITH STROKE
	{0x0493, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER GHE WITH STROKE
	{0x0494, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
	{0x0495, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
	{0x0496, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
	{0x0497, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ZHE WITH DESCENDER
	{0x0498, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
	{0x0499, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ZE WITH DESCENDER
	{0x049A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KA WITH DESCENDER
	{0x049B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KA WITH DESCENDER
	{0x049C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KA WITH VERTICAL STR
	{0x049D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KA WITH VERTICAL STROK
	{0x049E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KA WITH STROKE
	{0x049F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KA WITH STROKE
	{0x04A0, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER BASHKIR KA
	{0x04A1, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER BASHKIR KA
	{0x04A2, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER EN WITH DESCENDER
	{0x04A3, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER EN WITH DESCENDER
	{0x04A4, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LIGATURE EN GHE
	{0x04A5, 0x0, propertyPVALID},           // CYRILLIC SMALL LIGATURE EN GHE
	{0x04A6, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
	{0x04A7, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
	{0x04A8, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ABKHASIAN HA
	{0x04A9, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ABKHASIAN HA
	{0x04AA, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ES WITH DESCENDER
	{0x04AB, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ES WITH DESCENDER
	{0x04AC, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER TE WITH DESCENDER
	{0x04AD, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER TE WITH DESCENDER
	{0x04AE, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER STRAIGHT U
	{0x04AF, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER STRAIGHT U
	{0x04B0, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER STRAIGHT U WITH STRO
	{0x04B1, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
	{0x04B2, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER HA WITH DESCENDER
	{0x04B3, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER HA WITH DESCENDER
	{0x04B4, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LIGATURE TE TSE
	{0x04B5, 0x0, propertyPVALID},           // CYRILLIC SMALL LIGATURE TE TSE
	{0x04B6, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
	{0x04B7, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER CHE WITH DESCENDER
	{0x04B8, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER CHE WITH VERTICAL ST
	{0x04B9, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER CHE WITH VERTICAL STRO
	{0x04BA, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER SHHA
	{0x04BB, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SHHA
	{0x04BC, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ABKHASIAN CHE
	{0x04BD, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ABKHASIAN CHE
	{0x04BE, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH D
	{0x04BF, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DES
	{0x04C0, 0x04C1, propertyDISALLOWED},    // CYRILLIC LETTER PALOCHKA..CYRILLIC CAPITAL L
	{0x04C2, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ZHE WITH BREVE
	{0x04C3, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KA WITH HOOK
	{0x04C4, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KA WITH HOOK
	{0x04C5, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER EL WITH TAIL
	{0x04C6, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER EL WITH TAIL
	{0x04C7, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER EN WITH HOOK
	{0x04C8, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER EN WITH HOOK
	{0x04C9, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER EN WITH TAIL
	{0x04CA, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER EN WITH TAIL
	{0x04CB, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
	{0x04CC, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KHAKASSIAN CHE
	{0x04CD, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER EM WITH TAIL
	{0x04CE, 0x04CF, propertyPVALID},        // CYRILLIC SMALL LETTER EM WITH TAIL..CYRILLIC
	{0x04D0, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER A WITH BREVE
	{0x04D1, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER A WITH BREVE
	{0x04D2, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER A WITH DIAERESIS
	{0x04D3, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER A WITH DIAERESIS
	{0x04D4, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LIGATURE A IE
	{0x04D5, 0x0, propertyPVALID},           // CYRILLIC SMALL LIGATURE A IE
	{0x04D6, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IE WITH BREVE
	{0x04D7, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IE WITH BREVE
	{0x04D8, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER SCHWA
	{0x04D9, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SCHWA
	{0x04DA, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
	{0x04DB, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
	{0x04DC, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
	{0x04DD, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
	{0x04DE, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
	{0x04DF, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ZE WITH DIAERESIS
	{0x04E0, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ABKHASIAN DZE
	{0x04E1, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ABKHASIAN DZE
	{0x04E2, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER I WITH MACRON
	{0x04E3, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER I WITH MACRON
	{0x04E4, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER I WITH DIAERESIS
	{0x04E5, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER I WITH DIAERESIS
	{0x04E6, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER O WITH DIAERESIS
	{0x04E7, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER O WITH DIAERESIS
	{0x04E8, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER BARRED O
	{0x04E9, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER BARRED O
	{0x04EA, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER BARRED O WITH DIAERE
	{0x04EB, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER BARRED O WITH DIAERESI
	{0x04EC, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER E WITH DIAERESIS
	{0x04ED, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER E WITH DIAERESIS
	{0x04EE, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER U WITH MACRON
	{0x04EF, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER U WITH MACRON
	{0x04F0, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER U WITH DIAERESIS
	{0x04F1, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER U WITH DIAERESIS
	{0x04F2, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
	{0x04F3, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
	{0x04F4, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
	{0x04F5, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER CHE WITH DIAERESIS
	{0x04F6, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
	{0x04F7, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER GHE WITH DESCENDER
	{0x04F8, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
	{0x04F9, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER YERU WITH DIAERESIS
	{0x04FA, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER GHE WITH STROKE AND
	{0x04FB, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER GHE WITH STROKE AND HO
	{0x04FC, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER HA WITH HOOK
	{0x04FD, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER HA WITH HOOK
	{0x04FE, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER HA WITH STROKE
	{0x04FF, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER HA WITH STROKE
	{0x0500, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOMI DE
	{0x0501, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOMI DE
	{0x0502, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOMI DJE
	{0x0503, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOMI DJE
	{0x0504, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOMI ZJE
	{0x0505, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOMI ZJE
	{0x0506, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOMI DZJE
	{0x0507, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOMI DZJE
	{0x0508, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOMI LJE
	{0x0509, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOMI LJE
	{0x050A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOMI NJE
	{0x050B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOMI NJE
	{0x050C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOMI SJE
	{0x050D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOMI SJE
	{0x050E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER KOMI TJE
	{0x050F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER KOMI TJE
	{0x0510, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER REVERSED ZE
	{0x0511, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER REVERSED ZE
	{0x0512, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER EL WITH HOOK
	{0x0513, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER EL WITH HOOK
	{0x0514, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER LHA
	{0x0515, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER LHA
	{0x0516, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER RHA
	{0x0517, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER RHA
	{0x0518, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER YAE
	{0x0519, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER YAE
	{0x051A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER QA
	{0x051B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER QA
	{0x051C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER WE
	{0x051D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER WE
	{0x051E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ALEUT KA
	{0x051F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ALEUT KA
	{0x0520, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
	{0x0521, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER EL WITH MIDDLE HOOK
	{0x0522, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
	{0x0523, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER EN WITH MIDDLE HOOK
	{0x0524, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER PE WITH DESCENDER
	{0x0525, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER PE WITH DESCENDER
	{0x0526, 0x0530, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0531, 0x0556, propertyDISALLOWED},    // ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITA
	{0x0557, 0x0558, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0559, 0x0, propertyPVALID},           // ARMENIAN MODIFIER LETTER LEFT HALF RING
	{0x055A, 0x055F, propertyDISALLOWED},    // ARMENIAN APOSTROPHE..ARMENIAN ABBREVIATION M
	{0x0560, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0561, 0x0586, propertyPVALID},        // ARMENIAN SMALL LETTER AYB..ARMENIAN SMALL LE
	{0x0587, 0x0, propertyDISALLOWED},       // ARMENIAN SMALL LIGATURE ECH YIWN
	{0x0588, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0589, 0x058A, propertyDISALLOWED},    // ARMENIAN FULL STOP..ARMENIAN HYPHEN
	{0x058B, 0x0590, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0591, 0x05BD, propertyPVALID},        // HEBREW ACCENT ETNAHTA..HEBREW POINT METEG
	{0x05BE, 0x0, propertyDISALLOWED},       // HEBREW PUNCTUATION MAQAF
	{0x05BF, 0x0, propertyPVALID},           // HEBREW POINT RAFE
	{0x05C0, 0x0, propertyDISALLOWED},       // HEBREW PUNCTUATION PASEQ
	{0x05C1, 0x05C2, propertyPVALID},        // HEBREW POINT SHIN DOT..HEBREW POINT SIN DOT
	{0x05C3, 0x0, propertyDISALLOWED},       // HEBREW PUNCTUATION SOF PASUQ
	{0x05C4, 0x05C5, propertyPVALID},        // HEBREW MARK UPPER DOT..HEBREW MARK LOWER DOT
	{0x05C6, 0x0, propertyDISALLOWED},       // HEBREW PUNCTUATION NUN HAFUKHA
	{0x05C7, 0x0, propertyPVALID},           // HEBREW POINT QAMATS QATAN
	{0x05C8, 0x05CF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x05D0, 0x05EA, propertyPVALID},        // HEBREW LETTER ALEF..HEBREW LETTER TAV
	{0x05EB, 0x05EF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x05F0, 0x05F2, propertyPVALID},        // HEBREW LIGATURE YIDDISH DOUBLE VAV..HEBREW L
	{0x05F3, 0x05F4, propertyCONTEXTO},      // HEBREW PUNCTUATION GERESH..HEBREW PUNCTUATIO
	{0x05F5, 0x05FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0600, 0x0603, propertyDISALLOWED},    // ARABIC NUMBER SIGN..ARABIC SIGN SAFHA
	{0x0604, 0x0605, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0606, 0x060F, propertyDISALLOWED},    // ARABIC-INDIC CUBE ROOT..ARABIC SIGN MISRA
	{0x0610, 0x061A, propertyPVALID},        // ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM..AR
	{0x061B, 0x0, propertyDISALLOWED},       // ARABIC SEMICOLON
	{0x061C, 0x061D, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x061E, 0x061F, propertyDISALLOWED},    // ARABIC TRIPLE DOT PUNCTUATION MARK..ARABIC Q
	{0x0620, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0621, 0x063F, propertyPVALID},        // ARABIC LETTER HAMZA..ARABIC LETTER FARSI YEH
	{0x0640, 0x0, propertyDISALLOWED},       // ARABIC TATWEEL
	{0x0641, 0x065E, propertyPVALID},        // ARABIC LETTER FEH..ARABIC FATHA WITH TWO DOT
	{0x065F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0660, 0x0669, propertyCONTEXTO},      // ARABIC-INDIC DIGIT ZERO..ARABIC-INDIC DIGIT
	{0x066A, 0x066D, propertyDISALLOWED},    // ARABIC PERCENT SIGN..ARABIC FIVE POINTED STA
	{0x066E, 0x0674, propertyPVALID},        // ARABIC LETTER DOTLESS BEH..ARABIC LETTER HIG
	{0x0675, 0x0678, propertyDISALLOWED},    // ARABIC LETTER HIGH HAMZA ALEF..ARABIC LETTER
	{0x0679, 0x06D3, propertyPVALID},        // ARABIC LETTER TTEH..ARABIC LETTER YEH BARREE
	{0x06D4, 0x0, propertyDISALLOWED},       // ARABIC FULL STOP
	{0x06D5, 0x06DC, propertyPVALID},        // ARABIC LETTER AE..ARABIC SMALL HIGH SEEN
	{0x06DD, 0x06DE, propertyDISALLOWED},    // ARABIC END OF AYAH..ARABIC START OF RUB EL H
	{0x06DF, 0x06E8, propertyPVALID},        // ARABIC SMALL HIGH ROUNDED ZERO..ARABIC SMALL
	{0x06E9, 0x0, propertyDISALLOWED},       // ARABIC PLACE OF SAJDAH
	{0x06EA, 0x06EF, propertyPVALID},        // ARABIC EMPTY CENTRE LOW STOP..ARABIC LETTER
	{0x06F0, 0x06F9, propertyCONTEXTO},      // EXTENDED ARABIC-INDIC DIGIT ZERO..EXTENDED A
	{0x06FA, 0x06FF, propertyPVALID},        // ARABIC LETTER SHEEN WITH DOT BELOW..ARABIC L
	{0x0700, 0x070D, propertyDISALLOWED},    // SYRIAC END OF PARAGRAPH..SYRIAC HARKLEAN AST
	{0x070E, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x070F, 0x0, propertyDISALLOWED},       // SYRIAC ABBREVIATION MARK
	{0x0710, 0x074A, propertyPVALID},        // SYRIAC LETTER ALAPH..SYRIAC BARREKH
	{0x074B, 0x074C, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x074D, 0x07B1, propertyPVALID},        // SYRIAC LETTER SOGDIAN ZHAIN..THAANA LETTER N
	{0x07B2, 0x07BF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x07C0, 0x07F5, propertyPVALID},        // NKO DIGIT ZERO..NKO LOW TONE APOSTROPHE
	{0x07F6, 0x07FA, propertyDISALLOWED},    // NKO SYMBOL OO DENNEN..NKO LAJANYALAN
	{0x07FB, 0x07FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0800, 0x082D, propertyPVALID},        // SAMARITAN LETTER ALAF..SAMARITAN MARK NEQUDA
	{0x082E, 0x082F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0830, 0x083E, propertyDISALLOWED},    // SAMARITAN PUNCTUATION NEQUDAA..SAMARITAN PUN
	{0x083F, 0x08FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0900, 0x0939, propertyPVALID},        // DEVANAGARI SIGN INVERTED CANDRABINDU..DEVANA
	{0x093A, 0x093B, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x093C, 0x094E, propertyPVALID},        // DEVANAGARI SIGN NUKTA..DEVANAGARI VOWEL SIGN
	{0x094F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0950, 0x0955, propertyPVALID},        // DEVANAGARI OM..DEVANAGARI VOWEL SIGN CANDRA
	{0x0956, 0x0957, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0958, 0x095F, propertyDISALLOWED},    // DEVANAGARI LETTER QA..DEVANAGARI LETTER YYA
	{0x0960, 0x0963, propertyPVALID},        // DEVANAGARI LETTER VOCALIC RR..DEVANAGARI VOW
	{0x0964, 0x0965, propertyDISALLOWED},    // DEVANAGARI DANDA..DEVANAGARI DOUBLE DANDA
	{0x0966, 0x096F, propertyPVALID},        // DEVANAGARI DIGIT ZERO..DEVANAGARI DIGIT NINE
	{0x0970, 0x0, propertyDISALLOWED},       // DEVANAGARI ABBREVIATION SIGN
	{0x0971, 0x0972, propertyPVALID},        // DEVANAGARI SIGN HIGH SPACING DOT..DEVANAGARI
	{0x0973, 0x0978, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0979, 0x097F, propertyPVALID},        // DEVANAGARI LETTER ZHA..DEVANAGARI LETTER BBA
	{0x0980, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0981, 0x0983, propertyPVALID},        // BENGALI SIGN CANDRABINDU..BENGALI SIGN VISAR
	{0x0984, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0985, 0x098C, propertyPVALID},        // BENGALI LETTER A..BENGALI LETTER VOCALIC L
	{0x098D, 0x098E, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x098F, 0x0990, propertyPVALID},        // BENGALI LETTER E..BENGALI LETTER AI
	{0x0991, 0x0992, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0993, 0x09A8, propertyPVALID},        // BENGALI LETTER O..BENGALI LETTER NA
	{0x09A9, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x09AA, 0x09B0, propertyPVALID},        // BENGALI LETTER PA..BENGALI LETTER RA
	{0x09B1, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x09B2, 0x0, propertyPVALID},           // BENGALI LETTER LA
	{0x09B3, 0x09B5, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x09B6, 0x09B9, propertyPVALID},        // BENGALI LETTER SHA..BENGALI LETTER HA
	{0x09BA, 0x09BB, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x09BC, 0x09C4, propertyPVALID},        // BENGALI SIGN NUKTA..BENGALI VOWEL SIGN VOCAL
	{0x09C5, 0x09C6, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x09C7, 0x09C8, propertyPVALID},        // BENGALI VOWEL SIGN E..BENGALI VOWEL SIGN AI
	{0x09C9, 0x09CA, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x09CB, 0x09CE, propertyPVALID},        // BENGALI VOWEL SIGN O..BENGALI LETTER KHANDA
	{0x09CF, 0x09D6, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x09D7, 0x0, propertyPVALID},           // BENGALI AU LENGTH MARK
	{0x09D8, 0x09DB, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x09DC, 0x09DD, propertyDISALLOWED},    // BENGALI LETTER RRA..BENGALI LETTER RHA
	{0x09DE, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x09DF, 0x0, propertyDISALLOWED},       // BENGALI LETTER YYA
	{0x09E0, 0x09E3, propertyPVALID},        // BENGALI LETTER VOCALIC RR..BENGALI VOWEL SIG
	{0x09E4, 0x09E5, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x09E6, 0x09F1, propertyPVALID},        // BENGALI DIGIT ZERO..BENGALI LETTER RA WITH L
	{0x09F2, 0x09FB, propertyDISALLOWED},    // BENGALI RUPEE MARK..BENGALI GANDA MARK
	{0x09FC, 0x0A00, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A01, 0x0A03, propertyPVALID},        // GURMUKHI SIGN ADAK BINDI..GURMUKHI SIGN VISA
	{0x0A04, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A05, 0x0A0A, propertyPVALID},        // GURMUKHI LETTER A..GURMUKHI LETTER UU
	{0x0A0B, 0x0A0E, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A0F, 0x0A10, propertyPVALID},        // GURMUKHI LETTER EE..GURMUKHI LETTER AI
	{0x0A11, 0x0A12, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A13, 0x0A28, propertyPVALID},        // GURMUKHI LETTER OO..GURMUKHI LETTER NA
	{0x0A29, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A2A, 0x0A30, propertyPVALID},        // GURMUKHI LETTER PA..GURMUKHI LETTER RA
	{0x0A31, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A32, 0x0, propertyPVALID},           // GURMUKHI LETTER LA
	{0x0A33, 0x0, propertyDISALLOWED},       // GURMUKHI LETTER LLA
	{0x0A34, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A35, 0x0, propertyPVALID},           // GURMUKHI LETTER VA
	{0x0A36, 0x0, propertyDISALLOWED},       // GURMUKHI LETTER SHA
	{0x0A37, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A38, 0x0A39, propertyPVALID},        // GURMUKHI LETTER SA..GURMUKHI LETTER HA
	{0x0A3A, 0x0A3B, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A3C, 0x0, propertyPVALID},           // GURMUKHI SIGN NUKTA
	{0x0A3D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A3E, 0x0A42, propertyPVALID},        // GURMUKHI VOWEL SIGN AA..GURMUKHI VOWEL SIGN
	{0x0A43, 0x0A46, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A47, 0x0A48, propertyPVALID},        // GURMUKHI VOWEL SIGN EE..GURMUKHI VOWEL SIGN
	{0x0A49, 0x0A4A, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A4B, 0x0A4D, propertyPVALID},        // GURMUKHI VOWEL SIGN OO..GURMUKHI SIGN VIRAMA
	{0x0A4E, 0x0A50, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A51, 0x0, propertyPVALID},           // GURMUKHI SIGN UDAAT
	{0x0A52, 0x0A58, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A59, 0x0A5B, propertyDISALLOWED},    // GURMUKHI LETTER KHHA..GURMUKHI LETTER ZA
	{0x0A5C, 0x0, propertyPVALID},           // GURMUKHI LETTER RRA
	{0x0A5D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A5E, 0x0, propertyDISALLOWED},       // GURMUKHI LETTER FA
	{0x0A5F, 0x0A65, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A66, 0x0A75, propertyPVALID},        // GURMUKHI DIGIT ZERO..GURMUKHI SIGN YAKASH
	{0x0A76, 0x0A80, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0A81, 0x0A83, propertyPVALID},        // GUJARATI SIGN CANDRABINDU..GUJARATI SIGN VIS
	{0x0A84, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A85, 0x0A8D, propertyPVALID},        // GUJARATI LETTER A..GUJARATI VOWEL CANDRA E
	{0x0A8E, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A8F, 0x0A91, propertyPVALID},        // GUJARATI LETTER E..GUJARATI VOWEL CANDRA O
	{0x0A92, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0A93, 0x0AA8, propertyPVALID},        // GUJARATI LETTER O..GUJARATI LETTER NA
	{0x0AA9, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0AAA, 0x0AB0, propertyPVALID},        // GUJARATI LETTER PA..GUJARATI LETTER RA
	{0x0AB1, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0AB2, 0x0AB3, propertyPVALID},        // GUJARATI LETTER LA..GUJARATI LETTER LLA
	{0x0AB4, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0AB5, 0x0AB9, propertyPVALID},        // GUJARATI LETTER VA..GUJARATI LETTER HA
	{0x0ABA, 0x0ABB, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0ABC, 0x0AC5, propertyPVALID},        // GUJARATI SIGN NUKTA..GUJARATI VOWEL SIGN CAN
	{0x0AC6, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0AC7, 0x0AC9, propertyPVALID},        // GUJARATI VOWEL SIGN E..GUJARATI VOWEL SIGN C
	{0x0ACA, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0ACB, 0x0ACD, propertyPVALID},        // GUJARATI VOWEL SIGN O..GUJARATI SIGN VIRAMA
	{0x0ACE, 0x0ACF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0AD0, 0x0, propertyPVALID},           // GUJARATI OM
	{0x0AD1, 0x0ADF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0AE0, 0x0AE3, propertyPVALID},        // GUJARATI LETTER VOCALIC RR..GUJARATI VOWEL S
	{0x0AE4, 0x0AE5, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0AE6, 0x0AEF, propertyPVALID},        // GUJARATI DIGIT ZERO..GUJARATI DIGIT NINE
	{0x0AF0, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0AF1, 0x0, propertyDISALLOWED},       // GUJARATI RUPEE SIGN
	{0x0AF2, 0x0B00, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B01, 0x0B03, propertyPVALID},        // ORIYA SIGN CANDRABINDU..ORIYA SIGN VISARGA
	{0x0B04, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B05, 0x0B0C, propertyPVALID},        // ORIYA LETTER A..ORIYA LETTER VOCALIC L
	{0x0B0D, 0x0B0E, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B0F, 0x0B10, propertyPVALID},        // ORIYA LETTER E..ORIYA LETTER AI
	{0x0B11, 0x0B12, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B13, 0x0B28, propertyPVALID},        // ORIYA LETTER O..ORIYA LETTER NA
	{0x0B29, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B2A, 0x0B30, propertyPVALID},        // ORIYA LETTER PA..ORIYA LETTER RA
	{0x0B31, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B32, 0x0B33, propertyPVALID},        // ORIYA LETTER LA..ORIYA LETTER LLA
	{0x0B34, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B35, 0x0B39, propertyPVALID},        // ORIYA LETTER VA..ORIYA LETTER HA
	{0x0B3A, 0x0B3B, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B3C, 0x0B44, propertyPVALID},        // ORIYA SIGN NUKTA..ORIYA VOWEL SIGN VOCALIC R
	{0x0B45, 0x0B46, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B47, 0x0B48, propertyPVALID},        // ORIYA VOWEL SIGN E..ORIYA VOWEL SIGN AI
	{0x0B49, 0x0B4A, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B4B, 0x0B4D, propertyPVALID},        // ORIYA VOWEL SIGN O..ORIYA SIGN VIRAMA
	{0x0B4E, 0x0B55, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B56, 0x0B57, propertyPVALID},        // ORIYA AI LENGTH MARK..ORIYA AU LENGTH MARK
	{0x0B58, 0x0B5B, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B5C, 0x0B5D, propertyDISALLOWED},    // ORIYA LETTER RRA..ORIYA LETTER RHA
	{0x0B5E, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B5F, 0x0B63, propertyPVALID},        // ORIYA LETTER YYA..ORIYA VOWEL SIGN VOCALIC L
	{0x0B64, 0x0B65, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B66, 0x0B6F, propertyPVALID},        // ORIYA DIGIT ZERO..ORIYA DIGIT NINE
	{0x0B70, 0x0, propertyDISALLOWED},       // ORIYA ISSHAR
	{0x0B71, 0x0, propertyPVALID},           // ORIYA LETTER WA
	{0x0B72, 0x0B81, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B82, 0x0B83, propertyPVALID},        // TAMIL SIGN ANUSVARA..TAMIL SIGN VISARGA
	{0x0B84, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B85, 0x0B8A, propertyPVALID},        // TAMIL LETTER A..TAMIL LETTER UU
	{0x0B8B, 0x0B8D, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B8E, 0x0B90, propertyPVALID},        // TAMIL LETTER E..TAMIL LETTER AI
	{0x0B91, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B92, 0x0B95, propertyPVALID},        // TAMIL LETTER O..TAMIL LETTER KA
	{0x0B96, 0x0B98, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0B99, 0x0B9A, propertyPVALID},        // TAMIL LETTER NGA..TAMIL LETTER CA
	{0x0B9B, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B9C, 0x0, propertyPVALID},           // TAMIL LETTER JA
	{0x0B9D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0B9E, 0x0B9F, propertyPVALID},        // TAMIL LETTER NYA..TAMIL LETTER TTA
	{0x0BA0, 0x0BA2, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0BA3, 0x0BA4, propertyPVALID},        // TAMIL LETTER NNA..TAMIL LETTER TA
	{0x0BA5, 0x0BA7, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0BA8, 0x0BAA, propertyPVALID},        // TAMIL LETTER NA..TAMIL LETTER PA
	{0x0BAB, 0x0BAD, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0BAE, 0x0BB9, propertyPVALID},        // TAMIL LETTER MA..TAMIL LETTER HA
	{0x0BBA, 0x0BBD, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0BBE, 0x0BC2, propertyPVALID},        // TAMIL VOWEL SIGN AA..TAMIL VOWEL SIGN UU
	{0x0BC3, 0x0BC5, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0BC6, 0x0BC8, propertyPVALID},        // TAMIL VOWEL SIGN E..TAMIL VOWEL SIGN AI
	{0x0BC9, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0BCA, 0x0BCD, propertyPVALID},        // TAMIL VOWEL SIGN O..TAMIL SIGN VIRAMA
	{0x0BCE, 0x0BCF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0BD0, 0x0, propertyPVALID},           // TAMIL OM
	{0x0BD1, 0x0BD6, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0BD7, 0x0, propertyPVALID},           // TAMIL AU LENGTH MARK
	{0x0BD8, 0x0BE5, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0BE6, 0x0BEF, propertyPVALID},        // TAMIL DIGIT ZERO..TAMIL DIGIT NINE
	{0x0BF0, 0x0BFA, propertyDISALLOWED},    // TAMIL NUMBER TEN..TAMIL NUMBER SIGN
	{0x0BFB, 0x0C00, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0C01, 0x0C03, propertyPVALID},        // TELUGU SIGN CANDRABINDU..TELUGU SIGN VISARGA
	{0x0C04, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C05, 0x0C0C, propertyPVALID},        // TELUGU LETTER A..TELUGU LETTER VOCALIC L
	{0x0C0D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C0E, 0x0C10, propertyPVALID},        // TELUGU LETTER E..TELUGU LETTER AI
	{0x0C11, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C12, 0x0C28, propertyPVALID},        // TELUGU LETTER O..TELUGU LETTER NA
	{0x0C29, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C2A, 0x0C33, propertyPVALID},        // TELUGU LETTER PA..TELUGU LETTER LLA
	{0x0C34, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C35, 0x0C39, propertyPVALID},        // TELUGU LETTER VA..TELUGU LETTER HA
	{0x0C3A, 0x0C3C, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0C3D, 0x0C44, propertyPVALID},        // TELUGU SIGN AVAGRAHA..TELUGU VOWEL SIGN VOCA
	{0x0C45, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C46, 0x0C48, propertyPVALID},        // TELUGU VOWEL SIGN E..TELUGU VOWEL SIGN AI
	{0x0C49, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C4A, 0x0C4D, propertyPVALID},        // TELUGU VOWEL SIGN O..TELUGU SIGN VIRAMA
	{0x0C4E, 0x0C54, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0C55, 0x0C56, propertyPVALID},        // TELUGU LENGTH MARK..TELUGU AI LENGTH MARK
	{0x0C57, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C58, 0x0C59, propertyPVALID},        // TELUGU LETTER TSA..TELUGU LETTER DZA
	{0x0C5A, 0x0C5F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0C60, 0x0C63, propertyPVALID},        // TELUGU LETTER VOCALIC RR..TELUGU VOWEL SIGN
	{0x0C64, 0x0C65, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0C66, 0x0C6F, propertyPVALID},        // TELUGU DIGIT ZERO..TELUGU DIGIT NINE
	{0x0C70, 0x0C77, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0C78, 0x0C7F, propertyDISALLOWED},    // TELUGU FRACTION DIGIT ZERO FOR ODD POWERS OF
	{0x0C80, 0x0C81, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0C82, 0x0C83, propertyPVALID},        // KANNADA SIGN ANUSVARA..KANNADA SIGN VISARGA
	{0x0C84, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C85, 0x0C8C, propertyPVALID},        // KANNADA LETTER A..KANNADA LETTER VOCALIC L
	{0x0C8D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C8E, 0x0C90, propertyPVALID},        // KANNADA LETTER E..KANNADA LETTER AI
	{0x0C91, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0C92, 0x0CA8, propertyPVALID},        // KANNADA LETTER O..KANNADA LETTER NA
	{0x0CA9, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0CAA, 0x0CB3, propertyPVALID},        // KANNADA LETTER PA..KANNADA LETTER LLA
	{0x0CB4, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0CB5, 0x0CB9, propertyPVALID},        // KANNADA LETTER VA..KANNADA LETTER HA
	{0x0CBA, 0x0CBB, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0CBC, 0x0CC4, propertyPVALID},        // KANNADA SIGN NUKTA..KANNADA VOWEL SIGN VOCAL
	{0x0CC5, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0CC6, 0x0CC8, propertyPVALID},        // KANNADA VOWEL SIGN E..KANNADA VOWEL SIGN AI
	{0x0CC9, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0CCA, 0x0CCD, propertyPVALID},        // KANNADA VOWEL SIGN O..KANNADA SIGN VIRAMA
	{0x0CCE, 0x0CD4, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0CD5, 0x0CD6, propertyPVALID},        // KANNADA LENGTH MARK..KANNADA AI LENGTH MARK
	{0x0CD7, 0x0CDD, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0CDE, 0x0, propertyPVALID},           // KANNADA LETTER FA
	{0x0CDF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0CE0, 0x0CE3, propertyPVALID},        // KANNADA LETTER VOCALIC RR..KANNADA VOWEL SIG
	{0x0CE4, 0x0CE5, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0CE6, 0x0CEF, propertyPVALID},        // KANNADA DIGIT ZERO..KANNADA DIGIT NINE
	{0x0CF0, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0CF1, 0x0CF2, propertyDISALLOWED},    // KANNADA SIGN JIHVAMULIYA..KANNADA SIGN UPADH
	{0x0CF3, 0x0D01, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0D02, 0x0D03, propertyPVALID},        // MALAYALAM SIGN ANUSVARA..MALAYALAM SIGN VISA
	{0x0D04, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0D05, 0x0D0C, propertyPVALID},        // MALAYALAM LETTER A..MALAYALAM LETTER VOCALIC
	{0x0D0D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0D0E, 0x0D10, propertyPVALID},        // MALAYALAM LETTER E..MALAYALAM LETTER AI
	{0x0D11, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0D12, 0x0D28, propertyPVALID},        // MALAYALAM LETTER O..MALAYALAM LETTER NA
	{0x0D29, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0D2A, 0x0D39, propertyPVALID},        // MALAYALAM LETTER PA..MALAYALAM LETTER HA
	{0x0D3A, 0x0D3C, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0D3D, 0x0D44, propertyPVALID},        // MALAYALAM SIGN AVAGRAHA..MALAYALAM VOWEL SIG
	{0x0D45, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0D46, 0x0D48, propertyPVALID},        // MALAYALAM VOWEL SIGN E..MALAYALAM VOWEL SIGN
	{0x0D49, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0D4A, 0x0D4D, propertyPVALID},        // MALAYALAM VOWEL SIGN O..MALAYALAM SIGN VIRAM
	{0x0D4E, 0x0D56, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0D57, 0x0, propertyPVALID},           // MALAYALAM AU LENGTH MARK
	{0x0D58, 0x0D5F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0D60, 0x0D63, propertyPVALID},        // MALAYALAM LETTER VOCALIC RR..MALAYALAM VOWEL
	{0x0D64, 0x0D65, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0D66, 0x0D6F, propertyPVALID},        // MALAYALAM DIGIT ZERO..MALAYALAM DIGIT NINE
	{0x0D70, 0x0D75, propertyDISALLOWED},    // MALAYALAM NUMBER TEN..MALAYALAM FRACTION THR
	{0x0D76, 0x0D78, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0D79, 0x0, propertyDISALLOWED},       // MALAYALAM DATE MARK
	{0x0D7A, 0x0D7F, propertyPVALID},        // MALAYALAM LETTER CHILLU NN..MALAYALAM LETTER
	{0x0D80, 0x0D81, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0D82, 0x0D83, propertyPVALID},        // SINHALA SIGN ANUSVARAYA..SINHALA SIGN VISARG
	{0x0D84, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0D85, 0x0D96, propertyPVALID},        // SINHALA LETTER AYANNA..SINHALA LETTER AUYANN
	{0x0D97, 0x0D99, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0D9A, 0x0DB1, propertyPVALID},        // SINHALA LETTER ALPAPRAANA KAYANNA..SINHALA L
	{0x0DB2, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0DB3, 0x0DBB, propertyPVALID},        // SINHALA LETTER SANYAKA DAYANNA..SINHALA LETT
	{0x0DBC, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0DBD, 0x0, propertyPVALID},           // SINHALA LETTER DANTAJA LAYANNA
	{0x0DBE, 0x0DBF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0DC0, 0x0DC6, propertyPVALID},        // SINHALA LETTER VAYANNA..SINHALA LETTER FAYAN
	{0x0DC7, 0x0DC9, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0DCA, 0x0, propertyPVALID},           // SINHALA SIGN AL-LAKUNA
	{0x0DCB, 0x0DCE, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0DCF, 0x0DD4, propertyPVALID},        // SINHALA VOWEL SIGN AELA-PILLA..SINHALA VOWEL
	{0x0DD5, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0DD6, 0x0, propertyPVALID},           // SINHALA VOWEL SIGN DIGA PAA-PILLA
	{0x0DD7, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0DD8, 0x0DDF, propertyPVALID},        // SINHALA VOWEL SIGN GAETTA-PILLA..SINHALA VOW
	{0x0DE0, 0x0DF1, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0DF2, 0x0DF3, propertyPVALID},        // SINHALA VOWEL SIGN DIGA GAETTA-PILLA..SINHAL
	{0x0DF4, 0x0, propertyDISALLOWED},       // SINHALA PUNCTUATION KUNDDALIYA
	{0x0DF5, 0x0E00, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0E01, 0x0E32, propertyPVALID},        // THAI CHARACTER KO KAI..THAI CHARACTER SARA A
	{0x0E33, 0x0, propertyDISALLOWED},       // THAI CHARACTER SARA AM
	{0x0E34, 0x0E3A, propertyPVALID},        // THAI CHARACTER SARA I..THAI CHARACTER PHINTH
	{0x0E3B, 0x0E3E, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0E3F, 0x0, propertyDISALLOWED},       // THAI CURRENCY SYMBOL BAHT
	{0x0E40, 0x0E4E, propertyPVALID},        // THAI CHARACTER SARA E..THAI CHARACTER YAMAKK
	{0x0E4F, 0x0, propertyDISALLOWED},       // THAI CHARACTER FONGMAN
	{0x0E50, 0x0E59, propertyPVALID},        // THAI DIGIT ZERO..THAI DIGIT NINE
	{0x0E5A, 0x0E5B, propertyDISALLOWED},    // THAI CHARACTER ANGKHANKHU..THAI CHARACTER KH
	{0x0E5C, 0x0E80, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0E81, 0x0E82, propertyPVALID},        // LAO LETTER KO..LAO LETTER KHO SUNG
	{0x0E83, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0E84, 0x0, propertyPVALID},           // LAO LETTER KHO TAM
	{0x0E85, 0x0E86, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0E87, 0x0E88, propertyPVALID},        // LAO LETTER NGO..LAO LETTER CO
	{0x0E89, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0E8A, 0x0, propertyPVALID},           // LAO LETTER SO TAM
	{0x0E8B, 0x0E8C, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0E8D, 0x0, propertyPVALID},           // LAO LETTER NYO
	{0x0E8E, 0x0E93, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0E94, 0x0E97, propertyPVALID},        // LAO LETTER DO..LAO LETTER THO TAM
	{0x0E98, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0E99, 0x0E9F, propertyPVALID},        // LAO LETTER NO..LAO LETTER FO SUNG
	{0x0EA0, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0EA1, 0x0EA3, propertyPVALID},        // LAO LETTER MO..LAO LETTER LO LING
	{0x0EA4, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0EA5, 0x0, propertyPVALID},           // LAO LETTER LO LOOT
	{0x0EA6, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0EA7, 0x0, propertyPVALID},           // LAO LETTER WO
	{0x0EA8, 0x0EA9, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0EAA, 0x0EAB, propertyPVALID},        // LAO LETTER SO SUNG..LAO LETTER HO SUNG
	{0x0EAC, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0EAD, 0x0EB2, propertyPVALID},        // LAO LETTER O..LAO VOWEL SIGN AA
	{0x0EB3, 0x0, propertyDISALLOWED},       // LAO VOWEL SIGN AM
	{0x0EB4, 0x0EB9, propertyPVALID},        // LAO VOWEL SIGN I..LAO VOWEL SIGN UU
	{0x0EBA, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0EBB, 0x0EBD, propertyPVALID},        // LAO VOWEL SIGN MAI KON..LAO SEMIVOWEL SIGN N
	{0x0EBE, 0x0EBF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0EC0, 0x0EC4, propertyPVALID},        // LAO VOWEL SIGN E..LAO VOWEL SIGN AI
	{0x0EC5, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0EC6, 0x0, propertyPVALID},           // LAO KO LA
	{0x0EC7, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0EC8, 0x0ECD, propertyPVALID},        // LAO TONE MAI EK..LAO NIGGAHITA
	{0x0ECE, 0x0ECF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0ED0, 0x0ED9, propertyPVALID},        // LAO DIGIT ZERO..LAO DIGIT NINE
	{0x0EDA, 0x0EDB, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0EDC, 0x0EDD, propertyDISALLOWED},    // LAO HO NO..LAO HO MO
	{0x0EDE, 0x0EFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0F00, 0x0, propertyPVALID},           // TIBETAN SYLLABLE OM
	{0x0F01, 0x0F0A, propertyDISALLOWED},    // TIBETAN MARK GTER YIG MGO TRUNCATED A..TIBET
	{0x0F0B, 0x0, propertyPVALID},           // TIBETAN MARK INTERSYLLABIC TSHEG
	{0x0F0C, 0x0F17, propertyDISALLOWED},    // TIBETAN MARK DELIMITER TSHEG BSTAR..TIBETAN
	{0x0F18, 0x0F19, propertyPVALID},        // TIBETAN ASTROLOGICAL SIGN -KHYUD PA..TIBETAN
	{0x0F1A, 0x0F1F, propertyDISALLOWED},    // TIBETAN SIGN RDEL DKAR GCIG..TIBETAN SIGN RD
	{0x0F20, 0x0F29, propertyPVALID},        // TIBETAN DIGIT ZERO..TIBETAN DIGIT NINE
	{0x0F2A, 0x0F34, propertyDISALLOWED},    // TIBETAN DIGIT HALF ONE..TIBETAN MARK BSDUS R
	{0x0F35, 0x0, propertyPVALID},           // TIBETAN MARK NGAS BZUNG NYI ZLA
	{0x0F36, 0x0, propertyDISALLOWED},       // TIBETAN MARK CARET -DZUD RTAGS BZHI MIG CAN
	{0x0F37, 0x0, propertyPVALID},           // TIBETAN MARK NGAS BZUNG SGOR RTAGS
	{0x0F38, 0x0, propertyDISALLOWED},       // TIBETAN MARK CHE MGO
	{0x0F39, 0x0, propertyPVALID},           // TIBETAN MARK TSA -PHRU
	{0x0F3A, 0x0F3D, propertyDISALLOWED},    // TIBETAN MARK GUG RTAGS GYON..TIBETAN MARK AN
	{0x0F3E, 0x0F42, propertyPVALID},        // TIBETAN SIGN YAR TSHES..TIBETAN LETTER GA
	{0x0F43, 0x0, propertyDISALLOWED},       // TIBETAN LETTER GHA
	{0x0F44, 0x0F47, propertyPVALID},        // TIBETAN LETTER NGA..TIBETAN LETTER JA
	{0x0F48, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0F49, 0x0F4C, propertyPVALID},        // TIBETAN LETTER NYA..TIBETAN LETTER DDA
	{0x0F4D, 0x0, propertyDISALLOWED},       // TIBETAN LETTER DDHA
	{0x0F4E, 0x0F51, propertyPVALID},        // TIBETAN LETTER NNA..TIBETAN LETTER DA
	{0x0F52, 0x0, propertyDISALLOWED},       // TIBETAN LETTER DHA
	{0x0F53, 0x0F56, propertyPVALID},        // TIBETAN LETTER NA..TIBETAN LETTER BA
	{0x0F57, 0x0, propertyDISALLOWED},       // TIBETAN LETTER BHA
	{0x0F58, 0x0F5B, propertyPVALID},        // TIBETAN LETTER MA..TIBETAN LETTER DZA
	{0x0F5C, 0x0, propertyDISALLOWED},       // TIBETAN LETTER DZHA
	{0x0F5D, 0x0F68, propertyPVALID},        // TIBETAN LETTER WA..TIBETAN LETTER A
	{0x0F69, 0x0, propertyDISALLOWED},       // TIBETAN LETTER KSSA
	{0x0F6A, 0x0F6C, propertyPVALID},        // TIBETAN LETTER FIXED-FORM RA..TIBETAN LETTER
	{0x0F6D, 0x0F70, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0F71, 0x0F72, propertyPVALID},        // TIBETAN VOWEL SIGN AA..TIBETAN VOWEL SIGN I
	{0x0F73, 0x0, propertyDISALLOWED},       // TIBETAN VOWEL SIGN II
	{0x0F74, 0x0, propertyPVALID},           // TIBETAN VOWEL SIGN U
	{0x0F75, 0x0F79, propertyDISALLOWED},    // TIBETAN VOWEL SIGN UU..TIBETAN VOWEL SIGN VO
	{0x0F7A, 0x0F80, propertyPVALID},        // TIBETAN VOWEL SIGN E..TIBETAN VOWEL SIGN REV
	{0x0F81, 0x0, propertyDISALLOWED},       // TIBETAN VOWEL SIGN REVERSED II
	{0x0F82, 0x0F84, propertyPVALID},        // TIBETAN SIGN NYI ZLA NAA DA..TIBETAN MARK HA
	{0x0F85, 0x0, propertyDISALLOWED},       // TIBETAN MARK PALUTA
	{0x0F86, 0x0F8B, propertyPVALID},        // TIBETAN SIGN LCI RTAGS..TIBETAN SIGN GRU MED
	{0x0F8C, 0x0F8F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x0F90, 0x0F92, propertyPVALID},        // TIBETAN SUBJOINED LETTER KA..TIBETAN SUBJOIN
	{0x0F93, 0x0, propertyDISALLOWED},       // TIBETAN SUBJOINED LETTER GHA
	{0x0F94, 0x0F97, propertyPVALID},        // TIBETAN SUBJOINED LETTER NGA..TIBETAN SUBJOI
	{0x0F98, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0F99, 0x0F9C, propertyPVALID},        // TIBETAN SUBJOINED LETTER NYA..TIBETAN SUBJOI
	{0x0F9D, 0x0, propertyDISALLOWED},       // TIBETAN SUBJOINED LETTER DDHA
	{0x0F9E, 0x0FA1, propertyPVALID},        // TIBETAN SUBJOINED LETTER NNA..TIBETAN SUBJOI
	{0x0FA2, 0x0, propertyDISALLOWED},       // TIBETAN SUBJOINED LETTER DHA
	{0x0FA3, 0x0FA6, propertyPVALID},        // TIBETAN SUBJOINED LETTER NA..TIBETAN SUBJOIN
	{0x0FA7, 0x0, propertyDISALLOWED},       // TIBETAN SUBJOINED LETTER BHA
	{0x0FA8, 0x0FAB, propertyPVALID},        // TIBETAN SUBJOINED LETTER MA..TIBETAN SUBJOIN
	{0x0FAC, 0x0, propertyDISALLOWED},       // TIBETAN SUBJOINED LETTER DZHA
	{0x0FAD, 0x0FB8, propertyPVALID},        // TIBETAN SUBJOINED LETTER WA..TIBETAN SUBJOIN
	{0x0FB9, 0x0, propertyDISALLOWED},       // TIBETAN SUBJOINED LETTER KSSA
	{0x0FBA, 0x0FBC, propertyPVALID},        // TIBETAN SUBJOINED LETTER FIXED-FORM WA..TIBE
	{0x0FBD, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0FBE, 0x0FC5, propertyDISALLOWED},    // TIBETAN KU RU KHA..TIBETAN SYMBOL RDO RJE
	{0x0FC6, 0x0, propertyPVALID},           // TIBETAN SYMBOL PADMA GDAN
	{0x0FC7, 0x0FCC, propertyDISALLOWED},    // TIBETAN SYMBOL RDO RJE RGYA GRAM..TIBETAN SY
	{0x0FCD, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x0FCE, 0x0FD8, propertyDISALLOWED},    // TIBETAN SIGN RDEL NAG RDEL DKAR..LEFT-FACING
	{0x0FD9, 0x0FFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1000, 0x1049, propertyPVALID},        // MYANMAR LETTER KA..MYANMAR DIGIT NINE
	{0x104A, 0x104F, propertyDISALLOWED},    // MYANMAR SIGN LITTLE SECTION..MYANMAR SYMBOL
	{0x1050, 0x109D, propertyPVALID},        // MYANMAR LETTER SHA..MYANMAR VOWEL SIGN AITON
	{0x109E, 0x10C5, propertyDISALLOWED},    // MYANMAR SYMBOL SHAN ONE..GEORGIAN CAPITAL LE
	{0x10C6, 0x10CF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x10D0, 0x10FA, propertyPVALID},        // GEORGIAN LETTER AN..GEORGIAN LETTER AIN
	{0x10FB, 0x10FC, propertyDISALLOWED},    // GEORGIAN PARAGRAPH SEPARATOR..MODIFIER LETTE
	{0x10FD, 0x10FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1100, 0x11FF, propertyDISALLOWED},    // HANGUL CHOSEONG KIYEOK..HANGUL JONGSEONG SSA
	{0x1200, 0x1248, propertyPVALID},        // ETHIOPIC SYLLABLE HA..ETHIOPIC SYLLABLE QWA
	{0x1249, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x124A, 0x124D, propertyPVALID},        // ETHIOPIC SYLLABLE QWI..ETHIOPIC SYLLABLE QWE
	{0x124E, 0x124F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1250, 0x1256, propertyPVALID},        // ETHIOPIC SYLLABLE QHA..ETHIOPIC SYLLABLE QHO
	{0x1257, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1258, 0x0, propertyPVALID},           // ETHIOPIC SYLLABLE QHWA
	{0x1259, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x125A, 0x125D, propertyPVALID},        // ETHIOPIC SYLLABLE QHWI..ETHIOPIC SYLLABLE QH
	{0x125E, 0x125F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1260, 0x1288, propertyPVALID},        // ETHIOPIC SYLLABLE BA..ETHIOPIC SYLLABLE XWA
	{0x1289, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x128A, 0x128D, propertyPVALID},        // ETHIOPIC SYLLABLE XWI..ETHIOPIC SYLLABLE XWE
	{0x128E, 0x128F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1290, 0x12B0, propertyPVALID},        // ETHIOPIC SYLLABLE NA..ETHIOPIC SYLLABLE KWA
	{0x12B1, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x12B2, 0x12B5, propertyPVALID},        // ETHIOPIC SYLLABLE KWI..ETHIOPIC SYLLABLE KWE
	{0x12B6, 0x12B7, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x12B8, 0x12BE, propertyPVALID},        // ETHIOPIC SYLLABLE KXA..ETHIOPIC SYLLABLE KXO
	{0x12BF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x12C0, 0x0, propertyPVALID},           // ETHIOPIC SYLLABLE KXWA
	{0x12C1, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x12C2, 0x12C5, propertyPVALID},        // ETHIOPIC SYLLABLE KXWI..ETHIOPIC SYLLABLE KX
	{0x12C6, 0x12C7, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x12C8, 0x12D6, propertyPVALID},        // ETHIOPIC SYLLABLE WA..ETHIOPIC SYLLABLE PHAR
	{0x12D7, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x12D8, 0x1310, propertyPVALID},        // ETHIOPIC SYLLABLE ZA..ETHIOPIC SYLLABLE GWA
	{0x1311, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1312, 0x1315, propertyPVALID},        // ETHIOPIC SYLLABLE GWI..ETHIOPIC SYLLABLE GWE
	{0x1316, 0x1317, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1318, 0x135A, propertyPVALID},        // ETHIOPIC SYLLABLE GGA..ETHIOPIC SYLLABLE FYA
	{0x135B, 0x135E, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x135F, 0x0, propertyPVALID},           // ETHIOPIC COMBINING GEMINATION MARK
	{0x1360, 0x137C, propertyDISALLOWED},    // ETHIOPIC SECTION MARK..ETHIOPIC NUMBER TEN T
	{0x137D, 0x137F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1380, 0x138F, propertyPVALID},        // ETHIOPIC SYLLABLE SEBATBEIT MWA..ETHIOPIC SY
	{0x1390, 0x1399, propertyDISALLOWED},    // ETHIOPIC TONAL MARK YIZET..ETHIOPIC TONAL MA
	{0x139A, 0x139F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x13A0, 0x13F4, propertyPVALID},        // CHEROKEE LETTER A..CHEROKEE LETTER YV
	{0x13F5, 0x13FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1400, 0x0, propertyDISALLOWED},       // CANADIAN SYLLABICS HYPHEN
	{0x1401, 0x166C, propertyPVALID},        // CANADIAN SYLLABICS E..CANADIAN SYLLABICS CAR
	{0x166D, 0x166E, propertyDISALLOWED},    // CANADIAN SYLLABICS CHI SIGN..CANADIAN SYLLAB
	{0x166F, 0x167F, propertyPVALID},        // CANADIAN SYLLABICS QAI..CANADIAN SYLLABICS B
	{0x1680, 0x0, propertyDISALLOWED},       // OGHAM SPACE MARK
	{0x1681, 0x169A, propertyPVALID},        // OGHAM LETTER BEITH..OGHAM LETTER PEITH
	{0x169B, 0x169C, propertyDISALLOWED},    // OGHAM FEATHER MARK..OGHAM REVERSED FEATHER M
	{0x169D, 0x169F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x16A0, 0x16EA, propertyPVALID},        // RUNIC LETTER FEHU FEOH FE F..RUNIC LETTER X
	{0x16EB, 0x16F0, propertyDISALLOWED},    // RUNIC SINGLE PUNCTUATION..RUNIC BELGTHOR SYM
	{0x16F1, 0x16FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1700, 0x170C, propertyPVALID},        // TAGALOG LETTER A..TAGALOG LETTER YA
	{0x170D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x170E, 0x1714, propertyPVALID},        // TAGALOG LETTER LA..TAGALOG SIGN VIRAMA
	{0x1715, 0x171F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1720, 0x1734, propertyPVALID},        // HANUNOO LETTER A..HANUNOO SIGN PAMUDPOD
	{0x1735, 0x1736, propertyDISALLOWED},    // PHILIPPINE SINGLE PUNCTUATION..PHILIPPINE DO
	{0x1737, 0x173F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1740, 0x1753, propertyPVALID},        // BUHID LETTER A..BUHID VOWEL SIGN U
	{0x1754, 0x175F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1760, 0x176C, propertyPVALID},        // TAGBANWA LETTER A..TAGBANWA LETTER YA
	{0x176D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x176E, 0x1770, propertyPVALID},        // TAGBANWA LETTER LA..TAGBANWA LETTER SA
	{0x1771, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1772, 0x1773, propertyPVALID},        // TAGBANWA VOWEL SIGN I..TAGBANWA VOWEL SIGN U
	{0x1774, 0x177F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1780, 0x17B3, propertyPVALID},        // KHMER LETTER KA..KHMER INDEPENDENT VOWEL QAU
	{0x17B4, 0x17B5, propertyDISALLOWED},    // KHMER VOWEL INHERENT AQ..KHMER VOWEL INHEREN
	{0x17B6, 0x17D3, propertyPVALID},        // KHMER VOWEL SIGN AA..KHMER SIGN BATHAMASAT
	{0x17D4, 0x17D6, propertyDISALLOWED},    // KHMER SIGN KHAN..KHMER SIGN CAMNUC PII KUUH
	{0x17D7, 0x0, propertyPVALID},           // KHMER SIGN LEK TOO
	{0x17D8, 0x17DB, propertyDISALLOWED},    // KHMER SIGN BEYYAL..KHMER CURRENCY SYMBOL RIE
	{0x17DC, 0x17DD, propertyPVALID},        // KHMER SIGN AVAKRAHASANYA..KHMER SIGN ATTHACA
	{0x17DE, 0x17DF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x17E0, 0x17E9, propertyPVALID},        // KHMER DIGIT ZERO..KHMER DIGIT NINE
	{0x17EA, 0x17EF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x17F0, 0x17F9, propertyDISALLOWED},    // KHMER SYMBOL LEK ATTAK SON..KHMER SYMBOL LEK
	{0x17FA, 0x17FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1800, 0x180E, propertyDISALLOWED},    // MONGOLIAN BIRGA..MONGOLIAN VOWEL SEPARATOR
	{0x180F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1810, 0x1819, propertyPVALID},        // MONGOLIAN DIGIT ZERO..MONGOLIAN DIGIT NINE
	{0x181A, 0x181F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1820, 0x1877, propertyPVALID},        // MONGOLIAN LETTER A..MONGOLIAN LETTER MANCHU
	{0x1878, 0x187F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1880, 0x18AA, propertyPVALID},        // MONGOLIAN LETTER ALI GALI ANUSVARA ONE..MONG
	{0x18AB, 0x18AF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x18B0, 0x18F5, propertyPVALID},        // CANADIAN SYLLABICS OY..CANADIAN SYLLABICS CA
	{0x18F6, 0x18FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1900, 0x191C, propertyPVALID},        // LIMBU VOWEL-CARRIER LETTER..LIMBU LETTER HA
	{0x191D, 0x191F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1920, 0x192B, propertyPVALID},        // LIMBU VOWEL SIGN A..LIMBU SUBJOINED LETTER W
	{0x192C, 0x192F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1930, 0x193B, propertyPVALID},        // LIMBU SMALL LETTER KA..LIMBU SIGN SA-I
	{0x193C, 0x193F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1940, 0x0, propertyDISALLOWED},       // LIMBU SIGN LOO
	{0x1941, 0x1943, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1944, 0x1945, propertyDISALLOWED},    // LIMBU EXCLAMATION MARK..LIMBU QUESTION MARK
	{0x1946, 0x196D, propertyPVALID},        // LIMBU DIGIT ZERO..TAI LE LETTER AI
	{0x196E, 0x196F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1970, 0x1974, propertyPVALID},        // TAI LE LETTER TONE-2..TAI LE LETTER TONE-6
	{0x1975, 0x197F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1980, 0x19AB, propertyPVALID},        // NEW TAI LUE LETTER HIGH QA..NEW TAI LUE LETT
	{0x19AC, 0x19AF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x19B0, 0x19C9, propertyPVALID},        // NEW TAI LUE VOWEL SIGN VOWEL SHORTENER..NEW
	{0x19CA, 0x19CF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x19D0, 0x19DA, propertyPVALID},        // NEW TAI LUE DIGIT ZERO..NEW TAI LUE THAM DIG
	{0x19DB, 0x19DD, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x19DE, 0x19FF, propertyDISALLOWED},    // NEW TAI LUE SIGN LAE..KHMER SYMBOL DAP-PRAM
	{0x1A00, 0x1A1B, propertyPVALID},        // BUGINESE LETTER KA..BUGINESE VOWEL SIGN AE
	{0x1A1C, 0x1A1D, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1A1E, 0x1A1F, propertyDISALLOWED},    // BUGINESE PALLAWA..BUGINESE END OF SECTION
	{0x1A20, 0x1A5E, propertyPVALID},        // TAI THAM LETTER HIGH KA..TAI THAM CONSONANT
	{0x1A5F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1A60, 0x1A7C, propertyPVALID},        // TAI THAM SIGN SAKOT..TAI THAM SIGN KHUEN-LUE
	{0x1A7D, 0x1A7E, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1A7F, 0x1A89, propertyPVALID},        // TAI THAM COMBINING CRYPTOGRAMMIC DOT..TAI TH
	{0x1A8A, 0x1A8F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1A90, 0x1A99, propertyPVALID},        // TAI THAM THAM DIGIT ZERO..TAI THAM THAM DIGI
	{0x1A9A, 0x1A9F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1AA0, 0x1AA6, propertyDISALLOWED},    // TAI THAM SIGN WIANG..TAI THAM SIGN REVERSED
	{0x1AA7, 0x0, propertyPVALID},           // TAI THAM SIGN MAI YAMOK
	{0x1AA8, 0x1AAD, propertyDISALLOWED},    // TAI THAM SIGN KAAN..TAI THAM SIGN CAANG
	{0x1AAE, 0x1AFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1B00, 0x1B4B, propertyPVALID},        // BALINESE SIGN ULU RICEM..BALINESE LETTER ASY
	{0x1B4C, 0x1B4F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1B50, 0x1B59, propertyPVALID},        // BALINESE DIGIT ZERO..BALINESE DIGIT NINE
	{0x1B5A, 0x1B6A, propertyDISALLOWED},    // BALINESE PANTI..BALINESE MUSICAL SYMBOL DANG
	{0x1B6B, 0x1B73, propertyPVALID},        // BALINESE MUSICAL SYMBOL COMBINING TEGEH..BAL
	{0x1B74, 0x1B7C, propertyDISALLOWED},    // BALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DUG.
	{0x1B7D, 0x1B7F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1B80, 0x1BAA, propertyPVALID},        // SUNDANESE SIGN PANYECEK..SUNDANESE SIGN PAMA
	{0x1BAB, 0x1BAD, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1BAE, 0x1BB9, propertyPVALID},        // SUNDANESE LETTER KHA..SUNDANESE DIGIT NINE
	{0x1BBA, 0x1BFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1C00, 0x1C37, propertyPVALID},        // LEPCHA LETTER KA..LEPCHA SIGN NUKTA
	{0x1C38, 0x1C3A, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1C3B, 0x1C3F, propertyDISALLOWED},    // LEPCHA PUNCTUATION TA-ROL..LEPCHA PUNCTUATIO
	{0x1C40, 0x1C49, propertyPVALID},        // LEPCHA DIGIT ZERO..LEPCHA DIGIT NINE
	{0x1C4A, 0x1C4C, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1C4D, 0x1C7D, propertyPVALID},        // LEPCHA LETTER TTA..OL CHIKI AHAD
	{0x1C7E, 0x1C7F, propertyDISALLOWED},    // OL CHIKI PUNCTUATION MUCAAD..OL CHIKI PUNCTU
	{0x1C80, 0x1CCF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1CD0, 0x1CD2, propertyPVALID},        // VEDIC TONE KARSHANA..VEDIC TONE PRENKHA
	{0x1CD3, 0x0, propertyDISALLOWED},       // VEDIC SIGN NIHSHVASA
	{0x1CD4, 0x1CF2, propertyPVALID},        // VEDIC SIGN YAJURVEDIC MIDLINE SVARITA..VEDIC
	{0x1CF3, 0x1CFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1D00, 0x1D2B, propertyPVALID},        // LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTE
	{0x1D2C, 0x1D2E, propertyDISALLOWED},    // MODIFIER LETTER CAPITAL A..MODIFIER LETTER C
	{0x1D2F, 0x0, propertyPVALID},           // MODIFIER LETTER CAPITAL BARRED B
	{0x1D30, 0x1D3A, propertyDISALLOWED},    // MODIFIER LETTER CAPITAL D..MODIFIER LETTER C
	{0x1D3B, 0x0, propertyPVALID},           // MODIFIER LETTER CAPITAL REVERSED N
	{0x1D3C, 0x1D4D, propertyDISALLOWED},    // MODIFIER LETTER CAPITAL O..MODIFIER LETTER S
	{0x1D4E, 0x0, propertyPVALID},           // MODIFIER LETTER SMALL TURNED I
	{0x1D4F, 0x1D6A, propertyDISALLOWED},    // MODIFIER LETTER SMALL K..GREEK SUBSCRIPT SMA
	{0x1D6B, 0x1D77, propertyPVALID},        // LATIN SMALL LETTER UE..LATIN SMALL LETTER TU
	{0x1D78, 0x0, propertyDISALLOWED},       // MODIFIER LETTER CYRILLIC EN
	{0x1D79, 0x1D9A, propertyPVALID},        // LATIN SMALL LETTER INSULAR G..LATIN SMALL LE
	{0x1D9B, 0x1DBF, propertyDISALLOWED},    // MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER
	{0x1DC0, 0x1DE6, propertyPVALID},        // COMBINING DOTTED GRAVE ACCENT..COMBINING LAT
	{0x1DE7, 0x1DFC, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1DFD, 0x1DFF, propertyPVALID},        // COMBINING ALMOST EQUAL TO BELOW..COMBINING R
	{0x1E00, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH RING BELOW
	{0x1E01, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH RING BELOW
	{0x1E02, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER B WITH DOT ABOVE
	{0x1E03, 0x0, propertyPVALID},           // LATIN SMALL LETTER B WITH DOT ABOVE
	{0x1E04, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER B WITH DOT BELOW
	{0x1E05, 0x0, propertyPVALID},           // LATIN SMALL LETTER B WITH DOT BELOW
	{0x1E06, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER B WITH LINE BELOW
	{0x1E07, 0x0, propertyPVALID},           // LATIN SMALL LETTER B WITH LINE BELOW
	{0x1E08, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER C WITH CEDILLA AND ACUT
	{0x1E09, 0x0, propertyPVALID},           // LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
	{0x1E0A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER D WITH DOT ABOVE
	{0x1E0B, 0x0, propertyPVALID},           // LATIN SMALL LETTER D WITH DOT ABOVE
	{0x1E0C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER D WITH DOT BELOW
	{0x1E0D, 0x0, propertyPVALID},           // LATIN SMALL LETTER D WITH DOT BELOW
	{0x1E0E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER D WITH LINE BELOW
	{0x1E0F, 0x0, propertyPVALID},           // LATIN SMALL LETTER D WITH LINE BELOW
	{0x1E10, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER D WITH CEDILLA
	{0x1E11, 0x0, propertyPVALID},           // LATIN SMALL LETTER D WITH CEDILLA
	{0x1E12, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
	{0x1E13, 0x0, propertyPVALID},           // LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
	{0x1E14, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
	{0x1E15, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH MACRON AND GRAVE
	{0x1E16, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
	{0x1E17, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH MACRON AND ACUTE
	{0x1E18, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
	{0x1E19, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
	{0x1E1A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH TILDE BELOW
	{0x1E1B, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH TILDE BELOW
	{0x1E1C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CEDILLA AND BREV
	{0x1E1D, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CEDILLA AND BREVE
	{0x1E1E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER F WITH DOT ABOVE
	{0x1E1F, 0x0, propertyPVALID},           // LATIN SMALL LETTER F WITH DOT ABOVE
	{0x1E20, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER G WITH MACRON
	{0x1E21, 0x0, propertyPVALID},           // LATIN SMALL LETTER G WITH MACRON
	{0x1E22, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH DOT ABOVE
	{0x1E23, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH DOT ABOVE
	{0x1E24, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH DOT BELOW
	{0x1E25, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH DOT BELOW
	{0x1E26, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH DIAERESIS
	{0x1E27, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH DIAERESIS
	{0x1E28, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH CEDILLA
	{0x1E29, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH CEDILLA
	{0x1E2A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH BREVE BELOW
	{0x1E2B, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH BREVE BELOW
	{0x1E2C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH TILDE BELOW
	{0x1E2D, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH TILDE BELOW
	{0x1E2E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH DIAERESIS AND AC
	{0x1E2F, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH DIAERESIS AND ACUT
	{0x1E30, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH ACUTE
	{0x1E31, 0x0, propertyPVALID},           // LATIN SMALL LETTER K WITH ACUTE
	{0x1E32, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH DOT BELOW
	{0x1E33, 0x0, propertyPVALID},           // LATIN SMALL LETTER K WITH DOT BELOW
	{0x1E34, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH LINE BELOW
	{0x1E35, 0x0, propertyPVALID},           // LATIN SMALL LETTER K WITH LINE BELOW
	{0x1E36, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH DOT BELOW
	{0x1E37, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH DOT BELOW
	{0x1E38, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH DOT BELOW AND MA
	{0x1E39, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH DOT BELOW AND MACR
	{0x1E3A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH LINE BELOW
	{0x1E3B, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH LINE BELOW
	{0x1E3C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
	{0x1E3D, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
	{0x1E3E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER M WITH ACUTE
	{0x1E3F, 0x0, propertyPVALID},           // LATIN SMALL LETTER M WITH ACUTE
	{0x1E40, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER M WITH DOT ABOVE
	{0x1E41, 0x0, propertyPVALID},           // LATIN SMALL LETTER M WITH DOT ABOVE
	{0x1E42, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER M WITH DOT BELOW
	{0x1E43, 0x0, propertyPVALID},           // LATIN SMALL LETTER M WITH DOT BELOW
	{0x1E44, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER N WITH DOT ABOVE
	{0x1E45, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH DOT ABOVE
	{0x1E46, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER N WITH DOT BELOW
	{0x1E47, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH DOT BELOW
	{0x1E48, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER N WITH LINE BELOW
	{0x1E49, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH LINE BELOW
	{0x1E4A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
	{0x1E4B, 0x0, propertyPVALID},           // LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
	{0x1E4C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
	{0x1E4D, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH TILDE AND ACUTE
	{0x1E4E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH TILDE AND DIAERE
	{0x1E4F, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH TILDE AND DIAERESI
	{0x1E50, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
	{0x1E51, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH MACRON AND GRAVE
	{0x1E52, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
	{0x1E53, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH MACRON AND ACUTE
	{0x1E54, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER P WITH ACUTE
	{0x1E55, 0x0, propertyPVALID},           // LATIN SMALL LETTER P WITH ACUTE
	{0x1E56, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER P WITH DOT ABOVE
	{0x1E57, 0x0, propertyPVALID},           // LATIN SMALL LETTER P WITH DOT ABOVE
	{0x1E58, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH DOT ABOVE
	{0x1E59, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH DOT ABOVE
	{0x1E5A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH DOT BELOW
	{0x1E5B, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH DOT BELOW
	{0x1E5C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH DOT BELOW AND MA
	{0x1E5D, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH DOT BELOW AND MACR
	{0x1E5E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R WITH LINE BELOW
	{0x1E5F, 0x0, propertyPVALID},           // LATIN SMALL LETTER R WITH LINE BELOW
	{0x1E60, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH DOT ABOVE
	{0x1E61, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH DOT ABOVE
	{0x1E62, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH DOT BELOW
	{0x1E63, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH DOT BELOW
	{0x1E64, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH ACUTE AND DOT AB
	{0x1E65, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH ACUTE AND DOT ABOV
	{0x1E66, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH CARON AND DOT AB
	{0x1E67, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH CARON AND DOT ABOV
	{0x1E68, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER S WITH DOT BELOW AND DO
	{0x1E69, 0x0, propertyPVALID},           // LATIN SMALL LETTER S WITH DOT BELOW AND DOT
	{0x1E6A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH DOT ABOVE
	{0x1E6B, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH DOT ABOVE
	{0x1E6C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH DOT BELOW
	{0x1E6D, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH DOT BELOW
	{0x1E6E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH LINE BELOW
	{0x1E6F, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH LINE BELOW
	{0x1E70, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
	{0x1E71, 0x0, propertyPVALID},           // LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
	{0x1E72, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
	{0x1E73, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH DIAERESIS BELOW
	{0x1E74, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH TILDE BELOW
	{0x1E75, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH TILDE BELOW
	{0x1E76, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
	{0x1E77, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
	{0x1E78, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
	{0x1E79, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH TILDE AND ACUTE
	{0x1E7A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH MACRON AND DIAER
	{0x1E7B, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH MACRON AND DIAERES
	{0x1E7C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER V WITH TILDE
	{0x1E7D, 0x0, propertyPVALID},           // LATIN SMALL LETTER V WITH TILDE
	{0x1E7E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER V WITH DOT BELOW
	{0x1E7F, 0x0, propertyPVALID},           // LATIN SMALL LETTER V WITH DOT BELOW
	{0x1E80, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER W WITH GRAVE
	{0x1E81, 0x0, propertyPVALID},           // LATIN SMALL LETTER W WITH GRAVE
	{0x1E82, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER W WITH ACUTE
	{0x1E83, 0x0, propertyPVALID},           // LATIN SMALL LETTER W WITH ACUTE
	{0x1E84, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER W WITH DIAERESIS
	{0x1E85, 0x0, propertyPVALID},           // LATIN SMALL LETTER W WITH DIAERESIS
	{0x1E86, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER W WITH DOT ABOVE
	{0x1E87, 0x0, propertyPVALID},           // LATIN SMALL LETTER W WITH DOT ABOVE
	{0x1E88, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER W WITH DOT BELOW
	{0x1E89, 0x0, propertyPVALID},           // LATIN SMALL LETTER W WITH DOT BELOW
	{0x1E8A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER X WITH DOT ABOVE
	{0x1E8B, 0x0, propertyPVALID},           // LATIN SMALL LETTER X WITH DOT ABOVE
	{0x1E8C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER X WITH DIAERESIS
	{0x1E8D, 0x0, propertyPVALID},           // LATIN SMALL LETTER X WITH DIAERESIS
	{0x1E8E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH DOT ABOVE
	{0x1E8F, 0x0, propertyPVALID},           // LATIN SMALL LETTER Y WITH DOT ABOVE
	{0x1E90, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
	{0x1E91, 0x0, propertyPVALID},           // LATIN SMALL LETTER Z WITH CIRCUMFLEX
	{0x1E92, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Z WITH DOT BELOW
	{0x1E93, 0x0, propertyPVALID},           // LATIN SMALL LETTER Z WITH DOT BELOW
	{0x1E94, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Z WITH LINE BELOW
	{0x1E95, 0x1E99, propertyPVALID},        // LATIN SMALL LETTER Z WITH LINE BELOW..LATIN
	{0x1E9A, 0x1E9B, propertyDISALLOWED},    // LATIN SMALL LETTER A WITH RIGHT HALF RING..L
	{0x1E9C, 0x1E9D, propertyPVALID},        // LATIN SMALL LETTER LONG S WITH DIAGONAL STRO
	{0x1E9E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER SHARP S
	{0x1E9F, 0x0, propertyPVALID},           // LATIN SMALL LETTER DELTA
	{0x1EA0, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH DOT BELOW
	{0x1EA1, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH DOT BELOW
	{0x1EA2, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH HOOK ABOVE
	{0x1EA3, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH HOOK ABOVE
	{0x1EA4, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND A
	{0x1EA5, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACU
	{0x1EA6, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND G
	{0x1EA7, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRA
	{0x1EA8, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND H
	{0x1EA9, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOO
	{0x1EAA, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND T
	{0x1EAB, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH CIRCUMFLEX AND TIL
	{0x1EAC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND D
	{0x1EAD, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT
	{0x1EAE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
	{0x1EAF, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH BREVE AND ACUTE
	{0x1EB0, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
	{0x1EB1, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH BREVE AND GRAVE
	{0x1EB2, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH BREVE AND HOOK A
	{0x1EB3, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH BREVE AND HOOK ABO
	{0x1EB4, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH BREVE AND TILDE
	{0x1EB5, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH BREVE AND TILDE
	{0x1EB6, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER A WITH BREVE AND DOT BE
	{0x1EB7, 0x0, propertyPVALID},           // LATIN SMALL LETTER A WITH BREVE AND DOT BELO
	{0x1EB8, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH DOT BELOW
	{0x1EB9, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH DOT BELOW
	{0x1EBA, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH HOOK ABOVE
	{0x1EBB, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH HOOK ABOVE
	{0x1EBC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH TILDE
	{0x1EBD, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH TILDE
	{0x1EBE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND A
	{0x1EBF, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACU
	{0x1EC0, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND G
	{0x1EC1, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRA
	{0x1EC2, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND H
	{0x1EC3, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOO
	{0x1EC4, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND T
	{0x1EC5, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CIRCUMFLEX AND TIL
	{0x1EC6, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND D
	{0x1EC7, 0x0, propertyPVALID},           // LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT
	{0x1EC8, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH HOOK ABOVE
	{0x1EC9, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH HOOK ABOVE
	{0x1ECA, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER I WITH DOT BELOW
	{0x1ECB, 0x0, propertyPVALID},           // LATIN SMALL LETTER I WITH DOT BELOW
	{0x1ECC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH DOT BELOW
	{0x1ECD, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH DOT BELOW
	{0x1ECE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH HOOK ABOVE
	{0x1ECF, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH HOOK ABOVE
	{0x1ED0, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND A
	{0x1ED1, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACU
	{0x1ED2, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND G
	{0x1ED3, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRA
	{0x1ED4, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND H
	{0x1ED5, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOO
	{0x1ED6, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND T
	{0x1ED7, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH CIRCUMFLEX AND TIL
	{0x1ED8, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND D
	{0x1ED9, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT
	{0x1EDA, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH HORN AND ACUTE
	{0x1EDB, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH HORN AND ACUTE
	{0x1EDC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH HORN AND GRAVE
	{0x1EDD, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH HORN AND GRAVE
	{0x1EDE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH HORN AND HOOK AB
	{0x1EDF, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH HORN AND HOOK ABOV
	{0x1EE0, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH HORN AND TILDE
	{0x1EE1, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH HORN AND TILDE
	{0x1EE2, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH HORN AND DOT BEL
	{0x1EE3, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH HORN AND DOT BELOW
	{0x1EE4, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH DOT BELOW
	{0x1EE5, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH DOT BELOW
	{0x1EE6, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH HOOK ABOVE
	{0x1EE7, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH HOOK ABOVE
	{0x1EE8, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH HORN AND ACUTE
	{0x1EE9, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH HORN AND ACUTE
	{0x1EEA, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH HORN AND GRAVE
	{0x1EEB, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH HORN AND GRAVE
	{0x1EEC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH HORN AND HOOK AB
	{0x1EED, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH HORN AND HOOK ABOV
	{0x1EEE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH HORN AND TILDE
	{0x1EEF, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH HORN AND TILDE
	{0x1EF0, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER U WITH HORN AND DOT BEL
	{0x1EF1, 0x0, propertyPVALID},           // LATIN SMALL LETTER U WITH HORN AND DOT BELOW
	{0x1EF2, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH GRAVE
	{0x1EF3, 0x0, propertyPVALID},           // LATIN SMALL LETTER Y WITH GRAVE
	{0x1EF4, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH DOT BELOW
	{0x1EF5, 0x0, propertyPVALID},           // LATIN SMALL LETTER Y WITH DOT BELOW
	{0x1EF6, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH HOOK ABOVE
	{0x1EF7, 0x0, propertyPVALID},           // LATIN SMALL LETTER Y WITH HOOK ABOVE
	{0x1EF8, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH TILDE
	{0x1EF9, 0x0, propertyPVALID},           // LATIN SMALL LETTER Y WITH TILDE
	{0x1EFA, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER MIDDLE-WELSH LL
	{0x1EFB, 0x0, propertyPVALID},           // LATIN SMALL LETTER MIDDLE-WELSH LL
	{0x1EFC, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER MIDDLE-WELSH V
	{0x1EFD, 0x0, propertyPVALID},           // LATIN SMALL LETTER MIDDLE-WELSH V
	{0x1EFE, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Y WITH LOOP
	{0x1EFF, 0x1F07, propertyPVALID},        // LATIN SMALL LETTER Y WITH LOOP..GREEK SMALL
	{0x1F08, 0x1F0F, propertyDISALLOWED},    // GREEK CAPITAL LETTER ALPHA WITH PSILI..GREEK
	{0x1F10, 0x1F15, propertyPVALID},        // GREEK SMALL LETTER EPSILON WITH PSILI..GREEK
	{0x1F16, 0x1F17, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1F18, 0x1F1D, propertyDISALLOWED},    // GREEK CAPITAL LETTER EPSILON WITH PSILI..GRE
	{0x1F1E, 0x1F1F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1F20, 0x1F27, propertyPVALID},        // GREEK SMALL LETTER ETA WITH PSILI..GREEK SMA
	{0x1F28, 0x1F2F, propertyDISALLOWED},    // GREEK CAPITAL LETTER ETA WITH PSILI..GREEK C
	{0x1F30, 0x1F37, propertyPVALID},        // GREEK SMALL LETTER IOTA WITH PSILI..GREEK SM
	{0x1F38, 0x1F3F, propertyDISALLOWED},    // GREEK CAPITAL LETTER IOTA WITH PSILI..GREEK
	{0x1F40, 0x1F45, propertyPVALID},        // GREEK SMALL LETTER OMICRON WITH PSILI..GREEK
	{0x1F46, 0x1F47, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1F48, 0x1F4D, propertyDISALLOWED},    // GREEK CAPITAL LETTER OMICRON WITH PSILI..GRE
	{0x1F4E, 0x1F4F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1F50, 0x1F57, propertyPVALID},        // GREEK SMALL LETTER UPSILON WITH PSILI..GREEK
	{0x1F58, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1F59, 0x0, propertyDISALLOWED},       // GREEK CAPITAL LETTER UPSILON WITH DASIA
	{0x1F5A, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1F5B, 0x0, propertyDISALLOWED},       // GREEK CAPITAL LETTER UPSILON WITH DASIA AND
	{0x1F5C, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1F5D, 0x0, propertyDISALLOWED},       // GREEK CAPITAL LETTER UPSILON WITH DASIA AND
	{0x1F5E, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1F5F, 0x0, propertyDISALLOWED},       // GREEK CAPITAL LETTER UPSILON WITH DASIA AND
	{0x1F60, 0x1F67, propertyPVALID},        // GREEK SMALL LETTER OMEGA WITH PSILI..GREEK S
	{0x1F68, 0x1F6F, propertyDISALLOWED},    // GREEK CAPITAL LETTER OMEGA WITH PSILI..GREEK
	{0x1F70, 0x0, propertyPVALID},           // GREEK SMALL LETTER ALPHA WITH VARIA
	{0x1F71, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER ALPHA WITH OXIA
	{0x1F72, 0x0, propertyPVALID},           // GREEK SMALL LETTER EPSILON WITH VARIA
	{0x1F73, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER EPSILON WITH OXIA
	{0x1F74, 0x0, propertyPVALID},           // GREEK SMALL LETTER ETA WITH VARIA
	{0x1F75, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER ETA WITH OXIA
	{0x1F76, 0x0, propertyPVALID},           // GREEK SMALL LETTER IOTA WITH VARIA
	{0x1F77, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER IOTA WITH OXIA
	{0x1F78, 0x0, propertyPVALID},           // GREEK SMALL LETTER OMICRON WITH VARIA
	{0x1F79, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER OMICRON WITH OXIA
	{0x1F7A, 0x0, propertyPVALID},           // GREEK SMALL LETTER UPSILON WITH VARIA
	{0x1F7B, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER UPSILON WITH OXIA
	{0x1F7C, 0x0, propertyPVALID},           // GREEK SMALL LETTER OMEGA WITH VARIA
	{0x1F7D, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER OMEGA WITH OXIA
	{0x1F7E, 0x1F7F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1F80, 0x1FAF, propertyDISALLOWED},    // GREEK SMALL LETTER ALPHA WITH PSILI AND YPOG
	{0x1FB0, 0x1FB1, propertyPVALID},        // GREEK SMALL LETTER ALPHA WITH VRACHY..GREEK
	{0x1FB2, 0x1FB4, propertyDISALLOWED},    // GREEK SMALL LETTER ALPHA WITH VARIA AND YPOG
	{0x1FB5, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1FB6, 0x0, propertyPVALID},           // GREEK SMALL LETTER ALPHA WITH PERISPOMENI
	{0x1FB7, 0x1FC4, propertyDISALLOWED},    // GREEK SMALL LETTER ALPHA WITH PERISPOMENI AN
	{0x1FC5, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1FC6, 0x0, propertyPVALID},           // GREEK SMALL LETTER ETA WITH PERISPOMENI
	{0x1FC7, 0x1FCF, propertyDISALLOWED},    // GREEK SMALL LETTER ETA WITH PERISPOMENI AND
	{0x1FD0, 0x1FD2, propertyPVALID},        // GREEK SMALL LETTER IOTA WITH VRACHY..GREEK S
	{0x1FD3, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER IOTA WITH DIALYTIKA AND O
	{0x1FD4, 0x1FD5, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1FD6, 0x1FD7, propertyPVALID},        // GREEK SMALL LETTER IOTA WITH PERISPOMENI..GR
	{0x1FD8, 0x1FDB, propertyDISALLOWED},    // GREEK CAPITAL LETTER IOTA WITH VRACHY..GREEK
	{0x1FDC, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1FDD, 0x1FDF, propertyDISALLOWED},    // GREEK DASIA AND VARIA..GREEK DASIA AND PERIS
	{0x1FE0, 0x1FE2, propertyPVALID},        // GREEK SMALL LETTER UPSILON WITH VRACHY..GREE
	{0x1FE3, 0x0, propertyDISALLOWED},       // GREEK SMALL LETTER UPSILON WITH DIALYTIKA AN
	{0x1FE4, 0x1FE7, propertyPVALID},        // GREEK SMALL LETTER RHO WITH PSILI..GREEK SMA
	{0x1FE8, 0x1FEF, propertyDISALLOWED},    // GREEK CAPITAL LETTER UPSILON WITH VRACHY..GR
	{0x1FF0, 0x1FF1, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x1FF2, 0x1FF4, propertyDISALLOWED},    // GREEK SMALL LETTER OMEGA WITH VARIA AND YPOG
	{0x1FF5, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x1FF6, 0x0, propertyPVALID},           // GREEK SMALL LETTER OMEGA WITH PERISPOMENI
	{0x1FF7, 0x1FFE, propertyDISALLOWED},    // GREEK SMALL LETTER OMEGA WITH PERISPOMENI AN
	{0x1FFF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2000, 0x200B, propertyDISALLOWED},    // EN QUAD..ZERO WIDTH SPACE
	{0x200C, 0x200D, propertyCONTEXTJ},      // ZERO WIDTH NON-JOINER..ZERO WIDTH JOINER
	{0x200E, 0x2064, propertyDISALLOWED},    // LEFT-TO-RIGHT MARK..INVISIBLE PLUS
	{0x2065, 0x2069, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x206A, 0x2071, propertyDISALLOWED},    // INHIBIT SYMMETRIC SWAPPING..SUPERSCRIPT LATI
	{0x2072, 0x2073, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2074, 0x208E, propertyDISALLOWED},    // SUPERSCRIPT FOUR..SUBSCRIPT RIGHT PARENTHESI
	{0x208F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2090, 0x2094, propertyDISALLOWED},    // LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCR
	{0x2095, 0x209F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x20A0, 0x20B8, propertyDISALLOWED},    // EURO-CURRENCY SIGN..TENGE SIGN
	{0x20B9, 0x20CF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x20D0, 0x20F0, propertyDISALLOWED},    // COMBINING LEFT HARPOON ABOVE..COMBINING ASTE
	{0x20F1, 0x20FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2100, 0x214D, propertyDISALLOWED},    // ACCOUNT OF..AKTIESELSKAB
	{0x214E, 0x0, propertyPVALID},           // TURNED SMALL F
	{0x214F, 0x2183, propertyDISALLOWED},    // SYMBOL FOR SAMARITAN SOURCE..ROMAN NUMERAL R
	{0x2184, 0x0, propertyPVALID},           // LATIN SMALL LETTER REVERSED C
	{0x2185, 0x2189, propertyDISALLOWED},    // ROMAN NUMERAL SIX LATE FORM..VULGAR FRACTION
	{0x218A, 0x218F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2190, 0x23E8, propertyDISALLOWED},    // LEFTWARDS ARROW..DECIMAL EXPONENT SYMBOL
	{0x23E9, 0x23FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2400, 0x2426, propertyDISALLOWED},    // SYMBOL FOR NULL..SYMBOL FOR SUBSTITUTE FORM
	{0x2427, 0x243F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2440, 0x244A, propertyDISALLOWED},    // OCR HOOK..OCR DOUBLE BACKSLASH
	{0x244B, 0x245F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2460, 0x26CD, propertyDISALLOWED},    // CIRCLED DIGIT ONE..DISABLED CAR
	{0x26CE, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x26CF, 0x26E1, propertyDISALLOWED},    // PICK..RESTRICTED LEFT ENTRY-2
	{0x26E2, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x26E3, 0x0, propertyDISALLOWED},       // HEAVY CIRCLE WITH STROKE AND TWO DOTS ABOVE
	{0x26E4, 0x26E7, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x26E8, 0x26FF, propertyDISALLOWED},    // BLACK CROSS ON SHIELD..WHITE FLAG WITH HORIZ
	{0x2700, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2701, 0x2704, propertyDISALLOWED},    // UPPER BLADE SCISSORS..WHITE SCISSORS
	{0x2705, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2706, 0x2709, propertyDISALLOWED},    // TELEPHONE LOCATION SIGN..ENVELOPE
	{0x270A, 0x270B, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x270C, 0x2727, propertyDISALLOWED},    // VICTORY HAND..WHITE FOUR POINTED STAR
	{0x2728, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2729, 0x274B, propertyDISALLOWED},    // STRESS OUTLINED WHITE STAR..HEAVY EIGHT TEAR
	{0x274C, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x274D, 0x0, propertyDISALLOWED},       // SHADOWED WHITE CIRCLE
	{0x274E, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x274F, 0x2752, propertyDISALLOWED},    // LOWER RIGHT DROP-SHADOWED WHITE SQUARE..UPPE
	{0x2753, 0x2755, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2756, 0x275E, propertyDISALLOWED},    // BLACK DIAMOND MINUS WHITE X..HEAVY DOUBLE CO
	{0x275F, 0x2760, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2761, 0x2794, propertyDISALLOWED},    // CURVED STEM PARAGRAPH SIGN ORNAMENT..HEAVY W
	{0x2795, 0x2797, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2798, 0x27AF, propertyDISALLOWED},    // HEAVY SOUTH EAST ARROW..NOTCHED LOWER RIGHT-
	{0x27B0, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x27B1, 0x27BE, propertyDISALLOWED},    // NOTCHED UPPER RIGHT-SHADOWED WHITE RIGHTWARD
	{0x27BF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x27C0, 0x27CA, propertyDISALLOWED},    // THREE DIMENSIONAL ANGLE..VERTICAL BAR WITH H
	{0x27CB, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x27CC, 0x0, propertyDISALLOWED},       // LONG DIVISION
	{0x27CD, 0x27CF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x27D0, 0x2B4C, propertyDISALLOWED},    // WHITE DIAMOND WITH CENTRED DOT..RIGHTWARDS A
	{0x2B4D, 0x2B4F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2B50, 0x2B59, propertyDISALLOWED},    // WHITE MEDIUM STAR..HEAVY CIRCLED SALTIRE
	{0x2B5A, 0x2BFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2C00, 0x2C2E, propertyDISALLOWED},    // GLAGOLITIC CAPITAL LETTER AZU..GLAGOLITIC CA
	{0x2C2F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2C30, 0x2C5E, propertyPVALID},        // GLAGOLITIC SMALL LETTER AZU..GLAGOLITIC SMAL
	{0x2C5F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2C60, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH DOUBLE BAR
	{0x2C61, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH DOUBLE BAR
	{0x2C62, 0x2C64, propertyDISALLOWED},    // LATIN CAPITAL LETTER L WITH MIDDLE TILDE..LA
	{0x2C65, 0x2C66, propertyPVALID},        // LATIN SMALL LETTER A WITH STROKE..LATIN SMAL
	{0x2C67, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER H WITH DESCENDER
	{0x2C68, 0x0, propertyPVALID},           // LATIN SMALL LETTER H WITH DESCENDER
	{0x2C69, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH DESCENDER
	{0x2C6A, 0x0, propertyPVALID},           // LATIN SMALL LETTER K WITH DESCENDER
	{0x2C6B, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Z WITH DESCENDER
	{0x2C6C, 0x0, propertyPVALID},           // LATIN SMALL LETTER Z WITH DESCENDER
	{0x2C6D, 0x2C70, propertyDISALLOWED},    // LATIN CAPITAL LETTER ALPHA..LATIN CAPITAL LE
	{0x2C71, 0x0, propertyPVALID},           // LATIN SMALL LETTER V WITH RIGHT HOOK
	{0x2C72, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER W WITH HOOK
	{0x2C73, 0x2C74, propertyPVALID},        // LATIN SMALL LETTER W WITH HOOK..LATIN SMALL
	{0x2C75, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER HALF H
	{0x2C76, 0x2C7B, propertyPVALID},        // LATIN SMALL LETTER HALF H..LATIN LETTER SMAL
	{0x2C7C, 0x2C80, propertyDISALLOWED},    // LATIN SUBSCRIPT SMALL LETTER J..COPTIC CAPIT
	{0x2C81, 0x0, propertyPVALID},           // COPTIC SMALL LETTER ALFA
	{0x2C82, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER VIDA
	{0x2C83, 0x0, propertyPVALID},           // COPTIC SMALL LETTER VIDA
	{0x2C84, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER GAMMA
	{0x2C85, 0x0, propertyPVALID},           // COPTIC SMALL LETTER GAMMA
	{0x2C86, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER DALDA
	{0x2C87, 0x0, propertyPVALID},           // COPTIC SMALL LETTER DALDA
	{0x2C88, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER EIE
	{0x2C89, 0x0, propertyPVALID},           // COPTIC SMALL LETTER EIE
	{0x2C8A, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER SOU
	{0x2C8B, 0x0, propertyPVALID},           // COPTIC SMALL LETTER SOU
	{0x2C8C, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER ZATA
	{0x2C8D, 0x0, propertyPVALID},           // COPTIC SMALL LETTER ZATA
	{0x2C8E, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER HATE
	{0x2C8F, 0x0, propertyPVALID},           // COPTIC SMALL LETTER HATE
	{0x2C90, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER THETHE
	{0x2C91, 0x0, propertyPVALID},           // COPTIC SMALL LETTER THETHE
	{0x2C92, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER IAUDA
	{0x2C93, 0x0, propertyPVALID},           // COPTIC SMALL LETTER IAUDA
	{0x2C94, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER KAPA
	{0x2C95, 0x0, propertyPVALID},           // COPTIC SMALL LETTER KAPA
	{0x2C96, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER LAULA
	{0x2C97, 0x0, propertyPVALID},           // COPTIC SMALL LETTER LAULA
	{0x2C98, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER MI
	{0x2C99, 0x0, propertyPVALID},           // COPTIC SMALL LETTER MI
	{0x2C9A, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER NI
	{0x2C9B, 0x0, propertyPVALID},           // COPTIC SMALL LETTER NI
	{0x2C9C, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER KSI
	{0x2C9D, 0x0, propertyPVALID},           // COPTIC SMALL LETTER KSI
	{0x2C9E, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER O
	{0x2C9F, 0x0, propertyPVALID},           // COPTIC SMALL LETTER O
	{0x2CA0, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER PI
	{0x2CA1, 0x0, propertyPVALID},           // COPTIC SMALL LETTER PI
	{0x2CA2, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER RO
	{0x2CA3, 0x0, propertyPVALID},           // COPTIC SMALL LETTER RO
	{0x2CA4, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER SIMA
	{0x2CA5, 0x0, propertyPVALID},           // COPTIC SMALL LETTER SIMA
	{0x2CA6, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER TAU
	{0x2CA7, 0x0, propertyPVALID},           // COPTIC SMALL LETTER TAU
	{0x2CA8, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER UA
	{0x2CA9, 0x0, propertyPVALID},           // COPTIC SMALL LETTER UA
	{0x2CAA, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER FI
	{0x2CAB, 0x0, propertyPVALID},           // COPTIC SMALL LETTER FI
	{0x2CAC, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER KHI
	{0x2CAD, 0x0, propertyPVALID},           // COPTIC SMALL LETTER KHI
	{0x2CAE, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER PSI
	{0x2CAF, 0x0, propertyPVALID},           // COPTIC SMALL LETTER PSI
	{0x2CB0, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OOU
	{0x2CB1, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OOU
	{0x2CB2, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER DIALECT-P ALEF
	{0x2CB3, 0x0, propertyPVALID},           // COPTIC SMALL LETTER DIALECT-P ALEF
	{0x2CB4, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC AIN
	{0x2CB5, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC AIN
	{0x2CB6, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
	{0x2CB7, 0x0, propertyPVALID},           // COPTIC SMALL LETTER CRYPTOGRAMMIC EIE
	{0x2CB8, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER DIALECT-P KAPA
	{0x2CB9, 0x0, propertyPVALID},           // COPTIC SMALL LETTER DIALECT-P KAPA
	{0x2CBA, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER DIALECT-P NI
	{0x2CBB, 0x0, propertyPVALID},           // COPTIC SMALL LETTER DIALECT-P NI
	{0x2CBC, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
	{0x2CBD, 0x0, propertyPVALID},           // COPTIC SMALL LETTER CRYPTOGRAMMIC NI
	{0x2CBE, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC OOU
	{0x2CBF, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC OOU
	{0x2CC0, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER SAMPI
	{0x2CC1, 0x0, propertyPVALID},           // COPTIC SMALL LETTER SAMPI
	{0x2CC2, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER CROSSED SHEI
	{0x2CC3, 0x0, propertyPVALID},           // COPTIC SMALL LETTER CROSSED SHEI
	{0x2CC4, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC SHEI
	{0x2CC5, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC SHEI
	{0x2CC6, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC ESH
	{0x2CC7, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC ESH
	{0x2CC8, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER AKHMIMIC KHEI
	{0x2CC9, 0x0, propertyPVALID},           // COPTIC SMALL LETTER AKHMIMIC KHEI
	{0x2CCA, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER DIALECT-P HORI
	{0x2CCB, 0x0, propertyPVALID},           // COPTIC SMALL LETTER DIALECT-P HORI
	{0x2CCC, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC HORI
	{0x2CCD, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC HORI
	{0x2CCE, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC HA
	{0x2CCF, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC HA
	{0x2CD0, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER L-SHAPED HA
	{0x2CD1, 0x0, propertyPVALID},           // COPTIC SMALL LETTER L-SHAPED HA
	{0x2CD2, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC HEI
	{0x2CD3, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC HEI
	{0x2CD4, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC HAT
	{0x2CD5, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC HAT
	{0x2CD6, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC GANGIA
	{0x2CD7, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC GANGIA
	{0x2CD8, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC DJA
	{0x2CD9, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC DJA
	{0x2CDA, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD COPTIC SHIMA
	{0x2CDB, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD COPTIC SHIMA
	{0x2CDC, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
	{0x2CDD, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD NUBIAN SHIMA
	{0x2CDE, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD NUBIAN NGI
	{0x2CDF, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD NUBIAN NGI
	{0x2CE0, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD NUBIAN NYI
	{0x2CE1, 0x0, propertyPVALID},           // COPTIC SMALL LETTER OLD NUBIAN NYI
	{0x2CE2, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER OLD NUBIAN WAU
	{0x2CE3, 0x2CE4, propertyPVALID},        // COPTIC SMALL LETTER OLD NUBIAN WAU..COPTIC S
	{0x2CE5, 0x2CEB, propertyDISALLOWED},    // COPTIC SYMBOL MI RO..COPTIC CAPITAL LETTER C
	{0x2CEC, 0x0, propertyPVALID},           // COPTIC SMALL LETTER CRYPTOGRAMMIC SHEI
	{0x2CED, 0x0, propertyDISALLOWED},       // COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
	{0x2CEE, 0x2CF1, propertyPVALID},        // COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA..CO
	{0x2CF2, 0x2CF8, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2CF9, 0x2CFF, propertyDISALLOWED},    // COPTIC OLD NUBIAN FULL STOP..COPTIC MORPHOLO
	{0x2D00, 0x2D25, propertyPVALID},        // GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LET
	{0x2D26, 0x2D2F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2D30, 0x2D65, propertyPVALID},        // TIFINAGH LETTER YA..TIFINAGH LETTER YAZZ
	{0x2D66, 0x2D6E, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2D6F, 0x0, propertyDISALLOWED},       // TIFINAGH MODIFIER LETTER LABIALIZATION MARK
	{0x2D70, 0x2D7F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2D80, 0x2D96, propertyPVALID},        // ETHIOPIC SYLLABLE LOA..ETHIOPIC SYLLABLE GGW
	{0x2D97, 0x2D9F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2DA0, 0x2DA6, propertyPVALID},        // ETHIOPIC SYLLABLE SSA..ETHIOPIC SYLLABLE SSO
	{0x2DA7, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2DA8, 0x2DAE, propertyPVALID},        // ETHIOPIC SYLLABLE CCA..ETHIOPIC SYLLABLE CCO
	{0x2DAF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2DB0, 0x2DB6, propertyPVALID},        // ETHIOPIC SYLLABLE ZZA..ETHIOPIC SYLLABLE ZZO
	{0x2DB7, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2DB8, 0x2DBE, propertyPVALID},        // ETHIOPIC SYLLABLE CCHA..ETHIOPIC SYLLABLE CC
	{0x2DBF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2DC0, 0x2DC6, propertyPVALID},        // ETHIOPIC SYLLABLE QYA..ETHIOPIC SYLLABLE QYO
	{0x2DC7, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2DC8, 0x2DCE, propertyPVALID},        // ETHIOPIC SYLLABLE KYA..ETHIOPIC SYLLABLE KYO
	{0x2DCF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2DD0, 0x2DD6, propertyPVALID},        // ETHIOPIC SYLLABLE XYA..ETHIOPIC SYLLABLE XYO
	{0x2DD7, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2DD8, 0x2DDE, propertyPVALID},        // ETHIOPIC SYLLABLE GYA..ETHIOPIC SYLLABLE GYO
	{0x2DDF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2DE0, 0x2DFF, propertyPVALID},        // COMBINING CYRILLIC LETTER BE..COMBINING CYRI
	{0x2E00, 0x2E2E, propertyDISALLOWED},    // RIGHT ANGLE SUBSTITUTION MARKER..REVERSED QU
	{0x2E2F, 0x0, propertyPVALID},           // VERTICAL TILDE
	{0x2E30, 0x2E31, propertyDISALLOWED},    // RING POINT..WORD SEPARATOR MIDDLE DOT
	{0x2E32, 0x2E7F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2E80, 0x2E99, propertyDISALLOWED},    // CJK RADICAL REPEAT..CJK RADICAL RAP
	{0x2E9A, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x2E9B, 0x2EF3, propertyDISALLOWED},    // CJK RADICAL CHOKE..CJK RADICAL C-SIMPLIFIED
	{0x2EF4, 0x2EFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2F00, 0x2FD5, propertyDISALLOWED},    // KANGXI RADICAL ONE..KANGXI RADICAL FLUTE
	{0x2FD6, 0x2FEF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x2FF0, 0x2FFB, propertyDISALLOWED},    // IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO RI
	{0x2FFC, 0x2FFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x3000, 0x3004, propertyDISALLOWED},    // IDEOGRAPHIC SPACE..JAPANESE INDUSTRIAL STAND
	{0x3005, 0x3007, propertyPVALID},        // IDEOGRAPHIC ITERATION MARK..IDEOGRAPHIC NUMB
	{0x3008, 0x3029, propertyDISALLOWED},    // LEFT ANGLE BRACKET..HANGZHOU NUMERAL NINE
	{0x302A, 0x302D, propertyPVALID},        // IDEOGRAPHIC LEVEL TONE MARK..IDEOGRAPHIC ENT
	{0x302E, 0x303B, propertyDISALLOWED},    // HANGUL SINGLE DOT TONE MARK..VERTICAL IDEOGR
	{0x303C, 0x0, propertyPVALID},           // MASU MARK
	{0x303D, 0x303F, propertyDISALLOWED},    // PART ALTERNATION MARK..IDEOGRAPHIC HALF FILL
	{0x3040, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x3041, 0x3096, propertyPVALID},        // HIRAGANA LETTER SMALL A..HIRAGANA LETTER SMA
	{0x3097, 0x3098, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x3099, 0x309A, propertyPVALID},        // COMBINING KATAKANA-HIRAGANA VOICED SOUND MAR
	{0x309B, 0x309C, propertyDISALLOWED},    // KATAKANA-HIRAGANA VOICED SOUND MARK..KATAKAN
	{0x309D, 0x309E, propertyPVALID},        // HIRAGANA ITERATION MARK..HIRAGANA VOICED ITE
	{0x309F, 0x30A0, propertyDISALLOWED},    // HIRAGANA DIGRAPH YORI..KATAKANA-HIRAGANA DOU
	{0x30A1, 0x30FA, propertyPVALID},        // KATAKANA LETTER SMALL A..KATAKANA LETTER VO
	{0x30FB, 0x0, propertyCONTEXTO},         // KATAKANA MIDDLE DOT
	{0x30FC, 0x30FE, propertyPVALID},        // KATAKANA-HIRAGANA PROLONGED SOUND MARK..KATA
	{0x30FF, 0x0, propertyDISALLOWED},       // KATAKANA DIGRAPH KOTO
	{0x3100, 0x3104, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x3105, 0x312D, propertyPVALID},        // BOPOMOFO LETTER B..BOPOMOFO LETTER IH
	{0x312E, 0x3130, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x3131, 0x318E, propertyDISALLOWED},    // HANGUL LETTER KIYEOK..HANGUL LETTER ARAEAE
	{0x318F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x3190, 0x319F, propertyDISALLOWED},    // IDEOGRAPHIC ANNOTATION LINKING MARK..IDEOGRA
	{0x31A0, 0x31B7, propertyPVALID},        // BOPOMOFO LETTER BU..BOPOMOFO FINAL LETTER H
	{0x31B8, 0x31BF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x31C0, 0x31E3, propertyDISALLOWED},    // CJK STROKE T..CJK STROKE Q
	{0x31E4, 0x31EF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x31F0, 0x31FF, propertyPVALID},        // KATAKANA LETTER SMALL KU..KATAKANA LETTER SM
	{0x3200, 0x321E, propertyDISALLOWED},    // PARENTHESIZED HANGUL KIYEOK..PARENTHESIZED K
	{0x321F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x3220, 0x32FE, propertyDISALLOWED},    // PARENTHESIZED IDEOGRAPH ONE..CIRCLED KATAKAN
	{0x32FF, 0x0, propertyUNASSIGNED},       // <reserved>
	{0x3300, 0x33FF, propertyDISALLOWED},    // SQUARE APAATO..SQUARE GAL
	{0x3400, 0x4DB5, propertyPVALID},        // <CJK Ideograph Extension A>..<CJK Ideograph
	{0x4DB6, 0x4DBF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0x4DC0, 0x4DFF, propertyDISALLOWED},    // HEXAGRAM FOR THE CREATIVE HEAVEN..HEXAGRAM F
	{0x4E00, 0x9FCB, propertyPVALID},        // <CJK Ideograph>..<CJK Ideograph>
	{0x9FCC, 0x9FFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA000, 0xA48C, propertyPVALID},        // YI SYLLABLE IT..YI SYLLABLE YYR
	{0xA48D, 0xA48F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA490, 0xA4C6, propertyDISALLOWED},    // YI RADICAL QOT..YI RADICAL KE
	{0xA4C7, 0xA4CF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA4D0, 0xA4FD, propertyPVALID},        // LISU LETTER BA..LISU LETTER TONE MYA JEU
	{0xA4FE, 0xA4FF, propertyDISALLOWED},    // LISU PUNCTUATION COMMA..LISU PUNCTUATION FUL
	{0xA500, 0xA60C, propertyPVALID},        // VAI SYLLABLE EE..VAI SYLLABLE LENGTHENER
	{0xA60D, 0xA60F, propertyDISALLOWED},    // VAI COMMA..VAI QUESTION MARK
	{0xA610, 0xA62B, propertyPVALID},        // VAI SYLLABLE NDOLE FA..VAI SYLLABLE NDOLE DO
	{0xA62C, 0xA63F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA640, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ZEMLYA
	{0xA641, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ZEMLYA
	{0xA642, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER DZELO
	{0xA643, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER DZELO
	{0xA644, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER REVERSED DZE
	{0xA645, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER REVERSED DZE
	{0xA646, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IOTA
	{0xA647, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IOTA
	{0xA648, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER DJERV
	{0xA649, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER DJERV
	{0xA64A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER MONOGRAPH UK
	{0xA64B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER MONOGRAPH UK
	{0xA64C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER BROAD OMEGA
	{0xA64D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER BROAD OMEGA
	{0xA64E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER NEUTRAL YER
	{0xA64F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER NEUTRAL YER
	{0xA650, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER YERU WITH BACK YER
	{0xA651, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER YERU WITH BACK YER
	{0xA652, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IOTIFIED YAT
	{0xA653, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IOTIFIED YAT
	{0xA654, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER REVERSED YU
	{0xA655, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER REVERSED YU
	{0xA656, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IOTIFIED A
	{0xA657, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IOTIFIED A
	{0xA658, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS
	{0xA659, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER CLOSED LITTLE YUS
	{0xA65A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER BLENDED YUS
	{0xA65B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER BLENDED YUS
	{0xA65C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITT
	{0xA65D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER IOTIFIED CLOSED LITTLE
	{0xA65E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER YN
	{0xA65F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER YN
	{0xA660, 0xA661, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA662, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER SOFT DE
	{0xA663, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SOFT DE
	{0xA664, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER SOFT EL
	{0xA665, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SOFT EL
	{0xA666, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER SOFT EM
	{0xA667, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SOFT EM
	{0xA668, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER MONOCULAR O
	{0xA669, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER MONOCULAR O
	{0xA66A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER BINOCULAR O
	{0xA66B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER BINOCULAR O
	{0xA66C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O
	{0xA66D, 0xA66F, propertyPVALID},        // CYRILLIC SMALL LETTER DOUBLE MONOCULAR O..CO
	{0xA670, 0xA673, propertyDISALLOWED},    // COMBINING CYRILLIC TEN MILLIONS SIGN..SLAVON
	{0xA674, 0xA67B, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA67C, 0xA67D, propertyPVALID},        // COMBINING CYRILLIC KAVYKA..COMBINING CYRILLI
	{0xA67E, 0x0, propertyDISALLOWED},       // CYRILLIC KAVYKA
	{0xA67F, 0x0, propertyPVALID},           // CYRILLIC PAYEROK
	{0xA680, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER DWE
	{0xA681, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER DWE
	{0xA682, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER DZWE
	{0xA683, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER DZWE
	{0xA684, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER ZHWE
	{0xA685, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER ZHWE
	{0xA686, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER CCHE
	{0xA687, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER CCHE
	{0xA688, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER DZZE
	{0xA689, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER DZZE
	{0xA68A, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK
	{0xA68B, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER TE WITH MIDDLE HOOK
	{0xA68C, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER TWE
	{0xA68D, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER TWE
	{0xA68E, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER TSWE
	{0xA68F, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER TSWE
	{0xA690, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER TSSE
	{0xA691, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER TSSE
	{0xA692, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER TCHE
	{0xA693, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER TCHE
	{0xA694, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER HWE
	{0xA695, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER HWE
	{0xA696, 0x0, propertyDISALLOWED},       // CYRILLIC CAPITAL LETTER SHWE
	{0xA697, 0x0, propertyPVALID},           // CYRILLIC SMALL LETTER SHWE
	{0xA698, 0xA69F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA6A0, 0xA6E5, propertyPVALID},        // BAMUM LETTER A..BAMUM LETTER KI
	{0xA6E6, 0xA6EF, propertyDISALLOWED},    // BAMUM LETTER MO..BAMUM LETTER KOGHOM
	{0xA6F0, 0xA6F1, propertyPVALID},        // BAMUM COMBINING MARK KOQNDON..BAMUM COMBININ
	{0xA6F2, 0xA6F7, propertyDISALLOWED},    // BAMUM NJAEMLI..BAMUM QUESTION MARK
	{0xA6F8, 0xA6FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA700, 0xA716, propertyDISALLOWED},    // MODIFIER LETTER CHINESE TONE YIN PING..MODIF
	{0xA717, 0xA71F, propertyPVALID},        // MODIFIER LETTER DOT VERTICAL BAR..MODIFIER L
	{0xA720, 0xA722, propertyDISALLOWED},    // MODIFIER LETTER STRESS AND HIGH TONE..LATIN
	{0xA723, 0x0, propertyPVALID},           // LATIN SMALL LETTER EGYPTOLOGICAL ALEF
	{0xA724, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER EGYPTOLOGICAL AIN
	{0xA725, 0x0, propertyPVALID},           // LATIN SMALL LETTER EGYPTOLOGICAL AIN
	{0xA726, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER HENG
	{0xA727, 0x0, propertyPVALID},           // LATIN SMALL LETTER HENG
	{0xA728, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER TZ
	{0xA729, 0x0, propertyPVALID},           // LATIN SMALL LETTER TZ
	{0xA72A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER TRESILLO
	{0xA72B, 0x0, propertyPVALID},           // LATIN SMALL LETTER TRESILLO
	{0xA72C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER CUATRILLO
	{0xA72D, 0x0, propertyPVALID},           // LATIN SMALL LETTER CUATRILLO
	{0xA72E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER CUATRILLO WITH COMMA
	{0xA72F, 0xA731, propertyPVALID},        // LATIN SMALL LETTER CUATRILLO WITH COMMA..LAT
	{0xA732, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER AA
	{0xA733, 0x0, propertyPVALID},           // LATIN SMALL LETTER AA
	{0xA734, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER AO
	{0xA735, 0x0, propertyPVALID},           // LATIN SMALL LETTER AO
	{0xA736, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER AU
	{0xA737, 0x0, propertyPVALID},           // LATIN SMALL LETTER AU
	{0xA738, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER AV
	{0xA739, 0x0, propertyPVALID},           // LATIN SMALL LETTER AV
	{0xA73A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR
	{0xA73B, 0x0, propertyPVALID},           // LATIN SMALL LETTER AV WITH HORIZONTAL BAR
	{0xA73C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER AY
	{0xA73D, 0x0, propertyPVALID},           // LATIN SMALL LETTER AY
	{0xA73E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER REVERSED C WITH DOT
	{0xA73F, 0x0, propertyPVALID},           // LATIN SMALL LETTER REVERSED C WITH DOT
	{0xA740, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH STROKE
	{0xA741, 0x0, propertyPVALID},           // LATIN SMALL LETTER K WITH STROKE
	{0xA742, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH DIAGONAL STROKE
	{0xA743, 0x0, propertyPVALID},           // LATIN SMALL LETTER K WITH DIAGONAL STROKE
	{0xA744, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER K WITH STROKE AND DIAGO
	{0xA745, 0x0, propertyPVALID},           // LATIN SMALL LETTER K WITH STROKE AND DIAGONA
	{0xA746, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER BROKEN L
	{0xA747, 0x0, propertyPVALID},           // LATIN SMALL LETTER BROKEN L
	{0xA748, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER L WITH HIGH STROKE
	{0xA749, 0x0, propertyPVALID},           // LATIN SMALL LETTER L WITH HIGH STROKE
	{0xA74A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH LONG STROKE OVER
	{0xA74B, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH LONG STROKE OVERLA
	{0xA74C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER O WITH LOOP
	{0xA74D, 0x0, propertyPVALID},           // LATIN SMALL LETTER O WITH LOOP
	{0xA74E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER OO
	{0xA74F, 0x0, propertyPVALID},           // LATIN SMALL LETTER OO
	{0xA750, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER P WITH STROKE THROUGH D
	{0xA751, 0x0, propertyPVALID},           // LATIN SMALL LETTER P WITH STROKE THROUGH DES
	{0xA752, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER P WITH FLOURISH
	{0xA753, 0x0, propertyPVALID},           // LATIN SMALL LETTER P WITH FLOURISH
	{0xA754, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER P WITH SQUIRREL TAIL
	{0xA755, 0x0, propertyPVALID},           // LATIN SMALL LETTER P WITH SQUIRREL TAIL
	{0xA756, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Q WITH STROKE THROUGH D
	{0xA757, 0x0, propertyPVALID},           // LATIN SMALL LETTER Q WITH STROKE THROUGH DES
	{0xA758, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER Q WITH DIAGONAL STROKE
	{0xA759, 0x0, propertyPVALID},           // LATIN SMALL LETTER Q WITH DIAGONAL STROKE
	{0xA75A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER R ROTUNDA
	{0xA75B, 0x0, propertyPVALID},           // LATIN SMALL LETTER R ROTUNDA
	{0xA75C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER RUM ROTUNDA
	{0xA75D, 0x0, propertyPVALID},           // LATIN SMALL LETTER RUM ROTUNDA
	{0xA75E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER V WITH DIAGONAL STROKE
	{0xA75F, 0x0, propertyPVALID},           // LATIN SMALL LETTER V WITH DIAGONAL STROKE
	{0xA760, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER VY
	{0xA761, 0x0, propertyPVALID},           // LATIN SMALL LETTER VY
	{0xA762, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER VISIGOTHIC Z
	{0xA763, 0x0, propertyPVALID},           // LATIN SMALL LETTER VISIGOTHIC Z
	{0xA764, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER THORN WITH STROKE
	{0xA765, 0x0, propertyPVALID},           // LATIN SMALL LETTER THORN WITH STROKE
	{0xA766, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER THORN WITH STROKE THROU
	{0xA767, 0x0, propertyPVALID},           // LATIN SMALL LETTER THORN WITH STROKE THROUGH
	{0xA768, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER VEND
	{0xA769, 0x0, propertyPVALID},           // LATIN SMALL LETTER VEND
	{0xA76A, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER ET
	{0xA76B, 0x0, propertyPVALID},           // LATIN SMALL LETTER ET
	{0xA76C, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER IS
	{0xA76D, 0x0, propertyPVALID},           // LATIN SMALL LETTER IS
	{0xA76E, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER CON
	{0xA76F, 0x0, propertyPVALID},           // LATIN SMALL LETTER CON
	{0xA770, 0x0, propertyDISALLOWED},       // MODIFIER LETTER US
	{0xA771, 0xA778, propertyPVALID},        // LATIN SMALL LETTER DUM..LATIN SMALL LETTER U
	{0xA779, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER INSULAR D
	{0xA77A, 0x0, propertyPVALID},           // LATIN SMALL LETTER INSULAR D
	{0xA77B, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER INSULAR F
	{0xA77C, 0x0, propertyPVALID},           // LATIN SMALL LETTER INSULAR F
	{0xA77D, 0xA77E, propertyDISALLOWED},    // LATIN CAPITAL LETTER INSULAR G..LATIN CAPITA
	{0xA77F, 0x0, propertyPVALID},           // LATIN SMALL LETTER TURNED INSULAR G
	{0xA780, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER TURNED L
	{0xA781, 0x0, propertyPVALID},           // LATIN SMALL LETTER TURNED L
	{0xA782, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER INSULAR R
	{0xA783, 0x0, propertyPVALID},           // LATIN SMALL LETTER INSULAR R
	{0xA784, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER INSULAR S
	{0xA785, 0x0, propertyPVALID},           // LATIN SMALL LETTER INSULAR S
	{0xA786, 0x0, propertyDISALLOWED},       // LATIN CAPITAL LETTER INSULAR T
	{0xA787, 0xA788, propertyPVALID},        // LATIN SMALL LETTER INSULAR T..MODIFIER LETTE
	{0xA789, 0xA78B, propertyDISALLOWED},    // MODIFIER LETTER COLON..LATIN CAPITAL LETTER
	{0xA78C, 0x0, propertyPVALID},           // LATIN SMALL LETTER SALTILLO
	{0xA78D, 0xA7FA, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA7FB, 0xA827, propertyPVALID},        // LATIN EPIGRAPHIC LETTER REVERSED F..SYLOTI N
	{0xA828, 0xA82B, propertyDISALLOWED},    // SYLOTI NAGRI POETRY MARK-1..SYLOTI NAGRI POE
	{0xA82C, 0xA82F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA830, 0xA839, propertyDISALLOWED},    // NORTH INDIC FRACTION ONE QUARTER..NORTH INDI
	{0xA83A, 0xA83F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA840, 0xA873, propertyPVALID},        // PHAGS-PA LETTER KA..PHAGS-PA LETTER CANDRABI
	{0xA874, 0xA877, propertyDISALLOWED},    // PHAGS-PA SINGLE HEAD MARK..PHAGS-PA MARK DOU
	{0xA878, 0xA87F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA880, 0xA8C4, propertyPVALID},        // SAURASHTRA SIGN ANUSVARA..SAURASHTRA SIGN VI
	{0xA8C5, 0xA8CD, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA8CE, 0xA8CF, propertyDISALLOWED},    // SAURASHTRA DANDA..SAURASHTRA DOUBLE DANDA
	{0xA8D0, 0xA8D9, propertyPVALID},        // SAURASHTRA DIGIT ZERO..SAURASHTRA DIGIT NINE
	{0xA8DA, 0xA8DF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA8E0, 0xA8F7, propertyPVALID},        // COMBINING DEVANAGARI DIGIT ZERO..DEVANAGARI
	{0xA8F8, 0xA8FA, propertyDISALLOWED},    // DEVANAGARI SIGN PUSHPIKA..DEVANAGARI CARET
	{0xA8FB, 0x0, propertyPVALID},           // DEVANAGARI HEADSTROKE
	{0xA8FC, 0xA8FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA900, 0xA92D, propertyPVALID},        // KAYAH LI DIGIT ZERO..KAYAH LI TONE CALYA PLO
	{0xA92E, 0xA92F, propertyDISALLOWED},    // KAYAH LI SIGN CWI..KAYAH LI SIGN SHYA
	{0xA930, 0xA953, propertyPVALID},        // REJANG LETTER KA..REJANG VIRAMA
	{0xA954, 0xA95E, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA95F, 0xA97C, propertyDISALLOWED},    // REJANG SECTION MARK..HANGUL CHOSEONG SSANGYE
	{0xA97D, 0xA97F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA980, 0xA9C0, propertyPVALID},        // JAVANESE SIGN PANYANGGA..JAVANESE PANGKON
	{0xA9C1, 0xA9CD, propertyDISALLOWED},    // JAVANESE LEFT RERENGGAN..JAVANESE TURNED PAD
	{0xA9CE, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xA9CF, 0xA9D9, propertyPVALID},        // JAVANESE PANGRANGKEP..JAVANESE DIGIT NINE
	{0xA9DA, 0xA9DD, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xA9DE, 0xA9DF, propertyDISALLOWED},    // JAVANESE PADA TIRTA TUMETES..JAVANESE PADA I
	{0xA9E0, 0xA9FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xAA00, 0xAA36, propertyPVALID},        // CHAM LETTER A..CHAM CONSONANT SIGN WA
	{0xAA37, 0xAA3F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xAA40, 0xAA4D, propertyPVALID},        // CHAM LETTER FINAL K..CHAM CONSONANT SIGN FIN
	{0xAA4E, 0xAA4F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xAA50, 0xAA59, propertyPVALID},        // CHAM DIGIT ZERO..CHAM DIGIT NINE
	{0xAA5A, 0xAA5B, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xAA5C, 0xAA5F, propertyDISALLOWED},    // CHAM PUNCTUATION SPIRAL..CHAM PUNCTUATION TR
	{0xAA60, 0xAA76, propertyPVALID},        // MYANMAR LETTER KHAMTI GA..MYANMAR LOGOGRAM K
	{0xAA77, 0xAA79, propertyDISALLOWED},    // MYANMAR SYMBOL AITON EXCLAMATION..MYANMAR SY
	{0xAA7A, 0xAA7B, propertyPVALID},        // MYANMAR LETTER AITON RA..MYANMAR SIGN PAO KA
	{0xAA7C, 0xAA7F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xAA80, 0xAAC2, propertyPVALID},        // TAI VIET LETTER LOW KO..TAI VIET TONE MAI SO
	{0xAAC3, 0xAADA, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xAADB, 0xAADD, propertyPVALID},        // TAI VIET SYMBOL KON..TAI VIET SYMBOL SAM
	{0xAADE, 0xAADF, propertyDISALLOWED},    // TAI VIET SYMBOL HO HOI..TAI VIET SYMBOL KOI
	{0xAAE0, 0xABBF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xABC0, 0xABEA, propertyPVALID},        // MEETEI MAYEK LETTER KOK..MEETEI MAYEK VOWEL
	{0xABEB, 0x0, propertyDISALLOWED},       // MEETEI MAYEK CHEIKHEI
	{0xABEC, 0xABED, propertyPVALID},        // MEETEI MAYEK LUM IYEK..MEETEI MAYEK APUN IYE
	{0xABEE, 0xABEF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xABF0, 0xABF9, propertyPVALID},        // MEETEI MAYEK DIGIT ZERO..MEETEI MAYEK DIGIT
	{0xABFA, 0xABFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xAC00, 0xD7A3, propertyPVALID},        // <Hangul Syllable>..<Hangul Syllable>
	{0xD7A4, 0xD7AF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xD7B0, 0xD7C6, propertyDISALLOWED},    // HANGUL JUNGSEONG O-YEO..HANGUL JUNGSEONG ARA
	{0xD7C7, 0xD7CA, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xD7CB, 0xD7FB, propertyDISALLOWED},    // HANGUL JONGSEONG NIEUN-RIEUL..HANGUL JONGSEO
	{0xD7FC, 0xD7FF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xD800, 0xFA0D, propertyDISALLOWED},    // <Non Private Use High Surrogate>..CJK COMPAT
	{0xFA0E, 0xFA0F, propertyPVALID},        // CJK COMPATIBILITY IDEOGRAPH-FA0E..CJK COMPAT
	{0xFA10, 0x0, propertyDISALLOWED},       // CJK COMPATIBILITY IDEOGRAPH-FA10
	{0xFA11, 0x0, propertyPVALID},           // CJK COMPATIBILITY IDEOGRAPH-FA11
	{0xFA12, 0x0, propertyDISALLOWED},       // CJK COMPATIBILITY IDEOGRAPH-FA12
	{0xFA13, 0xFA14, propertyPVALID},        // CJK COMPATIBILITY IDEOGRAPH-FA13..CJK COMPAT
	{0xFA15, 0xFA1E, propertyDISALLOWED},    // CJK COMPATIBILITY IDEOGRAPH-FA15..CJK COMPAT
	{0xFA1F, 0x0, propertyPVALID},           // CJK COMPATIBILITY IDEOGRAPH-FA1F
	{0xFA20, 0x0, propertyDISALLOWED},       // CJK COMPATIBILITY IDEOGRAPH-FA20
	{0xFA21, 0x0, propertyPVALID},           // CJK COMPATIBILITY IDEOGRAPH-FA21
	{0xFA22, 0x0, propertyDISALLOWED},       // CJK COMPATIBILITY IDEOGRAPH-FA22
	{0xFA23, 0xFA24, propertyPVALID},        // CJK COMPATIBILITY IDEOGRAPH-FA23..CJK COMPAT
	{0xFA25, 0xFA26, propertyDISALLOWED},    // CJK COMPATIBILITY IDEOGRAPH-FA25..CJK COMPAT
	{0xFA27, 0xFA29, propertyPVALID},        // CJK COMPATIBILITY IDEOGRAPH-FA27..CJK COMPAT
	{0xFA2A, 0xFA2D, propertyDISALLOWED},    // CJK COMPATIBILITY IDEOGRAPH-FA2A..CJK COMPAT
	{0xFA2E, 0xFA2F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFA30, 0xFA6D, propertyDISALLOWED},    // CJK COMPATIBILITY IDEOGRAPH-FA30..CJK COMPAT
	{0xFA6E, 0xFA6F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFA70, 0xFAD9, propertyDISALLOWED},    // CJK COMPATIBILITY IDEOGRAPH-FA70..CJK COMPAT
	{0xFADA, 0xFAFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFB00, 0xFB06, propertyDISALLOWED},    // LATIN SMALL LIGATURE FF..LATIN SMALL LIGATUR
	{0xFB07, 0xFB12, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFB13, 0xFB17, propertyDISALLOWED},    // ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SM
	{0xFB18, 0xFB1C, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFB1D, 0x0, propertyDISALLOWED},       // HEBREW LETTER YOD WITH HIRIQ
	{0xFB1E, 0x0, propertyPVALID},           // HEBREW POINT JUDEO-SPANISH VARIKA
	{0xFB1F, 0xFB36, propertyDISALLOWED},    // HEBREW LIGATURE YIDDISH YOD YOD PATAH..HEBRE
	{0xFB37, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFB38, 0xFB3C, propertyDISALLOWED},    // HEBREW LETTER TET WITH DAGESH..HEBREW LETTER
	{0xFB3D, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFB3E, 0x0, propertyDISALLOWED},       // HEBREW LETTER MEM WITH DAGESH
	{0xFB3F, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFB40, 0xFB41, propertyDISALLOWED},    // HEBREW LETTER NUN WITH DAGESH..HEBREW LETTER
	{0xFB42, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFB43, 0xFB44, propertyDISALLOWED},    // HEBREW LETTER FINAL PE WITH DAGESH..HEBREW L
	{0xFB45, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFB46, 0xFBB1, propertyDISALLOWED},    // HEBREW LETTER TSADI WITH DAGESH..ARABIC LETT
	{0xFBB2, 0xFBD2, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFBD3, 0xFD3F, propertyDISALLOWED},    // ARABIC LETTER NG ISOLATED FORM..ORNATE RIGHT
	{0xFD40, 0xFD4F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFD50, 0xFD8F, propertyDISALLOWED},    // ARABIC LIGATURE TEH WITH JEEM WITH MEEM INIT
	{0xFD90, 0xFD91, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFD92, 0xFDC7, propertyDISALLOWED},    // ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INI
	{0xFDC8, 0xFDCF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFDD0, 0xFDFD, propertyDISALLOWED},    // <noncharacter>..ARABIC LIGATURE BISMILLAH AR
	{0xFDFE, 0xFDFF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFE00, 0xFE19, propertyDISALLOWED},    // VARIATION SELECTOR-1..PRESENTATION FORM FOR
	{0xFE1A, 0xFE1F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFE20, 0xFE26, propertyPVALID},        // COMBINING LIGATURE LEFT HALF..COMBINING CONJ
	{0xFE27, 0xFE2F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFE30, 0xFE52, propertyDISALLOWED},    // PRESENTATION FORM FOR VERTICAL TWO DOT LEADE
	{0xFE53, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFE54, 0xFE66, propertyDISALLOWED},    // SMALL SEMICOLON..SMALL EQUALS SIGN
	{0xFE67, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFE68, 0xFE6B, propertyDISALLOWED},    // SMALL REVERSE SOLIDUS..SMALL COMMERCIAL AT
	{0xFE6C, 0xFE6F, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFE70, 0xFE72, propertyDISALLOWED},    // ARABIC FATHATAN ISOLATED FORM..ARABIC DAMMAT
	{0xFE73, 0x0, propertyPVALID},           // ARABIC TAIL FRAGMENT
	{0xFE74, 0x0, propertyDISALLOWED},       // ARABIC KASRATAN ISOLATED FORM
	{0xFE75, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFE76, 0xFEFC, propertyDISALLOWED},    // ARABIC FATHA ISOLATED FORM..ARABIC LIGATURE
	{0xFEFD, 0xFEFE, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFEFF, 0x0, propertyDISALLOWED},       // ZERO WIDTH NO-BREAK SPACE
	{0xFF00, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFF01, 0xFFBE, propertyDISALLOWED},    // FULLWIDTH EXCLAMATION MARK..HALFWIDTH HANGUL
	{0xFFBF, 0xFFC1, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFFC2, 0xFFC7, propertyDISALLOWED},    // HALFWIDTH HANGUL LETTER A..HALFWIDTH HANGUL
	{0xFFC8, 0xFFC9, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFFCA, 0xFFCF, propertyDISALLOWED},    // HALFWIDTH HANGUL LETTER YEO..HALFWIDTH HANGU
	{0xFFD0, 0xFFD1, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFFD2, 0xFFD7, propertyDISALLOWED},    // HALFWIDTH HANGUL LETTER YO..HALFWIDTH HANGUL
	{0xFFD8, 0xFFD9, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFFDA, 0xFFDC, propertyDISALLOWED},    // HALFWIDTH HANGUL LETTER EU..HALFWIDTH HANGUL
	{0xFFDD, 0xFFDF, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFFE0, 0xFFE6, propertyDISALLOWED},    // FULLWIDTH CENT SIGN..FULLWIDTH WON SIGN
	{0xFFE7, 0x0, propertyUNASSIGNED},       // <reserved>
	{0xFFE8, 0xFFEE, propertyDISALLOWED},    // HALFWIDTH FORMS LIGHT VERTICAL..HALFWIDTH WH
	{0xFFEF, 0xFFF8, propertyUNASSIGNED},    // <reserved>..<reserved>
	{0xFFF9, 0xFFFF, propertyDISALLOWED},    // INTERLINEAR ANNOTATION ANCHOR..<noncharacter
	{0x10000, 0x1000B, propertyPVALID},      // LINEAR B SYLLABLE B008 A..LINEAR B SYLLABLE
	{0x1000C, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1000D, 0x10026, propertyPVALID},      // LINEAR B SYLLABLE B036 JO..LINEAR B SYLLABLE
	{0x10027, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x10028, 0x1003A, propertyPVALID},      // LINEAR B SYLLABLE B060 RA..LINEAR B SYLLABLE
	{0x1003B, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1003C, 0x1003D, propertyPVALID},      // LINEAR B SYLLABLE B017 ZA..LINEAR B SYLLABLE
	{0x1003E, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1003F, 0x1004D, propertyPVALID},      // LINEAR B SYLLABLE B020 ZO..LINEAR B SYLLABLE
	{0x1004E, 0x1004F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10050, 0x1005D, propertyPVALID},      // LINEAR B SYMBOL B018..LINEAR B SYMBOL B089
	{0x1005E, 0x1007F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10080, 0x100FA, propertyPVALID},      // LINEAR B IDEOGRAM B100 MAN..LINEAR B IDEOGRA
	{0x100FB, 0x100FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10100, 0x10102, propertyDISALLOWED},  // AEGEAN WORD SEPARATOR LINE..AEGEAN CHECK MAR
	{0x10103, 0x10106, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10107, 0x10133, propertyDISALLOWED},  // AEGEAN NUMBER ONE..AEGEAN NUMBER NINETY THOU
	{0x10134, 0x10136, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10137, 0x1018A, propertyDISALLOWED},  // AEGEAN WEIGHT BASE UNIT..GREEK ZERO SIGN
	{0x1018B, 0x1018F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10190, 0x1019B, propertyDISALLOWED},  // ROMAN SEXTANS SIGN..ROMAN CENTURIAL SIGN
	{0x1019C, 0x101CF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x101D0, 0x101FC, propertyDISALLOWED},  // PHAISTOS DISC SIGN PEDESTRIAN..PHAISTOS DISC
	{0x101FD, 0x0, propertyPVALID},          // PHAISTOS DISC SIGN COMBINING OBLIQUE STROKE
	{0x101FE, 0x1027F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10280, 0x1029C, propertyPVALID},      // LYCIAN LETTER A..LYCIAN LETTER X
	{0x1029D, 0x1029F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x102A0, 0x102D0, propertyPVALID},      // CARIAN LETTER A..CARIAN LETTER UUU3
	{0x102D1, 0x102FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10300, 0x1031E, propertyPVALID},      // OLD ITALIC LETTER A..OLD ITALIC LETTER UU
	{0x1031F, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x10320, 0x10323, propertyDISALLOWED},  // OLD ITALIC NUMERAL ONE..OLD ITALIC NUMERAL F
	{0x10324, 0x1032F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10330, 0x10340, propertyPVALID},      // GOTHIC LETTER AHSA..GOTHIC LETTER PAIRTHRA
	{0x10341, 0x0, propertyDISALLOWED},      // GOTHIC LETTER NINETY
	{0x10342, 0x10349, propertyPVALID},      // GOTHIC LETTER RAIDA..GOTHIC LETTER OTHAL
	{0x1034A, 0x0, propertyDISALLOWED},      // GOTHIC LETTER NINE HUNDRED
	{0x1034B, 0x1037F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10380, 0x1039D, propertyPVALID},      // UGARITIC LETTER ALPA..UGARITIC LETTER SSU
	{0x1039E, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1039F, 0x0, propertyDISALLOWED},      // UGARITIC WORD DIVIDER
	{0x103A0, 0x103C3, propertyPVALID},      // OLD PERSIAN SIGN A..OLD PERSIAN SIGN HA
	{0x103C4, 0x103C7, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x103C8, 0x103CF, propertyPVALID},      // OLD PERSIAN SIGN AURAMAZDAA..OLD PERSIAN SIG
	{0x103D0, 0x103D5, propertyDISALLOWED},  // OLD PERSIAN WORD DIVIDER..OLD PERSIAN NUMBER
	{0x103D6, 0x103FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10400, 0x10427, propertyDISALLOWED},  // DESERET CAPITAL LETTER LONG I..DESERET CAPIT
	{0x10428, 0x1049D, propertyPVALID},      // DESERET SMALL LETTER LONG I..OSMANYA LETTER
	{0x1049E, 0x1049F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x104A0, 0x104A9, propertyPVALID},      // OSMANYA DIGIT ZERO..OSMANYA DIGIT NINE
	{0x104AA, 0x107FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10800, 0x10805, propertyPVALID},      // CYPRIOT SYLLABLE A..CYPRIOT SYLLABLE JA
	{0x10806, 0x10807, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10808, 0x0, propertyPVALID},          // CYPRIOT SYLLABLE JO
	{0x10809, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1080A, 0x10835, propertyPVALID},      // CYPRIOT SYLLABLE KA..CYPRIOT SYLLABLE WO
	{0x10836, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x10837, 0x10838, propertyPVALID},      // CYPRIOT SYLLABLE XA..CYPRIOT SYLLABLE XE
	{0x10839, 0x1083B, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1083C, 0x0, propertyPVALID},          // CYPRIOT SYLLABLE ZA
	{0x1083D, 0x1083E, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1083F, 0x10855, propertyPVALID},      // CYPRIOT SYLLABLE ZO..IMPERIAL ARAMAIC LETTER
	{0x10856, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x10857, 0x1085F, propertyDISALLOWED},  // IMPERIAL ARAMAIC SECTION SIGN..IMPERIAL ARAM
	{0x10860, 0x108FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10900, 0x10915, propertyPVALID},      // PHOENICIAN LETTER ALF..PHOENICIAN LETTER TAU
	{0x10916, 0x1091B, propertyDISALLOWED},  // PHOENICIAN NUMBER ONE..PHOENICIAN NUMBER THR
	{0x1091C, 0x1091E, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1091F, 0x0, propertyDISALLOWED},      // PHOENICIAN WORD SEPARATOR
	{0x10920, 0x10939, propertyPVALID},      // LYDIAN LETTER A..LYDIAN LETTER C
	{0x1093A, 0x1093E, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1093F, 0x0, propertyDISALLOWED},      // LYDIAN TRIANGULAR MARK
	{0x10940, 0x109FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10A00, 0x10A03, propertyPVALID},      // KHAROSHTHI LETTER A..KHAROSHTHI VOWEL SIGN V
	{0x10A04, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x10A05, 0x10A06, propertyPVALID},      // KHAROSHTHI VOWEL SIGN E..KHAROSHTHI VOWEL SI
	{0x10A07, 0x10A0B, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10A0C, 0x10A13, propertyPVALID},      // KHAROSHTHI VOWEL LENGTH MARK..KHAROSHTHI LET
	{0x10A14, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x10A15, 0x10A17, propertyPVALID},      // KHAROSHTHI LETTER CA..KHAROSHTHI LETTER JA
	{0x10A18, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x10A19, 0x10A33, propertyPVALID},      // KHAROSHTHI LETTER NYA..KHAROSHTHI LETTER TTT
	{0x10A34, 0x10A37, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10A38, 0x10A3A, propertyPVALID},      // KHAROSHTHI SIGN BAR ABOVE..KHAROSHTHI SIGN D
	{0x10A3B, 0x10A3E, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10A3F, 0x0, propertyPVALID},          // KHAROSHTHI VIRAMA
	{0x10A40, 0x10A47, propertyDISALLOWED},  // KHAROSHTHI DIGIT ONE..KHAROSHTHI NUMBER ONE
	{0x10A48, 0x10A4F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10A50, 0x10A58, propertyDISALLOWED},  // KHAROSHTHI PUNCTUATION DOT..KHAROSHTHI PUNCT
	{0x10A59, 0x10A5F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10A60, 0x10A7C, propertyPVALID},      // OLD SOUTH ARABIAN LETTER HE..OLD SOUTH ARABI
	{0x10A7D, 0x10A7F, propertyDISALLOWED},  // OLD SOUTH ARABIAN NUMBER ONE..OLD SOUTH ARAB
	{0x10A80, 0x10AFF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10B00, 0x10B35, propertyPVALID},      // AVESTAN LETTER A..AVESTAN LETTER HE
	{0x10B36, 0x10B38, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10B39, 0x10B3F, propertyDISALLOWED},  // AVESTAN ABBREVIATION MARK..LARGE ONE RING OV
	{0x10B40, 0x10B55, propertyPVALID},      // INSCRIPTIONAL PARTHIAN LETTER ALEPH..INSCRIP
	{0x10B56, 0x10B57, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10B58, 0x10B5F, propertyDISALLOWED},  // INSCRIPTIONAL PARTHIAN NUMBER ONE..INSCRIPTI
	{0x10B60, 0x10B72, propertyPVALID},      // INSCRIPTIONAL PAHLAVI LETTER ALEPH..INSCRIPT
	{0x10B73, 0x10B77, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10B78, 0x10B7F, propertyDISALLOWED},  // INSCRIPTIONAL PAHLAVI NUMBER ONE..INSCRIPTIO
	{0x10B80, 0x10BFF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10C00, 0x10C48, propertyPVALID},      // OLD TURKIC LETTER ORKHON A..OLD TURKIC LETTE
	{0x10C49, 0x10E5F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x10E60, 0x10E7E, propertyDISALLOWED},  // RUMI DIGIT ONE..RUMI FRACTION TWO THIRDS
	{0x10E7F, 0x1107F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x11080, 0x110BA, propertyPVALID},      // KAITHI SIGN CANDRABINDU..KAITHI SIGN NUKTA
	{0x110BB, 0x110C1, propertyDISALLOWED},  // KAITHI ABBREVIATION SIGN..KAITHI DOUBLE DAND
	{0x110C2, 0x11FFF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x12000, 0x1236E, propertyPVALID},      // CUNEIFORM SIGN A..CUNEIFORM SIGN ZUM
	{0x1236F, 0x123FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x12400, 0x12462, propertyDISALLOWED},  // CUNEIFORM NUMERIC SIGN TWO ASH..CUNEIFORM NU
	{0x12463, 0x1246F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x12470, 0x12473, propertyDISALLOWED},  // CUNEIFORM PUNCTUATION SIGN OLD ASSYRIAN WORD
	{0x12474, 0x12FFF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x13000, 0x1342E, propertyPVALID},      // EGYPTIAN HIEROGLYPH A001..EGYPTIAN HIEROGLYP
	{0x1342F, 0x1CFFF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D000, 0x1D0F5, propertyDISALLOWED},  // BYZANTINE MUSICAL SYMBOL PSILI..BYZANTINE MU
	{0x1D0F6, 0x1D0FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D100, 0x1D126, propertyDISALLOWED},  // MUSICAL SYMBOL SINGLE BARLINE..MUSICAL SYMBO
	{0x1D127, 0x1D128, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D129, 0x1D1DD, propertyDISALLOWED},  // MUSICAL SYMBOL MULTIPLE MEASURE REST..MUSICA
	{0x1D1DE, 0x1D1FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D200, 0x1D245, propertyDISALLOWED},  // GREEK VOCAL NOTATION SYMBOL-1..GREEK MUSICAL
	{0x1D246, 0x1D2FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D300, 0x1D356, propertyDISALLOWED},  // MONOGRAM FOR EARTH..TETRAGRAM FOR FOSTERING
	{0x1D357, 0x1D35F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D360, 0x1D371, propertyDISALLOWED},  // COUNTING ROD UNIT DIGIT ONE..COUNTING ROD TE
	{0x1D372, 0x1D3FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D400, 0x1D454, propertyDISALLOWED},  // MATHEMATICAL BOLD CAPITAL A..MATHEMATICAL IT
	{0x1D455, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D456, 0x1D49C, propertyDISALLOWED},  // MATHEMATICAL ITALIC SMALL I..MATHEMATICAL SC
	{0x1D49D, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D49E, 0x1D49F, propertyDISALLOWED},  // MATHEMATICAL SCRIPT CAPITAL C..MATHEMATICAL
	{0x1D4A0, 0x1D4A1, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D4A2, 0x0, propertyDISALLOWED},      // MATHEMATICAL SCRIPT CAPITAL G
	{0x1D4A3, 0x1D4A4, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D4A5, 0x1D4A6, propertyDISALLOWED},  // MATHEMATICAL SCRIPT CAPITAL J..MATHEMATICAL
	{0x1D4A7, 0x1D4A8, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D4A9, 0x1D4AC, propertyDISALLOWED},  // MATHEMATICAL SCRIPT CAPITAL N..MATHEMATICAL
	{0x1D4AD, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D4AE, 0x1D4B9, propertyDISALLOWED},  // MATHEMATICAL SCRIPT CAPITAL S..MATHEMATICAL
	{0x1D4BA, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D4BB, 0x0, propertyDISALLOWED},      // MATHEMATICAL SCRIPT SMALL F
	{0x1D4BC, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D4BD, 0x1D4C3, propertyDISALLOWED},  // MATHEMATICAL SCRIPT SMALL H..MATHEMATICAL SC
	{0x1D4C4, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D4C5, 0x1D505, propertyDISALLOWED},  // MATHEMATICAL SCRIPT SMALL P..MATHEMATICAL FR
	{0x1D506, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D507, 0x1D50A, propertyDISALLOWED},  // MATHEMATICAL FRAKTUR CAPITAL D..MATHEMATICAL
	{0x1D50B, 0x1D50C, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D50D, 0x1D514, propertyDISALLOWED},  // MATHEMATICAL FRAKTUR CAPITAL J..MATHEMATICAL
	{0x1D515, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D516, 0x1D51C, propertyDISALLOWED},  // MATHEMATICAL FRAKTUR CAPITAL S..MATHEMATICAL
	{0x1D51D, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D51E, 0x1D539, propertyDISALLOWED},  // MATHEMATICAL FRAKTUR SMALL A..MATHEMATICAL D
	{0x1D53A, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D53B, 0x1D53E, propertyDISALLOWED},  // MATHEMATICAL DOUBLE-STRUCK CAPITAL D..MATHEM
	{0x1D53F, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D540, 0x1D544, propertyDISALLOWED},  // MATHEMATICAL DOUBLE-STRUCK CAPITAL I..MATHEM
	{0x1D545, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D546, 0x0, propertyDISALLOWED},      // MATHEMATICAL DOUBLE-STRUCK CAPITAL O
	{0x1D547, 0x1D549, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D54A, 0x1D550, propertyDISALLOWED},  // MATHEMATICAL DOUBLE-STRUCK CAPITAL S..MATHEM
	{0x1D551, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1D552, 0x1D6A5, propertyDISALLOWED},  // MATHEMATICAL DOUBLE-STRUCK SMALL A..MATHEMAT
	{0x1D6A6, 0x1D6A7, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D6A8, 0x1D7CB, propertyDISALLOWED},  // MATHEMATICAL BOLD CAPITAL ALPHA..MATHEMATICA
	{0x1D7CC, 0x1D7CD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1D7CE, 0x1D7FF, propertyDISALLOWED},  // MATHEMATICAL BOLD DIGIT ZERO..MATHEMATICAL M
	{0x1D800, 0x1EFFF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F000, 0x1F02B, propertyDISALLOWED},  // MAHJONG TILE EAST WIND..MAHJONG TILE BACK
	{0x1F02C, 0x1F02F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F030, 0x1F093, propertyDISALLOWED},  // DOMINO TILE HORIZONTAL BACK..DOMINO TILE VER
	{0x1F094, 0x1F0FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F100, 0x1F10A, propertyDISALLOWED},  // DIGIT ZERO FULL STOP..DIGIT NINE COMMA
	{0x1F10B, 0x1F10F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F110, 0x1F12E, propertyDISALLOWED},  // PARENTHESIZED LATIN CAPITAL LETTER A..CIRCLE
	{0x1F12F, 0x1F130, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F131, 0x0, propertyDISALLOWED},      // SQUARED LATIN CAPITAL LETTER B
	{0x1F132, 0x1F13C, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F13D, 0x0, propertyDISALLOWED},      // SQUARED LATIN CAPITAL LETTER N
	{0x1F13E, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1F13F, 0x0, propertyDISALLOWED},      // SQUARED LATIN CAPITAL LETTER P
	{0x1F140, 0x1F141, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F142, 0x0, propertyDISALLOWED},      // SQUARED LATIN CAPITAL LETTER S
	{0x1F143, 0x1F145, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F146, 0x0, propertyDISALLOWED},      // SQUARED LATIN CAPITAL LETTER W
	{0x1F147, 0x1F149, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F14A, 0x1F14E, propertyDISALLOWED},  // SQUARED HV..SQUARED PPV
	{0x1F14F, 0x1F156, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F157, 0x0, propertyDISALLOWED},      // NEGATIVE CIRCLED LATIN CAPITAL LETTER H
	{0x1F158, 0x1F15E, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F15F, 0x0, propertyDISALLOWED},      // NEGATIVE CIRCLED LATIN CAPITAL LETTER P
	{0x1F160, 0x1F178, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F179, 0x0, propertyDISALLOWED},      // NEGATIVE SQUARED LATIN CAPITAL LETTER J
	{0x1F17A, 0x0, propertyUNASSIGNED},      // <reserved>
	{0x1F17B, 0x1F17C, propertyDISALLOWED},  // NEGATIVE SQUARED LATIN CAPITAL LETTER L..NEG
	{0x1F17D, 0x1F17E, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F17F, 0x0, propertyDISALLOWED},      // NEGATIVE SQUARED LATIN CAPITAL LETTER P
	{0x1F180, 0x1F189, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F18A, 0x1F18D, propertyDISALLOWED},  // CROSSED NEGATIVE SQUARED LATIN CAPITAL LETTE
	{0x1F18E, 0x1F18F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F190, 0x0, propertyDISALLOWED},      // SQUARE DJ
	{0x1F191, 0x1F1FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F200, 0x0, propertyDISALLOWED},      // SQUARE HIRAGANA HOKA
	{0x1F201, 0x1F20F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F210, 0x1F231, propertyDISALLOWED},  // SQUARED CJK UNIFIED IDEOGRAPH-624B..SQUARED
	{0x1F232, 0x1F23F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1F240, 0x1F248, propertyDISALLOWED},  // TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRA
	{0x1F249, 0x1FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x1FFFE, 0x1FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0x20000, 0x2A6D6, propertyPVALID},      // <CJK Ideograph Extension B>..<CJK Ideograph
	{0x2A6D7, 0x2A6FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x2A700, 0x2B734, propertyPVALID},      // <CJK Ideograph Extension C>..<CJK Ideograph
	{0x2B735, 0x2F7FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x2F800, 0x2FA1D, propertyDISALLOWED},  // CJK COMPATIBILITY IDEOGRAPH-2F800..CJK COMPA
	{0x2FA1E, 0x2FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x2FFFE, 0x2FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0x30000, 0x3FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x3FFFE, 0x3FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0x40000, 0x4FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x4FFFE, 0x4FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0x50000, 0x5FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x5FFFE, 0x5FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0x60000, 0x6FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x6FFFE, 0x6FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0x70000, 0x7FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x7FFFE, 0x7FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0x80000, 0x8FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x8FFFE, 0x8FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0x90000, 0x9FFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0x9FFFE, 0x9FFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0xA0000, 0xAFFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0xAFFFE, 0xAFFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0xB0000, 0xBFFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0xBFFFE, 0xBFFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0xC0000, 0xCFFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0xCFFFE, 0xCFFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0xD0000, 0xDFFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0xDFFFE, 0xDFFFF, propertyDISALLOWED},  // <noncharacter>..<noncharacter>
	{0xE0000, 0x0, propertyUNASSIGNED},      // <reserved>
	{0xE0001, 0x0, propertyDISALLOWED},      // LANGUAGE TAG
	{0xE0002, 0xE001F, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0xE0020, 0xE007F, propertyDISALLOWED},  // TAG SPACE..CANCEL TAG
	{0xE0080, 0xE00FF, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0xE0100, 0xE01EF, propertyDISALLOWED},  // VARIATION SELECTOR-17..VARIATION SELECTOR-25
	{0xE01F0, 0xEFFFD, propertyUNASSIGNED},  // <reserved>..<reserved>
	{0xEFFFE, 0x10FFFF, propertyDISALLOWED}, // <noncharacter>..<noncharacter>
}
