// Package keyvault implements the Azure ARM Keyvault service API version 7.0.
//
// The key vault client performs cryptographic key operations and vault operations against the Key Vault service.
package keyvault

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"github.com/Azure/go-autorest/tracing"
	"net/http"
)

// BaseClient is the base client for Keyvault.
type BaseClient struct {
	autorest.Client
}

// New creates an instance of the BaseClient client.
func New() BaseClient {
	return NewWithoutDefaults()
}

// NewWithoutDefaults creates an instance of the BaseClient client.
func NewWithoutDefaults() BaseClient {
	return BaseClient{
		Client: autorest.NewClientWithUserAgent(UserAgent()),
	}
}

// BackupCertificate requests that a backup of the specified certificate be downloaded to the client. All versions of
// the certificate will be downloaded. This operation requires the certificates/backup permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
func (client BaseClient) BackupCertificate(ctx context.Context, vaultBaseURL string, certificateName string) (result BackupCertificateResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.BackupCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.BackupCertificatePreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.BackupCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.BackupCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupCertificate", resp, "Failure responding to request")
	}

	return
}

// BackupCertificatePreparer prepares the BackupCertificate request.
func (client BaseClient) BackupCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/backup", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// BackupCertificateSender sends the BackupCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) BackupCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// BackupCertificateResponder handles the response to the BackupCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) BackupCertificateResponder(resp *http.Response) (result BackupCertificateResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// BackupKey the Key Backup operation exports a key from Azure Key Vault in a protected form. Note that this operation
// does NOT return key material in a form that can be used outside the Azure Key Vault system, the returned key
// material is either protected to a Azure Key Vault HSM or to Azure Key Vault itself. The intent of this operation is
// to allow a client to GENERATE a key in one Azure Key Vault instance, BACKUP the key, and then RESTORE it into
// another Azure Key Vault instance. The BACKUP operation may be used to export, in protected form, any key type from
// Azure Key Vault. Individual versions of a key cannot be backed up. BACKUP / RESTORE can be performed within
// geographical boundaries only; meaning that a BACKUP from one geographical area cannot be restored to another
// geographical area. For example, a backup from the US geographical area cannot be restored in an EU geographical
// area. This operation requires the key/backup permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
func (client BaseClient) BackupKey(ctx context.Context, vaultBaseURL string, keyName string) (result BackupKeyResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.BackupKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.BackupKeyPreparer(ctx, vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.BackupKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupKey", resp, "Failure sending request")
		return
	}

	result, err = client.BackupKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupKey", resp, "Failure responding to request")
	}

	return
}

// BackupKeyPreparer prepares the BackupKey request.
func (client BaseClient) BackupKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/backup", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// BackupKeySender sends the BackupKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) BackupKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// BackupKeyResponder handles the response to the BackupKey request. The method always
// closes the http.Response Body.
func (client BaseClient) BackupKeyResponder(resp *http.Response) (result BackupKeyResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// BackupSecret requests that a backup of the specified secret be downloaded to the client. All versions of the secret
// will be downloaded. This operation requires the secrets/backup permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the secret.
func (client BaseClient) BackupSecret(ctx context.Context, vaultBaseURL string, secretName string) (result BackupSecretResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.BackupSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.BackupSecretPreparer(ctx, vaultBaseURL, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.BackupSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupSecret", resp, "Failure sending request")
		return
	}

	result, err = client.BackupSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupSecret", resp, "Failure responding to request")
	}

	return
}

// BackupSecretPreparer prepares the BackupSecret request.
func (client BaseClient) BackupSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/backup", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// BackupSecretSender sends the BackupSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) BackupSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// BackupSecretResponder handles the response to the BackupSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) BackupSecretResponder(resp *http.Response) (result BackupSecretResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// BackupStorageAccount requests that a backup of the specified storage account be downloaded to the client. This
// operation requires the storage/backup permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
func (client BaseClient) BackupStorageAccount(ctx context.Context, vaultBaseURL string, storageAccountName string) (result BackupStorageResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.BackupStorageAccount")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.BackupStorageAccountPreparer(ctx, vaultBaseURL, storageAccountName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.BackupStorageAccountSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.BackupStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupStorageAccount", resp, "Failure responding to request")
	}

	return
}

// BackupStorageAccountPreparer prepares the BackupStorageAccount request.
func (client BaseClient) BackupStorageAccountPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}/backup", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// BackupStorageAccountSender sends the BackupStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) BackupStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// BackupStorageAccountResponder handles the response to the BackupStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) BackupStorageAccountResponder(resp *http.Response) (result BackupStorageResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// CreateCertificate if this is the first version, the certificate resource is created. This operation requires the
// certificates/create permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
// parameters - the parameters to create a certificate.
func (client BaseClient) CreateCertificate(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateCreateParameters) (result CertificateOperation, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.CreateCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: certificateName,
			Constraints: []validation.Constraint{{Target: "certificateName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.CertificatePolicy", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.Null, Rule: false,
						Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}},
					}},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "CreateCertificate", err.Error())
	}

	req, err := client.CreateCertificatePreparer(ctx, vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.CreateCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateCertificate", resp, "Failure responding to request")
	}

	return
}

// CreateCertificatePreparer prepares the CreateCertificate request.
func (client BaseClient) CreateCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/create", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateCertificateSender sends the CreateCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CreateCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// CreateCertificateResponder handles the response to the CreateCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) CreateCertificateResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// CreateKey the create key operation can be used to create any key type in Azure Key Vault. If the named key already
// exists, Azure Key Vault creates a new version of the key. It requires the keys/create permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name for the new key. The system will generate the version name for the new key.
// parameters - the parameters to create a key.
func (client BaseClient) CreateKey(ctx context.Context, vaultBaseURL string, keyName string, parameters KeyCreateParameters) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.CreateKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: keyName,
			Constraints: []validation.Constraint{{Target: "keyName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "CreateKey", err.Error())
	}

	req, err := client.CreateKeyPreparer(ctx, vaultBaseURL, keyName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateKey", resp, "Failure sending request")
		return
	}

	result, err = client.CreateKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateKey", resp, "Failure responding to request")
	}

	return
}

// CreateKeyPreparer prepares the CreateKey request.
func (client BaseClient) CreateKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, parameters KeyCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/create", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateKeySender sends the CreateKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CreateKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// CreateKeyResponder handles the response to the CreateKey request. The method always
// closes the http.Response Body.
func (client BaseClient) CreateKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Decrypt the DECRYPT operation decrypts a well-formed block of ciphertext using the target encryption key and
// specified algorithm. This operation is the reverse of the ENCRYPT operation; only a single block of data may be
// decrypted, the size of this block is dependent on the target key and the algorithm to be used. The DECRYPT operation
// applies to asymmetric and symmetric keys stored in Azure Key Vault since it uses the private portion of the key.
// This operation requires the keys/decrypt permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
// keyVersion - the version of the key.
// parameters - the parameters for the decryption operation.
func (client BaseClient) Decrypt(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Decrypt")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "Decrypt", err.Error())
	}

	req, err := client.DecryptPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Decrypt", nil, "Failure preparing request")
		return
	}

	resp, err := client.DecryptSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Decrypt", resp, "Failure sending request")
		return
	}

	result, err = client.DecryptResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Decrypt", resp, "Failure responding to request")
	}

	return
}

// DecryptPreparer prepares the Decrypt request.
func (client BaseClient) DecryptPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/decrypt", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DecryptSender sends the Decrypt request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DecryptSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DecryptResponder handles the response to the Decrypt request. The method always
// closes the http.Response Body.
func (client BaseClient) DecryptResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificate deletes all versions of a certificate object along with its associated policy. Delete certificate
// cannot be used to remove individual versions of a certificate object. This operation requires the
// certificates/delete permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
func (client BaseClient) DeleteCertificate(ctx context.Context, vaultBaseURL string, certificateName string) (result DeletedCertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteCertificatePreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificate", resp, "Failure responding to request")
	}

	return
}

// DeleteCertificatePreparer prepares the DeleteCertificate request.
func (client BaseClient) DeleteCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCertificateSender sends the DeleteCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DeleteCertificateResponder handles the response to the DeleteCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteCertificateResponder(resp *http.Response) (result DeletedCertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateContacts deletes the certificate contacts for a specified key vault certificate. This operation
// requires the certificates/managecontacts permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
func (client BaseClient) DeleteCertificateContacts(ctx context.Context, vaultBaseURL string) (result Contacts, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteCertificateContacts")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteCertificateContactsPreparer(ctx, vaultBaseURL)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateContacts", resp, "Failure responding to request")
	}

	return
}

// DeleteCertificateContactsPreparer prepares the DeleteCertificateContacts request.
func (client BaseClient) DeleteCertificateContactsPreparer(ctx context.Context, vaultBaseURL string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCertificateContactsSender sends the DeleteCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteCertificateContactsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DeleteCertificateContactsResponder handles the response to the DeleteCertificateContacts request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateIssuer the DeleteCertificateIssuer operation permanently removes the specified certificate issuer
// from the vault. This operation requires the certificates/manageissuers/deleteissuers permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// issuerName - the name of the issuer.
func (client BaseClient) DeleteCertificateIssuer(ctx context.Context, vaultBaseURL string, issuerName string) (result IssuerBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteCertificateIssuer")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteCertificateIssuerPreparer(ctx, vaultBaseURL, issuerName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateIssuer", resp, "Failure responding to request")
	}

	return
}

// DeleteCertificateIssuerPreparer prepares the DeleteCertificateIssuer request.
func (client BaseClient) DeleteCertificateIssuerPreparer(ctx context.Context, vaultBaseURL string, issuerName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCertificateIssuerSender sends the DeleteCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DeleteCertificateIssuerResponder handles the response to the DeleteCertificateIssuer request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateOperation deletes the creation operation for a specified certificate that is in the process of
// being created. The certificate is no longer created. This operation requires the certificates/update permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
func (client BaseClient) DeleteCertificateOperation(ctx context.Context, vaultBaseURL string, certificateName string) (result CertificateOperation, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteCertificateOperation")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteCertificateOperationPreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateOperation", resp, "Failure responding to request")
	}

	return
}

// DeleteCertificateOperationPreparer prepares the DeleteCertificateOperation request.
func (client BaseClient) DeleteCertificateOperationPreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCertificateOperationSender sends the DeleteCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteCertificateOperationSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DeleteCertificateOperationResponder handles the response to the DeleteCertificateOperation request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteKey the delete key operation cannot be used to remove individual versions of a key. This operation removes the
// cryptographic material associated with the key, which means the key is not usable for Sign/Verify, Wrap/Unwrap or
// Encrypt/Decrypt operations. This operation requires the keys/delete permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key to delete.
func (client BaseClient) DeleteKey(ctx context.Context, vaultBaseURL string, keyName string) (result DeletedKeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteKeyPreparer(ctx, vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteKey", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteKey", resp, "Failure responding to request")
	}

	return
}

// DeleteKeyPreparer prepares the DeleteKey request.
func (client BaseClient) DeleteKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteKeySender sends the DeleteKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DeleteKeyResponder handles the response to the DeleteKey request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteKeyResponder(resp *http.Response) (result DeletedKeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteSasDefinition deletes a SAS definition from a specified storage account. This operation requires the
// storage/deletesas permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// sasDefinitionName - the name of the SAS definition.
func (client BaseClient) DeleteSasDefinition(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string) (result DeletedSasDefinitionBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteSasDefinition")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: sasDefinitionName,
			Constraints: []validation.Constraint{{Target: "sasDefinitionName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "DeleteSasDefinition", err.Error())
	}

	req, err := client.DeleteSasDefinitionPreparer(ctx, vaultBaseURL, storageAccountName, sasDefinitionName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSasDefinition", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSasDefinitionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSasDefinition", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteSasDefinitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSasDefinition", resp, "Failure responding to request")
	}

	return
}

// DeleteSasDefinitionPreparer prepares the DeleteSasDefinition request.
func (client BaseClient) DeleteSasDefinitionPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"sas-definition-name":  autorest.Encode("path", sasDefinitionName),
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}/sas/{sas-definition-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteSasDefinitionSender sends the DeleteSasDefinition request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteSasDefinitionSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DeleteSasDefinitionResponder handles the response to the DeleteSasDefinition request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteSasDefinitionResponder(resp *http.Response) (result DeletedSasDefinitionBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteSecret the DELETE operation applies to any secret stored in Azure Key Vault. DELETE cannot be applied to an
// individual version of a secret. This operation requires the secrets/delete permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the secret.
func (client BaseClient) DeleteSecret(ctx context.Context, vaultBaseURL string, secretName string) (result DeletedSecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteSecretPreparer(ctx, vaultBaseURL, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSecret", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSecret", resp, "Failure responding to request")
	}

	return
}

// DeleteSecretPreparer prepares the DeleteSecret request.
func (client BaseClient) DeleteSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteSecretSender sends the DeleteSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DeleteSecretResponder handles the response to the DeleteSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteSecretResponder(resp *http.Response) (result DeletedSecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteStorageAccount deletes a storage account. This operation requires the storage/delete permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
func (client BaseClient) DeleteStorageAccount(ctx context.Context, vaultBaseURL string, storageAccountName string) (result DeletedStorageBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteStorageAccount")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "DeleteStorageAccount", err.Error())
	}

	req, err := client.DeleteStorageAccountPreparer(ctx, vaultBaseURL, storageAccountName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteStorageAccountSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteStorageAccount", resp, "Failure responding to request")
	}

	return
}

// DeleteStorageAccountPreparer prepares the DeleteStorageAccount request.
func (client BaseClient) DeleteStorageAccountPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteStorageAccountSender sends the DeleteStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// DeleteStorageAccountResponder handles the response to the DeleteStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteStorageAccountResponder(resp *http.Response) (result DeletedStorageBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Encrypt the ENCRYPT operation encrypts an arbitrary sequence of bytes using an encryption key that is stored in
// Azure Key Vault. Note that the ENCRYPT operation only supports a single block of data, the size of which is
// dependent on the target key and the encryption algorithm to be used. The ENCRYPT operation is only strictly
// necessary for symmetric keys stored in Azure Key Vault since protection with an asymmetric key can be performed
// using public portion of the key. This operation is supported for asymmetric keys as a convenience for callers that
// have a key-reference but do not have access to the public key material. This operation requires the keys/encrypt
// permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
// keyVersion - the version of the key.
// parameters - the parameters for the encryption operation.
func (client BaseClient) Encrypt(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Encrypt")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "Encrypt", err.Error())
	}

	req, err := client.EncryptPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Encrypt", nil, "Failure preparing request")
		return
	}

	resp, err := client.EncryptSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Encrypt", resp, "Failure sending request")
		return
	}

	result, err = client.EncryptResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Encrypt", resp, "Failure responding to request")
	}

	return
}

// EncryptPreparer prepares the Encrypt request.
func (client BaseClient) EncryptPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/encrypt", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// EncryptSender sends the Encrypt request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) EncryptSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// EncryptResponder handles the response to the Encrypt request. The method always
// closes the http.Response Body.
func (client BaseClient) EncryptResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificate gets information about a specific certificate. This operation requires the certificates/get
// permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate in the given vault.
// certificateVersion - the version of the certificate.
func (client BaseClient) GetCertificate(ctx context.Context, vaultBaseURL string, certificateName string, certificateVersion string) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificatePreparer(ctx, vaultBaseURL, certificateName, certificateVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificate", resp, "Failure responding to request")
	}

	return
}

// GetCertificatePreparer prepares the GetCertificate request.
func (client BaseClient) GetCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, certificateVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name":    autorest.Encode("path", certificateName),
		"certificate-version": autorest.Encode("path", certificateVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/{certificate-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateSender sends the GetCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetCertificateResponder handles the response to the GetCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateContacts the GetCertificateContacts operation returns the set of certificate contact resources in the
// specified key vault. This operation requires the certificates/managecontacts permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
func (client BaseClient) GetCertificateContacts(ctx context.Context, vaultBaseURL string) (result Contacts, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateContacts")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificateContactsPreparer(ctx, vaultBaseURL)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateContacts", resp, "Failure responding to request")
	}

	return
}

// GetCertificateContactsPreparer prepares the GetCertificateContacts request.
func (client BaseClient) GetCertificateContactsPreparer(ctx context.Context, vaultBaseURL string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateContactsSender sends the GetCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateContactsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetCertificateContactsResponder handles the response to the GetCertificateContacts request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateIssuer the GetCertificateIssuer operation returns the specified certificate issuer resources in the
// specified key vault. This operation requires the certificates/manageissuers/getissuers permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// issuerName - the name of the issuer.
func (client BaseClient) GetCertificateIssuer(ctx context.Context, vaultBaseURL string, issuerName string) (result IssuerBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateIssuer")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificateIssuerPreparer(ctx, vaultBaseURL, issuerName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuer", resp, "Failure responding to request")
	}

	return
}

// GetCertificateIssuerPreparer prepares the GetCertificateIssuer request.
func (client BaseClient) GetCertificateIssuerPreparer(ctx context.Context, vaultBaseURL string, issuerName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateIssuerSender sends the GetCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetCertificateIssuerResponder handles the response to the GetCertificateIssuer request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateIssuers the GetCertificateIssuers operation returns the set of certificate issuer resources in the
// specified key vault. This operation requires the certificates/manageissuers/getissuers permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetCertificateIssuers(ctx context.Context, vaultBaseURL string, maxresults *int32) (result CertificateIssuerListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateIssuers")
		defer func() {
			sc := -1
			if result.cilr.Response.Response != nil {
				sc = result.cilr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetCertificateIssuers", err.Error())
	}

	result.fn = client.getCertificateIssuersNextResults
	req, err := client.GetCertificateIssuersPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuers", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateIssuersSender(req)
	if err != nil {
		result.cilr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuers", resp, "Failure sending request")
		return
	}

	result.cilr, err = client.GetCertificateIssuersResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuers", resp, "Failure responding to request")
	}

	return
}

// GetCertificateIssuersPreparer prepares the GetCertificateIssuers request.
func (client BaseClient) GetCertificateIssuersPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/issuers"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateIssuersSender sends the GetCertificateIssuers request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateIssuersSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetCertificateIssuersResponder handles the response to the GetCertificateIssuers request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateIssuersResponder(resp *http.Response) (result CertificateIssuerListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getCertificateIssuersNextResults retrieves the next set of results, if any.
func (client BaseClient) getCertificateIssuersNextResults(ctx context.Context, lastResults CertificateIssuerListResult) (result CertificateIssuerListResult, err error) {
	req, err := lastResults.certificateIssuerListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateIssuersNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetCertificateIssuersSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateIssuersNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetCertificateIssuersResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateIssuersNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetCertificateIssuersComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetCertificateIssuersComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result CertificateIssuerListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateIssuers")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetCertificateIssuers(ctx, vaultBaseURL, maxresults)
	return
}

// GetCertificateOperation gets the creation operation associated with a specified certificate. This operation requires
// the certificates/get permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
func (client BaseClient) GetCertificateOperation(ctx context.Context, vaultBaseURL string, certificateName string) (result CertificateOperation, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateOperation")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificateOperationPreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateOperation", resp, "Failure responding to request")
	}

	return
}

// GetCertificateOperationPreparer prepares the GetCertificateOperation request.
func (client BaseClient) GetCertificateOperationPreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateOperationSender sends the GetCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateOperationSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetCertificateOperationResponder handles the response to the GetCertificateOperation request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificatePolicy the GetCertificatePolicy operation returns the specified certificate policy resources in the
// specified key vault. This operation requires the certificates/get permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate in a given key vault.
func (client BaseClient) GetCertificatePolicy(ctx context.Context, vaultBaseURL string, certificateName string) (result CertificatePolicy, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificatePolicy")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificatePolicyPreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificatePolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificatePolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificatePolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificatePolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificatePolicy", resp, "Failure responding to request")
	}

	return
}

// GetCertificatePolicyPreparer prepares the GetCertificatePolicy request.
func (client BaseClient) GetCertificatePolicyPreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/policy", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificatePolicySender sends the GetCertificatePolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificatePolicySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetCertificatePolicyResponder handles the response to the GetCertificatePolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificatePolicyResponder(resp *http.Response) (result CertificatePolicy, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificates the GetCertificates operation returns the set of certificates resources in the specified key vault.
// This operation requires the certificates/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
// includePending - specifies whether to include certificates which are not completely provisioned.
func (client BaseClient) GetCertificates(ctx context.Context, vaultBaseURL string, maxresults *int32, includePending *bool) (result CertificateListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificates")
		defer func() {
			sc := -1
			if result.clr.Response.Response != nil {
				sc = result.clr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetCertificates", err.Error())
	}

	result.fn = client.getCertificatesNextResults
	req, err := client.GetCertificatesPreparer(ctx, vaultBaseURL, maxresults, includePending)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificates", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificatesSender(req)
	if err != nil {
		result.clr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificates", resp, "Failure sending request")
		return
	}

	result.clr, err = client.GetCertificatesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificates", resp, "Failure responding to request")
	}

	return
}

// GetCertificatesPreparer prepares the GetCertificates request.
func (client BaseClient) GetCertificatesPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32, includePending *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}
	if includePending != nil {
		queryParameters["includePending"] = autorest.Encode("query", *includePending)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificatesSender sends the GetCertificates request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificatesSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetCertificatesResponder handles the response to the GetCertificates request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificatesResponder(resp *http.Response) (result CertificateListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getCertificatesNextResults retrieves the next set of results, if any.
func (client BaseClient) getCertificatesNextResults(ctx context.Context, lastResults CertificateListResult) (result CertificateListResult, err error) {
	req, err := lastResults.certificateListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificatesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetCertificatesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificatesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetCertificatesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificatesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetCertificatesComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetCertificatesComplete(ctx context.Context, vaultBaseURL string, maxresults *int32, includePending *bool) (result CertificateListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificates")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetCertificates(ctx, vaultBaseURL, maxresults, includePending)
	return
}

// GetCertificateVersions the GetCertificateVersions operation returns the versions of a certificate in the specified
// key vault. This operation requires the certificates/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetCertificateVersions(ctx context.Context, vaultBaseURL string, certificateName string, maxresults *int32) (result CertificateListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateVersions")
		defer func() {
			sc := -1
			if result.clr.Response.Response != nil {
				sc = result.clr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetCertificateVersions", err.Error())
	}

	result.fn = client.getCertificateVersionsNextResults
	req, err := client.GetCertificateVersionsPreparer(ctx, vaultBaseURL, certificateName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateVersionsSender(req)
	if err != nil {
		result.clr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateVersions", resp, "Failure sending request")
		return
	}

	result.clr, err = client.GetCertificateVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateVersions", resp, "Failure responding to request")
	}

	return
}

// GetCertificateVersionsPreparer prepares the GetCertificateVersions request.
func (client BaseClient) GetCertificateVersionsPreparer(ctx context.Context, vaultBaseURL string, certificateName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateVersionsSender sends the GetCertificateVersions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateVersionsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetCertificateVersionsResponder handles the response to the GetCertificateVersions request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateVersionsResponder(resp *http.Response) (result CertificateListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getCertificateVersionsNextResults retrieves the next set of results, if any.
func (client BaseClient) getCertificateVersionsNextResults(ctx context.Context, lastResults CertificateListResult) (result CertificateListResult, err error) {
	req, err := lastResults.certificateListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateVersionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetCertificateVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateVersionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetCertificateVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateVersionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetCertificateVersionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetCertificateVersionsComplete(ctx context.Context, vaultBaseURL string, certificateName string, maxresults *int32) (result CertificateListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateVersions")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetCertificateVersions(ctx, vaultBaseURL, certificateName, maxresults)
	return
}

// GetDeletedCertificate the GetDeletedCertificate operation retrieves the deleted certificate information plus its
// attributes, such as retention interval, scheduled permanent deletion and the current deletion recovery level. This
// operation requires the certificates/get permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate
func (client BaseClient) GetDeletedCertificate(ctx context.Context, vaultBaseURL string, certificateName string) (result DeletedCertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetDeletedCertificatePreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeletedCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedCertificate", resp, "Failure responding to request")
	}

	return
}

// GetDeletedCertificatePreparer prepares the GetDeletedCertificate request.
func (client BaseClient) GetDeletedCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedcertificates/{certificate-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedCertificateSender sends the GetDeletedCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedCertificateResponder handles the response to the GetDeletedCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedCertificateResponder(resp *http.Response) (result DeletedCertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeletedCertificates the GetDeletedCertificates operation retrieves the certificates in the current vault which
// are in a deleted state and ready for recovery or purging. This operation includes deletion-specific information.
// This operation requires the certificates/get/list permission. This operation can only be enabled on soft-delete
// enabled vaults.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
// includePending - specifies whether to include certificates which are not completely provisioned.
func (client BaseClient) GetDeletedCertificates(ctx context.Context, vaultBaseURL string, maxresults *int32, includePending *bool) (result DeletedCertificateListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedCertificates")
		defer func() {
			sc := -1
			if result.dclr.Response.Response != nil {
				sc = result.dclr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetDeletedCertificates", err.Error())
	}

	result.fn = client.getDeletedCertificatesNextResults
	req, err := client.GetDeletedCertificatesPreparer(ctx, vaultBaseURL, maxresults, includePending)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedCertificates", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedCertificatesSender(req)
	if err != nil {
		result.dclr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedCertificates", resp, "Failure sending request")
		return
	}

	result.dclr, err = client.GetDeletedCertificatesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedCertificates", resp, "Failure responding to request")
	}

	return
}

// GetDeletedCertificatesPreparer prepares the GetDeletedCertificates request.
func (client BaseClient) GetDeletedCertificatesPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32, includePending *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}
	if includePending != nil {
		queryParameters["includePending"] = autorest.Encode("query", *includePending)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/deletedcertificates"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedCertificatesSender sends the GetDeletedCertificates request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedCertificatesSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedCertificatesResponder handles the response to the GetDeletedCertificates request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedCertificatesResponder(resp *http.Response) (result DeletedCertificateListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getDeletedCertificatesNextResults retrieves the next set of results, if any.
func (client BaseClient) getDeletedCertificatesNextResults(ctx context.Context, lastResults DeletedCertificateListResult) (result DeletedCertificateListResult, err error) {
	req, err := lastResults.deletedCertificateListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedCertificatesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetDeletedCertificatesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedCertificatesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetDeletedCertificatesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedCertificatesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetDeletedCertificatesComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetDeletedCertificatesComplete(ctx context.Context, vaultBaseURL string, maxresults *int32, includePending *bool) (result DeletedCertificateListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedCertificates")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetDeletedCertificates(ctx, vaultBaseURL, maxresults, includePending)
	return
}

// GetDeletedKey the Get Deleted Key operation is applicable for soft-delete enabled vaults. While the operation can be
// invoked on any vault, it will return an error if invoked on a non soft-delete enabled vault. This operation requires
// the keys/get permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
func (client BaseClient) GetDeletedKey(ctx context.Context, vaultBaseURL string, keyName string) (result DeletedKeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetDeletedKeyPreparer(ctx, vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedKey", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeletedKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedKey", resp, "Failure responding to request")
	}

	return
}

// GetDeletedKeyPreparer prepares the GetDeletedKey request.
func (client BaseClient) GetDeletedKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedkeys/{key-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedKeySender sends the GetDeletedKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedKeyResponder handles the response to the GetDeletedKey request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedKeyResponder(resp *http.Response) (result DeletedKeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeletedKeys retrieves a list of the keys in the Key Vault as JSON Web Key structures that contain the public part
// of a deleted key. This operation includes deletion-specific information. The Get Deleted Keys operation is
// applicable for vaults enabled for soft-delete. While the operation can be invoked on any vault, it will return an
// error if invoked on a non soft-delete enabled vault. This operation requires the keys/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetDeletedKeys(ctx context.Context, vaultBaseURL string, maxresults *int32) (result DeletedKeyListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedKeys")
		defer func() {
			sc := -1
			if result.dklr.Response.Response != nil {
				sc = result.dklr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetDeletedKeys", err.Error())
	}

	result.fn = client.getDeletedKeysNextResults
	req, err := client.GetDeletedKeysPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedKeys", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedKeysSender(req)
	if err != nil {
		result.dklr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedKeys", resp, "Failure sending request")
		return
	}

	result.dklr, err = client.GetDeletedKeysResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedKeys", resp, "Failure responding to request")
	}

	return
}

// GetDeletedKeysPreparer prepares the GetDeletedKeys request.
func (client BaseClient) GetDeletedKeysPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/deletedkeys"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedKeysSender sends the GetDeletedKeys request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedKeysSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedKeysResponder handles the response to the GetDeletedKeys request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedKeysResponder(resp *http.Response) (result DeletedKeyListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getDeletedKeysNextResults retrieves the next set of results, if any.
func (client BaseClient) getDeletedKeysNextResults(ctx context.Context, lastResults DeletedKeyListResult) (result DeletedKeyListResult, err error) {
	req, err := lastResults.deletedKeyListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedKeysNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetDeletedKeysSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedKeysNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetDeletedKeysResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedKeysNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetDeletedKeysComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetDeletedKeysComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result DeletedKeyListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedKeys")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetDeletedKeys(ctx, vaultBaseURL, maxresults)
	return
}

// GetDeletedSasDefinition the Get Deleted SAS Definition operation returns the specified deleted SAS definition along
// with its attributes. This operation requires the storage/getsas permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// sasDefinitionName - the name of the SAS definition.
func (client BaseClient) GetDeletedSasDefinition(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string) (result DeletedSasDefinitionBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedSasDefinition")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: sasDefinitionName,
			Constraints: []validation.Constraint{{Target: "sasDefinitionName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetDeletedSasDefinition", err.Error())
	}

	req, err := client.GetDeletedSasDefinitionPreparer(ctx, vaultBaseURL, storageAccountName, sasDefinitionName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSasDefinition", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedSasDefinitionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSasDefinition", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeletedSasDefinitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSasDefinition", resp, "Failure responding to request")
	}

	return
}

// GetDeletedSasDefinitionPreparer prepares the GetDeletedSasDefinition request.
func (client BaseClient) GetDeletedSasDefinitionPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"sas-definition-name":  autorest.Encode("path", sasDefinitionName),
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedstorage/{storage-account-name}/sas/{sas-definition-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedSasDefinitionSender sends the GetDeletedSasDefinition request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedSasDefinitionSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedSasDefinitionResponder handles the response to the GetDeletedSasDefinition request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedSasDefinitionResponder(resp *http.Response) (result DeletedSasDefinitionBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeletedSasDefinitions the Get Deleted Sas Definitions operation returns the SAS definitions that have been
// deleted for a vault enabled for soft-delete. This operation requires the storage/listsas permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetDeletedSasDefinitions(ctx context.Context, vaultBaseURL string, storageAccountName string, maxresults *int32) (result DeletedSasDefinitionListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedSasDefinitions")
		defer func() {
			sc := -1
			if result.dsdlr.Response.Response != nil {
				sc = result.dsdlr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetDeletedSasDefinitions", err.Error())
	}

	result.fn = client.getDeletedSasDefinitionsNextResults
	req, err := client.GetDeletedSasDefinitionsPreparer(ctx, vaultBaseURL, storageAccountName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSasDefinitions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedSasDefinitionsSender(req)
	if err != nil {
		result.dsdlr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSasDefinitions", resp, "Failure sending request")
		return
	}

	result.dsdlr, err = client.GetDeletedSasDefinitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSasDefinitions", resp, "Failure responding to request")
	}

	return
}

// GetDeletedSasDefinitionsPreparer prepares the GetDeletedSasDefinitions request.
func (client BaseClient) GetDeletedSasDefinitionsPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedstorage/{storage-account-name}/sas", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedSasDefinitionsSender sends the GetDeletedSasDefinitions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedSasDefinitionsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedSasDefinitionsResponder handles the response to the GetDeletedSasDefinitions request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedSasDefinitionsResponder(resp *http.Response) (result DeletedSasDefinitionListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getDeletedSasDefinitionsNextResults retrieves the next set of results, if any.
func (client BaseClient) getDeletedSasDefinitionsNextResults(ctx context.Context, lastResults DeletedSasDefinitionListResult) (result DeletedSasDefinitionListResult, err error) {
	req, err := lastResults.deletedSasDefinitionListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedSasDefinitionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetDeletedSasDefinitionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedSasDefinitionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetDeletedSasDefinitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedSasDefinitionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetDeletedSasDefinitionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetDeletedSasDefinitionsComplete(ctx context.Context, vaultBaseURL string, storageAccountName string, maxresults *int32) (result DeletedSasDefinitionListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedSasDefinitions")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetDeletedSasDefinitions(ctx, vaultBaseURL, storageAccountName, maxresults)
	return
}

// GetDeletedSecret the Get Deleted Secret operation returns the specified deleted secret along with its attributes.
// This operation requires the secrets/get permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the secret.
func (client BaseClient) GetDeletedSecret(ctx context.Context, vaultBaseURL string, secretName string) (result DeletedSecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetDeletedSecretPreparer(ctx, vaultBaseURL, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSecret", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeletedSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSecret", resp, "Failure responding to request")
	}

	return
}

// GetDeletedSecretPreparer prepares the GetDeletedSecret request.
func (client BaseClient) GetDeletedSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedsecrets/{secret-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedSecretSender sends the GetDeletedSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedSecretResponder handles the response to the GetDeletedSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedSecretResponder(resp *http.Response) (result DeletedSecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeletedSecrets the Get Deleted Secrets operation returns the secrets that have been deleted for a vault enabled
// for soft-delete. This operation requires the secrets/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetDeletedSecrets(ctx context.Context, vaultBaseURL string, maxresults *int32) (result DeletedSecretListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedSecrets")
		defer func() {
			sc := -1
			if result.dslr.Response.Response != nil {
				sc = result.dslr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetDeletedSecrets", err.Error())
	}

	result.fn = client.getDeletedSecretsNextResults
	req, err := client.GetDeletedSecretsPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSecrets", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedSecretsSender(req)
	if err != nil {
		result.dslr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSecrets", resp, "Failure sending request")
		return
	}

	result.dslr, err = client.GetDeletedSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedSecrets", resp, "Failure responding to request")
	}

	return
}

// GetDeletedSecretsPreparer prepares the GetDeletedSecrets request.
func (client BaseClient) GetDeletedSecretsPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/deletedsecrets"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedSecretsSender sends the GetDeletedSecrets request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedSecretsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedSecretsResponder handles the response to the GetDeletedSecrets request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedSecretsResponder(resp *http.Response) (result DeletedSecretListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getDeletedSecretsNextResults retrieves the next set of results, if any.
func (client BaseClient) getDeletedSecretsNextResults(ctx context.Context, lastResults DeletedSecretListResult) (result DeletedSecretListResult, err error) {
	req, err := lastResults.deletedSecretListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedSecretsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetDeletedSecretsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedSecretsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetDeletedSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedSecretsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetDeletedSecretsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetDeletedSecretsComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result DeletedSecretListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedSecrets")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetDeletedSecrets(ctx, vaultBaseURL, maxresults)
	return
}

// GetDeletedStorageAccount the Get Deleted Storage Account operation returns the specified deleted storage account
// along with its attributes. This operation requires the storage/get permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
func (client BaseClient) GetDeletedStorageAccount(ctx context.Context, vaultBaseURL string, storageAccountName string) (result DeletedStorageBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedStorageAccount")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetDeletedStorageAccount", err.Error())
	}

	req, err := client.GetDeletedStorageAccountPreparer(ctx, vaultBaseURL, storageAccountName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedStorageAccountSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.GetDeletedStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedStorageAccount", resp, "Failure responding to request")
	}

	return
}

// GetDeletedStorageAccountPreparer prepares the GetDeletedStorageAccount request.
func (client BaseClient) GetDeletedStorageAccountPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedstorage/{storage-account-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedStorageAccountSender sends the GetDeletedStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedStorageAccountResponder handles the response to the GetDeletedStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedStorageAccountResponder(resp *http.Response) (result DeletedStorageBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDeletedStorageAccounts the Get Deleted Storage Accounts operation returns the storage accounts that have been
// deleted for a vault enabled for soft-delete. This operation requires the storage/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetDeletedStorageAccounts(ctx context.Context, vaultBaseURL string, maxresults *int32) (result DeletedStorageListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedStorageAccounts")
		defer func() {
			sc := -1
			if result.dslr.Response.Response != nil {
				sc = result.dslr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetDeletedStorageAccounts", err.Error())
	}

	result.fn = client.getDeletedStorageAccountsNextResults
	req, err := client.GetDeletedStorageAccountsPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedStorageAccounts", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDeletedStorageAccountsSender(req)
	if err != nil {
		result.dslr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedStorageAccounts", resp, "Failure sending request")
		return
	}

	result.dslr, err = client.GetDeletedStorageAccountsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetDeletedStorageAccounts", resp, "Failure responding to request")
	}

	return
}

// GetDeletedStorageAccountsPreparer prepares the GetDeletedStorageAccounts request.
func (client BaseClient) GetDeletedStorageAccountsPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/deletedstorage"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDeletedStorageAccountsSender sends the GetDeletedStorageAccounts request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetDeletedStorageAccountsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetDeletedStorageAccountsResponder handles the response to the GetDeletedStorageAccounts request. The method always
// closes the http.Response Body.
func (client BaseClient) GetDeletedStorageAccountsResponder(resp *http.Response) (result DeletedStorageListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getDeletedStorageAccountsNextResults retrieves the next set of results, if any.
func (client BaseClient) getDeletedStorageAccountsNextResults(ctx context.Context, lastResults DeletedStorageListResult) (result DeletedStorageListResult, err error) {
	req, err := lastResults.deletedStorageListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedStorageAccountsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetDeletedStorageAccountsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedStorageAccountsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetDeletedStorageAccountsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getDeletedStorageAccountsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetDeletedStorageAccountsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetDeletedStorageAccountsComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result DeletedStorageListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetDeletedStorageAccounts")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetDeletedStorageAccounts(ctx, vaultBaseURL, maxresults)
	return
}

// GetKey the get key operation is applicable to all key types. If the requested key is symmetric, then no key material
// is released in the response. This operation requires the keys/get permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key to get.
// keyVersion - adding the version parameter retrieves a specific version of a key.
func (client BaseClient) GetKey(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetKeyPreparer(ctx, vaultBaseURL, keyName, keyVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKey", resp, "Failure sending request")
		return
	}

	result, err = client.GetKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKey", resp, "Failure responding to request")
	}

	return
}

// GetKeyPreparer prepares the GetKey request.
func (client BaseClient) GetKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetKeySender sends the GetKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetKeyResponder handles the response to the GetKey request. The method always
// closes the http.Response Body.
func (client BaseClient) GetKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetKeys retrieves a list of the keys in the Key Vault as JSON Web Key structures that contain the public part of a
// stored key. The LIST operation is applicable to all key types, however only the base key identifier, attributes, and
// tags are provided in the response. Individual versions of a key are not listed in the response. This operation
// requires the keys/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetKeys(ctx context.Context, vaultBaseURL string, maxresults *int32) (result KeyListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKeys")
		defer func() {
			sc := -1
			if result.klr.Response.Response != nil {
				sc = result.klr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetKeys", err.Error())
	}

	result.fn = client.getKeysNextResults
	req, err := client.GetKeysPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeys", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeysSender(req)
	if err != nil {
		result.klr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeys", resp, "Failure sending request")
		return
	}

	result.klr, err = client.GetKeysResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeys", resp, "Failure responding to request")
	}

	return
}

// GetKeysPreparer prepares the GetKeys request.
func (client BaseClient) GetKeysPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/keys"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetKeysSender sends the GetKeys request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetKeysSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetKeysResponder handles the response to the GetKeys request. The method always
// closes the http.Response Body.
func (client BaseClient) GetKeysResponder(resp *http.Response) (result KeyListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getKeysNextResults retrieves the next set of results, if any.
func (client BaseClient) getKeysNextResults(ctx context.Context, lastResults KeyListResult) (result KeyListResult, err error) {
	req, err := lastResults.keyListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeysNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetKeysSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeysNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetKeysResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeysNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetKeysComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetKeysComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result KeyListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKeys")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetKeys(ctx, vaultBaseURL, maxresults)
	return
}

// GetKeyVersions the full key identifier, attributes, and tags are provided in the response. This operation requires
// the keys/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetKeyVersions(ctx context.Context, vaultBaseURL string, keyName string, maxresults *int32) (result KeyListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKeyVersions")
		defer func() {
			sc := -1
			if result.klr.Response.Response != nil {
				sc = result.klr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetKeyVersions", err.Error())
	}

	result.fn = client.getKeyVersionsNextResults
	req, err := client.GetKeyVersionsPreparer(ctx, vaultBaseURL, keyName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeyVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeyVersionsSender(req)
	if err != nil {
		result.klr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeyVersions", resp, "Failure sending request")
		return
	}

	result.klr, err = client.GetKeyVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeyVersions", resp, "Failure responding to request")
	}

	return
}

// GetKeyVersionsPreparer prepares the GetKeyVersions request.
func (client BaseClient) GetKeyVersionsPreparer(ctx context.Context, vaultBaseURL string, keyName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetKeyVersionsSender sends the GetKeyVersions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetKeyVersionsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetKeyVersionsResponder handles the response to the GetKeyVersions request. The method always
// closes the http.Response Body.
func (client BaseClient) GetKeyVersionsResponder(resp *http.Response) (result KeyListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getKeyVersionsNextResults retrieves the next set of results, if any.
func (client BaseClient) getKeyVersionsNextResults(ctx context.Context, lastResults KeyListResult) (result KeyListResult, err error) {
	req, err := lastResults.keyListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeyVersionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetKeyVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeyVersionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetKeyVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeyVersionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetKeyVersionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetKeyVersionsComplete(ctx context.Context, vaultBaseURL string, keyName string, maxresults *int32) (result KeyListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKeyVersions")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetKeyVersions(ctx, vaultBaseURL, keyName, maxresults)
	return
}

// GetSasDefinition gets information about a SAS definition for the specified storage account. This operation requires
// the storage/getsas permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// sasDefinitionName - the name of the SAS definition.
func (client BaseClient) GetSasDefinition(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string) (result SasDefinitionBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSasDefinition")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: sasDefinitionName,
			Constraints: []validation.Constraint{{Target: "sasDefinitionName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetSasDefinition", err.Error())
	}

	req, err := client.GetSasDefinitionPreparer(ctx, vaultBaseURL, storageAccountName, sasDefinitionName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSasDefinition", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSasDefinitionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSasDefinition", resp, "Failure sending request")
		return
	}

	result, err = client.GetSasDefinitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSasDefinition", resp, "Failure responding to request")
	}

	return
}

// GetSasDefinitionPreparer prepares the GetSasDefinition request.
func (client BaseClient) GetSasDefinitionPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"sas-definition-name":  autorest.Encode("path", sasDefinitionName),
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}/sas/{sas-definition-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSasDefinitionSender sends the GetSasDefinition request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetSasDefinitionSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetSasDefinitionResponder handles the response to the GetSasDefinition request. The method always
// closes the http.Response Body.
func (client BaseClient) GetSasDefinitionResponder(resp *http.Response) (result SasDefinitionBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetSasDefinitions list storage SAS definitions for the given storage account. This operation requires the
// storage/listsas permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetSasDefinitions(ctx context.Context, vaultBaseURL string, storageAccountName string, maxresults *int32) (result SasDefinitionListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSasDefinitions")
		defer func() {
			sc := -1
			if result.sdlr.Response.Response != nil {
				sc = result.sdlr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetSasDefinitions", err.Error())
	}

	result.fn = client.getSasDefinitionsNextResults
	req, err := client.GetSasDefinitionsPreparer(ctx, vaultBaseURL, storageAccountName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSasDefinitions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSasDefinitionsSender(req)
	if err != nil {
		result.sdlr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSasDefinitions", resp, "Failure sending request")
		return
	}

	result.sdlr, err = client.GetSasDefinitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSasDefinitions", resp, "Failure responding to request")
	}

	return
}

// GetSasDefinitionsPreparer prepares the GetSasDefinitions request.
func (client BaseClient) GetSasDefinitionsPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}/sas", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSasDefinitionsSender sends the GetSasDefinitions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetSasDefinitionsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetSasDefinitionsResponder handles the response to the GetSasDefinitions request. The method always
// closes the http.Response Body.
func (client BaseClient) GetSasDefinitionsResponder(resp *http.Response) (result SasDefinitionListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getSasDefinitionsNextResults retrieves the next set of results, if any.
func (client BaseClient) getSasDefinitionsNextResults(ctx context.Context, lastResults SasDefinitionListResult) (result SasDefinitionListResult, err error) {
	req, err := lastResults.sasDefinitionListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSasDefinitionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetSasDefinitionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSasDefinitionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetSasDefinitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSasDefinitionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetSasDefinitionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetSasDefinitionsComplete(ctx context.Context, vaultBaseURL string, storageAccountName string, maxresults *int32) (result SasDefinitionListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSasDefinitions")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetSasDefinitions(ctx, vaultBaseURL, storageAccountName, maxresults)
	return
}

// GetSecret the GET operation is applicable to any secret stored in Azure Key Vault. This operation requires the
// secrets/get permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the secret.
// secretVersion - the version of the secret.
func (client BaseClient) GetSecret(ctx context.Context, vaultBaseURL string, secretName string, secretVersion string) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetSecretPreparer(ctx, vaultBaseURL, secretName, secretVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecret", resp, "Failure sending request")
		return
	}

	result, err = client.GetSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecret", resp, "Failure responding to request")
	}

	return
}

// GetSecretPreparer prepares the GetSecret request.
func (client BaseClient) GetSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string, secretVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name":    autorest.Encode("path", secretName),
		"secret-version": autorest.Encode("path", secretVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/{secret-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSecretSender sends the GetSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetSecretResponder handles the response to the GetSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) GetSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetSecrets the Get Secrets operation is applicable to the entire vault. However, only the base secret identifier and
// its attributes are provided in the response. Individual secret versions are not listed in the response. This
// operation requires the secrets/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified, the service will return up to
// 25 results.
func (client BaseClient) GetSecrets(ctx context.Context, vaultBaseURL string, maxresults *int32) (result SecretListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecrets")
		defer func() {
			sc := -1
			if result.slr.Response.Response != nil {
				sc = result.slr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetSecrets", err.Error())
	}

	result.fn = client.getSecretsNextResults
	req, err := client.GetSecretsPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecrets", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretsSender(req)
	if err != nil {
		result.slr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecrets", resp, "Failure sending request")
		return
	}

	result.slr, err = client.GetSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecrets", resp, "Failure responding to request")
	}

	return
}

// GetSecretsPreparer prepares the GetSecrets request.
func (client BaseClient) GetSecretsPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/secrets"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSecretsSender sends the GetSecrets request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetSecretsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetSecretsResponder handles the response to the GetSecrets request. The method always
// closes the http.Response Body.
func (client BaseClient) GetSecretsResponder(resp *http.Response) (result SecretListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getSecretsNextResults retrieves the next set of results, if any.
func (client BaseClient) getSecretsNextResults(ctx context.Context, lastResults SecretListResult) (result SecretListResult, err error) {
	req, err := lastResults.secretListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetSecretsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetSecretsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetSecretsComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result SecretListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecrets")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetSecrets(ctx, vaultBaseURL, maxresults)
	return
}

// GetSecretVersions the full secret identifier and attributes are provided in the response. No values are returned for
// the secrets. This operations requires the secrets/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the secret.
// maxresults - maximum number of results to return in a page. If not specified, the service will return up to
// 25 results.
func (client BaseClient) GetSecretVersions(ctx context.Context, vaultBaseURL string, secretName string, maxresults *int32) (result SecretListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecretVersions")
		defer func() {
			sc := -1
			if result.slr.Response.Response != nil {
				sc = result.slr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetSecretVersions", err.Error())
	}

	result.fn = client.getSecretVersionsNextResults
	req, err := client.GetSecretVersionsPreparer(ctx, vaultBaseURL, secretName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecretVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretVersionsSender(req)
	if err != nil {
		result.slr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecretVersions", resp, "Failure sending request")
		return
	}

	result.slr, err = client.GetSecretVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecretVersions", resp, "Failure responding to request")
	}

	return
}

// GetSecretVersionsPreparer prepares the GetSecretVersions request.
func (client BaseClient) GetSecretVersionsPreparer(ctx context.Context, vaultBaseURL string, secretName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSecretVersionsSender sends the GetSecretVersions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetSecretVersionsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetSecretVersionsResponder handles the response to the GetSecretVersions request. The method always
// closes the http.Response Body.
func (client BaseClient) GetSecretVersionsResponder(resp *http.Response) (result SecretListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getSecretVersionsNextResults retrieves the next set of results, if any.
func (client BaseClient) getSecretVersionsNextResults(ctx context.Context, lastResults SecretListResult) (result SecretListResult, err error) {
	req, err := lastResults.secretListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretVersionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetSecretVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretVersionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetSecretVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretVersionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetSecretVersionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetSecretVersionsComplete(ctx context.Context, vaultBaseURL string, secretName string, maxresults *int32) (result SecretListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecretVersions")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetSecretVersions(ctx, vaultBaseURL, secretName, maxresults)
	return
}

// GetStorageAccount gets information about a specified storage account. This operation requires the storage/get
// permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
func (client BaseClient) GetStorageAccount(ctx context.Context, vaultBaseURL string, storageAccountName string) (result StorageBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetStorageAccount")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetStorageAccount", err.Error())
	}

	req, err := client.GetStorageAccountPreparer(ctx, vaultBaseURL, storageAccountName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetStorageAccountSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.GetStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetStorageAccount", resp, "Failure responding to request")
	}

	return
}

// GetStorageAccountPreparer prepares the GetStorageAccount request.
func (client BaseClient) GetStorageAccountPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetStorageAccountSender sends the GetStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetStorageAccountResponder handles the response to the GetStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) GetStorageAccountResponder(resp *http.Response) (result StorageBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetStorageAccounts list storage accounts managed by the specified key vault. This operation requires the
// storage/list permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetStorageAccounts(ctx context.Context, vaultBaseURL string, maxresults *int32) (result StorageListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetStorageAccounts")
		defer func() {
			sc := -1
			if result.slr.Response.Response != nil {
				sc = result.slr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetStorageAccounts", err.Error())
	}

	result.fn = client.getStorageAccountsNextResults
	req, err := client.GetStorageAccountsPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetStorageAccounts", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetStorageAccountsSender(req)
	if err != nil {
		result.slr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetStorageAccounts", resp, "Failure sending request")
		return
	}

	result.slr, err = client.GetStorageAccountsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetStorageAccounts", resp, "Failure responding to request")
	}

	return
}

// GetStorageAccountsPreparer prepares the GetStorageAccounts request.
func (client BaseClient) GetStorageAccountsPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/storage"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetStorageAccountsSender sends the GetStorageAccounts request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetStorageAccountsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// GetStorageAccountsResponder handles the response to the GetStorageAccounts request. The method always
// closes the http.Response Body.
func (client BaseClient) GetStorageAccountsResponder(resp *http.Response) (result StorageListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getStorageAccountsNextResults retrieves the next set of results, if any.
func (client BaseClient) getStorageAccountsNextResults(ctx context.Context, lastResults StorageListResult) (result StorageListResult, err error) {
	req, err := lastResults.storageListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getStorageAccountsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetStorageAccountsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getStorageAccountsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetStorageAccountsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getStorageAccountsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetStorageAccountsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetStorageAccountsComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result StorageListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetStorageAccounts")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetStorageAccounts(ctx, vaultBaseURL, maxresults)
	return
}

// ImportCertificate imports an existing valid certificate, containing a private key, into Azure Key Vault. The
// certificate to be imported can be in either PFX or PEM format. If the certificate is in PEM format the PEM file must
// contain the key as well as x509 certificates. This operation requires the certificates/import permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
// parameters - the parameters to import the certificate.
func (client BaseClient) ImportCertificate(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateImportParameters) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.ImportCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: certificateName,
			Constraints: []validation.Constraint{{Target: "certificateName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Base64EncodedCertificate", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.CertificatePolicy", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties", Name: validation.Null, Rule: false,
						Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.Null, Rule: false,
							Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}},
						}},
					}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "ImportCertificate", err.Error())
	}

	req, err := client.ImportCertificatePreparer(ctx, vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.ImportCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.ImportCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportCertificate", resp, "Failure responding to request")
	}

	return
}

// ImportCertificatePreparer prepares the ImportCertificate request.
func (client BaseClient) ImportCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateImportParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/import", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ImportCertificateSender sends the ImportCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ImportCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// ImportCertificateResponder handles the response to the ImportCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) ImportCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ImportKey the import key operation may be used to import any key type into an Azure Key Vault. If the named key
// already exists, Azure Key Vault creates a new version of the key. This operation requires the keys/import
// permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - name for the imported key.
// parameters - the parameters to import a key.
func (client BaseClient) ImportKey(ctx context.Context, vaultBaseURL string, keyName string, parameters KeyImportParameters) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.ImportKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: keyName,
			Constraints: []validation.Constraint{{Target: "keyName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Key", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "ImportKey", err.Error())
	}

	req, err := client.ImportKeyPreparer(ctx, vaultBaseURL, keyName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.ImportKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportKey", resp, "Failure sending request")
		return
	}

	result, err = client.ImportKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportKey", resp, "Failure responding to request")
	}

	return
}

// ImportKeyPreparer prepares the ImportKey request.
func (client BaseClient) ImportKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, parameters KeyImportParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ImportKeySender sends the ImportKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ImportKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// ImportKeyResponder handles the response to the ImportKey request. The method always
// closes the http.Response Body.
func (client BaseClient) ImportKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// MergeCertificate the MergeCertificate operation performs the merging of a certificate or certificate chain with a
// key pair currently available in the service. This operation requires the certificates/create permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
// parameters - the parameters to merge certificate.
func (client BaseClient) MergeCertificate(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateMergeParameters) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.MergeCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.X509Certificates", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "MergeCertificate", err.Error())
	}

	req, err := client.MergeCertificatePreparer(ctx, vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "MergeCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.MergeCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "MergeCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.MergeCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "MergeCertificate", resp, "Failure responding to request")
	}

	return
}

// MergeCertificatePreparer prepares the MergeCertificate request.
func (client BaseClient) MergeCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateMergeParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending/merge", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// MergeCertificateSender sends the MergeCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) MergeCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// MergeCertificateResponder handles the response to the MergeCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) MergeCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusCreated),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// PurgeDeletedCertificate the PurgeDeletedCertificate operation performs an irreversible deletion of the specified
// certificate, without possibility for recovery. The operation is not available if the recovery level does not specify
// 'Purgeable'. This operation requires the certificate/purge permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate
func (client BaseClient) PurgeDeletedCertificate(ctx context.Context, vaultBaseURL string, certificateName string) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.PurgeDeletedCertificate")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.PurgeDeletedCertificatePreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.PurgeDeletedCertificateSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.PurgeDeletedCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedCertificate", resp, "Failure responding to request")
	}

	return
}

// PurgeDeletedCertificatePreparer prepares the PurgeDeletedCertificate request.
func (client BaseClient) PurgeDeletedCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedcertificates/{certificate-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// PurgeDeletedCertificateSender sends the PurgeDeletedCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) PurgeDeletedCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// PurgeDeletedCertificateResponder handles the response to the PurgeDeletedCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) PurgeDeletedCertificateResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByClosing())
	result.Response = resp
	return
}

// PurgeDeletedKey the Purge Deleted Key operation is applicable for soft-delete enabled vaults. While the operation
// can be invoked on any vault, it will return an error if invoked on a non soft-delete enabled vault. This operation
// requires the keys/purge permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key
func (client BaseClient) PurgeDeletedKey(ctx context.Context, vaultBaseURL string, keyName string) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.PurgeDeletedKey")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.PurgeDeletedKeyPreparer(ctx, vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.PurgeDeletedKeySender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedKey", resp, "Failure sending request")
		return
	}

	result, err = client.PurgeDeletedKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedKey", resp, "Failure responding to request")
	}

	return
}

// PurgeDeletedKeyPreparer prepares the PurgeDeletedKey request.
func (client BaseClient) PurgeDeletedKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedkeys/{key-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// PurgeDeletedKeySender sends the PurgeDeletedKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) PurgeDeletedKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// PurgeDeletedKeyResponder handles the response to the PurgeDeletedKey request. The method always
// closes the http.Response Body.
func (client BaseClient) PurgeDeletedKeyResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByClosing())
	result.Response = resp
	return
}

// PurgeDeletedSecret the purge deleted secret operation removes the secret permanently, without the possibility of
// recovery. This operation can only be enabled on a soft-delete enabled vault. This operation requires the
// secrets/purge permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the secret.
func (client BaseClient) PurgeDeletedSecret(ctx context.Context, vaultBaseURL string, secretName string) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.PurgeDeletedSecret")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.PurgeDeletedSecretPreparer(ctx, vaultBaseURL, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.PurgeDeletedSecretSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedSecret", resp, "Failure sending request")
		return
	}

	result, err = client.PurgeDeletedSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedSecret", resp, "Failure responding to request")
	}

	return
}

// PurgeDeletedSecretPreparer prepares the PurgeDeletedSecret request.
func (client BaseClient) PurgeDeletedSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedsecrets/{secret-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// PurgeDeletedSecretSender sends the PurgeDeletedSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) PurgeDeletedSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// PurgeDeletedSecretResponder handles the response to the PurgeDeletedSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) PurgeDeletedSecretResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByClosing())
	result.Response = resp
	return
}

// PurgeDeletedStorageAccount the purge deleted storage account operation removes the secret permanently, without the
// possibility of recovery. This operation can only be performed on a soft-delete enabled vault. This operation
// requires the storage/purge permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
func (client BaseClient) PurgeDeletedStorageAccount(ctx context.Context, vaultBaseURL string, storageAccountName string) (result autorest.Response, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.PurgeDeletedStorageAccount")
		defer func() {
			sc := -1
			if result.Response != nil {
				sc = result.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "PurgeDeletedStorageAccount", err.Error())
	}

	req, err := client.PurgeDeletedStorageAccountPreparer(ctx, vaultBaseURL, storageAccountName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.PurgeDeletedStorageAccountSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.PurgeDeletedStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "PurgeDeletedStorageAccount", resp, "Failure responding to request")
	}

	return
}

// PurgeDeletedStorageAccountPreparer prepares the PurgeDeletedStorageAccount request.
func (client BaseClient) PurgeDeletedStorageAccountPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedstorage/{storage-account-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// PurgeDeletedStorageAccountSender sends the PurgeDeletedStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) PurgeDeletedStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// PurgeDeletedStorageAccountResponder handles the response to the PurgeDeletedStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) PurgeDeletedStorageAccountResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RecoverDeletedCertificate the RecoverDeletedCertificate operation performs the reversal of the Delete operation. The
// operation is applicable in vaults enabled for soft-delete, and must be issued during the retention interval
// (available in the deleted certificate's attributes). This operation requires the certificates/recover permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the deleted certificate
func (client BaseClient) RecoverDeletedCertificate(ctx context.Context, vaultBaseURL string, certificateName string) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RecoverDeletedCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.RecoverDeletedCertificatePreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverDeletedCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverDeletedCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedCertificate", resp, "Failure responding to request")
	}

	return
}

// RecoverDeletedCertificatePreparer prepares the RecoverDeletedCertificate request.
func (client BaseClient) RecoverDeletedCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedcertificates/{certificate-name}/recover", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverDeletedCertificateSender sends the RecoverDeletedCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverDeletedCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RecoverDeletedCertificateResponder handles the response to the RecoverDeletedCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverDeletedCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RecoverDeletedKey the Recover Deleted Key operation is applicable for deleted keys in soft-delete enabled vaults. It
// recovers the deleted key back to its latest version under /keys. An attempt to recover an non-deleted key will
// return an error. Consider this the inverse of the delete operation on soft-delete enabled vaults. This operation
// requires the keys/recover permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the deleted key.
func (client BaseClient) RecoverDeletedKey(ctx context.Context, vaultBaseURL string, keyName string) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RecoverDeletedKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.RecoverDeletedKeyPreparer(ctx, vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverDeletedKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedKey", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverDeletedKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedKey", resp, "Failure responding to request")
	}

	return
}

// RecoverDeletedKeyPreparer prepares the RecoverDeletedKey request.
func (client BaseClient) RecoverDeletedKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedkeys/{key-name}/recover", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverDeletedKeySender sends the RecoverDeletedKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverDeletedKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RecoverDeletedKeyResponder handles the response to the RecoverDeletedKey request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverDeletedKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RecoverDeletedSasDefinition recovers the deleted SAS definition for the specified storage account. This operation
// can only be performed on a soft-delete enabled vault. This operation requires the storage/recover permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// sasDefinitionName - the name of the SAS definition.
func (client BaseClient) RecoverDeletedSasDefinition(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string) (result SasDefinitionBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RecoverDeletedSasDefinition")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: sasDefinitionName,
			Constraints: []validation.Constraint{{Target: "sasDefinitionName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "RecoverDeletedSasDefinition", err.Error())
	}

	req, err := client.RecoverDeletedSasDefinitionPreparer(ctx, vaultBaseURL, storageAccountName, sasDefinitionName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedSasDefinition", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverDeletedSasDefinitionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedSasDefinition", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverDeletedSasDefinitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedSasDefinition", resp, "Failure responding to request")
	}

	return
}

// RecoverDeletedSasDefinitionPreparer prepares the RecoverDeletedSasDefinition request.
func (client BaseClient) RecoverDeletedSasDefinitionPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"sas-definition-name":  autorest.Encode("path", sasDefinitionName),
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedstorage/{storage-account-name}/sas/{sas-definition-name}/recover", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverDeletedSasDefinitionSender sends the RecoverDeletedSasDefinition request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverDeletedSasDefinitionSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RecoverDeletedSasDefinitionResponder handles the response to the RecoverDeletedSasDefinition request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverDeletedSasDefinitionResponder(resp *http.Response) (result SasDefinitionBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RecoverDeletedSecret recovers the deleted secret in the specified vault. This operation can only be performed on a
// soft-delete enabled vault. This operation requires the secrets/recover permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the deleted secret.
func (client BaseClient) RecoverDeletedSecret(ctx context.Context, vaultBaseURL string, secretName string) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RecoverDeletedSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.RecoverDeletedSecretPreparer(ctx, vaultBaseURL, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverDeletedSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedSecret", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverDeletedSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedSecret", resp, "Failure responding to request")
	}

	return
}

// RecoverDeletedSecretPreparer prepares the RecoverDeletedSecret request.
func (client BaseClient) RecoverDeletedSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedsecrets/{secret-name}/recover", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverDeletedSecretSender sends the RecoverDeletedSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverDeletedSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RecoverDeletedSecretResponder handles the response to the RecoverDeletedSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverDeletedSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RecoverDeletedStorageAccount recovers the deleted storage account in the specified vault. This operation can only be
// performed on a soft-delete enabled vault. This operation requires the storage/recover permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
func (client BaseClient) RecoverDeletedStorageAccount(ctx context.Context, vaultBaseURL string, storageAccountName string) (result StorageBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RecoverDeletedStorageAccount")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "RecoverDeletedStorageAccount", err.Error())
	}

	req, err := client.RecoverDeletedStorageAccountPreparer(ctx, vaultBaseURL, storageAccountName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.RecoverDeletedStorageAccountSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.RecoverDeletedStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RecoverDeletedStorageAccount", resp, "Failure responding to request")
	}

	return
}

// RecoverDeletedStorageAccountPreparer prepares the RecoverDeletedStorageAccount request.
func (client BaseClient) RecoverDeletedStorageAccountPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/deletedstorage/{storage-account-name}/recover", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RecoverDeletedStorageAccountSender sends the RecoverDeletedStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RecoverDeletedStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RecoverDeletedStorageAccountResponder handles the response to the RecoverDeletedStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) RecoverDeletedStorageAccountResponder(resp *http.Response) (result StorageBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RegenerateStorageAccountKey regenerates the specified key value for the given storage account. This operation
// requires the storage/regeneratekey permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// parameters - the parameters to regenerate storage account key.
func (client BaseClient) RegenerateStorageAccountKey(ctx context.Context, vaultBaseURL string, storageAccountName string, parameters StorageAccountRegenerteKeyParameters) (result StorageBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RegenerateStorageAccountKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.KeyName", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "RegenerateStorageAccountKey", err.Error())
	}

	req, err := client.RegenerateStorageAccountKeyPreparer(ctx, vaultBaseURL, storageAccountName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RegenerateStorageAccountKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.RegenerateStorageAccountKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RegenerateStorageAccountKey", resp, "Failure sending request")
		return
	}

	result, err = client.RegenerateStorageAccountKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RegenerateStorageAccountKey", resp, "Failure responding to request")
	}

	return
}

// RegenerateStorageAccountKeyPreparer prepares the RegenerateStorageAccountKey request.
func (client BaseClient) RegenerateStorageAccountKeyPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, parameters StorageAccountRegenerteKeyParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}/regeneratekey", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RegenerateStorageAccountKeySender sends the RegenerateStorageAccountKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RegenerateStorageAccountKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RegenerateStorageAccountKeyResponder handles the response to the RegenerateStorageAccountKey request. The method always
// closes the http.Response Body.
func (client BaseClient) RegenerateStorageAccountKeyResponder(resp *http.Response) (result StorageBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RestoreCertificate restores a backed up certificate, and all its versions, to a vault. This operation requires the
// certificates/restore permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// parameters - the parameters to restore the certificate.
func (client BaseClient) RestoreCertificate(ctx context.Context, vaultBaseURL string, parameters CertificateRestoreParameters) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RestoreCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.CertificateBundleBackup", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "RestoreCertificate", err.Error())
	}

	req, err := client.RestoreCertificatePreparer(ctx, vaultBaseURL, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestoreCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.RestoreCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreCertificate", resp, "Failure responding to request")
	}

	return
}

// RestoreCertificatePreparer prepares the RestoreCertificate request.
func (client BaseClient) RestoreCertificatePreparer(ctx context.Context, vaultBaseURL string, parameters CertificateRestoreParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/restore"),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RestoreCertificateSender sends the RestoreCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RestoreCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RestoreCertificateResponder handles the response to the RestoreCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) RestoreCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RestoreKey imports a previously backed up key into Azure Key Vault, restoring the key, its key identifier,
// attributes and access control policies. The RESTORE operation may be used to import a previously backed up key.
// Individual versions of a key cannot be restored. The key is restored in its entirety with the same key name as it
// had when it was backed up. If the key name is not available in the target Key Vault, the RESTORE operation will be
// rejected. While the key name is retained during restore, the final key identifier will change if the key is restored
// to a different vault. Restore will restore all versions and preserve version identifiers. The RESTORE operation is
// subject to security constraints: The target Key Vault must be owned by the same Microsoft Azure Subscription as the
// source Key Vault The user must have RESTORE permission in the target Key Vault. This operation requires the
// keys/restore permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// parameters - the parameters to restore the key.
func (client BaseClient) RestoreKey(ctx context.Context, vaultBaseURL string, parameters KeyRestoreParameters) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RestoreKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.KeyBundleBackup", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "RestoreKey", err.Error())
	}

	req, err := client.RestoreKeyPreparer(ctx, vaultBaseURL, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestoreKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreKey", resp, "Failure sending request")
		return
	}

	result, err = client.RestoreKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreKey", resp, "Failure responding to request")
	}

	return
}

// RestoreKeyPreparer prepares the RestoreKey request.
func (client BaseClient) RestoreKeyPreparer(ctx context.Context, vaultBaseURL string, parameters KeyRestoreParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/keys/restore"),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RestoreKeySender sends the RestoreKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RestoreKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RestoreKeyResponder handles the response to the RestoreKey request. The method always
// closes the http.Response Body.
func (client BaseClient) RestoreKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RestoreSecret restores a backed up secret, and all its versions, to a vault. This operation requires the
// secrets/restore permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// parameters - the parameters to restore the secret.
func (client BaseClient) RestoreSecret(ctx context.Context, vaultBaseURL string, parameters SecretRestoreParameters) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RestoreSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.SecretBundleBackup", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "RestoreSecret", err.Error())
	}

	req, err := client.RestoreSecretPreparer(ctx, vaultBaseURL, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestoreSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreSecret", resp, "Failure sending request")
		return
	}

	result, err = client.RestoreSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreSecret", resp, "Failure responding to request")
	}

	return
}

// RestoreSecretPreparer prepares the RestoreSecret request.
func (client BaseClient) RestoreSecretPreparer(ctx context.Context, vaultBaseURL string, parameters SecretRestoreParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/secrets/restore"),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RestoreSecretSender sends the RestoreSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RestoreSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RestoreSecretResponder handles the response to the RestoreSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) RestoreSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RestoreStorageAccount restores a backed up storage account to a vault. This operation requires the storage/restore
// permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// parameters - the parameters to restore the storage account.
func (client BaseClient) RestoreStorageAccount(ctx context.Context, vaultBaseURL string, parameters StorageRestoreParameters) (result StorageBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RestoreStorageAccount")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.StorageBundleBackup", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "RestoreStorageAccount", err.Error())
	}

	req, err := client.RestoreStorageAccountPreparer(ctx, vaultBaseURL, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestoreStorageAccountSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.RestoreStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreStorageAccount", resp, "Failure responding to request")
	}

	return
}

// RestoreStorageAccountPreparer prepares the RestoreStorageAccount request.
func (client BaseClient) RestoreStorageAccountPreparer(ctx context.Context, vaultBaseURL string, parameters StorageRestoreParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/storage/restore"),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RestoreStorageAccountSender sends the RestoreStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RestoreStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// RestoreStorageAccountResponder handles the response to the RestoreStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) RestoreStorageAccountResponder(resp *http.Response) (result StorageBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetCertificateContacts sets the certificate contacts for the specified key vault. This operation requires the
// certificates/managecontacts permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// contacts - the contacts for the key vault certificate.
func (client BaseClient) SetCertificateContacts(ctx context.Context, vaultBaseURL string, contacts Contacts) (result Contacts, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.SetCertificateContacts")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.SetCertificateContactsPreparer(ctx, vaultBaseURL, contacts)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.SetCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateContacts", resp, "Failure responding to request")
	}

	return
}

// SetCertificateContactsPreparer prepares the SetCertificateContacts request.
func (client BaseClient) SetCertificateContactsPreparer(ctx context.Context, vaultBaseURL string, contacts Contacts) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	contacts.ID = nil
	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithJSON(contacts),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetCertificateContactsSender sends the SetCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SetCertificateContactsSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// SetCertificateContactsResponder handles the response to the SetCertificateContacts request. The method always
// closes the http.Response Body.
func (client BaseClient) SetCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetCertificateIssuer the SetCertificateIssuer operation adds or updates the specified certificate issuer. This
// operation requires the certificates/setissuers permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// issuerName - the name of the issuer.
// parameter - certificate issuer set parameter.
func (client BaseClient) SetCertificateIssuer(ctx context.Context, vaultBaseURL string, issuerName string, parameter CertificateIssuerSetParameters) (result IssuerBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.SetCertificateIssuer")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameter,
			Constraints: []validation.Constraint{{Target: "parameter.Provider", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "SetCertificateIssuer", err.Error())
	}

	req, err := client.SetCertificateIssuerPreparer(ctx, vaultBaseURL, issuerName, parameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.SetCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateIssuer", resp, "Failure responding to request")
	}

	return
}

// SetCertificateIssuerPreparer prepares the SetCertificateIssuer request.
func (client BaseClient) SetCertificateIssuerPreparer(ctx context.Context, vaultBaseURL string, issuerName string, parameter CertificateIssuerSetParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithJSON(parameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetCertificateIssuerSender sends the SetCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SetCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// SetCertificateIssuerResponder handles the response to the SetCertificateIssuer request. The method always
// closes the http.Response Body.
func (client BaseClient) SetCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetSasDefinition creates or updates a new SAS definition for the specified storage account. This operation requires
// the storage/setsas permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// sasDefinitionName - the name of the SAS definition.
// parameters - the parameters to create a SAS definition.
func (client BaseClient) SetSasDefinition(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string, parameters SasDefinitionCreateParameters) (result SasDefinitionBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.SetSasDefinition")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: sasDefinitionName,
			Constraints: []validation.Constraint{{Target: "sasDefinitionName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.TemplateURI", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.ValidityPeriod", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "SetSasDefinition", err.Error())
	}

	req, err := client.SetSasDefinitionPreparer(ctx, vaultBaseURL, storageAccountName, sasDefinitionName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSasDefinition", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetSasDefinitionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSasDefinition", resp, "Failure sending request")
		return
	}

	result, err = client.SetSasDefinitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSasDefinition", resp, "Failure responding to request")
	}

	return
}

// SetSasDefinitionPreparer prepares the SetSasDefinition request.
func (client BaseClient) SetSasDefinitionPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string, parameters SasDefinitionCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"sas-definition-name":  autorest.Encode("path", sasDefinitionName),
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}/sas/{sas-definition-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetSasDefinitionSender sends the SetSasDefinition request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SetSasDefinitionSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// SetSasDefinitionResponder handles the response to the SetSasDefinition request. The method always
// closes the http.Response Body.
func (client BaseClient) SetSasDefinitionResponder(resp *http.Response) (result SasDefinitionBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetSecret the SET operation adds a secret to the Azure Key Vault. If the named secret already exists, Azure Key
// Vault creates a new version of that secret. This operation requires the secrets/set permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the secret.
// parameters - the parameters for setting the secret.
func (client BaseClient) SetSecret(ctx context.Context, vaultBaseURL string, secretName string, parameters SecretSetParameters) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.SetSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: secretName,
			Constraints: []validation.Constraint{{Target: "secretName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "SetSecret", err.Error())
	}

	req, err := client.SetSecretPreparer(ctx, vaultBaseURL, secretName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSecret", resp, "Failure sending request")
		return
	}

	result, err = client.SetSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSecret", resp, "Failure responding to request")
	}

	return
}

// SetSecretPreparer prepares the SetSecret request.
func (client BaseClient) SetSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string, parameters SecretSetParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetSecretSender sends the SetSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SetSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// SetSecretResponder handles the response to the SetSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) SetSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetStorageAccount creates or updates a new storage account. This operation requires the storage/set permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// parameters - the parameters to create a storage account.
func (client BaseClient) SetStorageAccount(ctx context.Context, vaultBaseURL string, storageAccountName string, parameters StorageAccountCreateParameters) (result StorageBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.SetStorageAccount")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.ResourceID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.ActiveKeyName", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.AutoRegenerateKey", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "SetStorageAccount", err.Error())
	}

	req, err := client.SetStorageAccountPreparer(ctx, vaultBaseURL, storageAccountName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetStorageAccountSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.SetStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetStorageAccount", resp, "Failure responding to request")
	}

	return
}

// SetStorageAccountPreparer prepares the SetStorageAccount request.
func (client BaseClient) SetStorageAccountPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, parameters StorageAccountCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetStorageAccountSender sends the SetStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SetStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// SetStorageAccountResponder handles the response to the SetStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) SetStorageAccountResponder(resp *http.Response) (result StorageBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Sign the SIGN operation is applicable to asymmetric and symmetric keys stored in Azure Key Vault since this
// operation uses the private portion of the key. This operation requires the keys/sign permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
// keyVersion - the version of the key.
// parameters - the parameters for the signing operation.
func (client BaseClient) Sign(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeySignParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Sign")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "Sign", err.Error())
	}

	req, err := client.SignPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Sign", nil, "Failure preparing request")
		return
	}

	resp, err := client.SignSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Sign", resp, "Failure sending request")
		return
	}

	result, err = client.SignResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Sign", resp, "Failure responding to request")
	}

	return
}

// SignPreparer prepares the Sign request.
func (client BaseClient) SignPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeySignParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/sign", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SignSender sends the Sign request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SignSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// SignResponder handles the response to the Sign request. The method always
// closes the http.Response Body.
func (client BaseClient) SignResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UnwrapKey the UNWRAP operation supports decryption of a symmetric key using the target key encryption key. This
// operation is the reverse of the WRAP operation. The UNWRAP operation applies to asymmetric and symmetric keys stored
// in Azure Key Vault since it uses the private portion of the key. This operation requires the keys/unwrapKey
// permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
// keyVersion - the version of the key.
// parameters - the parameters for the key operation.
func (client BaseClient) UnwrapKey(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UnwrapKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "UnwrapKey", err.Error())
	}

	req, err := client.UnwrapKeyPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UnwrapKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.UnwrapKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UnwrapKey", resp, "Failure sending request")
		return
	}

	result, err = client.UnwrapKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UnwrapKey", resp, "Failure responding to request")
	}

	return
}

// UnwrapKeyPreparer prepares the UnwrapKey request.
func (client BaseClient) UnwrapKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/unwrapkey", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UnwrapKeySender sends the UnwrapKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UnwrapKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UnwrapKeyResponder handles the response to the UnwrapKey request. The method always
// closes the http.Response Body.
func (client BaseClient) UnwrapKeyResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificate the UpdateCertificate operation applies the specified update on the given certificate; the only
// elements updated are the certificate's attributes. This operation requires the certificates/update permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate in the given key vault.
// certificateVersion - the version of the certificate.
// parameters - the parameters for certificate update.
func (client BaseClient) UpdateCertificate(ctx context.Context, vaultBaseURL string, certificateName string, certificateVersion string, parameters CertificateUpdateParameters) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateCertificatePreparer(ctx, vaultBaseURL, certificateName, certificateVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificate", resp, "Failure responding to request")
	}

	return
}

// UpdateCertificatePreparer prepares the UpdateCertificate request.
func (client BaseClient) UpdateCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, certificateVersion string, parameters CertificateUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name":    autorest.Encode("path", certificateName),
		"certificate-version": autorest.Encode("path", certificateVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/{certificate-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCertificateSender sends the UpdateCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateCertificateSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UpdateCertificateResponder handles the response to the UpdateCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificateIssuer the UpdateCertificateIssuer operation performs an update on the specified certificate issuer
// entity. This operation requires the certificates/setissuers permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// issuerName - the name of the issuer.
// parameter - certificate issuer update parameter.
func (client BaseClient) UpdateCertificateIssuer(ctx context.Context, vaultBaseURL string, issuerName string, parameter CertificateIssuerUpdateParameters) (result IssuerBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateCertificateIssuer")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateCertificateIssuerPreparer(ctx, vaultBaseURL, issuerName, parameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateIssuer", resp, "Failure responding to request")
	}

	return
}

// UpdateCertificateIssuerPreparer prepares the UpdateCertificateIssuer request.
func (client BaseClient) UpdateCertificateIssuerPreparer(ctx context.Context, vaultBaseURL string, issuerName string, parameter CertificateIssuerUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithJSON(parameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCertificateIssuerSender sends the UpdateCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UpdateCertificateIssuerResponder handles the response to the UpdateCertificateIssuer request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificateOperation updates a certificate creation operation that is already in progress. This operation
// requires the certificates/update permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate.
// certificateOperation - the certificate operation response.
func (client BaseClient) UpdateCertificateOperation(ctx context.Context, vaultBaseURL string, certificateName string, certificateOperation CertificateOperationUpdateParameter) (result CertificateOperation, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateCertificateOperation")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateCertificateOperationPreparer(ctx, vaultBaseURL, certificateName, certificateOperation)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateOperation", resp, "Failure responding to request")
	}

	return
}

// UpdateCertificateOperationPreparer prepares the UpdateCertificateOperation request.
func (client BaseClient) UpdateCertificateOperationPreparer(ctx context.Context, vaultBaseURL string, certificateName string, certificateOperation CertificateOperationUpdateParameter) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithJSON(certificateOperation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCertificateOperationSender sends the UpdateCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateCertificateOperationSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UpdateCertificateOperationResponder handles the response to the UpdateCertificateOperation request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificatePolicy set specified members in the certificate policy. Leave others as null. This operation
// requires the certificates/update permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// certificateName - the name of the certificate in the given vault.
// certificatePolicy - the policy for the certificate.
func (client BaseClient) UpdateCertificatePolicy(ctx context.Context, vaultBaseURL string, certificateName string, certificatePolicy CertificatePolicy) (result CertificatePolicy, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateCertificatePolicy")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateCertificatePolicyPreparer(ctx, vaultBaseURL, certificateName, certificatePolicy)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificatePolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificatePolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificatePolicy", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificatePolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificatePolicy", resp, "Failure responding to request")
	}

	return
}

// UpdateCertificatePolicyPreparer prepares the UpdateCertificatePolicy request.
func (client BaseClient) UpdateCertificatePolicyPreparer(ctx context.Context, vaultBaseURL string, certificateName string, certificatePolicy CertificatePolicy) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	certificatePolicy.ID = nil
	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/policy", pathParameters),
		autorest.WithJSON(certificatePolicy),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCertificatePolicySender sends the UpdateCertificatePolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateCertificatePolicySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UpdateCertificatePolicyResponder handles the response to the UpdateCertificatePolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateCertificatePolicyResponder(resp *http.Response) (result CertificatePolicy, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateKey in order to perform this operation, the key must already exist in the Key Vault. Note: The cryptographic
// material of a key itself cannot be changed. This operation requires the keys/update permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of key to update.
// keyVersion - the version of the key to update.
// parameters - the parameters of the key to update.
func (client BaseClient) UpdateKey(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyUpdateParameters) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateKeyPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateKey", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateKey", resp, "Failure responding to request")
	}

	return
}

// UpdateKeyPreparer prepares the UpdateKey request.
func (client BaseClient) UpdateKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateKeySender sends the UpdateKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UpdateKeyResponder handles the response to the UpdateKey request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateSasDefinition updates the specified attributes associated with the given SAS definition. This operation
// requires the storage/setsas permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// sasDefinitionName - the name of the SAS definition.
// parameters - the parameters to update a SAS definition.
func (client BaseClient) UpdateSasDefinition(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string, parameters SasDefinitionUpdateParameters) (result SasDefinitionBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateSasDefinition")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}},
		{TargetValue: sasDefinitionName,
			Constraints: []validation.Constraint{{Target: "sasDefinitionName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "UpdateSasDefinition", err.Error())
	}

	req, err := client.UpdateSasDefinitionPreparer(ctx, vaultBaseURL, storageAccountName, sasDefinitionName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSasDefinition", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateSasDefinitionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSasDefinition", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateSasDefinitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSasDefinition", resp, "Failure responding to request")
	}

	return
}

// UpdateSasDefinitionPreparer prepares the UpdateSasDefinition request.
func (client BaseClient) UpdateSasDefinitionPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, sasDefinitionName string, parameters SasDefinitionUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"sas-definition-name":  autorest.Encode("path", sasDefinitionName),
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}/sas/{sas-definition-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateSasDefinitionSender sends the UpdateSasDefinition request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateSasDefinitionSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UpdateSasDefinitionResponder handles the response to the UpdateSasDefinition request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateSasDefinitionResponder(resp *http.Response) (result SasDefinitionBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateSecret the UPDATE operation changes specified attributes of an existing stored secret. Attributes that are not
// specified in the request are left unchanged. The value of a secret itself cannot be changed. This operation requires
// the secrets/set permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// secretName - the name of the secret.
// secretVersion - the version of the secret.
// parameters - the parameters for update secret operation.
func (client BaseClient) UpdateSecret(ctx context.Context, vaultBaseURL string, secretName string, secretVersion string, parameters SecretUpdateParameters) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateSecretPreparer(ctx, vaultBaseURL, secretName, secretVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSecret", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSecret", resp, "Failure responding to request")
	}

	return
}

// UpdateSecretPreparer prepares the UpdateSecret request.
func (client BaseClient) UpdateSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string, secretVersion string, parameters SecretUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name":    autorest.Encode("path", secretName),
		"secret-version": autorest.Encode("path", secretVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/{secret-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateSecretSender sends the UpdateSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateSecretSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UpdateSecretResponder handles the response to the UpdateSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateStorageAccount updates the specified attributes associated with the given storage account. This operation
// requires the storage/set/update permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// storageAccountName - the name of the storage account.
// parameters - the parameters to update a storage account.
func (client BaseClient) UpdateStorageAccount(ctx context.Context, vaultBaseURL string, storageAccountName string, parameters StorageAccountUpdateParameters) (result StorageBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateStorageAccount")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: storageAccountName,
			Constraints: []validation.Constraint{{Target: "storageAccountName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "UpdateStorageAccount", err.Error())
	}

	req, err := client.UpdateStorageAccountPreparer(ctx, vaultBaseURL, storageAccountName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateStorageAccount", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateStorageAccountSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateStorageAccount", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateStorageAccountResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateStorageAccount", resp, "Failure responding to request")
	}

	return
}

// UpdateStorageAccountPreparer prepares the UpdateStorageAccount request.
func (client BaseClient) UpdateStorageAccountPreparer(ctx context.Context, vaultBaseURL string, storageAccountName string, parameters StorageAccountUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"storage-account-name": autorest.Encode("path", storageAccountName),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/storage/{storage-account-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateStorageAccountSender sends the UpdateStorageAccount request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateStorageAccountSender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// UpdateStorageAccountResponder handles the response to the UpdateStorageAccount request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateStorageAccountResponder(resp *http.Response) (result StorageBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Verify the VERIFY operation is applicable to symmetric keys stored in Azure Key Vault. VERIFY is not strictly
// necessary for asymmetric keys stored in Azure Key Vault since signature verification can be performed using the
// public portion of the key but this operation is supported as a convenience for callers that only have a
// key-reference and not the public portion of the key. This operation requires the keys/verify permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
// keyVersion - the version of the key.
// parameters - the parameters for verify operations.
func (client BaseClient) Verify(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyVerifyParameters) (result KeyVerifyResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Verify")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Digest", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.Signature", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "Verify", err.Error())
	}

	req, err := client.VerifyPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Verify", nil, "Failure preparing request")
		return
	}

	resp, err := client.VerifySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Verify", resp, "Failure sending request")
		return
	}

	result, err = client.VerifyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Verify", resp, "Failure responding to request")
	}

	return
}

// VerifyPreparer prepares the Verify request.
func (client BaseClient) VerifyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyVerifyParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/verify", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// VerifySender sends the Verify request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) VerifySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// VerifyResponder handles the response to the Verify request. The method always
// closes the http.Response Body.
func (client BaseClient) VerifyResponder(resp *http.Response) (result KeyVerifyResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// WrapKey the WRAP operation supports encryption of a symmetric key using a key encryption key that has previously
// been stored in an Azure Key Vault. The WRAP operation is only strictly necessary for symmetric keys stored in Azure
// Key Vault since protection with an asymmetric key can be performed using the public portion of the key. This
// operation is supported for asymmetric keys as a convenience for callers that have a key-reference but do not have
// access to the public key material. This operation requires the keys/wrapKey permission.
// Parameters:
// vaultBaseURL - the vault name, for example https://myvault.vault.azure.net.
// keyName - the name of the key.
// keyVersion - the version of the key.
// parameters - the parameters for wrap operation.
func (client BaseClient) WrapKey(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.WrapKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "WrapKey", err.Error())
	}

	req, err := client.WrapKeyPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "WrapKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.WrapKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "WrapKey", resp, "Failure sending request")
		return
	}

	result, err = client.WrapKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "WrapKey", resp, "Failure responding to request")
	}

	return
}

// WrapKeyPreparer prepares the WrapKey request.
func (client BaseClient) WrapKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "7.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/wrapkey", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// WrapKeySender sends the WrapKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) WrapKeySender(req *http.Request) (*http.Response, error) {
	sd := autorest.GetSendDecorators(req.Context(), autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
	return autorest.SendWithSender(client, req, sd...)
}

// WrapKeyResponder handles the response to the WrapKey request. The method always
// closes the http.Response Body.
func (client BaseClient) WrapKeyResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}
