package upgrade

import (
	"context"
	"reflect"
	"strings"

	log "github.com/sirupsen/logrus"
	"github.com/spf13/viper"
	"go.opentelemetry.io/otel/global"
	"sigs.k8s.io/controller-runtime/pkg/client"

	v1 "github.com/jaegertracing/jaeger-operator/pkg/apis/jaegertracing/v1"
	"github.com/jaegertracing/jaeger-operator/pkg/tracing"
)

// ManagedInstances finds all the Jaeger instances for the current operator and upgrades them, if necessary
func ManagedInstances(ctx context.Context, c client.Client, reader client.Reader) error {
	tracer := global.TraceProvider().GetTracer(v1.ReconciliationTracer)
	ctx, span := tracer.Start(ctx, "ManagedInstances")
	defer span.End()

	log.Debug("Checking for Jaeger instances to be upgraded ...")

	list := &v1.JaegerList{}
	// Related to https://issues.redhat.com/browse/TRACING-1143
	// Matching label commented out as was not picking up 1.13.1 jaeger instances.
	// Jaeger operator is cluster-wide, so only one operator will be managing all instances currently.
	//identity := viper.GetString(v1.ConfigIdentity)
	opts := []client.ListOption{
		//client.MatchingLabels(map[string]string{
		//	v1.LabelOperatedBy: identity,
		//}),
	}

	if watchNamespaces := viper.GetString(v1.ConfigWatchNamespace); watchNamespaces != v1.WatchAllNamespaces {
		for _, namespace := range strings.Split(watchNamespaces, ",") {
			nsOpts := append(opts, client.InNamespace(namespace))
			nsList := &v1.JaegerList{}
			if err := reader.List(ctx, nsList, nsOpts...); err != nil {
				return tracing.HandleError(err, span)
			}
			list.Items = append(list.Items, nsList.Items...)
		}
	} else {
		if err := reader.List(ctx, list, opts...); err != nil {
			return tracing.HandleError(err, span)
		}
	}

	for _, j := range list.Items {
		// this check shouldn't have been necessary, as I'd expect the list of items to come filtered out already
		// but apparently, at least the fake client used in the unit tests doesn't filter it out... so, let's double-check
		// that we indeed own the item

		// Related to https://issues.redhat.com/browse/TRACING-1143
		// Matching label commented out as was not picking up 1.13.1 jaeger instances.
		// Jaeger operator is cluster-wide, so only one operator will be managing all instances currently.
		/*
		owner := j.Labels[v1.LabelOperatedBy]
		if owner != identity {
			log.WithFields(log.Fields{
				"our-identity":   identity,
				"owner-identity": owner,
			}).Debug("skipping CR upgrade as we are not owners")
			continue
		}
		*/

		jaeger, err := ManagedInstance(ctx, c, j)
		if err != nil {
			log.WithFields(log.Fields{
				"instance":  jaeger.Name,
				"namespace": jaeger.Namespace,
			}).WithError(err).Debug("Nothing to do at this level, just go to the next instance")

			// nothing to do at this level, just go to the next instance
			continue
		}

		if !reflect.DeepEqual(jaeger, j) {
			// the CR has changed, store it!
			log.WithFields(log.Fields{
				"instance":  jaeger.Name,
				"namespace": jaeger.Namespace,
				"newVersion": jaeger.Status.Version,
			}).Debug("CR has changed, so will store it")

			if err := c.Update(ctx, &jaeger); err != nil {
				log.WithFields(log.Fields{
					"instance":  jaeger.Name,
					"namespace": jaeger.Namespace,
				}).WithError(err).Error("failed to store the upgraded instance")
				tracing.HandleError(err, span)
			}
		} else {
			log.WithFields(log.Fields{
				"instance":  jaeger.Name,
				"namespace": jaeger.Namespace,
			}).Debug("No changes detected")
		}
	}

	log.Debug("Upgrade check completed")

	return nil
}

// ManagedInstance performs the necessary changes to bring the given Jaeger instance to the current version
func ManagedInstance(ctx context.Context, client client.Client, jaeger v1.Jaeger) (v1.Jaeger, error) {
	tracer := global.TraceProvider().GetTracer(v1.ReconciliationTracer)
	ctx, span := tracer.Start(ctx, "ManagedInstance")
	defer span.End()

	if jaeger.Status.Version == "" {
		// Set to first product version - 1.13.1 didn't set the Status.Version field
		jaeger.Status.Version = "1.13.1";

		log.WithFields(log.Fields{
			"instance":  jaeger.Name,
			"namespace": jaeger.Namespace,
			"version": jaeger.Status.Version,
		}).Debug("Instance version not set, so defaulting to 1.13.1")
	}

	log.WithFields(log.Fields{
		"instance":  jaeger.Name,
		"namespace": jaeger.Namespace,
		"version": jaeger.Status.Version,
	}).Debug("Upgrading instance")

	if v, ok := versions[jaeger.Status.Version]; ok {
		// we don't need to run the upgrade function for the version 'v', only the next ones
		for n := v.next; n != nil; n = n.next {
			// performs the upgrade to version 'n'
			log.WithFields(log.Fields{
				"instance":  jaeger.Name,
				"namespace": jaeger.Namespace,
				"current": jaeger.Status.Version,
				"next": n.v,
			}).Debug("Performing the upgrade to next version")

			upgraded, err := n.upgrade(ctx, client, jaeger)
			if err != nil {
				log.WithFields(log.Fields{
					"instance":  jaeger.Name,
					"namespace": jaeger.Namespace,
					"to":        n.v,
				}).WithError(err).Warn("failed to upgrade managed instance")
				return jaeger, tracing.HandleError(err, span)
			}
			log.WithFields(log.Fields{
				"instance":  jaeger.Name,
				"namespace": jaeger.Namespace,
				"to":        n.v,
			}).Info("Upgraded managed instance")

			upgraded.Status.Version = n.v
			jaeger = upgraded
		}
	}

	return jaeger, nil
}
